#!/usr/bin/perl -w

#############################################################
# Project gravy
# Pascal Haakmat june 1998, januari 2000
# see accompanying README.html for info
# $Log: gravy.pl,v $
# Revision 1.3  2001/09/03 20:35:45  groomed
# First jabs at icon support, using Markus' code.
#
# Revision 1.4  2001/07/23 01:02:06  p
# Added recursive @INCLUDE.
#
# Revision 1.3  2001/07/22 21:14:14  p
# Added @INCLUDE tag.
#
# Revision 1.2  2000/01/03 22:53:56  p
# added substitution capability for layout files.
#
#############################################################

$DEFINITION_FILE = "tree.def";
$LAYOUT_FILE = "default.layout";
$STRUCTURE_FILE = "default.struct";

#############################################################
# Initialization
#############################################################
# Structural terms
$GENERATOR_BEGIN_STRUCT = '\s*BEGIN GENERATOR\(Layout="(.*)"\s+Structure="(.*)"\)';
$GENERATOR_END_STRUCT = '\s*END GENERATOR';

$ROOT_BEGIN_STRUCT = '\s*BEGIN ROOT\((.*)\)';
$ROOT_END_STRUCT = '\s*END ROOT';

$TREE_BEGIN_STRUCT = '\s*BEGIN TREE\((.*)\)';
$TREE_END_STRUCT = '\s*END TREE';

$ELEMENT_STRUCT = '\s*ELEMENT\((.*)\)\s+';

# Attributes
$DESCRIPTION_STRUCT = '\s*Description="(.*?)"';
$FILE_STRUCT = '\s*File="(.*?)"';
$TITLE_STRUCT = '\s*Title="(.*?)"';
$TERMINAL_STRUCT = '\s*Terminal="(.*?)"';

# These are overridden in the $STRUCTURE_FILE and defined here only
# to shut up -w
$TITLE_BEGIN = "";
$TITLE_END = "";
$ROOT_BEGIN = '';
$ROOT_END = '';
$TREE_BEGIN = "";
$TREE_END = "";
$ELEMENT_BEGIN = "";
$ELEMENT_END = "";
$ELEMENT_BODY = '';
$SELECTED_ELEMENT_BEGIN = "";
$SELECTED_ELEMENT_END = "";
$SELECTED_ELEMENT_BODY = '';

$VERSION = '$Revision: 1.3 $';
$NOTICE = "This is gravy $VERSION\nPascal Haakmat ".'$Date: 2001/09/03 20:35:45 $'." <ahaakmat\@wins.uva.nl>\n";

#############################################################
# Program entry point
#
# A tree is used to represent the structure of the information. 
# Various functions return or accept references to trees.
#
# A tree is implemented as an array containing references to
# hashes. Every hash defines the "Children" key which contains
# a reference to another tree, which may be an empty list.
#############################################################
#chdir('vine:code:perl:gravy:');
#chdir('boot:Desktop Folder:gravy 1.0:');
print $NOTICE;
#$ARGV[0] = 'papa.def';
if(@ARGV > 0) { $DEFINITION_FILE = $ARGV[0]; }
else { print "usage: gravy tree-definition-file\n"; exit; }
print "using definition file $DEFINITION_FILE\n";

# parse the tree from the tree definition file
$treeRef = parseDefinitionFile($DEFINITION_FILE);
if($treeRef == 0) { print "couldn't parse definition file\n"; exit; }
dumpTree($treeRef,0);

# load the structure file
require $STRUCTURE_FILE;

# get all paths for the tree
my($pathsRef) = getAllPaths($treeRef);
dumpPaths($pathsRef);
my(@paths) = @{copyPaths($pathsRef)};
my(@paths2) = @{copyPaths($pathsRef)};

my($counter) = 0;
print "applying structure: $STRUCTURE_FILE, layout: $LAYOUT_FILE\n";
foreach(@paths) {
	my($thisPath) = \@{$paths2[$counter]};
	# apply a path to the tree resulting in a pruned tree
	$treeAppliedRef = applyPath($_,$treeRef);

	# generate a filename and contents for this pruned tree, and write
	my($file,$out) = generateFile(${@$pathsRef}[$counter],$treeAppliedRef,$thisPath);
	print "writing        ".fileWriter($file,$out)."\n";

	# write 'simples' (ie. nodes that have no children)
	my($children) = getChildrenOfPath(${@$pathsRef}[$counter],$treeRef);
	my($simples_counter) = 0;
	foreach(@$children) {
		my(@selection) = @$thisPath;
		my(%s) = %{$_};
		push @selection, $simples_counter;
		if(@{$s{"Children"}} == 0) {
			# generate the contents for a simple, and write
			my($out) = generateSimpleFile(${@$pathsRef}[$counter],$treeAppliedRef,\%s, \@selection);
			my($file) = $s{"File"};
			print "writing simple ".fileWriter($file,$out)."\n";
		}
		$simples_counter++;
	}
	$counter++;
}


#############################################################
# Given a path and the tree that results from it, 
# generateFile will evaluate the file name and contents.
# usage: $file,$content = generateFile($pathRef,$treeRef,$selectionPath)
# 	$pathRef is an array-reference for a path
#	$treeRef is an array-ref for a tree
#   $selectionPath is an array-ref to a path pointing at the 
#   selected element
# returns a filename and contents
#############################################################
sub generateFile {
	my($pathRef, $treeRef, $selectionPath) = @_;
	my(%s) = %{${@$treeRef}[0]};
	my(%s2) = %{getRecordOfPath($pathRef,$treeRef)};
	my($structure) = generateStructure($treeRef,$s{"File"},"IsRoot", $selectionPath);
	my($output) = generateLayout($structure,$s2{"Terminal"},$s2{"Title"},\%s);
	return ($s2{"File"},$output);
}

#############################################################
# generateSimpleFile evaluates the content for a simple
# usage: $content = generateSimpleFile($pathRef,$treeRef,$simpleRef,$selectionPath)
# 	$pathRef is an array-reference for a path
#	$treeRef is an array-ref for a tree
#   $simpleRef is a hash-ref for the simple node
#   $selectionPath is an array-ref to a path pointing at the 
#   selected element
# returns contents
#############################################################
sub generateSimpleFile {
	my($pathRef, $treeRef, $simpleRef, $selectionPath) = @_;
	my(%s) = %{$simpleRef};
	my(%s2) = %{${@$treeRef}[0]};
	my($structure) = generateStructure($treeRef,$s2{"File"},"IsRoot", $selectionPath);
	my($output) = generateLayout($structure,$s{"Terminal"},$s{"Title"},\%s);
	return ($output);
}

#############################################################
# generateLayout
#############################################################
sub generateLayout {
	my($structure,$terminalFile,$title,$propsRef) = @_;
	my($layout) = fileLoader($LAYOUT_FILE);
	my($terminal) = fileLoader($terminalFile);
	my($output) = '';
	my(%properties) = %{$propsRef};

	$layout =~ s/<\@TITLE>/$TITLE_BEGIN$title$TITLE_END/g;
	$layout =~ s/<\@TERMINAL>/$terminal/g;
	$layout =~ s/<\@STRUCTURE>/$structure/g;

	foreach(keys(%properties)) {
	    $find = uc($_);
	    $replace = $properties{$_};
	    $layout =~ s/<\@$find>/$replace/g;
	}

	$layout = applyIncludes($layout);

	return $layout;
}

#############################################################
# applyIncludes
#############################################################
sub applyIncludes {
    my($layout) = @_;
    my(%includes);

    # Apply @INCLUDE directives.
    while($layout =~ /<\@INCLUDE (.*?)>/g) {
	
	my($f) = $1;
	$i = applyIncludes(fileLoader($f));
	$includes{$f} = $i;
	
    }
    
    $layout =~ s/<\@INCLUDE (.*?)>/$includes{$1}/g;
    
    return $layout;
    
}

#############################################################
# getRecordOfPath
#############################################################
sub getRecordOfPath {
	my($pathRef,$treeRef) = @_;
	my(%s);
	foreach(@$pathRef) {
		my($index) = $_;
		my(@a) = @$treeRef;
		%s = %{$a[$index]};
		$treeRef = $s{"Children"};
	}
	return \%s;
}

#############################################################
# getChildrenOfPath
#############################################################
sub getChildrenOfPath {
	my($pathRef,$treeRef) = @_;
	my(%s) = %{getRecordOfPath($pathRef,$treeRef)};
	return $s{"Children"};
}

#############################################################
# dumpPaths
#############################################################
sub dumpPaths {
	my($pathsRef) = @_;
	foreach(@$pathsRef) {
		croak("path: ");
		foreach(@$_) {
			croak("$_ ");
		}
		croak("\n");
	}
	croak("\n");
}
	
#############################################################
# copyPaths
#############################################################
sub copyPaths {
	my($pathsRef) = @_;
	my(@paths) = ();
	foreach(@$pathsRef) {
		my(@p) = ();
		foreach(@$_) {
			push @p,$_;
		}
		push @paths,\@p;
	}
	return \@paths;
}

#############################################################
# generateStructure produces the tree layouts for the different
# pages.
# usage: $structure = generateStructure($treeRef, $topLevelFile,
#						$root, $selectionPath)
# 	$treeRef is a tree reference
#	$topLevelFile specifies the file that contains the parent level
#	$root should be string "IsRoot" when called
#	$selectionPath a path specifying the selected element
# returns the structure code in a scalar
#############################################################
sub generateStructure {
	my($treeRef,$topLevelFile,$root,$selectionPath) = @_;
	my(@tree) = @$treeRef;
	my(@a);
	my(@selection) = @{$selectionPath};

	# if the treeRef is the root, the tree is it's children
	if($root eq "IsRoot") {
	@a = @{${%{$tree[0]}}{"Children"}};
	shift @selection;
	# otherwise the tree is just the tree
	} else {
	@a = @tree;
	}

	my($selected) = shift @selection;

	my($structure) = '';
	my($counter) = 0;
	if(@a > 0) {
		$structure .= ($root eq "IsRoot" ? $ROOT_BEGIN : $TREE_BEGIN);
		foreach(@a) {
			my(%s) = %{$_};

			my($e) = ($#selection == -1 && defined($selected) && $selected eq $counter ? 
						$SELECTED_ELEMENT_BEGIN.$SELECTED_ELEMENT_BODY.$SELECTED_ELEMENT_END :
						$ELEMENT_BEGIN.$ELEMENT_BODY.$ELEMENT_END);
			$e =~ s/<\@DESCRIPTION>/$s{"Description"}/g;

			if(@{$s{"Children"}} > 0) {
				$e =~ s/<\@FILE>/$topLevelFile/g;
			}
			else {
			
				$e =~ s/<\@FILE>/$s{"File"}/g;
			}

			$structure .= $e;
			if(@{$s{"Children"}} > 0) {
				$topLevelFile = $s{"File"};
				$structure .= generateStructure($s{"Children"},$topLevelFile,"NotRoot", \@selection);
			}
			$counter++;
		}
		$structure .= ($root eq "IsRoot" ? $ROOT_END : $TREE_END);
	}
	
	return $structure;
}


#############################################################
# Applies a path to a tree generating a new tree.
# usage: $appliedTreeRef = applyPath($pathRef,$treeRef)
# 	$pathRef is an array-reference for a path
#	$treeRef is an array-reference for a tree
# returns the tree resulting from the application of a path
# onto the source tree. 
# remarks: applyPath destroys all data in the array referenced
# by pathRef
#############################################################
sub applyPath {
	my($pathRef, $treeRef) = @_;
	my(@a) = @$treeRef;
	my(@result) = ();
	local(@emptyList) = ();
	
	croak("\napplyPath @$pathRef\n");
	
	my($index) = shift @$pathRef;
	my(%s) = %{$a[$index]};
	my(@children) = ( );

	croak("node ".$s{"Description"}." index = $index\n");
	
	foreach(@{$s{'Children'}}) {
		my($rec) = copyRecord($_);
		croak(${%{$rec}}{"Description"}." is a child\n");
		${%{$rec}}{"Children"} = \@emptyList;
		push @children, $rec;
	}

	my($rec) = &copyRecord(\%s);

	# reduce the path if there are more nodes
	if(@$pathRef > 0) {
		my($idx2) = ${@$pathRef}[0];

		my(%sd) = %{$children[$idx2]};
		$sd{"Children"} = applyPath($pathRef, $s{"Children"});	# reduce

		$children[$idx2] = @{$sd{"Children"}}[0];

		croak("putting $children[$idx2] in $idx2 th child of ".$s{'Description'}."\n");		
	}

	${%{$rec}}{"Children"} = \@children;
	push @result, $rec;
	
	croak("contents of returned record: ".${%{$rec}}{"Description"}."\n");
	croak("returning @result\n\n");
	return \@result;
}

#############################################################
# copyRecord
#############################################################
sub copyRecord {
	my($s) = @_;
	my(%src) = %{$s};
	return  { 
			 'Description'	=> $src{'Description'},
			 'File'			=> $src{'File'},
			 'Terminal'		=> $src{'Terminal'},
			 'Title'		=> $src{'Title'},
			 'Children'		=> $src{'Children'},
			};
}

#############################################################
# Returns the list of all paths that can be constructed
# from a given tree.
# usage: @paths = getAllPaths($treeRef)
# returns a list of references to the path lists
#############################################################
sub getAllPaths {
	my($treeRef) = @_;
	my(@list) = ( 0 );
	my($listRef) = \@list;
	my(@paths) = ( $listRef );

	my($c) = 0;	
	do {
		$listRef = getNextPath($treeRef,$listRef);
		if(defined(@$listRef) && @$listRef != 0) {
			$paths[$#paths+1] = $listRef;
		}
	} while(@$listRef != 0);
	
	return \@paths;
}

#############################################################
# Returns every path that can be constructed from the 
# structured array representing a tree.
# usage: $pathRef = getNextPath($aRef,$rootRef)
#   $aRef is a reference to a tree
# 	$rootRef is a reference to a path array
# returns a reference to the path array that succeeds the 
# root array. Use this value on subsequent calls to 
# getNextPath to get all paths from the tree. Returns the 
# empty list if no more paths exist.
#############################################################
sub getNextPath {
	my($aRef,$listRef) = @_;
	my(@list) = @$listRef;
	my(@a) = @$aRef;
	
	&croak("\ngetNextPath @list\n");
	&croak("# listRef = ".@$listRef."\n");
	&croak("# aRef = ".@$aRef." \n");
	
	# if the path is more than one node, reduce it
	if(@$listRef > 1) {		
		my($i) = ${@$listRef}[0];
		my(%s) = %{$a[$i]};
		my($childRef) = $s{"Children"};
		my(@newlist) = @list;
		shift @newlist;
		&croak("Descending ".$s{"Description"}." with list @newlist\n");
		my($r) = getNextPath($childRef,\@newlist);	# reduce 
		my(@returnedList);
		if(defined($r)) {
		@returnedList = @{$r} if(defined($r));
		}
		else {
		@returnedList = ();
		}
		pop @list;
		&croak("Descent complete resulted in @returnedList\n");
		# if the reduction was unsuccesful, backtrack
		if(@returnedList == 0) {		
			my($idx) = shift @newlist;
			&croak("$idx = $idx\n");
			if($idx < scalar(@{$childRef})-1) {
				@newlist = (++$idx);
				&croak("this is @list\n");
				&croak("Descent resulted in nothing, finding another path from $idx\n");
				my(@children) = @{$s{"Children"}};
				my($i) = $idx;
				for($i=$idx; $i<@children; $i++) {
					my(%sc) = %{$children[$i]};
					my(@cchildren) = @{$sc{"Children"}};
					if(@cchildren != 0) {
						&croak($sc{"Description"}," has children\n");
						@list = ( $list[0], $i );
						&croak("returning (0,idx) @list\n");
						return \@list;
						
					}
					&croak($sc{"Description"}," \n");
					if($i == @children-1) {
						return ();
					}
				}
			}
			else {
				&croak("returning empty list\n");
				return ();
			}
		}
		if(@list > 1) {
		pop @list;
		}
		&croak("list = @list\n");
		push @list,@returnedList;
		&croak("returning @list\n");
		return \@list;

	}
	# the path is reduced to a single node
	else {					
		
		my($index) = ${@$listRef}[0];
		&croak("index = $index\n");
		&croak("# a = ".@$aRef."\n");

		# if the node cannot exist we return an empty list
		if($index >= @$aRef) { 
			&croak("Index >= # aRef; returning empty list\n");
			return (); 
		}
		
		my(%s) = %{$a[$index]};
		my(@l) = @{$s{"Children"}};
		
		&croak("children of ".$s{"Description"}.": @l\n");

		my($valid) = 0;
		foreach(@l) {
			my(%sc) = %{$_};
			my(@lc) = @{$sc{"Children"}};
			# if one of the children of this node has children,
			# we return the path for that child
			if(scalar(@lc) != 0) {
				&croak($sc{"Description"}." has children: @lc\n");

				push @list,$valid;
				&croak("returning  @list\n");
				return \@list;
			}
			else {
			
				&croak($sc{"Description"}." doesn't have children\n");
			}
			
			$valid++;
		}
		
		&croak("no valid nodes; returning empty list\n");
		return ();
		
	}


}

sub croak {
#print; 
}

#############################################################
# Parse a definition file;
# usage: parseDefinitionFile($filename)
# returns a reference to an array of structured hash 
# references
#############################################################
sub parseDefinitionFile {
	my($f) = @_;
	my(@stack);
	my(@root);
	my($childRef);
	
	open(FILE, $f) or return 0;

	while(<FILE>) {
	
		/$GENERATOR_BEGIN_STRUCT/ && do {
			$LAYOUT_FILE = $1;
			$STRUCTURE_FILE = $2;
			$childRef = \@root;
		};
		
		(/$ROOT_BEGIN_STRUCT/ || /$TREE_BEGIN_STRUCT/) && do {
			my(@attribs) = parseAttributes($1);
			local(@new_children) = ();
			
			my($treeRef) = {
				'Description' => $attribs[0],
				'Title' => $attribs[1],
				'File' => $attribs[2],
				'Terminal' => $attribs[3],
				'Children' => \@new_children
			};
			push @$childRef,$treeRef;
			push @stack,$childRef;
			$childRef = \@new_children;
		};
		
		/$GENERATOR_END_STRUCT/ && do {
		};
				
		/$ROOT_END_STRUCT/ && do {
		};
		
		/$TREE_END_STRUCT/ && do {
			$childRef = pop @stack;
		};
		
		/$ELEMENT_STRUCT/ && do {
		};
	}

	close(FILE);

	return \@root;
}

#############################################################
# Parses the attributes that come with a term.
# usage: parseAttributes($string)
# returns $description,$title,$file,$terminal
#############################################################
sub parseAttributes {
	my($string) = @_;
	$string =~ /$DESCRIPTION_STRUCT/;
	$desc = $1;
	$string =~ /$TERMINAL_STRUCT/;
	$terminal = $1;
	$string =~ /$TITLE_STRUCT/;
	$title = $1;
	$string =~ /$FILE_STRUCT/;
	$file = $1;

	return $desc,$title,$file,$terminal;
}

#############################################################
# Dumps a tree.
# trees are arrays of hash references, in which every
# hash reference defines a key 'Children' that points
# to the array of hash references containing the children
# of the node.
# usage: dumpTree($treeRef,0) ('0' is for indentation purposes)
#############################################################
sub dumpTree {
	my($root, $d) = @_;
	
	foreach(@$root) {
		my(%s) = %$_;
		
		croak(" " x ($d * 4));
		croak($s{"Description"}."\n");
		if(defined($s{"Children"})) {
			my($a) = $s{"Children"};
			dumpTree($a,$d+1);
		}
	}
	croak("\n");
}

#############################################################
# fileLoader
#############################################################
sub fileLoader {
	my($f) = @_;
	my($in);
	$oldSep = $/;
	$/ = undef;
	open(FILE, $f) or return "could not read $f";
	$in = <FILE>;
	close(FILE);
	$/ = $oldSep;
	return $in;
}

#############################################################
# fileWriter
#############################################################
sub fileWriter {
	my($file,$data) = @_;
	open(WRITE,">$file") or return "could not write $file";
	print WRITE $data;
	close(WRITE);
	return $file;
}
