
#include "oops.h"
#include <ctype.h>

#define		INCL_NOPMAPI
#define		INCL_DOSSEMAPHORES
#define		INCL_DOSMODULEMGR
#define		INCL_DOSPROCESS
#define		INCL_DOSERRORS
#include	<os2.h>

#define PortTraceCall(file, line) PortTrace("called from: %s(%d)", file, line)

int yywrap(void)
{
	return 1;
}

static void PortTrace(const char *format, ...)
{
	FILE    *f;
	va_list  a;

	f = fopen("_trace.log", "at");
	if (f)
	{
		va_start(a, format);
		fprintf(f, "[%p] ", pthread_self());
		vfprintf(f, format, a);
		fprintf(f, "\n");
		fclose(f);
		va_end(a);
	}
}

/*
 * Module loading
 */

#ifdef __WATCOMC__
extern int __set_errno_dos(APIRET apiret);
#else
#error Please write code for __set_errno_dos
/*
 * __set_errno_dos should decode OS/2 error code and set correct C errno,
 * then return -1. If you do not care about real value of errno, use this:
 */

/* #define __set_errno_dos(n)  (errno = EINVAL, -1) */
#endif

void * dlopen(const char *name, int flags)
{
	APIRET	i;
	HMODULE handle;
	char    errbuf[64];

	if (flags != RTLD_NOW)
		PortTrace("dlopen: flag is not 2 (RTLD_NOW): %d", flags);

	i = DosLoadModule(errbuf, sizeof(errbuf), (PSZ) name, &handle);
	if (i)
	{
		PortTrace("dlopen: DosLoadModule('%s') = %d, problem: '%s'", name, i, errbuf);
		__set_errno_dos(i);
		return NULL;
	}

	return (void *) handle;
}

void * dlsym(void *module, const char *name)
{
	APIRET i;
	PFN    proc;

	i = DosQueryProcAddr((HMODULE) module, 0, (PSZ)name, &proc);
	if (i)
	{
		char buf[200];

		if (i != 127)
		{
			if (DosQueryModuleName((HMODULE) module, sizeof(buf), buf))
				strcpy(buf, "*err*");

			PortTrace("dlsym: DosQueryProcAddr('%s:%s') = %d", buf, name, i);
		}
		__set_errno_dos(i);
		return NULL;
	}
	return (void *) proc;
}

int	dlclose(void *p)
{
	APIRET i;

	i = DosFreeModule((HMODULE) p);
	if (i)
	{
		PortTrace("dlclose: DosFreeModule() = %d", i);
		i = __set_errno_dos(i);
	}
	return i;
}

char *dlerror(void)
{
	return strerror(errno);
}

/*
 * Signal
 */

int sigaddset(sigset_t *set, int sig)
{
	if (sig < _SIGMIN || sig > _SIGMAX)
	{
		/* errno = EINVAL; */
		return -1;
	}

	*set |= 1 << sig;
	return 0;
}

int sigemptyset(sigset_t *set)
{
	*set = (sigset_t) 0;
	return 0;
}

/*
 * This function is useless because only thread 1 receives signals
 * which are processed by usual "signal" function
 */

int pthread_sigmask(int how, sigset_t const * set, sigset_t * oset)
{
	(void) how;
	(void) set;
	(void) oset;
	return 0;
}

/*
 * Glob
 */

static char *extract_filename(const char *s)
{
	const char *name;

	for (name = s; *s; s++)
		if (*s == ':' || *s == '/' || *s == '\\')
			name = s + 1;
	return name;
}

struct glob_list
{
	struct glob_list *next;
	char   file[1];
};

int glob(const char *mask, int n, int (*f)(const char *, int), glob_t *g)
{
	HDIR          hdir;
	FILEFINDBUF3  buf;
	ULONG         count;
	APIRET        i;
	struct glob_list *entry, *prevEntry;
	int           namepos;

	// PortTrace("glob: scanning '%s'", mask);
	if (n || f)
	{
		PortTrace("glob: unsupported arguments n=%d f=%p", n, f);
		errno = EINVAL;
		return -1;
	}

	bzero(g, sizeof(*g));

	namepos  = extract_filename(mask) - mask;
	hdir     = HDIR_CREATE;
	count    = 1;
	for (i = DosFindFirst((PSZ)mask, &hdir, FILE_ARCHIVED | FILE_SYSTEM | FILE_READONLY,
						  &buf, sizeof(buf), &count, FIL_STANDARD);
		 i == 0;
		 count = 1, i = DosFindNext(hdir, &buf, sizeof(buf), &count)
		)
	{
		if (count == 0)
			break;
	//	PortTrace("glob: found '%s'", buf.achName);

		entry = malloc(sizeof(*entry) + namepos + strlen(buf.achName));
		if (entry == NULL)
		{
			i = ERROR_NOT_ENOUGH_MEMORY;
			break;
		}
		entry->next = NULL;
		if (namepos)
			memcpy(entry->file, mask, namepos);
		strcpy(entry->file + namepos, buf.achName);

		if (g->entries == NULL)
			g->entries = entry;
		else
			prevEntry->next = entry;
		prevEntry = entry;
		g->gl_pathc++;
	}

	if (i == ERROR_NO_MORE_FILES)
		i = 0;

	if (hdir != HDIR_CREATE)
	{
		i = DosFindClose(hdir);
		if (i)
			PortTrace("glob: DosFindClose() = %d", i);
	}

	if (g->gl_pathc && i == 0)
	{
		g->gl_pathv = malloc(g->gl_pathc * sizeof(char *));
		if (g->gl_pathv == 0)
			i = ERROR_NOT_ENOUGH_MEMORY;
		else
		{
			int n;

			for (n = 0, entry = g->entries; entry; entry = entry->next, n++)
				g->gl_pathv[n] = entry->file;
		}
	}

	if (i != 0 && i != ERROR_NO_MORE_FILES)
	{
		PortTrace("glob: DosFindXXX('%s') = %d", mask, i);
		globfree(g);
		return __set_errno_dos(i);
	}

	return 0;
}

void globfree(glob_t *g)
{
	struct glob_list *cur, *next;

	for (cur = g->entries; cur; cur = next)
	{
		next = cur->next;
		free(cur);
	}
	if (g->gl_pathv)
		free(g->gl_pathv);
}


/*
 * Fake pthreads.
 * Note: these function will not set errno, they're just returning it.
 */

pthread_t pthread_self(void)
{
	return (pthread_t) *_threadid;  /* Hack: return TID instead of pointer */
}

int pthread_create(pthread_t * tid, const pthread_attr_t * attr,
				   void *(*start) (void *), void *arg)
{
	int i;

	if ((int)(*attr) != PTHREAD_CREATE_DETACHED)
		PortTrace("pthread_create: unsupported attr %d", (int)(*attr));

	i = _beginthread(*start, NULL, 32*1024, arg);
	if (i == -1)
		return errno;

	*tid = (pthread_t) i;
	return 0;
}

/*
 * Condition Variable Functions
 */

struct pthread_cond_item
{
	struct pthread_cond_item *prev, *next;
	HEV    eventsem;
};

struct pthread_cond_t_
{
	pthread_mutex_t  globalmutex;
	struct pthread_cond_item  *listfirst, *listlast;

};

int pthread_cond_init(pthread_cond_t * cond, const pthread_condattr_t * attr)
{
	int				ret;
	pthread_cond_t  thecond;

	if (attr)
		PortTrace("pthread_cond_init: non-NULL 'attr' unsupported");

	if ((thecond = calloc(sizeof(*thecond), 1)) == NULL)
		return ENOMEM;

	if ((ret = pthread_mutex_init(&thecond->globalmutex, NULL)) != 0)
	{
		free(thecond);
		return ret;
	}

	*cond = thecond;
	return 0;
}

int pthread_cond_destroy(pthread_cond_t * cond)
{
	pthread_cond_t thecond = *cond;
	int ret;

	if (thecond->listfirst)
	{
		PortTrace("pthread_cond_destroy: deleting cond with active waiters");
		ret = EINVAL;
	}
	else
	{
		ret = pthread_mutex_destroy(&thecond->globalmutex);
		free(thecond);
	}
	*cond = 0;
	return ret;
}

static int pthread_cond_timedwait_ms(pthread_cond_t * cond, pthread_mutex_t * mutex,
									 unsigned long milliseconds)
{
	int				ret, ret2;
	ULONG			postCount;
	pthread_cond_t	thecond = *cond;
	struct pthread_cond_item * item;

	/* Prepare structure for list entry */

	if ((item = calloc(sizeof(*item), 1)) == NULL)
		return ENOMEM;

	if ((ret = DosCreateEventSem(NULL, &item->eventsem, 0, FALSE)) != 0)
	{
		PortTrace("pthread_cond_timedwait_ms: DosCreateEventSem() = %d", ret);
		free(item);
		return EINVAL;
	}

	/* Link new entry into the list of waiters. Lock list before access. */

	if ((ret = pthread_mutex_lock(&thecond->globalmutex)) != 0)
		return ret;

	if (thecond->listfirst == NULL)
		thecond->listfirst = item;
	if (thecond->listlast)
		thecond->listlast->next = item;
	item->prev = thecond->listlast;
	thecond->listlast = item;

	pthread_mutex_unlock(&thecond->globalmutex);

	/* List entry is ok, just run the worker (release mutex) and wait for event */

	pthread_mutex_unlock(mutex);
	ret = DosWaitEventSem(item->eventsem, milliseconds);  /* [a] */
	/* Possible problem: at this point worker is still active (mutex clear)
	 * and can errorneously continue to run in loop.
	 * Solution: use global list of semaphores?
	 */
	pthread_mutex_lock(mutex);

	if (ret == ERROR_TIMEOUT)
	{
		PortTrace("pthread_cond_timedwait_ms: timed out (%d ms)", milliseconds);
		ret = ETIMEDOUT;
	} else if (ret != 0)
	{
		PortTrace("pthread_cond_timedwait_ms: DosWaitEventSem() = %d", ret);
		ret = EINVAL;
	}

	/* Remove our entry from list if not done yet */

	pthread_mutex_lock(&thecond->globalmutex); /* [b] */
	/* We must check that semaphore still hasn't been posted between [a] & [b] */
	if ((ret2 = DosQueryEventSem(item->eventsem, &postCount)) != 0)
		PortTrace("pthread_cond_timedwait_ms: DosQueryEventSem() = %d", ret2);
	if (ret2 != 0 || postCount == 0)  /* Not posted, remove from list */
	{
		if (thecond->listfirst == item)
			thecond->listfirst = item->next;
		if (thecond->listlast == item)
			thecond->listlast = item->prev;
		if (item->prev)
			item->prev->next = item->next;
		if (item->next)
			item->next->prev = item->prev;
	}
	pthread_mutex_unlock(&thecond->globalmutex);

	if ((ret2 = DosCloseEventSem(item->eventsem)) != 0)
		PortTrace("pthread_cond_timedwait_ms: DosCloseMutexSem() = %d", ret2);
	free(item);

	return ret;
}

#ifndef PTHREAD_DEBUG
int pthread_cond_wait(pthread_cond_t * cond, pthread_mutex_t * mutex)
#else
int pthread_cond_wait_debug(pthread_cond_t * cond, pthread_mutex_t * mutex,
			const char *file, int line)
#endif
{
	int ret;

	ret = pthread_cond_timedwait_ms(cond, mutex, SEM_INDEFINITE_WAIT);
#ifdef PTHREAD_DEBUG
	if (ret)
		PortTraceCall(file, line);
#endif
	return ret;
}

#ifndef PTHREAD_DEBUG
int pthread_cond_timedwait(pthread_cond_t * cond, pthread_mutex_t * mutex,
			const struct timespec *abstime)
#else
int pthread_cond_timedwait_debug(pthread_cond_t * cond, pthread_mutex_t * mutex,
			const struct timespec *abstime,
			const char *file, int line)
#endif
{
	struct timeval tv;
	unsigned abs_usec;
	unsigned long timeout_ms;
	int ret;

	abs_usec = abstime->tv_nsec / 1000;

	gettimeofday(&tv, NULL);
	if (tv.tv_sec > abstime->tv_sec ||
		(tv.tv_sec == abstime->tv_sec && tv.tv_usec > abs_usec)
	   )
		timeout_ms = 0;   /* too late */
	else
	{
		timeout_ms = (abstime->tv_sec - tv.tv_sec) * 1000;
		if (abs_usec > tv.tv_usec)  /* 0.300 to 1.900 = 1600 */
			timeout_ms += (abs_usec - tv.tv_usec) / 1000;
		else  /* 0.900 to 1.300 = 400 */
			timeout_ms -= (tv.tv_usec - abs_usec) / 1000;
	}

	ret = pthread_cond_timedwait_ms(cond, mutex, timeout_ms);
#ifdef PTHREAD_DEBUG
	if (ret)
		PortTraceCall(file, line);
#endif
	return ret;
}

static int pthread_cond_awake(pthread_cond_t * cond, int all)
{
	int		ret, ret2;
	struct pthread_cond_item	*item;
	pthread_cond_t				thecond = *cond;

	if ((ret = pthread_mutex_lock(&thecond->globalmutex)) != 0)
		return ret;

	item = thecond->listfirst;
	if (item == NULL)
		/* PortTrace("pthread_cond_signal: no waiters to awake!") */ ;
	else
		for (;;)
		{
			if ((ret2 = DosPostEventSem(item->eventsem)) != 0)
			{
				PortTrace("pthread_cond_awake: DosPostEventSem() = %d", ret2);
				ret = EINVAL;
			}
			/* we're holding global lock and item still can be accessed */
			if (thecond->listlast == item)
				thecond->listlast = NULL;
			item = item->next;
			thecond->listfirst = item;
			if (item == NULL)
				break;
			item->prev = NULL;
			if (!all)
				break;
		}

	pthread_mutex_unlock(&thecond->globalmutex);
	return ret;
}

int pthread_cond_signal(pthread_cond_t * cond)
{
	return pthread_cond_awake(cond, 0);
}

int pthread_cond_broadcast(pthread_cond_t * cond)
{
	return pthread_cond_awake(cond, 1);
}


/*
 * Thread attributes (hack: keep attribute inside pointer)
 */

int pthread_attr_init(pthread_attr_t * attr)
{
	*(int *) attr = -1;
	return 0;
}

int pthread_attr_setdetachstate(pthread_attr_t * attr, int detachstate)
{
	*(int *) attr = detachstate;
	return 0;
}

int pthread_attr_destroy(pthread_attr_t * attr)
{
	*(int *) attr = -1;
	return 0;
}


int pthread_mutex_init(pthread_mutex_t * mutex, const pthread_mutexattr_t * attr)
{
	int i;

	if (attr)
		PortTrace("pthread_mutex_init: non-NULL 'attr' unsupported");

	*(PHMTX) mutex = 0;
	i = DosCreateMutexSem(NULL, (PHMTX) mutex, 0, FALSE);
	if (i)
	{
		PortTrace("pthread_mutex_init: DosCreateMutexSem() = %d", i);
		return EINVAL;
	}
	return 0;
}

int pthread_mutex_lock(pthread_mutex_t * mutex)
{
	int i;

	i = DosRequestMutexSem(*(PHMTX) mutex, SEM_INDEFINITE_WAIT);
	if (i)
	{
		PortTrace("pthread_mutex_lock: DosRequestMutexSem() = %d", i);
		return EINVAL;
	}
	return 0;
}

int pthread_mutex_unlock(pthread_mutex_t * mutex)
{
	int i;

	i = DosReleaseMutexSem(*(PHMTX) mutex);
	if (i)
	{
		PortTrace("pthread_mutex_unlock: DosReleaseMutexSem() = %d", i);
		return EINVAL;
	}
	return 0;
}

#ifdef PTHREAD_DEBUG
int pthread_mutex_destroy_debug(pthread_mutex_t * mutex, const char *file, int line)
#else
int pthread_mutex_destroy_debug(pthread_mutex_t * mutex)
#endif
{
	int i;

	i = DosCloseMutexSem(*(PHMTX) mutex);
	if (i)
	{
		PortTrace("pthread_mutex_destroy: DosCloseMutexSem() = %d", i);
#ifdef PTHREAD_DEBUG
		PortTraceCall(file, line);
#endif
		return EINVAL;
	}
	return 0;
}

/*
 * New functions
 */

int strerror_r(int error, char *errbuf, size_t lerrbuf)
{
	strncpy(errbuf, strerror(error), lerrbuf);
	errbuf[lerrbuf-1] = 0;
	return 0;
}

int fcntl(int socket, int cmd, ...)
{
	int      arg, i;
	va_list  args;

	if (cmd == F_GETFL)
		return 0;

	if (cmd == F_SETFL)
	{
		va_start(args, cmd);
		arg = va_arg(args, int);
		va_end(args);

		if (arg != O_NONBLOCK)
		{
			PortTrace("fcntl: bad argument %d", arg);
			errno = EINVAL;
			return -1;
		}

		cmd = 1;
		i = ioctl(socket, FIONBIO, (char *) &cmd, sizeof(cmd));
		if (i == -1)
			PortTrace("fcntl: ioctl(FIONBIO) failed for socket %d", socket);
		return i;
	}

	PortTrace("fcntl: bad command %d", cmd);
	errno = EINVAL;
	return -1;
}

char * os2_oops_libdir(void)
{
	static char buf[_MAX_PATH];
	char *s;
	PTIB  tib;
	PPIB  pib;

	if (*buf == 0)
	{
		DosGetInfoBlocks(&tib, &pib);
		for (s = pib->pib_pchenv; *s; s += strlen(s) + 1)
			;
		strcpy(buf, s + 1);
		strcpy(extract_filename(buf), "modules");
	//	PortTrace("os2_oops_libdir: set to '%s'", buf);
	}

	return buf;
}

/* disabled, now handled in runtime DLL */
#if	0 && defined(__OS2__) && defined(__WATCOMC__)
/*
 * Convert VAC (?) timezone to Watcom format
 */

#define bool  char
#define true  1
#define false 0

static bool isnumber(char *str)
{
	if (str == NULL || *str == 0)
		return false;
	if (*str == '-')
		str++;

	for (; *str; str++)
		if (!isdigit(*str))
			return false;

	return true;
}

static char *strip(char *str)
{
	bool body = false;
	char *lastspace = NULL;
	char *firstspace = NULL;
	char *save = str;

	for (; *str; str++)
	{
		if (*str == ' ' || *str == '\t' || *str == '\n')
		{
			if (!body)
				firstspace = str;
			else
				if (lastspace == NULL)
					lastspace = str;
		}
		else
		{
			body      = true;
			lastspace = NULL;
		}
	}

	if (lastspace)
		*lastspace = 0;
	if (firstspace)
		strcpy(save, firstspace+1);
	return save;
}

void convert_watcom_tz_format(void)
{
	int   state;
	int   values[4];
	char  *s, *tz, buf[100], newtz[100];

	tz = getenv("TZ");
	if (tz == NULL)
		return;

	PortTrace("Current TZ: '%s'", tz);

	strcpy(newtz, "TZ=");
	for (state = 0;; state++)
	{
		s = strtok((state == 0 ? tz : NULL), ",");
		if (s == NULL)
			break;
		switch (state)
		{
		case 0:  /* MSK-3MSD */
			strcat(newtz, s);
			break;
		case 1: case 5: /* month */
		case 2: case 6: /* week  */
		case 3: case 7: /* day   */
		case 4: case 8: /* time  */
			if (!isnumber(strip(s)))
				state = 1000; /* failure */
			else
			{
				values[state - (state >= 5 ? 5 : 1)] = atoi(s);
				if (state == 4 || state == 8)
				{
					if (values[1] == -1) /* convert value of "last week" */
						values[1] = 5;
					sprintf(buf, ",M%d.%d.%d/%02d:%02d:%02d", values[0], values[1], values[2],
							values[3]/(60 * 60), (values[3] / 60) % 60, values[3] % 60
						   );
					strcat(newtz, buf);
				}
			}
			break;
		case 9: /* offset - skipped */
			break;
		}
	}

	if (state == 10)
	{
		PortTrace("Converting TZ to Watcom format for correct DST switch");
		PortTrace("Watcom  TZ: '%s'", newtz);
		putenv(strdup(newtz));
		tzset();
	}
	else
		PortTrace("Using current TZ value");
}
#endif

/*
 * We must override time() function because "global_sec_timer" updated
 * irregularly (with 1 sec. pauses). We always ending up in situation
 * when time(NULL) is "123" but "global_sec_timer" is still "122"
 */

#undef time

time_t os2_incremental_time(time_t *t)
{
	time_t result;

	result = time(t);
	if (result > global_sec_timer)
		global_sec_timer = result;

	return result;
}
