module: 	self-organizing-list
rcs-header:	$Header: /afs/cs.cmu.edu/project/gwydion/hackers/nkramer/mindy/mindy-1.3/libraries/coll-ext/RCS/solist.dylan,v 1.3 95/02/13 03:58:05 rgs Exp $
author: 	Robert Stockton (rgs@cs.cmu.edu)
synopsis:	Provides "self-organizing lists".  These explicit key
		collections provide roughly the semantics of hash tables, but
		use a probabilistic implementation which provides O(n) worst
		case performance but can provide very fast constant time
		access in the best case.

//======================================================================
//
// Copyright (c) 1994  Carnegie Mellon University
// All rights reserved.
// 
// Use and copying of this software and preparation of derivative
// works based on this software are permitted, including commercial
// use, provided that the following conditions are observed:
// 
// 1. This copyright notice must be retained in full on any copies
//    and on appropriate parts of any derivative works.
// 2. Documentation (paper or online) accompanying any system that
//    incorporates this software, or any part of it, must acknowledge
//    the contribution of the Gwydion Project at Carnegie Mellon
//    University.
// 
// This software is made available "as is".  Neither the authors nor
// Carnegie Mellon University make any warranty about the software,
// its performance, or its conformity to any specification.
// 
// Bug reports, questions, comments, and suggestions should be sent by
// E-mail to the Internet address "gwydion-bugs@cs.cmu.edu".
//
//======================================================================

//======================================================================
// The "Self Organizing List" is a "poor man's hash table".  More
// precisely, <self-organizing-list> is a subclass of <dictionary> for
// which addition and retrieval are both linear in the worst case, but
// which use a probabilistic strategy which yields nearly constant
// time in the best case.
//
// Because they have a very low overhead, self-organizing lists may
// provide better peformance than hash tables in cases where
// references have a high degree of temporal locality.  They may also
// be useful in situations where it is difficult to create a proper
// hash function.
//
// Define new self-organizing-lists with 
//
//   make(<self-organizing-list>, test: test)
//
// Test is expected to be an equality function.  In particular, it is
// expected to satisfy the identity and transitivity requirements
// described in chapter 5.  If not specified, test defaults to \==.
//======================================================================

define class <self-organizing-list> (<dictionary>)
  slot data :: <list>, init-value: #();
  // slot accessor provides method for standard collection op "key-test"
  slot key-test :: <function>, init-value: \==, init-keyword: test:;
end class;

define constant sol-fip-next-state =
  method (list :: <self-organizing-list>, state :: <list>) 
   => (result :: <list>);
    tail(state);
  end method;

define constant sol-fip-finished-state? =
  method (list :: <self-organizing-list>, state :: <list>, limit)
    state == #();
  end method;

define constant sol-fip-current-key =
  method (list :: <self-organizing-list>, state :: <list>) 
   => (result :: <object>);
    head(head(state));
  end method;


define constant sol-fip-current-element =
  method (list :: <self-organizing-list>, state :: <list>) 
   => (result :: <object>);
    tail(head(state));
  end method;

define constant sol-fip-current-element-setter =
  method (value :: <object>,
	  list :: <self-organizing-list>, state :: <list>) 
   => (result :: <object>);
    tail(head(state)) := value;
  end method;

define constant sol-fip-copy-state =
  method (list :: <self-organizing-list>, state :: <list>) 
   => (result :: <list>);
    state;
  end method;

define method forward-iteration-protocol (table :: <self-organizing-list>)
  values(table.data, #f, sol-fip-next-state, sol-fip-finished-state?,
	 sol-fip-current-key, sol-fip-current-element,
	 sol-fip-current-element-setter, sol-fip-copy-state);
end method forward-iteration-protocol;

define constant sol-no-default = pair(#f, #f);

// Auxiliary function for element.  Search the tail of the list for an element
// for which test(elem, key) is true, and then return the pair which
// *precedes* that element (or #() if not found)
//
define constant elem-search
  = method (prev :: <list>, test :: <function>, key)
      let list = prev.tail;
      if (list == #())
	#();
      else
	let elem = list.head;
	if (test(elem.head, key))
	  prev;
	else
	  elem-search(list, test, key);
	end if;
      end if;
    end method;

define method element(table :: <self-organizing-list>, key :: <object>,
		      #key default: default = sol-no-default)
  let list :: <list> = table.data;
  let test :: <function> = table.key-test;

  let first = list.head;	// depend upon head(#()) being defined
  case
    list == #() =>
      if (default == sol-no-default)
	error("Key %= not in %=", key, table)
      else
	default
      end if;
    test(first.head, key) =>
      first.tail;
    otherwise =>
      let cell :: <list> = elem-search(list, test, key);
      if (cell ~= #())
	// Move the matching element to the beginning
	let next = cell.tail;
	cell.tail := next.tail;
	next.tail := list;
	table.data := next;
	// Return the actual value
	next.head.tail;
      elseif (default == sol-no-default)
	error("Key %= not in %=", key, table)
      else
	default;
      end if;
  end case;
end method element;

define constant sol-no-value = pair(#f, #f);

define method element-setter(value, table :: <self-organizing-list>, 
			     key :: <object>)
  // Bring the existing element (if any) to the front of the list
  if (element(table, key, default: sol-no-value) == sol-no-value)
    // It wasn't there, so add it
    table.data := pair(pair(key, value), table.data);
  else
    // It was there, so change the value of the first element.
    tail(head(table.data)) := value;
  end if;
end method element-setter;

define method map-as
    (cls == <self-organizing-list>, proc :: <function>,
     coll :: <explicit-key-collection>,
     #next next-method, #rest more-collections)
 => (result :: <collection>);
  let test = key-test(coll);
  let sz = size(coll);
  if (~empty?(more-collections) | sz == #f)
    next-method();
  else 
    let result = make(cls);
    let (init, limit, next, done?, curkey, curelt)
      = forward-iteration-protocol(coll);
    for (list = #() then pair(pair(curkey(coll, state),
				   proc(curelt(coll, state))), list),
	 state = init then next(coll, state),
	 until done?(coll, state, limit))
    finally
      result.data := list;
    end for;
    result;
  end if;
end method map-as;

define method remove-key!
    (table :: <self-organizing-list>, key :: <object>)
 => (table :: <self-organizing-list>);
  // Bring the existing element (if any) to the front of the list
  if (element(table, key, default: sol-no-value) ~= sol-no-value)
    // It was there, so remove it.
    table.data := table.data.tail;
  end if;
  table;
end method remove-key!;
  