/* LVB (c) Copyright 2003-2006 by Daniel Barker.
 * Permission is granted to copy and use this program provided that no fee is
 * charged for it and provided that this copyright notice is not removed. */

/**********

=head1 NAME

mops.c - basic memory operation

Version Tag $Id: mops.c,v 1.19 2006/02/06 19:55:46 db60 Exp $

=cut

*********/

#include "lvb.h"

static const char *rcsid = "$Id: mops.c,v 1.19 2006/02/06 19:55:46 db60 Exp $";

/**********

=head1 alloc - ALLOCATE DYNAMIC HEAP MEMORY

=head2 SYNOPSIS

void *alloc(const size_t bytes, const char *const msg);

=head2 DESCRIPTION

B<alloc> is a wrapper for the standard library function C<malloc>. It
allocates new memory, or, on failure, crashes with an error message. It
also assures a system-independent return value, C<NULL>, if a zero-byte
allocation is requested.

B<alloc> will always fail without calling C<malloc> if the request
exceeds C<MAX_ALLOC> bytes, where C<MAX_ALLOC> is defined in C<lvb.h>.
This avoids straining C<malloc> beyond typical use. C<MAX_ALLOC> is a
large number that will comfortably fit in a 32-bit signed integer.

The allocated memory may be freed with the standard library function
C<free>.

The new memory is not initialized.

=head2 PARAMETERS

=head3 INPUT

=over4

=item bytes

The number of bytes to allocate. If C<bytes> is zero, no memory is
allocated.

=item msg

Pointer to the first text character in a string that describes the
object being allocated. On allocation failure, B<alloc> will crash with
an error message of 'FATAL ERROR: out of memory: cannot allocate for ',
followed by this string.

=back

=head2 RETURN

Returns pointer to the first byte of the newly allocated memory, which
is suitably aligned for storage of any object, or C<NULL> if a
zero-byte allocation was requested.

=cut

**********/

void *alloc(const size_t bytes, const char *const msg)
{
    void *p;	/* pointer to first byte of new memory, if any */

    if ((bytes == 0) || (bytes > MAX_ALLOC))
        p = NULL;
    else
    {
        p = malloc(bytes);
        if (p == NULL)
        crash("out of memory: cannot allocate for %s", msg);
    }
    return p;

}	/* end alloc() */
