/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) @(#) FrameShadow.java 1.58 - last change made 08/02/96
 */

package sunsoft.jws.visual.rt.shadow.java.awt;

import sunsoft.jws.visual.rt.base.Global;
import sunsoft.jws.visual.rt.base.DesignerAccess;
import sunsoft.jws.visual.rt.base.*;
import sunsoft.jws.visual.rt.type.ImageRef;
import sunsoft.jws.visual.rt.awt.RootFrame;

import java.awt.*;

/**
 * Wraps an AWT widget.  The attributes available for this
 * class are listed below.  In the type column, type names beginning
 * with "sunsoft.jws.visual.rt" have been abbreviated to begin with "rt".
 *
 * <pre>
    name            type                      default value
    -----------------------------------------------------------------------
  + background      java.awt.Color            #c0c0c0 
    icon            rt.type.ImageRef          null
    menubar         rt.type.MenuBarRef        null
    resizable       java.lang.Boolean         true 
    title           java.lang.String          "Unnamed Frame" 
 * </pre>
 *
 * + = this attribute overrides one inherited from an ancestor class.
 * <p>
 * Check the super class for additional attributes.
 *
 * @see Frame
 * @version 	1.58, 08/02/96
 */
public class FrameShadow extends WindowShadow {

  private int cursorCount = 0;
  private int prevCursor = Frame.DEFAULT_CURSOR;

  public FrameShadow() {
    attributes.add("title", "java.lang.String", "Unnamed Frame", NOEDITOR);
    attributes.alias("text", "title");
    attributes.add("resizable", "java.lang.Boolean", Boolean.TRUE, 0);
    attributes.add("icon", "sunsoft.jws.visual.rt.type.ImageRef", null, 0);
    attributes.add("menubar",
		   "sunsoft.jws.visual.rt.shadow.java.awt.MenuBarShadow",
		   null, DEFAULT | TRANSIENT);

    if (Global.isIrix())
      attributes.add("font", "java.awt.Font",
		     new Font("Helvetica", Font.PLAIN, 12), DONTFETCH);

    if (Global.isWindows()) {
      attributes.add("background", "java.awt.Color",
		     Color.lightGray, DONTFETCH);
      attributes.add("font", "java.awt.Font",
		     new Font("Dialog", Font.PLAIN, 12), DONTFETCH);
    }
  }

  protected Object getOnBody(String key) {
    Frame frame = (Frame)body;

    if (key.equals("title"))
      return frame.getTitle();
    else if (key.equals("resizable"))
      return new Boolean(frame.isResizable());
    else if (key.equals("icon"))
      return getFromTable("icon");
    else if (key.equals("menubar")) {
      MenuBar menuBar = frame.getMenuBar();
      if (menuBar == null)
	return null;
      else
	return DesignerAccess.getShadowTable().get(menuBar);
    }
    else
      return(super.getOnBody(key));
  }

  protected void setOnBody(String key, Object value) {
    Frame frame = (Frame)body;

    if (key.equals("title")) {
      frame.setTitle((String) value);
    }
    else if (key.equals("resizable")) {
      frame.setResizable(((Boolean) value).booleanValue());
    }
    else if (key.equals("menubar")) {
      MenuBarShadow s = (MenuBarShadow)getOnBody("menubar");
      if (s != null) {
	remove(s);
	s.destroy();
      }

      if (value != null) {
	s = (MenuBarShadow)value;
	add(s);
	s.create();
      }
    }
    else if (key.equals("icon")) {
      if (value != null) {
	try {
	  frame.setIconImage(((ImageRef) value).
			     getImage(frame, getGroup().getApplet()));
	}
	catch (VJException ex) {
	  if (isLive())
	    System.out.println("Error: " + ex.getMessage());
	  else
	    throw ex;
	}
      }
      else if (frame.getPeer() == null)
	frame.setIconImage(null);
    }
    else
      super.setOnBody(key, value);
  } 

  protected String getUserTypeName() {
    return("frame");
  }

  public void createBody() {
    if (!inDesignerRoot())
      body = new RootFrame();
    else {
      try {
	body = DesignerAccess.getFrameClass().newInstance();
      } catch (InstantiationException ex) {
	throw new Error(ex.toString());
      } catch (IllegalAccessException ex) {
	throw new Error(ex.toString());
      }
    }
  }

  // Overridden AMContainer interfaces

  public void addChildBody(Shadow child) {
    // frames will have a single central child and maybe a menubar
    if (body != null) {
      if (child instanceof MenuBarShadow) {
	// add the menubar
	if (((Frame) body).getMenuBar() != null) {
	  throw new Error("frame already has a menubar while trying to add "
			  + child.get("name"));
	}
	else {
	  ((Frame) body).setMenuBar((MenuBar) child.getBody());
	}
      }
      else {
	// add a component
	if (((Container) body).getLayout() instanceof BorderLayout) {
	  ((Container) body).add("Center", (Component) child.getBody());
	}
	else {
	  super.addChildBody(child);
	}
      }
    }
  }

  public void removeChildBody(Shadow child) {
    if (body != null) {
      if ((child instanceof MenuBarShadow)) {
	if (!((Frame) body).getMenuBar().equals((MenuBar) child.getBody())) {
	  throw new Error("this menubar was never installed: "
			  + child.get("name"));
	}
	else {
	  // remove the  menubar
	  ((Frame) body).remove((MenuComponent) child.getBody());
	}
      }
      else {
	// remove a component
	((Frame) body).remove((Component) child.getBody());
      }
    }
  }

  int incrCursor() {
    cursorCount++;
    return cursorCount;
  }

  int decrCursor() {
    cursorCount--;
    if (cursorCount < 0)
      cursorCount = 0;
    return cursorCount;
  }

  void setPrevCursor(int cursor) {
    prevCursor = cursor;
  }

  int getPrevCursor() {
    return prevCursor;
  }
}
