/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) ComponentShadow.java 1.71 - last change made 07/30/96
 */

package sunsoft.jws.visual.rt.shadow.java.awt;

import sunsoft.jws.visual.rt.base.*;
import sunsoft.jws.visual.rt.awt.GBLayout;
import sunsoft.jws.visual.rt.awt.GBConstraints;
import sunsoft.jws.visual.rt.type.AnchorEnum;

import java.awt.Component;
import java.awt.Container;
import java.awt.Window;
import java.awt.LayoutManager;
import java.awt.Color;
import java.awt.Font;

/**
 * Shadow class that wraps the AWT Component class.  The attributes
 * available for this class are listed below.  Check the super class
 * for additional attributes.
 *
 * <pre>
    name            type                      default value
    -----------------------------------------------------------------------
    anchor          rt.type.AnchorEnum        center 
    background      java.awt.Color            null
    enabled         java.lang.Boolean         true 
    font            java.awt.Font             null
    foreground      java.awt.Color            null
    GBConstraints   rt.awt.GBConstraints      new GBConstraints()
    insets          java.awt.Insets           null
    visible         java.lang.Boolean         true 
 * </pre>
 *
 * All shadow classes (except for the menu-related ones) include the
 * attributes from the ComponentShadow class in addition to their own,
 * and they do so by subclassing it.  This class is a super class and
 * isn't available directly from the palette.
 *
 * @see Component
 * @version 1.71, 07/30/96
 */
public class ComponentShadow extends Shadow {

  // Set to true while a show operation is in progress
  protected boolean doingShow = false;

  ComponentShadow() {
    GBConstraints c = new GBConstraints();
    c.gridx = 0;
    c.gridy = 0;
    attributes.add("GBConstraints", "sunsoft.jws.visual.rt.awt.GBConstraints",
		   c, HIDDEN | NONBODY | CONTAINER);

    attributes.add("layoutName", "java.lang.String", null,
		   HIDDEN | NONBODY | CONTAINER);
    attributes.add("anchor", "sunsoft.jws.visual.rt.type.AnchorEnum",
		   new AnchorEnum(GBConstraints.CENTER), NONBODY | CONTAINER);
    attributes.add("insets", "java.awt.Insets", null, NONBODY | CONTAINER);

    attributes.add("visible", "java.lang.Boolean", Boolean.TRUE,
		   HIDDEN | DEFAULT);
    attributes.add("enabled", "java.lang.Boolean", Boolean.TRUE, DEFAULT);
    attributes.add("foreground", "java.awt.Color", null,
		   DEFAULT | DONTFETCH);
    attributes.add("background", "java.awt.Color", null,
		   DEFAULT | DONTFETCH);
    attributes.add("font", "java.awt.Font", null,
		   DEFAULT | DONTFETCH);
  }

  protected Object getOnBody(String key) {
    Component comp = (Component)body;

    if (key.equals("visible"))
      return(getFromTable("visible"));
    else if (key.equals("enabled"))
      return(new Boolean(comp.isEnabled()));
    else if (key.equals("foreground"))
      return(comp.getForeground());
    else if (key.equals("background"))
      return(comp.getBackground());
    else if (key.equals("font"))
      return(comp.getFont());
    else
      return(super.getOnBody(key));
  }

  protected void setOnBody(String key, Object value) {
    Component comp = (Component)body;

    if (key.equals("visible")) {
      // Don't let visible be set to false if we are the main container
      // and we are running inside vjava.
      if (inDesignerRoot() && isMainContainer() &&
	  !((Boolean)value).booleanValue()) {
	throw new VJException(Global.newline() +
		"    It is illegal to set a panel's visible attribute" +
		Global.newline() +
		"    to false if it is the main panel for its group.");
      }

      // We don't need to use the doingShow() method, because the "visible"
      // attribute has the DEFAULT flag set, therefore setOnBody will
      // be called during creation for the "visible" attribute only if
      // the "visible" attribute is false.
      if (!doingShow)
	showComponent(((Boolean) value).booleanValue());
    }
    else if (key.equals("enabled"))
      comp.enable(((Boolean) value).booleanValue());
    else if (key.equals("foreground")) {
      comp.setForeground((Color) value);
      if (Global.isWindows()) {
	comp.repaint();
      }
    }
    else if (key.equals("background")) {
      comp.setBackground((Color) value);
      if (Global.isWindows()) {
	comp.repaint();
      }
    }
    else if (key.equals("font"))
      comp.setFont((Font) value);
    else
      super.setOnBody(key, value);
  }

  protected boolean isMainContainer() {
    Root r = getRoot();
    if (r == null)
      return false;

    AttributeManager mgr = r.getMainChild();
    if (mgr instanceof WindowShadow) {
      WindowShadow win = (WindowShadow)mgr;
      if (win.isPanel())
	mgr = win.getPanel();
    }

    return (mgr == this);
  }

  public void createBody() {};

  /**
   * Overrides destroyBody() in Shadow so that removeNotify() gets called on
   * AWT components when there will be no more references to them.
   */
  protected void destroyBody() {
    ((Component)body).removeNotify();
    body = null;
  }

  /**
   * Calls show or hide, depending on the value of cond.
   */
  public void show(boolean cond) {
    if (cond)
      show();
    else
      hide();
  }

  /**
   * Sets the visible attribute to true.
   */
  public void show() {
    set("visible", Boolean.TRUE);
  }

  /**
   * Sets the visible attribute to false.
   */
  public void hide() {
    set("visible", Boolean.FALSE);
  }

  /**
   * Calls showComponent or hideComponent, depending on the value of cond.
   */
  public void showComponent(boolean cond) {
    if (cond)
      showComponent();
    else
      hideComponent();
  }

  /**
   * Shows the component.  Calling showComponent does not affect the
   * value of the visible attrbute.  You should use "show" instead of
   * "showComponent".  The only reason this method exists is that
   * Visual Java needs to use it in certain situations.
   */
  public void showComponent() {
    // Call create if it hasn't already been called.
    if (!isCreated()) {
      doingShow = true;
      create();
      doingShow = false;
    }

    ((Component)body).show();
  }

  /**
   * Hides the component.  Calling hideComponent does not affect the
   * value of the visible attrbute.  You should use "hide" instead of
   * "hideComponent".  The only reason this method exists is that
   * Visual Java needs to use it in certain situations.
   */
  public void hideComponent() {
    if (body != null)
      ((Component)body).hide();
  }

  /**
   * Returns true if we are doing a create operation in the
   * middle of a show operation.  Create likes to call show if the
   * visible attribute is set to true, but create shouldn't call
   * show if show caused create to be called if the first place.
   */
  protected boolean doingShow() {
    if (doingShow) {
      return true;
    }
    else {
      Group g = getGroup();
      if (g != null)
	return DesignerAccess.doingShow(g);
      else
	return false;
    }
  }

  /**
   * Call validate to lay out the component and its children if they
   * are not valid.
   */
  public void validate() {
    if (body != null)
      ((Component)body).validate();
  }

  /**
   * Call invalidate to force the component to not be valid, so that
   * it will be layed out again when validate is called.
   */
  public void invalidate() {
    if (body != null)
      ((Component)body).invalidate();
  }

  /**
   * Returns the result from calling isShowing on the body.
   */
  public boolean isShowing() {
    if (body != null)
      return ((Component)body).isShowing();
    else
      return false;
  }
}
