// --------------------------------------------------------------------------
// The jSyncManager Project -- Source File.
// Copyright (c) 1998 - 2003 Brad BARCLAY <bbarclay@jsyncmanager.org>
// --------------------------------------------------------------------------
// OSI Certified Open Source Software
// --------------------------------------------------------------------------
//
// This library is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License as published 
// by the Free Software Foundation; either version 2.1 of the License, or 
// (at your option) any later version.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public 
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
// --------------------------------------------------------------------------
// $Id: SerialPeer.java,v 1.2 2003/11/20 17:50:58 yaztromo Exp $
// --------------------------------------------------------------------------

package org.jSyncManager.JSerial;

import java.io.IOException;

// ==========================================================================

/** A class to define native code headers for accessing a serial port in Java.
  * This class will provide the native code headers for accessing serial ports
  * under Java.  Note that normally a programmer should <B>not</B> directly utilize
  * this class - all functionality is available through the org.jSyncManager.JSerialInterface
  * class.
  * @author Brad BARCLAY &lt;bbarclay@jsyncmanager.org&gt;
  * <br>Last modified by: $Author: yaztromo $ on $Date: 2003/11/20 17:50:58 $.
  * @version $Revision: 1.2 $
  */

public class SerialPeer {

   // The following are static values used for this class
   public final static String NOPORT=new String("*** Requested Port does not exist.");
   public final static String PORTUNAVAILABLE=new String("*** The requested port is unavailable.");


   // The following are the private variables for instances of this class
   private byte port;               // Used to store the port number
   private int speed;               // Used to store the currently/last selected speed
   private int databits;            // Used to store the current databits per byte
   private int parity;              // Used to store the current parity per byte
   private int stopbits;            // Used to store the current stopbits per byte
   private boolean status=false;    // Used to store the current grab status.

// ==========================================================================

   /** The peer constructor.
     * @param port the port ID number for this serial peer.
     * @exception SerialInitializationException thrown if the port requested does not exist.
     */
   public SerialPeer(byte port) throws SerialInitializationException {

      // Check to see if the port exists
      if (checkPort(port)) {
         this.port=port;
      } else {
         // If not, throw an exception.
         throw new SerialInitializationException(NOPORT);
      } /* endif */

   } // end-constructor

// --------------------------------------------------------------------------

   /** Native method to gain exclusive access to a serial port.
     * @exception SerialGrabException thrown if the port cannot be grabbed by this process.
     */
   public native void grabPort() throws SerialGrabException;

// --------------------------------------------------------------------------

   /** Native method to set the speed for the serial port.
     * @param speed a valid speed setting.
     * @exception SerialSpeedException thrown if the given speed is invalid, or not supported.
     */
   public native void setSpeed(int speed) throws SerialSpeedException;

// --------------------------------------------------------------------------

   /** Sets the number of Data Bits per transmission byte.
     * @param databits the desired number of data bits per transmission byte.
     * @exception SerialDataBitException thrown if the specified value is invalid or unsupported.
     */
   public native void setDataBits(int databits) throws SerialDataBitException;

// --------------------------------------------------------------------------

   /** Sets the Parity bit type used for each transmission unit.
     * @param parity a value representing the parity type.
     * @exception SerialParityException thrown if the given parity is invalid or unsupported.
     */
   public native void setParity(int parity) throws SerialParityException;

// --------------------------------------------------------------------------

   /** Sets the number of Stop Bits per transmission unit.
     * @param stopbits the number of stop bits per transmission unit.
     * @exception SerialStopBitException thrown if the value specified is invalid or unsupported.
     */
   public native void setStopBits(int stopbits) throws SerialStopBitException;

// --------------------------------------------------------------------------

   /** Gets the next available character from the incoming serial stream.
     * @return an integer representing a character value from 0 - 255, or -1 if no character is available.
     * @exception IOException thrown if an IOException occurs while attempting to read the serial port.
     */
   public native int getChar() throws IOException;

// --------------------------------------------------------------------------

   /** Puts a character into the outgoing serial stream.
     * @param b a character value between 0 - 255.
     * @exception IOException thrown if an IOException occurs while attempting to read the serial port.
     */
   public native void putChar(int b) throws IOException;

// --------------------------------------------------------------------------

   /** A static method to return the name of the platform for which a working native code
     * library has been installed.
     * @return a String representing the name of the platform.
     */
   public static native String getPlatformName();

// --------------------------------------------------------------------------

   /** A static method to return the version number of the installed native code implementation.
     * @return a strong representing major_version.minor_version.
     */
   public static native String getVersion();

// --------------------------------------------------------------------------

   /** A method to return the ID value of the current port.
     * @return a byte representing the port ID.
     */
   public byte getPort() {
      return port;
   } // end-method

// --------------------------------------------------------------------------

   /** A method to get the currently set serial speed.
     * @return the current serial speed for the current port.
     */
   public int getSpeed() {
      return speed;
   } // end-method

// --------------------------------------------------------------------------

   /** A method to retreive the current databit setting.
     * @return the current data bit size per byte for the current port.
     */
   public int getDataBits() {
      return databits;
   } // end-method

// --------------------------------------------------------------------------

   /** A method to retreive the current parity setting.
     * @return the current parity type for the current port.
     */
   public int getParity() {
      return parity;
   } // end-method

// --------------------------------------------------------------------------

   /** A method to retreive the current stopbit setting.
     * @return the current stop bit size per byte for the current port.
     */
   public int getStopBits() {
      return stopbits;
   } // end-method

// --------------------------------------------------------------------------

   /** A method to release control of the serial port to the underlying OS.
     * @exception SerialReleaseException thrown if the port cannot be released.
     */
   public native void releasePort() throws SerialReleaseException;

// --------------------------------------------------------------------------

   /** A method to determine wether or not the current port is in the "grabbed" state.
     * @return the "grab" status - <B>true</B> if the port is released, <B>false</B> otherwise.
     */
   public boolean portStatus() {
      return !status;
   } // end-method

// --------------------------------------------------------------------------

   /** A method to check if the requested port exists on this machine.
     * @return wether or not this port is supported - <B>true</B> if the port exists, <B>false</B> otherwise.
     */
   public native boolean checkPort(byte port);

// --------------------------------------------------------------------------

   /** A method to set the state of the RTS/CTS handshaking.
     * Sets the state of RTS/CTS hardware handshaking.
     * @param state if <B>true</B>, enables RTS/CTS hardware handshaking.
     * @throws SerialHandshakingException thrown if the serial driver is unable to change modes.
     */
   public native void setRTSCTS(boolean state) throws SerialHandshakingException;

// --------------------------------------------------------------------------

   /** A method to set the state of the DSR/DTR handshaking.
     * Sets the state of DSR/DTR hardware handshaking.
     * @param state if <B>true</B>, enables hardware handshaking.
     * @throws SerialHandshakingException thrown if the serial driver is unable to change modes.
     */
   public native void setDSRDTR(boolean state) throws SerialHandshakingException;

// --------------------------------------------------------------------------

   /** A method to set the state of the XON/XOFF handshaking.
     * Sets the state of XON/XOFF software handshaking.  Note that this
     * function will set both the transmit and receive to use XON/XOFF control.
     * @param state if <B>true</B>, enables software handshaking.
     * @throws SerialHandshakingException thrown if the serial driver is unable to change modes.
     */
   public native void setXONXOFF(boolean state) throws SerialHandshakingException;

// --------------------------------------------------------------------------

   /** Starts transmitting a break signal.
     */
   public native void startBreak();

// --------------------------------------------------------------------------

   /** Stop transmission of a break signal.
     */
   public native void stopBreak();

// --------------------------------------------------------------------------

   /** Queries the state of the RTS/CTS handshaking.
     * @return <B>true</B> if RTS/CTS is enabled.
     */
   public native boolean getRTSCTSState();

// --------------------------------------------------------------------------

   /** Queries the state of the DSR/DTR handshaking.
     * @return <B>true</B> if DSR/DTR is enabled.
     */
   public native boolean getDSRDTRState();

// --------------------------------------------------------------------------

   /** Queries the state of the XON/XOFF handshaking.
     * @return <B>true</B> if XON/XOFF is enabled.
     */
   public native boolean getXONXOFFState();

// ==========================================================================

   static {
      System.loadLibrary("jserial");
   } // end-static-initializer

// ==========================================================================

} // end-class

