/*
 * 01/09/2002 - 20:43:57
 *
 * PluginManager.java -
 * Copyright (C) 2002 Csaba Kertsz
 * kcsaba@jdictionary.info
 * www.jdictionary.info
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


package info.jdictionary.pluginstuff;

import java.util.*;
import java.io.*;
import info.jdictionary.*;
import info.jdictionary.pluginstuff.*;
import java.util.zip.*;
import info.jdictionary.events.PluginStructureChangeEvent;
import info.jdictionary.events.PluginFilesChangeEvent;
import info.jdictionary.listeners.PluginStructureChangeListener;
import info.jdictionary.listeners.PluginFilesChangeListener;
import info.jdictionary.events.PluginSelectionEvent;
import info.jdictionary.listeners.PluginSelectionListener;
import info.jdictionary.events.PluginScanFinishedEvent;
import info.jdictionary.listeners.PluginScanFinishedListener;

public class PluginManager {

    HashMap plugins = new HashMap();
    String[] pluginFiles;
    JDictionary jDictionary;
    PluginInfoSheet selectedPlugin;
    String selectedSubPlugin;
    Vector pluginStructureChangeListeners = new Vector();
    Vector pluginFilesChangeListeners = new Vector();
    Vector pluginSelectionListeners = new Vector();
    Vector pluginScanFinishedListeners = new Vector();
    ScanThread scanThread;

    public PluginManager(JDictionary jDictionary) {
        this.jDictionary = jDictionary;
    }


    void pushPlugin(PluginInfoSheet infoSheet, Plugin plugin) {
        if (!plugins.containsKey(infoSheet))
            plugins.put(infoSheet, new ArrayList());
        ((ArrayList)plugins.get(infoSheet)).add(plugin);
    }


    public ArrayList getPlugin(PluginInfoSheet infoSheet) {
        if (plugins.containsKey(infoSheet))
            return (ArrayList) plugins.get(infoSheet);
        return null;
    }


    void removePlugin(PluginInfoSheet infoSheet) {
        Iterator it = getPlugin(infoSheet).iterator();
        while (it.hasNext())
            ((Plugin) it.next()).stop();
        plugins.remove(infoSheet);
        notifyPluginRemovedListeners(infoSheet);
    }


    public void removeAllPlugins() {
        Object[] sheets = getSheets();
        int n = 0;
        while (n < sheets.length)
            removePlugin((PluginInfoSheet) sheets[n++]);
    }


    public Object[] getSheets() {
        return plugins.keySet().toArray();
    }


    public int getNumberOfLoadedPlugins() {
        return plugins.size();
    }

    public int getNumberOfAvailablePlugins() {
        return getAvailablePluginFileNames().length;
    }

    public void checkPluginDirForChanges() {
        File f = new File(JDictionary.getPluginDirPath());
        String[] fileList = f.list(new DirFilter(".jar"));
        String[] knownPluginFiles = getAvailablePluginFileNames();

            if(knownPluginFiles.length != fileList.length)
                notifyPluginFilesChangedListeners(fileList);
            else
                for(int i = 0; i < knownPluginFiles.length; i++) {
                    int j;
                    for(j = 0; j < fileList.length; j++)
                        if(fileList[j].equals(knownPluginFiles[i]))
                            break;
                    if(j == fileList.length) {
                        notifyPluginFilesChangedListeners(fileList);
                        break;
                    }
                }
        pluginFiles = fileList;
    }


    public String[] getAvailablePluginFileNames() {
        if(pluginFiles == null) {
            File f = new File(JDictionary.getPluginDirPath());
            String[] fileList = f.list(new DirFilter(".jar"));
            pluginFiles = fileList;
        }
        return pluginFiles;
    }


    public ZipFile[] getAvailablePluginFiles() {
        String[] fileNames = getAvailablePluginFileNames();
        ZipFile[] pluginFiles = new ZipFile[fileNames.length];
        for(int i = 0; i < fileNames.length; i++) {
            try {
            ZipFile z = new ZipFile(JDictionary.getPluginDirPath() + JDictionary.getFileSeparator() + fileNames[i]);
            pluginFiles[i] = z;
            }
            catch (java.io.IOException e) {}
        }
        return pluginFiles;
    }

    //this method is a bit deprecated
    public boolean isItInstalled(PluginInfoSheet infoSheet) {
        Iterator sheetsIterator = plugins.keySet().iterator();
        while (sheetsIterator.hasNext()) {
            PluginInfoSheet tempSheet = (PluginInfoSheet)sheetsIterator.next();
            if (infoSheet.getFileName().equals(tempSheet.getFileName()))
                if (extractVersionNumberFromFileName(infoSheet.getFileName()) == extractVersionNumberFromFileName(tempSheet.getFileName()))
                    return true;
        }
        return false;
    }

    public float getNewestVersion(PluginInfoSheet infoSheet) {
        Iterator sheetsIterator = plugins.keySet().iterator();
        float newestVersion = 0.0f;
        while (sheetsIterator.hasNext()) {
            PluginInfoSheet tempSheet = (PluginInfoSheet) sheetsIterator.next();
            if (infoSheet.getFileName().equals(tempSheet.getFileName()))
                if (Float.parseFloat(tempSheet.getVersion()) > newestVersion)
                    newestVersion = Float.parseFloat(tempSheet.getVersion());
        }
        return newestVersion;
    }

  public float extractVersionNumberFromFileName(String fileName) {
        int s = fileName.lastIndexOf("-");
        int e = fileName.lastIndexOf(".jar");
        if(s > -1 && (s < e))
            return Float.parseFloat(new String(fileName.substring(s + 1, e)).replace('_', '.'));
        else
            return 0.0f;
      }

    public String getNameWithoutVersion(String fileName) {
        int s = fileName.lastIndexOf("-");
        if(s > -1)
            return fileName.substring(0, s);
        s = fileName.lastIndexOf(".jar");
        if(s > -1)
            return fileName.substring(0, s);
        return fileName;
    }

    public boolean isPluginFileLoaded(String fileName) {
        Object[] sheets = getSheets();
        for (int n = 0; n < sheets.length; n++) {
            String name = ((PluginInfoSheet) sheets[n]).getFileName();
            if (name.equals(fileName))
                return true;
        }
        return false;
    }


    public boolean isItActive(String fileName) {
        String name = getNameWithoutVersion(fileName);
        Iterator it = jDictionary.getPrefs().InActivePlugins.iterator();
        while (it.hasNext())
            if(((String)it.next()).equals(name))
                return false;
        return true;
    }


    public void selectPlugin(PluginInfoSheet sheet) {
        selectPlugin(sheet, null);
    }

    public void selectPlugin(PluginInfoSheet sheet, String subPluginName) {
        ArrayList plugin = getPlugin(sheet);
        if((sheet == selectedPlugin && subPluginName == selectedSubPlugin) || (sheet == selectedPlugin && subPluginName == null))
            return;
        if(plugin != null)
            selectedPlugin = sheet;
        else {
            selectedPlugin = null;
            selectedSubPlugin = null;
            jDictionary.getPrefs().pluginInfoSheet = selectedPlugin;
            jDictionary.getPrefs().selectedNodeName = selectedSubPlugin;
            notifyPluginSelectedListeners();
            return;
        }
        for(int i = 0; i < plugin.size(); i++) {
            if(plugin.get(i).toString().equals(subPluginName)) {
                selectedSubPlugin = subPluginName;
                notifyPluginSelectedListeners();
                jDictionary.getPrefs().pluginInfoSheet = selectedPlugin;
                jDictionary.getPrefs().selectedNodeName = selectedSubPlugin;
                return;
            }
        }
        selectedSubPlugin = null;
        jDictionary.getPrefs().pluginInfoSheet = selectedPlugin;
        jDictionary.getPrefs().selectedNodeName = selectedSubPlugin;
        notifyPluginSelectedListeners();
    }


    public void setActive(String fileName) {
        String name = getNameWithoutVersion(fileName);
        jDictionary.getPrefs().InActivePlugins.remove(name);
    }


    public void setInActive(String fileName) {
        String name = getNameWithoutVersion(fileName);
        jDictionary.getPrefs().InActivePlugins.add(name);
    }


    public void cleanUpInactivePluginList() {
        Iterator inactivePlugins = jDictionary.getPrefs().InActivePlugins.iterator();
        while(inactivePlugins.hasNext()) {
            if(!isItAvailable(inactivePlugins.next().toString(), false))
                inactivePlugins.remove();
        }
    }


    public boolean isThereNewerPluginFileAvailable(String fileName) {
        String[] pluginDir = getAvailablePluginFileNames();
        float version = extractVersionNumberFromFileName(fileName);
        for(int i = 0; i < pluginDir.length; i++) {
            if(getNameWithoutVersion(pluginDir[i]).equals(getNameWithoutVersion(fileName))) {
                float tempVersion = extractVersionNumberFromFileName(pluginDir[i]);
                if (tempVersion > version)
                    return true;
            }
        }
        return false;
    }


    public boolean isItAvailable(String fileName, boolean careWithVersion) {
            String[] fileNames = getAvailablePluginFileNames();
            for(int i = 0; i < fileNames.length; i++)
                if(careWithVersion) {
                    if(fileNames[i].equals(fileName))
                        return true;
                }
                else
                    if(getNameWithoutVersion(fileNames[i]).equals(getNameWithoutVersion(fileName)))
                        return true;
            return false;
    }


    public void syncPluginFiles() {
        File f = new File(JDictionary.getGlobalPluginDirPath());
        if(f.exists() == false) {
            cleanUpInactivePluginList();
            checkPluginDirForChanges();
            return;
        }
        String[] globalPluginDirFiles = f.list();
        for (int i = 0; i < globalPluginDirFiles.length; i++) {
            if(isItAvailable(globalPluginDirFiles[i], true))
                continue;
            if(isItAvailable(globalPluginDirFiles[i], false) && isThereNewerPluginFileAvailable(globalPluginDirFiles[i]))
                continue;

            File sourceFile = new File(JDictionary.getGlobalPluginDirPath() + JDictionary.getFileSeparator() + globalPluginDirFiles[i]);
            try {
                FileInputStream sourceStream = new FileInputStream(sourceFile);
                int offset = 0;
                int loadedBytesCount = 0;
                int fileLength = (int) sourceFile.length();
                byte[] b = new byte[fileLength];

                while (loadedBytesCount < fileLength) {
                    fileLength -= loadedBytesCount;
                    offset += loadedBytesCount;
                    loadedBytesCount = sourceStream.read(b, offset, fileLength);
                    if (loadedBytesCount == -1)
                        System.err.println("PluginLoader.syncPluginFiles(): Wrong thing happened when I attempted to copy the file named: " + sourceFile.getName());
                }
                File destFile = new File(JDictionary.getPluginDirPath() + JDictionary.getFileSeparator() + globalPluginDirFiles[i]);
                FileOutputStream outStream = new FileOutputStream(destFile);
                System.out.println("PluginLoader.syncPluginFiles(): writing: " + destFile.getName());
                outStream.write(b);
            }
            catch(java.io.FileNotFoundException e) {}
            catch(java.io.IOException e) {}
        }
            cleanUpInactivePluginList();
            checkPluginDirForChanges();
    }


    public void scanPlugins() {
        if(scanThread != null)
            try{
                scanThread.join();
            }
            catch(java.lang.InterruptedException e) {}

            scanThread = new ScanThread();
            scanThread.start();
    }


    void cleanupPlugins() {
        Object[] sheets = getSheets();
        for (int n = 0; n < sheets.length; n++) {
            PluginInfoSheet sheet = (PluginInfoSheet)sheets[n];
            String name = sheet.getFileName();

            //if pluginfile was deleted, this will delete the
            //appropriate node from the plugintree and the infos about
            //this plugin from the PluginManager
            if (!isItAvailable(name, true)) {
                removePlugin(sheet);
            }
            else {
                //okay, pluginfile is exist for the appropriate node,
                //but what if somebody replaced it for a newer one?
                //(For example JDictionary's PluginManager) If this
                //happened, we should delete the node from plugintree and
                //the info about this plugin from the PluginManager
                //and it will be redetected and reloaded in the run method.
                if (isThereNewerPluginFileAvailable(name) || isItActive(name) == false) {
                    removePlugin(sheet);
                }
            }
        }

    }


 long getSize(ZipFile zipFile) {
        File f = new File(zipFile.getName());
        return f.length();
    }


    void notifyPluginAddedListeners(PluginInfoSheet sheet) {
        Vector tmpList;
        PluginStructureChangeEvent event = new PluginStructureChangeEvent(this, sheet);
        synchronized(this) {
            tmpList = (Vector)pluginStructureChangeListeners.clone();
        }
        for(int i = 0; i < tmpList.size(); i++) {
            ((PluginStructureChangeListener)tmpList.elementAt(i)).pluginAdded(event);
        }
    }

     void notifyPluginRemovedListeners(PluginInfoSheet sheet) {
        Vector tmpList;
        PluginStructureChangeEvent event = new PluginStructureChangeEvent(this, sheet);
        synchronized(this) {
            tmpList = (Vector)pluginStructureChangeListeners.clone();
        }
        for(int i = 0; i < tmpList.size(); i++) {
            ((PluginStructureChangeListener)tmpList.elementAt(i)).pluginRemoved(event);
        }
    }

    void notifyPluginFilesChangedListeners(String[] fileList) {
        Vector tmpList;
        PluginFilesChangeEvent event = new PluginFilesChangeEvent(this, fileList);
        synchronized(this) {
            tmpList = (Vector)pluginFilesChangeListeners.clone();
        }
        for(int i = 0; i < tmpList.size(); i++) {
            ((PluginFilesChangeListener)tmpList.elementAt(i)).pluginFilesChanged(event);
        }
    }

     void notifyPluginSelectedListeners() {
        Vector tmpList;
        PluginSelectionEvent event = new PluginSelectionEvent(this, selectedPlugin, selectedSubPlugin);
        synchronized(this) {
            tmpList = (Vector)pluginSelectionListeners.clone();
        }
        for(int i = 0; i < tmpList.size(); i++) {
            ((PluginSelectionListener)tmpList.elementAt(i)).pluginSelected(event);
        }
    }

    void notifyPluginScanFinishedListeners() {
        Vector tmpList;
        PluginScanFinishedEvent event = new PluginScanFinishedEvent(this, plugins);
        synchronized(this) {
            tmpList = (Vector)pluginScanFinishedListeners.clone();
        }
        for(int i = 0; i < tmpList.size(); i++) {
            ((PluginScanFinishedListener)tmpList.elementAt(i)).pluginScanFinished(event);
        }
    }



    public synchronized void addPluginStructureChangeListener(PluginStructureChangeListener listener) {
        pluginStructureChangeListeners.addElement(listener);
    }

    public synchronized void removePluginStructureChangeListener(PluginStructureChangeListener listener) {
        pluginStructureChangeListeners.removeElement(listener);
    }

    public synchronized void addPluginFilesChangeListener(PluginFilesChangeListener listener) {
        pluginFilesChangeListeners.addElement(listener);
    }

    public synchronized void removePluginFilesChangeListener(PluginFilesChangeListener listener) {
        pluginFilesChangeListeners.removeElement(listener);
    }

    public synchronized void addPluginSelectionListener(PluginSelectionListener listener) {
        pluginSelectionListeners.addElement(listener);
    }

    public synchronized void removePluginSelectionListener(PluginSelectionListener listener) {
        pluginSelectionListeners.removeElement(listener);
    }

    public synchronized void addPluginScanFinishedListener(PluginScanFinishedListener listener) {
        pluginScanFinishedListeners.addElement(listener);
    }

    public synchronized void removePluginScanFinishedListener(PluginScanFinishedListener listener) {
        pluginScanFinishedListeners.removeElement(listener);
    }


class ScanThread extends Thread {
      public void run() {

        syncPluginFiles();

        cleanupPlugins();

        boolean loadLastUsed = true;
        int i = 0;

        String[] baseDirFiles = getAvailablePluginFileNames();

        //this will run through the pluginfiles, and asks
        //the PluginManager, whether the forthcoming pluginfile
        //has been loaded or not. If not, this will do the loading
        //and plugin registering process.
        while (i < baseDirFiles.length) {
            String fileNameToLoad = baseDirFiles[i];
            try {

                //trying to load the last used plugin first
                if(loadLastUsed) {
                    loadLastUsed = false;
                    Prefs prefs = jDictionary.getPrefs();
                    if(prefs.pluginInfoSheet != null && isItAvailable(prefs.pluginInfoSheet.getFileName(), true) && isItActive(prefs.pluginInfoSheet.getFileName())) {
                        fileNameToLoad = jDictionary.getPrefs().pluginInfoSheet.getFileName();
                        i = -1;
                    }
                }

                //if there is newer plugin file available than the fileNameToLoad
                //then the plugin file named fileNameToLoad will be deleted.
                if(isThereNewerPluginFileAvailable(fileNameToLoad)) {
                    File f = new File(JDictionary.getPluginDirPath() + JDictionary.getFileSeparator() + fileNameToLoad);
                    f.delete();
                    i++;
                    continue;
                }

                //skipping the files which belong to inactive plugins.
                if(isItActive(fileNameToLoad) == false) {
                    i++;
                    continue;
                }

                //if plugin file named fileNameToLoad is loaded then skipping this file.
                if (isPluginFileLoaded(fileNameToLoad)) {
                        System.out.println("Plugin " + fileNameToLoad + " is already loaded");
                        i++;
                        continue;
                }
                File f = new File(JDictionary.getPluginDirPath() + JDictionary.getFileSeparator() + fileNameToLoad);
                ZipFile jarFile = new ZipFile(f);
                System.out.println("\nscanPlugins(): Processing file: " + jarFile.getName());
                if (jarFile != null) {
                        System.out.println("Loading InfoSheet from file named: " + jarFile.getName());
                        final PluginInfoSheet sheet = PluginInfoSheet.CreateInfoSheet(jarFile);
                        if (sheet == null) { //skipping file if no infosheet file was fount in it.
                            System.out.println("Unable 0to load InfoSheet from file named: " + jarFile.getName() + " Skipping...");
                            i++;
                            try {
                                jarFile.close();
                            } catch (java.io.IOException ex) {}
                            continue;
                        }
                        if(Float.parseFloat(sheet.getMinJDictionaryVersion()) > JDictionary.getJDictionaryVersion()) {
                        System.out.println("Unable to load plugin named " + sheet.getName() + " (JDictionary version " + sheet.getMinJDictionaryVersion() + " or higher needed) Skipping...");
                            i++;
                            try {
                                jarFile.close();
                            } catch (java.io.IOException ex) {}
                            continue;
                        }
                        sheet.setFileName(fileNameToLoad); //corecting fileName
                        sheet.setSize(new Long(getSize(jarFile)).toString()); //finding out and setting filesize

                        PluginLoader pluginLoader = new PluginLoader(jarFile.getName());
                        Enumeration entries = jarFile.entries();
                        while (entries.hasMoreElements()) {
                            String entryName = ((ZipEntry) entries.nextElement()).getName();
                            if (entryName.endsWith("Plugin.class")) {
                                System.out.println("scanPlugins(): I want the PluginLoader to load the class named: " + entryName);
                                Class pluginClass = pluginLoader.loadClass(entryName);
                                final Object plugin = pluginClass.newInstance();
                                if (plugin instanceof Plugin) {
                                    pushPlugin(sheet, (Plugin) plugin);
                                }
                            }
                        }
                        notifyPluginAddedListeners(sheet);
                    }
                i++;
                try {
                    jarFile.close();
                } catch (java.io.IOException ex) {
                    System.out.println("Unable to close jarFile");
                }
            } catch (Exception e) {
                System.out.println("scanPlugins(): An error occurred while loading plugin named: " + fileNameToLoad + " Skipping...");
                e.printStackTrace();
                i++;

            }
        }
    notifyPluginScanFinishedListeners();
    checkPluginDirForChanges();
    }

    }
}

class DirFilter implements java.io.FilenameFilter {
    String afn;
    DirFilter(String afn) {
        this.afn = afn;
    }

    public boolean accept(File dir, String name) {
        String f = new File(name).getName();
        //return f.indexOf(afn) != -1;
        return f.endsWith(afn);
    }
}

