/* util.c: libshout utility/portability functions
 *
 *  Copyright (C) 2002-2003 the Icecast team <team@icecast.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
 #include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <sys/types.h>
#ifdef HAVE_SYS_SOCKET_H
#ifdef __WATCOMC__
#include <types.h>
#endif
#include <sys/socket.h>
#endif
#ifdef HAVE_WINSOCK2_H
#include <winsock2.h>
#endif

#include <shout/shout.h>
#include "util.h"
#include "debug.h"

char *_shout_util_strdup(const char *s)
{
	if (!s)
		return NULL;

	return debugStrDup(s);
}

int _shout_util_read_header(int sock, char *buff, unsigned long len)
{
	int read_bytes, ret;
	unsigned long pos;
	char c;

	read_bytes = 1;
	pos = 0;
	ret = 0;

	while ((read_bytes == 1) && (pos < (len - 1))) {
		read_bytes = 0;

		if ((read_bytes = recv(sock, &c, 1, 0))) {
			if (c != '\r')
				buff[pos++] = c;
			if ((pos > 1) && (buff[pos - 1] == '\n' && buff[pos - 2] == '\n')) {
				ret = 1;
				break;
			}
		} else {
			break;
		}
	}

	if (ret) buff[pos] = '\0';

	return ret;
}

static char base64table[65] = {
    'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P',
    'Q','R','S','T','U','V','W','X','Y','Z','a','b','c','d','e','f',
    'g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v',
    'w','x','y','z','0','1','2','3','4','5','6','7','8','9','+','/',
};

/* This isn't efficient, but it doesn't need to be */
char *_shout_util_base64_encode(char *data)
{
	int len = strlen(data);
	char *out = debugMAlloc(len*4/3 + 4);
	char *result = out;
	int chunk;

	while(len > 0) {
		chunk = (len >3)?3:len;
		*out++ = base64table[(*data & 0xFC)>>2];
		*out++ = base64table[((*data & 0x03)<<4) | ((*(data+1) & 0xF0) >> 4)];

		switch(chunk) {
		case 3:
			*out++ = base64table[((*(data+1) & 0x0F)<<2) | ((*(data+2) & 0xC0)>>6)];
			*out++ = base64table[(*(data+2)) & 0x3F];
			break;
		case 2:
			*out++ = base64table[((*(data+1) & 0x0F)<<2)];
			*out++ = '=';
			break;
		case 1:
			*out++ = '=';
			*out++ = '=';
			break;
		}
		data += chunk;
		len -= chunk;
	}
	*out = 0;

	return result;
}

static char urltable[16] = {
	'0','1','2','3','4','5','6','7','8','9','a','b','c','d','e','f'
};

static char safechars[256] = {
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,
      0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
      1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,
      0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
      1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
};

#ifdef __WATCOMC__
static int _url_encode_len(const char *data)
{
	int n;
  const char *p;

	for (p = data, n = 0; *p; p++)
	{
		n++;
		if (!safechars[(unsigned char)(*p)])
			n += 2;
	}

	return n;
}

static int _url_encode(char *buf, size_t buf_len, const char *data)
{
	int enc_len = _url_encode_len( data );
	const char *src = data;
	char *dst = buf;

	if ( enc_len >= buf_len )
		return -1;

	while( *src != '\0' )
	{
		if ( safechars[(unsigned char)(*src)] )
			*(dst++) = *src;
		else
		{
			*(dst++) = '%';
			*(dst++) = urltable[ (*src >> 4) & 0x0F ];
			*(dst++) = urltable[ *src & 0x0F ];
		}
    src++;
	}

	*dst = '\0';
	return enc_len;
}

char *_shout_util_url_encode(const char *data) {
	int enc_len = _url_encode_len( data );
  char *enc;

  if ( enc_len == 0 )
    return NULL;

  enc_len++;                    // One byte for zero.
  enc = debugMAlloc( enc_len );
  if ( enc != NULL )
    _url_encode( enc, enc_len, data );

  return enc;
}

#else

/* modified from libshout1, which credits Rick Franchuk <rickf@transpect.net>.
 * Caller must free result. */
char *_shout_util_url_encode(const char *data) {
	const char *p;
	char *q, *dest;
	int digit;
	size_t n;

	for (p = data, n = 0; *p; p++) {
		n++;
		if (!safechars[(unsigned char)(*p)])
			n += 2;
	}
	if (!(dest = debugMAlloc(n+1)))
		return NULL;
		
	for (p = data, q = dest; *p; p++, q++) {
		if (safechars[(unsigned char)(*p)]) {
			*q = *p;
		} else {
			*q++ = '%';
			digit = (*p >> 4) & 0xF;
			*q++ = urltable[digit];
			digit = *p & 0xf;
			*q = urltable[digit];
			n += 2;
		}
	}
	*q = '\0';

	return dest;
}
#endif

util_dict *_shout_util_dict_new(void)
{
  debugInc( "shout_dict" );
	return (util_dict *)debugCAlloc(1, sizeof(util_dict));
}

void _shout_util_dict_free(util_dict *dict)
{
	util_dict *next;

	while (dict) {
		next = dict->next;

		if (dict->key)
			debugFree (dict->key);
		if (dict->val)
			debugFree (dict->val);
		debugFree (dict);

		dict = next;
	}
  debugDec( "shout_dict" );
}

const char *_shout_util_dict_get(util_dict *dict, const char *key)
{
	while (dict) {
		if (dict->key && !strcmp(key, dict->key))
			return dict->val;
		dict = dict->next;
	}

	return NULL;
}

int _shout_util_dict_set(util_dict *dict, const char *key, const char *val)
{
	util_dict *prev;

	if (!dict || !key)
		return SHOUTERR_INSANE;

	prev = NULL;
	while (dict) {
		if (!dict->key || !strcmp(dict->key, key))
			break;
		prev = dict;
		dict = dict->next;
	}

	if (!dict) {
		dict = _shout_util_dict_new();
		if (!dict)
			return SHOUTERR_MALLOC;
		if (prev)
			prev->next = dict;
	}

	if (dict->key)
		debugFree (dict->val);
	else if (!(dict->key = debugStrDup(key))) {
		if (prev)
			prev->next = NULL;
		_shout_util_dict_free (dict);

		return SHOUTERR_MALLOC;
	}

#ifdef __WATCOMC__
	dict->val = val == NULL ? NULL : debugStrDup(val);
#else
	dict->val = debugStrDup(val);
#endif
	if (!dict->val) {
		return SHOUTERR_MALLOC;
	}

	return SHOUTERR_SUCCESS;
}

/* given a dictionary, URL-encode each key and val and stringify them in order as
  key=val&key=val... if val is set, or just key&key if val is NULL.
  TODO: Memory management needs overhaul. */
char *_shout_util_dict_urlencode(util_dict *dict, char delim)
{
#ifdef __WATCOMC__
  util_dict *dict_scan;
  int enc_len = 0;
  int len;
  char *enc;
  char *dst;

  // Calculating the length of the resulting string.

  for( dict_scan = dict; dict_scan != NULL; dict_scan = dict_scan->next )
  {
    if ( dict_scan->key == NULL )
      continue;

    enc_len += strlen( dict_scan->key ) + 1; // key's length + '&'
    if ( dict_scan->val != NULL )
      enc_len += 1 + _url_encode_len( dict_scan->val ); // '=' + value's length
  }

  // enc_len reserved one extra byte - for last '&'. And now we need one byte
  // for terminating zero -- use enc_len as is.
  enc = debugMAlloc( enc_len );
  if ( enc == NULL )
    return NULL;

  // Encoding.

  dst = enc;
  for( dict_scan = dict; dict_scan != NULL; dict_scan = dict_scan->next )
  {
    if ( dict_scan->key == NULL )
      continue;

    if ( dict_scan != dict )
    {
      *(dst++) = '&';
      enc_len--;
    }

    len = strlen( dict_scan->key );
    strcpy( dst, dict_scan->key );
    dst += len;
    enc_len -= len;

    if ( dict_scan->val != NULL )
    {
      *(dst++) = '=';
      enc_len--;
      len = _url_encode( dst, enc_len, dict_scan->val );
      dst += len;
      enc_len -= len;
    }
  }

  return enc;
#else
	char *res, *tmp;
	char *enc;
	int start = 1;

	for (res = NULL; dict; dict = dict->next) {
		/* encode key */
		if (!dict->key)
			continue;
		if (!(enc = _shout_util_url_encode(dict->key))) {
			if (res)
				debugFree(res);
			return NULL;
		}
		if (start) {
			if (!(res = debugMAlloc(strlen(enc) + 1))) {
				debugFree(enc);
				return NULL;
			}
			sprintf(res, "%s", enc);
			debugFree(enc);
			start = 0;
		} else {
			if (!(tmp = debugReAlloc(res, strlen(res) + strlen(enc) + 2))) {
				debugFree(enc);
				debugFree(res);
				return NULL;
			} else
				res = tmp;
			sprintf(res + strlen(res), "%c%s", delim, enc);
			debugFree(enc);
		}

		/* encode value */
		if (!dict->val)
			continue;
		if (!(enc = _shout_util_url_encode(dict->val))) {
			debugFree(res);
			return NULL;
		}

		if (!(tmp = debugReAlloc(res, strlen(res) + strlen(enc) + 2))) {
			debugFree(enc);
			debugFree(res);
			return NULL;
		} else
			res = tmp;
		sprintf(res + strlen(res), "=%s", enc);
		debugFree(enc);
	}

	return res;
#endif
}
