#define INCL_DOSSEMAPHORES   /* Semaphore values */
#define INCL_DOSDATETIME     /* Timer support    */
#define INCL_DOSERRORS       /* DOS error values */
#define INCL_DOSNMPIPES
#include <os2.h>
#include "definitions.h"
#include "setup.h"
#include "log.h"
#include "metadata.h"
#include "cue.h"
#include "sysctrl.h"
#include <debug.h>

#define SEM_BASE_PATH            "\\SEM32\\ICES-CC"
#define PIPE_BASE_PATH           "\\PIPE\\ICES-CC"
#define PIPE_CUE_BUF_SIZE        4094

#ifdef DEBUG_FILE
#  define SEMID_DBGCNT     4
#  define SEMID_PIPECUE    5 // Not index of apszCommands, for pipe's semaphore.
#  define SEM_COUNT        6
#else
#  define SEMID_PIPECUE    4 // Not index of apszCommands, for pipe's semaphore.
#  define SEM_COUNT        5
#endif

// PIPECUE_xxxxxx - values of ulPipeCUEState
#define PIPECUE_NOT_CREATED      0
#define PIPECUE_BUSY             1
#define PIPECUE_LISTEN           2

extern ices_config_t ices_config;
extern int ices_cue_lineno;         // cue.c

static HMUX            hmuxCommands = NULLHANDLE;
static HPIPE           hPipeCUE = NULLHANDLE;
static ULONG           ulPipeCUEState = PIPECUE_NOT_CREATED;

// Commands for ices_sysctrl_command()
static PSZ             apszCommands[] =
{
  "STOP",       // SEMID_STOP
  "NEXT",       // SEMID_NEXT
  "RELOAD",     // SEMID_RELOAD
  "ROTATE"      // SEMID_ROTATE
#ifdef DEBUG_FILE
  ,"DBGCNT"      // SEMID_DBGCNT
#endif
};

static HSEM _createSemaphore(PSZ pszName)
{
  HEV        hevNew;
  ULONG      ulRC;

  ulRC = DosCreateEventSem( pszName, &hevNew, DC_SEM_SHARED, FALSE );
  if ( ulRC != NO_ERROR )
  {
    if ( ulRC == ERROR_DUPLICATE_NAME )
      ices_log_error_output( "Duplicate name for an event semaphore %s "
                             "Does instance %s already launched?",
                             pszName,
                             ices_config.instance_id == NULL
                             ? "(default)"
                             : ices_config.instance_id );
    else
      ices_log_error_output( "Cannot create an event semaphore %s, rc = %u",
                             pszName == NULL ? "(no name)" : pszName, ulRC );
    return NULLHANDLE;
  }

  return (HSEM)hevNew;
}

static LONG _makeInstanceBaseName(PCHAR pcBuf, PSZ pszBase)
{
  return ices_config.instance_id != NULL
           ? _snprintf( pcBuf, CCHMAXPATH - 8, "%s\\%s\\",
                        pszBase, ices_config.instance_id )
           : _snprintf( pcBuf, CCHMAXPATH - 8, "%s\\", pszBase );
}


void ices_sysctrl_initialize(void)
{
  ULONG      ulRC;
  CHAR       szBuf[CCHMAXPATH];
  LONG       cbBase;
  SEMRECORD  aSemRec[SEM_COUNT] = { 0 };
  BOOL       fError = FALSE;
  ULONG      ulIdx;
  HEV        hevPipeCUE = NULLHANDLE;

  // Semaphores.

  cbBase = _makeInstanceBaseName( &szBuf, SEM_BASE_PATH );
  if ( cbBase <= 0 )
    return;

  // Create event semaphores and fill SEMRECORD array.
  for( ulIdx = 0; ulIdx < sizeof(apszCommands) / sizeof(PSZ); ulIdx++ )
  {
    strcpy( &szBuf[cbBase], apszCommands[ulIdx] );
    aSemRec[ulIdx].hsemCur = _createSemaphore( &szBuf );
    if ( aSemRec[ulIdx].hsemCur == 0 )
    {
      fError = TRUE;
      break;
    }
    aSemRec[ulIdx].ulUser = ulIdx;
  }

  if ( !fError )
  {
    // Create named pipe for CUE and event semaphore for this pipe.

    ulPipeCUEState = PIPECUE_NOT_CREATED;
    cbBase = _makeInstanceBaseName( &szBuf, PIPE_BASE_PATH );
    if ( cbBase > 0 )
    do
    {
      strcpy( &szBuf[cbBase], "CUE" );

      ulRC = DosCreateNPipe( &szBuf, &hPipeCUE, NP_NOINHERIT | NP_ACCESS_OUTBOUND,
                             NP_NOWAIT | NP_TYPE_BYTE |NP_READMODE_BYTE | 1,
                             PIPE_CUE_BUF_SIZE, 1024, 0 );
      if ( ulRC != NO_ERROR )  
      {
        ices_log_error_output( "Cannot create named pipe %s, rc = %u",
                               &szBuf, ulRC );
        break;
      }

      ulRC = DosConnectNPipe( hPipeCUE );
      if ( ulRC != NO_ERROR && ulRC != ERROR_PIPE_NOT_CONNECTED )
      {
        ices_log_error_output( "Cannot connect named pipe %s, rc = %u",
                               &szBuf, ulRC );
        break;
      }

      aSemRec[SEMID_PIPECUE].hsemCur = _createSemaphore( NULL );
      if ( aSemRec[SEMID_PIPECUE].hsemCur == 0 )
        break;
      aSemRec[ulIdx].ulUser = SEMID_PIPECUE;

      ulRC = DosSetNPipeSem( hPipeCUE, aSemRec[SEMID_PIPECUE].hsemCur, 1 );
      if ( ulRC != NO_ERROR )
      {
        ices_log_error_output( "DosSetNPipeSem(), rc = %u", ulRC );
        DosCloseEventSem( (HEV)aSemRec[SEMID_PIPECUE].hsemCur );
        break;
      }

      ulPipeCUEState = PIPECUE_LISTEN;
    }
    while( FALSE );

    if ( ulPipeCUEState == PIPECUE_NOT_CREATED && hPipeCUE != NULLHANDLE )
    {
      DosClose( hPipeCUE );
      hPipeCUE = NULLHANDLE;
    }

    // Create a multiple wait (muxwait) semaphore.

    ulRC = DosCreateMuxWaitSem( NULL, &hmuxCommands,
                                hPipeCUE != NULLHANDLE
                                  ? SEM_COUNT : (SEM_COUNT - 1),
                                &aSemRec, DCMW_WAIT_ANY );
    if ( ulRC != NO_ERROR )  
    {
      ices_log_error_output( "Cannot create a multiple wait (muxwait) semaphore,"
                             " rc = %u", ulRC );
      fError = TRUE;
    }
  }

  if ( fError )
  {
    // Error - close created event semaphores and shutdown.

    for( ulIdx = 0; ulIdx < SEM_COUNT; ulIdx++ )
      if ( aSemRec[ulIdx].hsemCur != 0 )
        DosCloseEventSem( (HEV)aSemRec[ulIdx].hsemCur );

    hmuxCommands = NULLHANDLE;
    ices_setup_shutdown();
  }
}

void ices_sysctrl_shutdown(void)
{
  ULONG      ulRC;
  SEMRECORD  aSemRec[SEM_COUNT];
  ULONG      cSemRec = SEM_COUNT;
  ULONG      ulIdx;

  if ( hmuxCommands != NULLHANDLE )
  {
    ulRC = DosQueryMuxWaitSem( hmuxCommands, &cSemRec, &aSemRec, &ulIdx );

    if ( ulRC != NO_ERROR )
      ices_log_debug( "DosQueryMuxWaitSem(), rc = %u", ulRC );
    else
    {
      for( ulIdx = 0; ulIdx < cSemRec; ulIdx++ )
      {
        ulRC = DosCloseEventSem( (HEV)aSemRec[ulIdx].hsemCur );
        if ( ulRC != NO_ERROR )
          ices_log_debug( "DosCloseEventSem(), rc = %u", ulRC );
      }
    }

    DosCloseMuxWaitSem( hmuxCommands );
    hmuxCommands = NULLHANDLE;
  }

  if ( hPipeCUE != NULLHANDLE )
  {
    ulRC = DosClose( hPipeCUE );
    if ( ulRC != NO_ERROR )  
      ices_log_error_output( "Pipe close, rc = %u", ulRC );

    hPipeCUE = NULLHANDLE;
  }
}

unsigned long ices_sysctrl_check(unsigned long timeout, input_stream_t* source)
{
  ULONG      ulRC;
  ULONG      ulSemId = (ULONG)(-1);
  ULONG      ulState;

  if ( ( source != NULL ) && ( hPipeCUE != NULLHANDLE ) &&
       ( ulPipeCUEState == PIPECUE_LISTEN ) )
  {
    // Does client connected to the pipe?

    ulRC = DosConnectNPipe( hPipeCUE );
    if ( ulRC == NO_ERROR )
    {
      // Ok, client connected to the pipe. We can send a data.
      CHAR             szArtist[255];
      CHAR             szTitle[255];
      CHAR             szFileTime[32];
      PSZ              pszBuf;

      szArtist[0] = '\0';
      szTitle[0]  = '\0';
      ices_metadata_get( &szArtist, sizeof(szArtist), &szTitle, sizeof(szTitle) );

      // Allocate buffer for reply text.
      pszBuf = debugMAlloc( PIPE_CUE_BUF_SIZE );
      if ( pszBuf == NULL )
      {
        ices_log_error_output( "Not enough memory" );
        DosDisConnectNPipe( hPipeCUE );
      }
      else
      {
        // Make reply text.
        LONG            cbBytes =
          _snprintf( pszBuf, PIPE_CUE_BUF_SIZE,
                     "%s\n%d\n%d\n%s\n%f\n%d\n%s\n%s\n",
                     source->path, (int)source->filesize, source->bitrate,
                     ices_util_file_time( source->bitrate, source->filesize,
                                          &szFileTime ),
                     ices_util_percent( source->bytes_read, source->filesize ),
                     ices_cue_lineno, &szArtist, &szTitle );

        // Write reply to the named pipe.
        ulRC = DosWrite( hPipeCUE, pszBuf,
                         cbBytes < 0 ? PIPE_CUE_BUF_SIZE : cbBytes, &ulState );
        // Free reply text buffer.
        debugFree( pszBuf );

        if ( ulRC != NO_ERROR )
        {
          ices_log_error_output( "DosWrite(), rc = %u", ulRC );
          ulRC = DosDisConnectNPipe( hPipeCUE );
          if ( ulRC != NO_ERROR )
            ices_log_error_output( "DosDisConnectNPipe(), rc = %u", ulRC );
        }
        else
          // Now we will wait while client closes pipe (semaphore SEMID_PIPECUE).
          ulPipeCUEState = PIPECUE_BUSY;
      }
    }
  }

  // Wait any of semaphores.
  ulRC = DosWaitMuxWaitSem( hmuxCommands, timeout, &ulSemId );
  if ( ulRC != ERROR_TIMEOUT )
  {
    SEMRECORD          aSemRec[SEM_COUNT];
    ULONG              cSemRec = SEM_COUNT;
    ULONG              ulAttr;

    if ( ulRC == ERROR_INTERRUPT )  
    {
      debug( "Interrupt signal" );
      return SEMID_STOP;
    }
      
    if ( ulRC != NO_ERROR )  
      ices_log_error_output( "Wait for a muxwait semaphore: error %u", ulRC );
    else
    {
      // Query list of all semaphores.
      ulRC = DosQueryMuxWaitSem( hmuxCommands, &cSemRec, &aSemRec, &ulAttr );
      if ( ulRC != NO_ERROR )
        ices_log_debug( "DosQueryMuxWaitSem(), rc = %u", ulRC );
      else if ( ulSemId != SEMID_STOP )
      {
        // Reset posted semaphore (except semaphore "stop").
        ulRC = DosResetEventSem( (HEV)aSemRec[ulSemId].hsemCur, &ulAttr );
        if ( ulRC != NO_ERROR )
          ices_log_debug( "DosResetEventSem(), rc = %u", ulRC );
      }

      switch( ulSemId )
      {
        case SEMID_STOP:
          debug( "An event semaphore STOP posted" );
/*          ices_log_debug( "An event semaphore STOP posted, shutting down..." );
          ices_setup_shutdown();*/
          break;

        case SEMID_NEXT:
          ices_log_debug( "An event semaphore NEXT posted, "
                          "skipping to next track...");
          ices_stream_next();
          break;

        case SEMID_RELOAD:
          ices_log_debug( "An event semaphore RELOAD posted, "
                          "Reloading playlist..." );
          ices_playlist_reload();
          break;

        case SEMID_ROTATE:
          ices_log_debug( "An event semaphore ROTATE posted, "
                          "Rotate log files..." );
          ices_log_reopen_logfile();
          break;
#ifdef DEBUG_FILE
        case SEMID_DBGCNT:
          debugStat();
          break;
#endif
       case SEMID_PIPECUE:
          // Client closes pipe - disconnect and back to listen pipe again.

          if ( ulPipeCUEState == PIPECUE_BUSY )
          {
            ulRC = DosDisConnectNPipe( hPipeCUE );
            if ( ulRC != NO_ERROR )
              ices_log_error_output( "DosDisConnectNPipe(), rc = %u", ulRC );

            ulPipeCUEState = PIPECUE_LISTEN;
          }
          break;
      }
    }
  }

  return ulSemId;
}

// Post shared semaphore related with command cmd.
void ices_sysctrl_command(char *cmd)
{
  ULONG      ulRC;
  ULONG      ulIdx;
  BOOL       fFound = FALSE;
  CHAR       szBuf[CCHMAXPATH];
  LONG       cbBase;
  PSZ        pszName;
  HEV        hevSem = NULLHANDLE;

  if ( stricmp( cmd, "CUE" ) == 0 )
  {
    HPIPE              hPipeCUE;
    CHAR               acMessage[PIPE_CUE_BUF_SIZE];

    // Make name for named pipe.
    cbBase = _makeInstanceBaseName( &szBuf, PIPE_BASE_PATH );
    if ( cbBase <= 0 )
      return;
    strcpy( &szBuf[cbBase], "CUE" );

    // Wait for the named pipe to become available. 
    ulRC = DosWaitNPipe( &szBuf, 500 );
    if ( ulRC != NO_ERROR )
    {
      if ( ulRC == ERROR_FILE_NOT_FOUND || ulRC == ERROR_PATH_NOT_FOUND )
        ices_log_error_output( "No pipe. Does instance %s launched?",
                               ices_config.instance_id == NULL
                               ? "(default)"
                               : ices_config.instance_id );
      else
        ices_log_error_output( "DosWaitNPipe(), rc = %u", ulRC );

      return;
    }

    // Open named pipe and read data.

    ulRC = DosOpen( &szBuf, &hPipeCUE, &ulIdx, 0, FILE_NORMAL, FILE_OPEN,
                    OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE, NULL );
    if ( ulRC != NO_ERROR )
    {
      ices_log_error_output( "DosOpen(), rc = %u", ulRC );
      return;
    }

    ulRC = DosRead( hPipeCUE, &acMessage, sizeof(acMessage) - 1, &ulIdx );
    if ( ulRC != NO_ERROR )
      ices_log_error_output( "DosRead(), rc = %u", ulRC );
    else
    {
      // Print obtained data.
      acMessage[ulIdx] = '\0';
      puts( &acMessage );
    }

    DosClose( hPipeCUE );
    return;
  }

  for( ulIdx = 0; ulIdx < sizeof(apszCommands) / sizeof(PSZ); ulIdx++ )
  {
    if ( stricmp( apszCommands[ulIdx], cmd ) == 0 )
    {
      fFound = TRUE;
      break;
    }
  }

  if ( !fFound )
  {
    ices_log_error_output( "Unknown command \"%s\"", cmd );
    return;
  }

  // Make "base" name (path) for semaphore.
  cbBase = _makeInstanceBaseName( &szBuf, SEM_BASE_PATH );
  if ( cbBase <= 0 )
    return;

  // Add name of semaphore for given command.
  strcpy( &szBuf[cbBase], apszCommands[ulIdx] );

  // Post semaphore.
  ulRC = DosOpenEventSem( &szBuf, &hevSem );
  if ( ulRC != NO_ERROR )
  {
    if ( ulRC == ERROR_SEM_NOT_FOUND  )
      ices_log_error_output( "Semaphore does not exist. Does instance %s "
                             "launched?",
                             ices_config.instance_id == NULL ?
                               "(default)" : ices_config.instance_id );
    else
      ices_log_error_output( "DosOpenEventSem(), rc = %u", ulRC );
  }
  else
  {
    ulRC = DosPostEventSem( hevSem );
    DosCloseEventSem( ulRC );
    ices_log_error_output( "Ok." );
  }
}

// Post STOP semaphore for _this_ process.
void ices_sysctrl_post(unsigned long ulSemaphoreNumber)
{
  ULONG      ulAttr;
  ULONG      cSemRec;
  SEMRECORD  aSemRec[ sizeof(apszCommands) / sizeof(apszCommands[0]) ];
  ULONG      ulRC;

  do
  {
    ulRC = DosQueryMuxWaitSem( hmuxCommands, &cSemRec, aSemRec, &ulAttr );
    if ( ulRC != NO_ERROR )
    {
      debug( "DosQueryMuxWaitSem(), rc = %u", ulRC );
      break;
    }

    if ( ulSemaphoreNumber >= cSemRec )
    {
      debug( "Invalid semaphore number" );
      break;
    }

    ulRC = DosPostEventSem( (HEV)aSemRec[ulSemaphoreNumber].hsemCur );
    if ( ulRC != NO_ERROR )
    {
      debug( "DosPostEventSem(), rc = %u", ulRC );
      break;
    }

    return;
  }
  while( FALSE );

  ices_log_error_output( "Post an event semaphore failed." );
}
