/*
   [Digi] A new log module for ices.
*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h> 
#include <sys/stat.h> 
#include <fcntl.h>
#include <io.h>
#include <time.h>
#ifdef __NT__
#include <malloc.h>
#include <windows.h>
#endif
#include "definitions.h"
#include "xpl.h"

// It seems Open Watcom ignores access mode O_NOINHERIT for open().
#ifdef __OS2__
#define WATCOM_NOINHERIT_BUG
#endif

#ifdef WATCOM_NOINHERIT_BUG
# define INCL_DOSFILEMGR
# define INCL_DOSERRORS
# include <os2.h>
#endif
#include <debug.h>

// OPENLOG_xxxxxx - flags for _openLogFile()
#define OPENLOG_NEW      (O_WRONLY | O_CREAT | O_TRUNC | O_TEXT | O_NOINHERIT)
#define OPENLOG_APPEND   (O_WRONLY | O_CREAT | O_APPEND | O_TEXT | O_NOINHERIT)

// RECTYPE_xxx - types of record for _logString()
#define RECTYPE_MESSAGE          ((unsigned long)' GSM')
#define RECTYPE_ERROR            ((unsigned long)' RRE')
#define RECTYPE_DEBUG            ((unsigned long)' GBD')

extern ices_config_t ices_config;

static int             hLogFile = -1;
static CHAR            szError[1024];
static HMTX            hmtxLog;

// int _makeBaseFilename(char *pcBuf)
//
// Makes log file name without extension, like D:\ices\ices_myinstance.
// Returns length of string or negative value if result string too long.

static int _makeBaseFilename(char *pcBuf)
{
  int        iLen;

  if ( ices_config.instance_id == NULL )
    iLen = _snprintf( pcBuf, _MAX_PATH - 3, "%s\\ices.",
                      ices_config.base_directory );
  else
    iLen = _snprintf( pcBuf, _MAX_PATH - 3, "%s\\ices_%s.",
                      ices_config.base_directory, ices_config.instance_id );

  return iLen;
}

// int _openLogFile(int iFlags)
//
// Opens log file. iFlags: OPENLOG_NEW or OPENLOG_APPEND
// Returns 1 on success and 0 when error occurs.

static int _openLogFile(int iFlags)
{
  char       szName[_MAX_PATH];
  int        cbName;
  char       szBuf[256];
  int        hNewFile;

  if ( ices_config.log_max_size == 0 || ices_config.log_rotate == 0 )
    // No logs...
    return 1;

  // Make file name without extension.
  cbName = _makeBaseFilename( &szName );
  if ( cbName <= 0 )
  {
    ices_log_error_output( "Too long name for the logfile" );
    return 0;
  }

  // Append extension to the file name.
  strcpy( &szName[cbName], "log" );

  // Open/create file.

#ifndef WATCOM_NOINHERIT_BUG
  hNewFile = open( &szName, iFlags, S_IREAD | S_IWRITE );
#else
  {
    // It seems Open Watcom ignores access mode O_NOINHERIT. We open file
    // via system API with open mode OPEN_FLAGS_NOINHERIT.
    // We must use O_NOINHERIT to avoid keeps file opened after close() with
    // external programs (playlist program). It not impossible rename log file
    // while rotation without O_NOINHERIT while child process runned.

    HFILE       hFile;
    ULONG       ulRC;
    ULONG       ulAction;

    ulRC = DosOpen( &szName, &hFile, &ulAction, 0, FILE_NORMAL,

                    iFlags == OPENLOG_NEW
                    ? OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS
                    : OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,

                    OPEN_FLAGS_FAIL_ON_ERROR | OPEN_SHARE_DENYNONE |
                    OPEN_ACCESS_WRITEONLY | OPEN_FLAGS_NOINHERIT, NULL );
    if ( ulRC != NO_ERROR )
    {
      ices_log_error_output( "DosOpen(%s,,,,,,,) rc = %u", &szName, ulRC );
      return 0;
    }

    hNewFile = _open_osfhandle( hFile, iFlags );
  }
#endif

	if ( hNewFile == -1 )
  {
    ices_log_error_output( "Error while opening %s: %s", &szName,
                           ices_util_strerror( errno, &szBuf, sizeof(szBuf) ) );
		return 0;
	}

  hLogFile = hNewFile;
  return 1;
}

static BOOL _reopenLogFile()
{
  char       szName[_MAX_PATH];
  char       szNewName[_MAX_PATH];
  char       *pcSlash;
  int        cbName;
  int        cbNewName;
  int        iIdx;
  char       szBuf[256];

  if ( hLogFile == -1 || ices_config.log_rotate == 0 )
    // No logs...
    return TRUE;

  close( hLogFile );
  hLogFile = -1;

  if ( ices_config.log_rotate > 1 )
  {
    // Rotate log files.

    cbName = _makeBaseFilename( &szName );
    if ( cbName <= 0 )
      return FALSE;

    pcSlash = strrchr( &szName, '\\' );
    strcpy( &szNewName, pcSlash + 1 );
    cbNewName = strlen( &szNewName );

    // Remove oldest log file.
    sprintf( &szName[cbName], "%.2d", ices_config.log_rotate - 1 );
    unlink( &szName );

    if ( ices_config.log_rotate > 2 )
    {
      // Rename "archived" logs *.01 to *.02, *.02 to *03, e.t.c.

      for( iIdx = ices_config.log_rotate - 2; iIdx > 0; iIdx-- )
      {
        sprintf( &szName[cbName], "%.2d", iIdx );
        sprintf( &szNewName[cbNewName], "%.2d", iIdx + 1 );
        rename( &szName, &szNewName );
      }
    }

    // Rename current log file to "archived" name *.01
    strcpy( &szName[cbName], "log" );
    strcpy( &szNewName[cbNewName], "01" );
    if ( rename( &szName, &szNewName ) != 0 )
      ices_log_error_output( "Cannot rename %s to %s: %s", &szName, &szNewName,
                          ices_util_strerror( errno, &szBuf, sizeof(szBuf) ) );
  }

  // Start new log file.
  _openLogFile( OPENLOG_NEW );

  return TRUE;
}

static VOID _checkLogSize()
{
  if ( hLogFile == -1 )
    return;

  if ( filelength( hLogFile ) >= ices_config.log_max_size )
    _reopenLogFile();
}

static void	_logString(ULONG ulType, PSZ pszRecord)
{
  xplMutexLock( hmtxLog, XPL_INDEFINITE_WAIT );

  if ( hLogFile != NULL )
  {
    CHAR       szBuf[32];
    time_t	   t = time( NULL );
    ULONG      cbBuf = strftime( &szBuf, sizeof(szBuf) - 1, "%F %T ",
                                 localtime( &t ) );
    ULONG      cbRecord = strlen( pszRecord );
    
    // Remove trailing end-of-line, tab and space characters.
    while( ( cbRecord > 0 ) && isspace( pszRecord[cbRecord - 1] ) )
      cbRecord--;

    write( hLogFile, &szBuf, cbBuf );                   // Timestamp.
    write( hLogFile, &ulType, 4 );                      // Type.
    write( hLogFile, pszRecord, cbRecord );             // Text.
    write( hLogFile, "\n", 1 );                         // New line.
    _checkLogSize();
  }

  {
#ifdef __NT__
    // Convert message to the Windows console codepage.
    int        cbRecord = strlen( pszRecord );
    char       *pcOEMStr = alloca( cbRecord + 1 );
    
    if ( pcOEMStr != NULL )
    {
      CharToOemBuff( pszRecord, pcOEMStr, cbRecord );
      pcOEMStr[cbRecord] = '\0';
      pszRecord = pcOEMStr;
    }
#endif
    puts( pszRecord );
  }

  xplMutexUnlock( hmtxLog );
}



void ices_log_initialize(void)
{
  if ( _openLogFile( OPENLOG_APPEND ) )
  {
    // Rotate logs if file size exceeds the specified value.
    _checkLogSize();
  }

  xplMutexCreate( &hmtxLog, FALSE );
  if ( hmtxLog == NULLHANDLE )
    debug( "xplMutexCreate() failed" );
}

void ices_log_shutdown(void)
{
  close( hLogFile );
  hLogFile = -1;

  xplMutexDestroy( hmtxLog );
}

// void ices_log(const char *fmt, ...)
//
// Log to console and file.

void ices_log(const char *fmt, ...)
{
	char       szBuf[1024];
	va_list    args;
  
	va_start( args, fmt );
	vsnprintf( &szBuf, sizeof(szBuf), fmt, args );
	va_end( args );

	_logString( RECTYPE_MESSAGE, &szBuf );
}

// void ices_log_error(const char *fmt, ...)
//
// Store error information in module memory. Thread unsafe!

void ices_log_error(const char *fmt, ...)
{
	va_list    args;

	va_start( args, fmt );
	vsnprintf( &szError, sizeof(szError), fmt, args );
	va_end( args );
}

// char *ices_log_get_error(void)
//
// Get last error from log module.

char *ices_log_get_error(void)
{
	return &szError;
}

// void ices_log_error_output(const char *fmt, ...)
//
// (Store error information in module memory and) write to log.

void ices_log_error_output(const char *fmt, ...)
{
	va_list    args;
#if 1
  // Do not set global error message.
	char       szError[1024];
#endif

	va_start( args, fmt );
	vsnprintf( &szError, sizeof(szError), fmt, args );
	va_end( args );

	_logString( RECTYPE_ERROR, &szError );
}

// void ices_log_debug(const char *fmt, ...)
//
// Log only if verbose mode is set. Prepend output with DEBUG:

void ices_log_debug(const char *fmt, ...)
{
	char       szBuf[1024];
	va_list    args;
  
	if ( !ices_config.verbose )
		return;

//  strcpy( &szBuf, "DEBUG: " );
szBuf[0] = '\0';
	va_start( args, fmt );
	_vsnprintf( strchr( &szBuf, '\0' ), sizeof(szBuf) - 7, fmt, args );
	va_end( args );

	_logString( RECTYPE_DEBUG, &szBuf );
}

// int ices_log_reopen_logfile(void)
//
// Rotates log files (not reopen for our version of ices).
// Returns !=0 on success or 0 on error. Btw, ices don't uses return value.

int ices_log_reopen_logfile(void)
{
  BOOL       fSuccess;

  xplMutexLock( hmtxLog, XPL_INDEFINITE_WAIT );
  fSuccess = _reopenLogFile();
  xplMutexUnlock( hmtxLog );

  return (int)fSuccess;
}
