/* im_dart.c
 * - Raw PCM input from OS/2 DART
 *
 * by Vasilkin Andrey <digi@os2.snc.ru>
 *
 * This program is distributed under the terms of the GNU General
 * Public License, version 2. You may use, modify, and redistribute
 * it under the terms of this license. A copy should be included
 * with this source.
 */

#ifdef HAVE_CONFIG_H
 #include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#include "cfgparse.h"
#include "stream.h"
#include "metadata.h"
#include "inputmodule.h"

#define INCL_OS2MM  // for os2.h included at im_dart.h -> ./os2/thread/thread.h
#include "im_dart.h"
#include <os2me.h>

#define MODULE "input-dart/"
#include "logging.h"


static void close_module(input_module_t *mod)
{
  if ( mod != NULL )
  {
    if ( mod->internal != NULL )
    {
      im_dart_state              *s = mod->internal;
      MCI_GENERIC_PARMS	         mci_generic = { 0 };

      mciSendCommand( s->id, MCI_STOP, MCI_WAIT, &mci_generic, 0 );

      mciSendCommand( s->id, MCI_BUFFER, MCI_WAIT | MCI_DEALLOCATE_MEMORY,
                      &s->mci_buffer, 0 );

      mciSendCommand( s->id, MCI_RELEASEDEVICE, MCI_NOTIFY, &mci_generic, 0 );

      mciSendCommand( s->id, MCI_CLOSE, MCI_WAIT, &mci_generic, 0 );

      if ( s->mci_buffer.pBufList != NULL )
        free( s->mci_buffer.pBufList );

      thread_mutex_destroy( &s->mtx_metadata );
      thread_mutex_destroy( &s->mtx_readycnt );
      thread_cond_destroy( &s->ev_nextblk );

      free( s );
    }
    free( mod );
  }
}

static int event_handler(input_module_t *mod, enum event_type ev, void *param)
{
  im_dart_state *s = mod->internal;

  switch( ev )
  {
    case EVENT_SHUTDOWN:
      close_module( mod );
      break;

    case EVENT_NEXTTRACK:
      s->newtrack = 1;
      break;

    case EVENT_METADATAUPDATE:
      thread_mutex_lock( &s->mtx_metadata );

      if( s->metadata )
      {
        char **md = s->metadata;

        while( *md != NULL )
          free( *md++ );

        free( s->metadata );
      }
      s->metadata = (char **)param;
      s->newtrack = 1;

      thread_mutex_unlock( &s->mtx_metadata );
      break;

    default:
      LOG_WARN1( "Unhandled event %d", ev );
      return -1;
  }

  return 0;
}

static void metadata_update(void *self, vorbis_comment *vc)
{
  im_dart_state        *s = self;
  char                 **md;

  thread_mutex_lock(&s->mtx_metadata);

  md = s->metadata;

  if(md)
  {
      while(*md)
          vorbis_comment_add(vc, *md++);
  }

  thread_mutex_unlock(&s->mtx_metadata);
}

static LONG APIENTRY mix_read_handler(ULONG status, PMCI_MIX_BUFFER pmci_mix_buf,
                                      ULONG flags)
{
  im_dart_state        *s;

  if ( flags != MIX_READ_COMPLETE )
    return 0;

  s = (im_dart_state *)pmci_mix_buf->ulUserParm;
  s->mci_mix_setup.pmixRead( s->mci_mix_setup.ulMixHandle, pmci_mix_buf, 1 );

  thread_mutex_lock( &s->mtx_readycnt );
  s->readycnt++;
  thread_mutex_unlock( &s->mtx_readycnt );
  thread_cond_broadcast( &s->ev_nextblk );

  return 1;
}

/* Core streaming function for this module
 * This is what actually produces the data which gets streamed.
 *
 * returns:  >0  Number of bytes read
 *            0  Non-fatal error.
 *           <0  Fatal error.
 */
static int dart_read(void *self, ref_buffer *rb)
{
  int            result;
  im_dart_state  *s = self;

  rb->buf = malloc( s->mci_buffer.ulBufferSize );
  if ( !rb->buf )
    return -1;

  while( 1 )
  {
    thread_mutex_lock( &s->mtx_readycnt );
    if ( s->readycnt > 0 )
    {
      s->readycnt--;
      thread_mutex_unlock( &s->mtx_readycnt );
      break;
    }
    thread_mutex_unlock( &s->mtx_readycnt );
    thread_cond_wait( &s->ev_nextblk );
  }

  memcpy( rb->buf,
          ((MCI_MIX_BUFFER *)(s->mci_buffer.pBufList))[s->curbuf].pBuffer,
          s->mci_buffer.ulBufferSize );

  s->curbuf++;
  if ( s->curbuf == s->mci_buffer.ulNumBuffers )
    s->curbuf = 0;

  rb->len = s->mci_buffer.ulBufferSize;
  rb->aux_data = s->mci_mix_setup.ulSamplesPerSec *
                 s->mci_mix_setup.ulChannels * 2;
  if ( s->newtrack )
  {
    rb->critical = 1;
    s->newtrack = 0;
  }

  return rb->len;
}

input_module_t *dart_open_module(module_param_t *params)
{
  input_module_t       *mod = calloc( 1, sizeof(input_module_t) );
  im_dart_state        *s;
  module_param_t       *current;
  int                  device = 0;       // default device 
  int                  use_metadata = 1; // Default to On
  int                  rc;
  char                 *errstr = 0;
  int                  monitor = 1;      // By default, monitor is On
  int                  volume = 100;     // Default volume - 100%
  char                 buf[128];
  MCI_AMP_OPEN_PARMS   mci_amp_open = { 0 };
  MCI_GENERIC_PARMS    mci_generic = { 0 };
  MCI_CONNECTOR_PARMS  mci_connector = { 0 };
  MCI_AMP_SET_PARMS    mci_amp_set = { 0 };

  mod->type            = ICES_INPUT_PCM;
  mod->subtype         = INPUT_PCM_LE_16;
  mod->getdata         = dart_read;
  mod->handle_event    = event_handler;
  mod->metadata_update = metadata_update;

  mod->internal = calloc( 1, sizeof(im_dart_state) );
  s = mod->internal;

  s->mci_mix_setup.ulBitsPerSample = 16;
  s->mci_mix_setup.ulSamplesPerSec = 44100;
  s->mci_mix_setup.ulFormatTag     = MCI_WAVE_FORMAT_PCM;
  s->mci_mix_setup.ulChannels      = 2;
  s->mci_mix_setup.ulFormatMode    = MCI_RECORD;
  s->mci_mix_setup.ulDeviceType    = MCI_DEVTYPE_WAVEFORM_AUDIO;
  s->mci_mix_setup.pmixEvent       = mix_read_handler;
  mci_connector.ulConnectorType    = MCI_LINE_IN_CONNECTOR;

  thread_mutex_create( &s->mtx_metadata );
  thread_mutex_create( &s->mtx_readycnt );
  thread_cond_create( &s->ev_nextblk );

  current = params;

  // Set parameters.

  while( current )
  {
    if ( strcmp( current->name, "rate" ) == 0 )
      s->mci_mix_setup.ulSamplesPerSec = atoi( current->value );
    else if ( strcmp( current->name, "channels" ) == 0 )
      s->mci_mix_setup.ulChannels = atoi( current->value );
    else if ( strcmp( current->name, "device" ) == 0 )
      device = atoi( current->value );
    else if( strcmp( current->name, "mic" ) == 0 )
      mci_connector.ulConnectorType = atoi( current->value )
                                      ? MCI_MICROPHONE_CONNECTOR
                                      : MCI_LINE_IN_CONNECTOR;
    else if( strcmp( current->name, "monitor" ) == 0 )
      monitor = atoi( current->value );
    else if( strcmp(current->name, "metadata") == 0 )
      use_metadata = atoi( current->value );
    else if( strcmp( current->name, "metadatafilename" ) == 0 )
      ices_config->metadata_filename = current->value;
    else if( strcmp( current->name, "input-level" ) == 0 )
      volume = atoi( current->value );
    else
      LOG_WARN1( "Unknown parameter %s for dart module", current->name );

    current = current->next;
  }

  // Open audio device.
  mci_amp_open.pszDeviceType = (PSZ)MAKEULONG(MCI_DEVTYPE_AUDIO_AMPMIX, device);
  rc = mciSendCommand( 0, MCI_OPEN, MCI_WAIT | MCI_OPEN_TYPE_ID,
                       &mci_amp_open, 0 );

  if ( ULONG_LOWD(rc) != MCIERR_SUCCESS )
  {
      errstr = "Failed to open audio device";
      goto fail_log;
  }

  s->id = mci_amp_open.usDeviceID;

  // Set particular connector.
  mciSendCommand( mci_amp_open.usDeviceID, MCI_CONNECTOR,
                  MCI_WAIT | MCI_ENABLE_CONNECTOR |
                  MCI_CONNECTOR_TYPE, &mci_connector, 0 );

  // Enable/disable monitor.
  mci_amp_set.ulItem = MCI_AMP_SET_MONITOR;
  mciSendCommand( mci_amp_open.usDeviceID, MCI_SET,
                  monitor
                  ? MCI_WAIT | MCI_SET_ON | MCI_SET_ITEM
                  : MCI_WAIT | MCI_SET_OFF | MCI_SET_ITEM,
                  &mci_amp_set, 0 );

  // Ser record volume.
  mci_amp_set.ulAudio = MCI_SET_AUDIO_ALL;
  mci_amp_set.ulLevel = volume;
  mciSendCommand( mci_amp_open.usDeviceID, MCI_SET,
                  MCI_WAIT | MCI_SET_AUDIO | MCI_AMP_SET_GAIN,
                  &mci_amp_set, 0 );

  mciSendCommand( mci_amp_open.usDeviceID, MCI_ACQUIREDEVICE,
                  MCI_EXCLUSIVE_INSTANCE, &mci_generic, 0 );

  // Inform the mixer device that we want to read buffers directly and set up
  // the device in the correct mode.
  rc = mciSendCommand( mci_amp_open.usDeviceID, MCI_MIXSETUP,
                       MCI_WAIT | MCI_MIXSETUP_INIT, &s->mci_mix_setup, 0 );

  if ( ULONG_LOWD(rc) != MCIERR_SUCCESS )
  {
    errstr = "Error setting HW params";
    goto fail_log;
  }

  // Allocate buffers.
  s->mci_buffer.ulNumBuffers = 5;//s->mci_mix_setup.ulNumBuffers;
  s->mci_buffer.ulBufferSize = 1024*32;//s->mci_mix_setup.ulBufferSize;
  s->mci_buffer.pBufList =
      (PVOID)malloc( s->mci_buffer.ulNumBuffers * sizeof(MCI_MIX_BUFFER) );

  for( rc = 0; rc < s->mci_buffer.ulNumBuffers; rc++ )
  {
    ((MCI_MIX_BUFFER *)(s->mci_buffer.pBufList))[rc].ulUserParm = (ULONG)s;
    ((MCI_MIX_BUFFER *)(s->mci_buffer.pBufList))[rc].ulBufferLength =
      s->mci_buffer.ulBufferSize;
  }

  rc = mciSendCommand( mci_amp_open.usDeviceID, MCI_BUFFER,
                       MCI_WAIT | MCI_ALLOCATE_MEMORY, &s->mci_buffer, 0 );

  if ( ULONG_LOWD(rc) != MCIERR_SUCCESS )
  {
      errstr = "Dart buffers allocate error";
      goto fail_log;
  }

  // Start the record.
  rc = s->mci_mix_setup.pmixRead( s->mci_mix_setup.ulMixHandle,
                                  (MCI_MIX_BUFFER *)(s->mci_buffer.pBufList),
                                  s->mci_buffer.ulNumBuffers );

  if ( ULONG_LOWD(rc) != MCIERR_SUCCESS )
  {
    errstr = "Failed to start record";
    goto fail_log;
  }

  /* We're done, and we didn't fail! */
  LOG_INFO1( "Opened audio device %d", device );
  LOG_INFO2( "using %d channel(s), %d Hz ",
             s->mci_mix_setup.ulChannels, s->mci_mix_setup.ulSamplesPerSec );

  if ( use_metadata )
  {
    LOG_INFO0( "Starting metadata update thread" );
    if ( ices_config->metadata_filename != NULL )
      thread_create( "im_dart-metadata", metadata_thread_signal, mod, 1 );
/*    else
      thread_create( "im_dart-metadata", metadata_thread_stdin, mod, 1 );*/
  }

  return mod;

fail_log:
  mciGetErrorString( rc, (PSZ)&buf, sizeof(buf) );
  LOG_ERROR2( "%s: %s", errstr, &buf );
//fail:
  close_module( mod ); /* safe, this checks for valid contents */
  return NULL;
}
