/*
 *      IPF2HTEXT.CMD - HyperText/2 - V2.01 - A. Schnellbacher 2007 - 2009
 *
 *    Syntax: ipf2htext <ipf_file> [<htext_file>] [/Aps:<aps_file>]
 *
 *    By default, ipf2htext creates an HTEXT file with the same name as
 *    the IPF file, but with the extension ".htext". Optionally you can
 *    specify the HTEXT file to override the default name (and directory).
 *
 *    The APS file is required to resolve IPF symbols. When not specified,
 *    the default APS filename "apsymbol.aps" is searched. It is searched
 *    in the current path, in the path of this CMD file and in the value of
 *    the IPFC and DPATH environment variables.
 */
/* The first comment is used as online help text */

 SIGNAL ON HALT NAME HALT;

 TitleLine = STRIP( SUBSTR( SourceLine(2), 3));
 PARSE VAR TitleLine CmdName'.CMD 'Info;
 Title = CmdName Info;
 env       = 'OS2ENVIRONMENT';
 TRUE      = (1 = 1);
 FALSE     = (0 = 1);
 CrLf      = '0d0a'x;
 Redirection = '1>NUL 2>&1';
 '@ECHO OFF';
 rcx = SETLOCAL();

 /* some OS/2 Error codes */
 ERROR.NO_ERROR           =   0;
 ERROR.INVALID_FUNCTION   =   1;
 ERROR.FILE_NOT_FOUND     =   2;
 ERROR.PATH_NOT_FOUND     =   3;
 ERROR.ACCESS_DENIED      =   5;
 ERROR.NOT_ENOUGH_MEMORY  =   8;
 ERROR.INVALID_FORMAT     =  11;
 ERROR.INVALID_DATA       =  13;
 ERROR.NO_MORE_FILES      =  18;
 ERROR.WRITE_FAULT        =  29;
 ERROR.READ_FAULT         =  30;
 ERROR.SHARING_VIOLATION  =  32;
 ERROR.GEN_FAILURE        =  31;
 ERROR.INVALID_PARAMETER  =  87;
 ERROR.ENVVAR_NOT_FOUND   = 204;

 GlobalVars = 'Title CmdName env TRUE FALSE CrLf Redirection ERROR.';

 /* load RexxUtil */
 CALL RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs';
 CALL SysLoadFuncs;

 GlobalVars = GlobalVars 'Flag. LineCount ListIndent ListDepth fLinkOpen fTagOpen' ||,
                         ' ThisLine Link LastLine fXmpOpen DocState' ||,
                         ' IPFC. HTEXT. SourceFile fSingleNoteOpen' ||,
                         ' fNewRow Pages. ResId. RefId. ThisArtWork' ||,
                         ' ThisArtLink fArtLinkOpen Bitmap. LastIndex' ||,
                         ' LinkDataMark';

 rc                = ERROR.NO_ERROR;
 Flag.fHelp        = FALSE;
 SourceFile        = '';
 TargetFile        = '';
 LinkDataMark      = '010101'x;  /* for to differ between link data and link text */

 ListIndent        = 2;  /* number of spaces */
 HTEXT._EscapeChars = '_*#[]';
 HTEXT._Urls        = 'HTTP HTTPS FTP MAILTO IRC';
 /* The following string is used for an HTEXT .HRULE command: */
 HTEXT._HruleLine   = COPIES( '_', 80);
 IPFC._EnvVar      = 'IPFC';
 IPFC._ApsFile     = '';
 IPFC._DefaultApsFile    = 'apsymbol.aps';  /* default symbol translation table */
 IPFC._fApsFileProcessed = FALSE;
 IPFC._ApsLine.0   = 0;
 IPFC._StartTags   = 'acviewport artlink artwork caution cgraphic color ctrl ctrldef' ||,
                     ' ddf dl dthd ddhd dt dd docprof fig figcap font fn h1 h2 h3 h4 h5 h6' ||,
                     ' hdref hide hp1 hp2 hp3 hp4 hp5 hp6 hp7 hp8 hp9 i1 i2 icmd isyn' ||,
                     ' li lines link lm lp note nt ol p parml pbutton pd pt rm sl' ||,
                     ' table row c title ul userdoc warning xmp';
 IPFC._EndTags     = 'eartlink ecaution ecgraphic ectrldef' ||,
                     ' edl edocprof efig efn' ||,
                     ' ehide ehp1 ehp2 ehp3 ehp4 ehp5 ehp6 ehp7 ehp8 ehp9' ||,
                     ' elines elink ent eol eparml esl' ||,
                     ' etable eul euserdoc ewarning exmp';
 /* These tags require following text after the dot closing the tag; */
 /* text may be specified on the following lines:                     */
 IPFC._RequireTextTags = 'h1 h2 h3 h4 h5 h6 title li dthd ddhd dt dd pt pd c'
 ThisLine          = '';  /* holds unwritten IPF data (until the next tag or line end is processed) */
 LastLine          = '';  /* last written HTEXT line */
 LineCount         = 0;
 DocState          = 0;  /* :USERDOC. ==> 1, :EUSERDOC. ==> 2 */
 Pages.0           = 0;
 rcx = InitPage();
 /* The following global vars are reset by InitPage: */
 /*
    Link            = '';  /* holds unwritten HTEXT link data */
    ThisArtWork     = '';  /* holds unwritten IPF artwork attribs */
    ThisArtLink     = '';  /* holds unwritten IPF link within an artlink */
    fLinkOpen       = FALSE;
    fArtLinkOpen    = FALSE;
    fTagOpen        = FALSE;
    fXmpOpen        = FALSE;
    fSingleNoteOpen = FALSE;
    ListDepth       = 0;
    LastIndex       = '';  /* holds last index for a panel, used to enable subindex */
 */
 Pages._AnchorList = '';

 /* set default values for env vars if not already set */
 NameList = 'WEBBMP EMAILBMP   INFBMP   NOTE'
 ValList  = 'ns.bmp nsmail.bmp book.bmp *Note:*'
 DO i = 1 TO WORDS( NameList)
    Name = WORD( NameList, i);
    Val  = WORD( ValList, i);
    IF (VALUE( Name,, env) = '') THEN rcx = VALUE( Name, Val, env);
 END;
 /* check for bitmap files only once at first usage */
 Bitmap.                 = '';
 Bitmap._fBitmapsChecked = FALSE;

 /* read commandline parameters */
 PARSE ARG Parms;
 Rest = Parms;
 DO WHILE (Rest \= '')
    Rest = STRIP( Rest);

    IF (LEFT( Rest, 1) = '"') THEN
       PARSE VAR Rest '"'ThisParm'"' Rest;
    ELSE
       PARSE VAR Rest ThisParm Rest;

    PARSE VAR ThisParm ThisTag':'ThisValue;
    ThisTag = TRANSLATE( ThisTag);

    SELECT
       /*
       WHEN (POS( ThisTag, '/DEBUG') = 1) THEN
          Flag.fDebug = TRUE;

       WHEN (POS( ThisTag, '/VERBOSE') = 1) THEN
          Flag.fVerbose = TRUE;
       */

       WHEN (POS( ThisTag, '/APS') = 1) THEN
          IF (ThisValue > 0) THEN
          DO
             rc = FindApsFile( ThisValue);
             /* FindApsFile types the error msg on error itself */
             IF (rc \= ERROR.NO_ERROR) THEN
                LEAVE;
          END;
          ELSE
          DO
             SAY 'Error: No APS file given.';
             rc = ERROR.INVALID_PARAMETER;
             LEAVE;
          END;

       WHEN (POS( ThisTag, '/?') = 1) THEN
          Flag.fHelp =TRUE;

    OTHERWISE
       SELECT
          WHEN (SourceFile = '') THEN
              SourceFile = ThisParm;

          WHEN (TargetFile = '') THEN
              TargetFile = ThisParm;

       OTHERWISE
          SAY 'Error: Invalid parameters.';
          rc = ERROR.INVALID_PARAMETER;
          LEAVE;
       END;
    END;
 END;

 DO UNTIL (TRUE)

    IF (rc \= ERROR.NO_ERROR) THEN
       LEAVE;

    /* display help */
    IF (Flag.fHelp) THEN
    DO
       rcx = SHowHelp();
       LEAVE;
    END;

    /* check source file */
    IF (SourceFile = '') THEN
    DO
       SAY 'Error: No sourcefile given.';
       rc = ERROR.INVALID_PARAMETER;
       LEAVE;
    END;
    IF (\FileExist( SourceFile)) THEN
    DO
       SAY 'Error: Sourcefile "'SourceFile'" not found.';
       rc = ERROR.FILE_NOT_FOUND;
       LEAVE;
    END;

    /* get the APS file, if not already */
    IF (IPFC._ApsFile = '') THEN
    DO
       rc = FindApsFile( IPFC._DefaultApsFile);
       /* FindApsFile writes the error msg on error itself */
       IF (rc \= ERROR.NO_ERROR) THEN
          LEAVE;
    END;

    /* determine target file */
    IF (TargetFile = '') THEN
       TargetFile = ChangeExtension( SourceFile, '.htext');

    /* process source file */
    rcProcess = ProcessSourceFile( SourceFile, TargetFile);
    IF (rcProcess > 0) THEN
    DO
       rc = rcProcess;
       SAY 'Aborting due to error, rc =' rc'.';
       LEAVE;
    END;

    IF (Pages.0 > 0) THEN
    DO
       /* write header of target file */
       rcx = SysFileDelete( TargetFile);
       rcx = LINEOUT( TargetFile, '.. Generated by IPF2HTEXT from 'FILESPEC( 'N', SourceFile)' - 'DATE() TIME());

       /* write pages of target file */
       DO p = 0 TO Pages.0
          next = Pages.p._HeaderLines;
          IF (LENGTH( next) > 0) THEN
             rcx = LINEOUT( TargetFile, next);

          next = Pages.p._AnchorName;
          IF (LENGTH( next) > 0) THEN
             rcx = LINEOUT( TargetFile, '.an' next);

          next = Pages.p._ResId;
          IF (LENGTH( next) > 0) THEN
          DO
             rcx = LINEOUT( TargetFile, '.re' next);

             /* check if dimensions where set by a :link reftype=hd res=. */
             Dims = GetDimsForResId( next);
             IF (LENGTH( Dims) > 0) THEN
                Pages.p._Dimensions = Dims;
          END;

          /* check if dimensions where set by a :link reftype=hd refid=. */
          next = Pages.p._RefId;
          IF (LENGTH( next) > 0) THEN
          DO
             Dims = GetDimsForRefId( next);
             IF (LENGTH( Dims) > 0) THEN
                Pages.p._Dimensions = Dims;
          END;

          next = Pages.p._Dimensions;
          IF (LENGTH( next) > 0) THEN
             rcx = LINEOUT( TargetFile, '.di' next);

          IndexLines = Pages.p._IndexLines;
          DO WHILE (LENGTH( IndexLines) > 0)
             PARSE VAR IndexLines next(CrLf)IndexLines;
             rcx = LINEOUT( TargetFile, '.in' next);
          END;

          IF (Pages.p._fHide) THEN
             rcx = LINEOUT( TargetFile, '.hi');

          rcx = LINEOUT( TargetFile, '');

          next = Pages.p._TextLines;
          IF (LENGTH( next) > 0) THEN
          DO
             /* replace numeric anchor ids in links with unique names */
             /* the unique names were determined from the header titles during */
             /* header processing */
             next = ReplaceNumAnchorLinks( next);
             rcx = LINEOUT( TargetFile, next);
             IF (RIGHT( next, 2) \= CrLf) THEN
                rcx = LINEOUT( TargetFile, '');
          END;
       END;

       rcx = STREAM( TargetFile, 'c', 'close')
    END;
    ELSE
    DO
       SAY 'Error: Target file not written, because there''s no data.'
       rc = ERROR.INVALID_DATA;
       LEAVE;
    END;

 END;

 IF (rc \= ERROR.NO_ERROR) THEN
 DO
    SAY;
    'PAUSE';
 END;

 EXIT( rc);

/* ------------------------------------------------------------------------- */
InitPage: PROCEDURE EXPOSE (GlobalVars)

 p = Pages.0;
 Pages.p._Title       = '';
 Pages.p._HeaderLines = '';
 Pages.p._AnchorName  = '';
 Pages.p._RefId       = '';
 Pages.p._ResId       = '';
 Pages.p._Dimensions  = '';
 Pages.p._IndexLines  = '';
 Pages.p._fHide       = FALSE;
 Pages.p._TextLines   = '';

 /* (emergency) reset for global vars */
 Link            = '';  /* holds unwritten HTEXT link data */
 ThisArtWork     = '';  /* holds unwritten IPF artwork attribs */
 ThisArtLink     = '';  /* holds unwritten IPF link within an artlink */
 fLinkOpen       = FALSE;
 fArtLinkOpen    = FALSE;
 fTagOpen        = FALSE;
 fXmpOpen        = FALSE;
 fSingleNoteOpen = FALSE;
 ListDepth       = 0;
 LastIndex       = '';  /* holds last index for a panel, used to enable subindex */

 RETURN( '');

/* ------------------------------------------------------------------------- */
FindApsFile: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG ThisParm;

 rc = ERROR.NO_ERROR;

 DO UNTIL (TRUE)

    /* first search in current path */
    next = STREAM( ThisParm, 'C', 'QUERY EXISTS');
    IF (next \= '') THEN
    DO
       IPFC._ApsFile = next;
       LEAVE;
    END;

    /* accept no fully qualified or relative filenames from here */
    IF (POS( '\', ThisParm) \= 0) THEN
    DO
       rc = ERROR.FILE_NOT_FOUND;
       SAY 'Error: APS file not found.';
       LEAVE;
    END;

    /* next search in path of current CMD file */
    next = STREAM( GetCallDir()'\'ThisParm, 'C', 'QUERY EXISTS');
    IF (next \= '') THEN
    DO
       IPFC._ApsFile = next;
       LEAVE;
    END;

    /* search in IPFC path */
    next = SysSearchPath( IPFC._EnvVar, ThisParm);
    if (next \= '') THEN
    DO
       IPFC._ApsFile = next;
       LEAVE;
    END;

    /* search APS file in DPATH */
    next = SysSearchPath( 'DPATH', ThisParm);
    IF (next \= '') THEN
    DO
       IPFC._ApsFile = next;
       LEAVE;
    END;

    rc = ERROR.FILE_NOT_FOUND;
    SAY 'Error: APS file not found in current, IPFC or DPATH path.';
    LEAVE;

 END;

 RETURN( rc);

/* ------------------------------------------------------------------------- */
ProcessSourceFile: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG SourceFile, TargetFile;

 rc = ERROR.NO_ERROR;

 fTagOpen = FALSE;
 RestTagLine = '';
 HRuleLine = '';
 rcx = STREAM( SourceFile, 'c', 'open read');
 DO WHILE (LINES( SourceFile) > 0)

    /* stop at :EUSERDOC. */
    IF (DocState = 2) THEN
       LEAVE;

    /* read line */
    LineCount = LineCount + 1;
    Line = LINEIN( SourceFile);
    Line = TRANSLATE( Line, ' ', '09'x);
    Line = STRIP( Line, 'T');

    /* ignore empty lines */
    IF (Line = '' & \fXmpOpen) THEN
       ITERATE;

    IF (fLinkOpen & RIGHT( Link, 1) \= ' ') THEN
       /* convert previous lineend to a space */
       Link = Link' ';

    IF (\fLinkOpen & \fTagOpen) THEN
       /* reset current line */
       ThisLine = '';

    /* recognize .HRULE */
    IF (Line = HTEXT._HRuleLine) THEN
    DO
       HRuleLine = '.hr';
       ITERATE;
    END;
    IF (HRuleLine = '.hr') THEN
    DO
       IF (Line = '.br') THEN
          ITERATE;
    END;

    IF ((fTagOpen) & (LENGTH( RestTagLine) > 0)) THEN
       RestLine = RestTagLine Line;
    ELSE
       RestLine = Line;
    RestTagLine = '';

    /* parse Line at first column */
    PARSE VAR RestLine Chr1 2 Chr2 3 RestChr;
    /* handle IPF control words */
    IF (Chr1 = '.') THEN
    DO
       Wrd1 = TRANSLATE( WORD( Line, 1));
       RestWrd = SUBWORD( Line, 2);
       RestLine = ProcessIpfCommand( Chr2, RestChr, Wrd1, RestWrd);
    END;

    /* parse rest of line in parts (this gives speed) */
    DO WHILE (LENGTH( RestLine) > 0)

       /* first check if a pending bitmap should be added to the output */
       DO UNTIL (TRUE)
          IF (fLinkOpen | fTagOpen) THEN
             LEAVE;

          /* parse IPF tag, if any */
          CheckTag = '';
          IF (LEFT( RestLine, 1) = ':') THEN
             CheckTag = SUBSTR( TRANSLATE( WORD( TRANSLATE( RestLine, ' ', '.'), 1)), 2);

          /* no bitmap pending? */
          IF (ThisArtWork = '') THEN
          DO
             /* ignore artlink not preceded by an artwork */
             IF (CheckTag = 'EARTLINK') THEN
                ThisArtLink = '';
             LEAVE;
          END;

          /* comments after a bitmap are allowed for artlink recognition */
          IF (LEFT( RestLine, 2) == '.*') THEN
             LEAVE;

          /* trailing spaces after a bitmap are allowed for artlink recognition */
          IF (STRIP( RestLine, 'T') == '') THEN
             LEAVE;

          /* artlink following a bitmap? */
          IF (CheckTag = 'ARTLINK') THEN
             LEAVE;
          IF (CheckTag = 'EARTLINK') THEN
             LEAVE;

          /* link following a bitmap? */
          IF (CheckTag = 'LINK') THEN
             LEAVE;
          IF ((CheckTag = 'ELINK') & \fArtLinkOpen) THEN
             LEAVE;

          /* bitmap without a following text link and with an optional artlink */
          rcx = ProcessArtWork();
          rcx = FlushLine();
       END;

       /* search symbols or tags */
       p = VERIFY( RestLine, HTEXT._EscapeChars'&:', 'M');
       IF (p = 0) THEN
       DO
          /* nothing found, so append rest */
          rcx = AppendToLine( RestLine);
          RestLine = '';
          LEAVE;
       END;

       /* parse at found symbol or tag */
       Chr = SUBSTR( RestLine, p, 1);
       PARSE VAR RestLine Next(Chr)RestLine;

       /* escape literal period in column 1 with a leading space */
       IF ((ThisLine = '') & \(fLinkOpen) & (LEFT( Next, 1) = '.')) THEN
          Next = ' 'Next;

       IF (LENGTH( Next) > 0) THEN
          rcx = AppendToLine( Next);

       SELECT

          /* double HTEXT escape chars */
          WHEN (POS( Chr, HTEXT._EscapeChars) > 0) THEN
             rcx = AppendToLine( Chr''Chr);

          /* replace IPF symbols */
          WHEN (Chr = '&') THEN
             RestLine = ProcessIpfSymbol( RestLine);

          /* handle ':' chars */
          WHEN (Chr = ':') THEN
          DO
             /* only process next tag if previous tag was closed */
             IF \(fTagOpen) THEN
             DO
                /* check if first word is a known tag word */
                check = WORD( TRANSLATE( RestLine, ' ', '.'), 1);
                check = TRANSLATE( check);
                wp = WORDPOS( check, TRANSLATE( IPFC._StartTags IPFC._EndTags));
                fTagOpen = (wp > 0);
             END;

             IF \(fTagOpen) THEN
                /* not a known tag word, so add colon to the output and parse further */
                rcx = AppendToLine( ':');

             pStart = 1;
             p = 0;
             DO WHILE (fTagOpen)
                /* search closing dot, but ignore when enquoted */
                p = POS( '.', RestLine, pStart);

                IF (p = 0) THEN
                DO
                   RestLine = Chr''RestLine;
                   LEAVE;  /* WHILE (fTagOpen) */
                END;

                /* count quotes before '.'; if odd, then '.' must belong to a string */
                nQ = 0;
                pStartQ = 1;
                DO WHILE (pStartQ < p)
                   pQ = POS( "'", RestLine, pStartQ);
                   IF (pQ = 0 | pQ > p) THEN
                      LEAVE;
                   nQ = nQ + 1;
                   pStartQ = pQ + 1;
                END;

                IF (nQ // 2 = 0) THEN
                DO
                   /* this dot must end a tag */
                   fTagOpen = FALSE;
                   SavedLine = RestLine
                   PARSE VAR RestLine ThisTag=(p) '.'RestLine;
                   ThisTagWord = WORD( ThisTag, 1);
                   RestLine = STRIP( RestLine, 'T');

                   IF ((WORDPOS( TRANSLATE( ThisTagWord),,
                                 TRANSLATE( IPFC._RequireTextTags)) > 0) &,
                       (RestLine = '')) THEN
                   DO
                      /* headers require title text after the tag, so read next line */
                      fTagOpen = TRUE;
                      /* process the entire tag at once after title text data is read */
                      RestLine = Chr''SavedLine;
                      LEAVE;  /* WHILE (fTagOpen) */
                   END;

                   /* process just closed tag */
                   /* maybe process RestLine only until the next tag */
                   /* check if first word is a known tag word */
                   PARSE VAR RestLine LeftPart':'RightPart
                   check = WORD( TRANSLATE( RightPart, ' ', '.'), 1);
                   check = TRANSLATE( check);
                   wp = WORDPOS( check, TRANSLATE( IPFC._StartTags IPFC._EndTags));
                   IF (wp > 0) THEN
                   DO
                      NextPart = LeftPart;
                      RemainingPart = ':'RightPart;
                   END;
                   ELSE
                   DO
                      NextPart = RestLine;
                      RemainingPart = '';
                   END;

                   NextPart = ProcessIpfTag( ThisTag, NextPart);
                   RestLine = NextPart''RemainingPart;

                   LEAVE;  /* WHILE (fTagOpen) */
                END;  /* if dot not enquoted */

                /* if dot is enquoted, search next */
                pStart = p + 1;
             END;  /* WHILE (fTagOpen) */

          END;  /* WHEN (Chr = ':') */

       OTHERWISE
          NOP;
       END;  /* SELECT */

       /* for open tags, set RestTagLine to append the following line later */
       IF (fTagOpen) THEN
       DO
          RestTagLine = RestLine;
          RestLine = '';
       END;

       IF (HRuleLine \= '') THEN
       DO
          /* this is a HRule title, so add it to the output */
          /* strip default bold attrib only if no other     */
          /* the leading ':HP2.' is here already converted to '*', if any */
          IF ((ThisLine = '*') & (TRANSLATE( LEFT(RestLine, 3)) \= ':HP') &,
              (TRANSLATE( RIGHT(RestLine, 6)) = ':EHP2.')) THEN
          DO
             ThisLine = '';
             RestLine = LEFT( RestLine, LENGTH( RestLine) - 6);
          END;
          ThisLine = HRuleLine ThisLine;
          HRuleLine = '';
       END;

    END;  /* DO WHILE (LENGTH( RestLine) > 0) */

    /* read next line for opened link or tag */
    IF ((fLinkOpen) | (fTagOpen)) THEN
       ITERATE;

    /* add line to the output, if not already processed */
    IF (LENGTH( ThisLine) > 0) THEN
       rcx = FlushLine( ThisLine);
 END;

 /* add bitmap to the output or link, if pending */
 IF (ThisArtWork \= '') THEN
    rcx = ProcessArtWork();

 rcx = STREAM( SourceFile, 'c', 'close');

 RETURN( rc);

/* ------------------------------------------------------------------------- */
ProcessIpfSymbol: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG RestLine;
 /* The '&' is already stripped off here */

 fProcessed = FALSE;
 IF (POS( '.', RestLine) > 0) THEN
 DO
    PARSE VAR RestLine Sym'.'After;
    /* Replace symbol, returns empty if unknown */
    ReplacedSymbol = ReplaceSymbol( Sym);
    IF (ReplacedSymbol = '.' & ThisLine = '' & Link = '') THEN
       /* Indent '.' when in first column */
       rcx = AppendToLine( ' 'ReplacedSymbol);
    ELSE
       rcx = AppendToLine( ReplacedSymbol);
    RestLine = After;
    fProcessed = TRUE;
 END;

 IF \(fProcessed) THEN
    rcx = AppendToLine( '&');

 RETURN( RestLine);

/* ------------------------------------------------------------------------- */
/* Process IPF command words, starting with a dot */
ProcessIpfCommand: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Chr2, RestChr, Wrd1, RestWrd;

 RestLine = '';
 SELECT

    /* process comments first */
    WHEN (Chr2 = '*') THEN
    DO
       IF ((fLinkOpen) | (fTagOpen)) THEN
          /* ignore comments in links or tags */
          NOP;
       ELSE
          ThisLine = '..'RestChr;
    END;

    /* check for IPF control words (dot tags) */
    WHEN (Wrd1 = '.BR') THEN
       ThisLine = '.';

    WHEN (Wrd1 = '.CE') THEN
       /* RestWrd can't contain tags, so add it to the output */
       ThisLine = '.ce 'MakeHtextLine( RestWrd);

    WHEN (Wrd1 = '.IM') THEN
       /* RestWrd can't contain tags, so add it to the output */
       ThisLine = '.include 'ChangeExtension( RestWrd, '.htext');

    WHEN (Wrd1 = '.NAMEIT') THEN
    DO
       IpfAttr  = '';
       VarName  = '';
       VarValue = '';
       Rest = RestWrd;
       DO WHILE (LENGTH( Rest) > 0)
           Rest = STRIP( Rest);
           p = POS( '=', Rest);
           IF (p = 0) THEN
              LEAVE;
           PARSE VAR Rest IpfAttr'='Rest;
           IF (LEFT( Rest, 1) = "'") THEN
              PARSE VAR Rest "'"Val"'" Rest;
           ELSE
              PARSE VAR Rest Val Rest;
           SELECT
              WHEN (TRANSLATE( IpfAttr) = 'SYMBOL') THEN
                 VarName = Val;
              WHEN (TRANSLATE( IpfAttr) = 'TEXT') THEN
                 VarValue = Val;
           OTHERWISE
              NOP;
           END;
       END;

       IF ((VarName \= '') & (VarValue \= '')) THEN
          /* convert to a .set command */
          ThisLine = '.set 'VarName'='MakeHtextLine( VarValue);
       ELSE
       DO
          SAY 'Warning: .NAMEIT ignored, command not complete.'
          ThisLine = '.. 'ThisLine;
       END;
    END;

 OTHERWISE
    /* indent dots at start of line */
    ThisLine = ' .';
    RestLine = Chr2''RestChr;
 END;

 /* ThisLine is global */

 RETURN( RestLine);

/* ------------------------------------------------------------------------- */
/* Resolve IPF symbols and double HTEXT escape chars. */
/* Don't process IPF tags here. */
/* This proc is called by tags and control words to process the rest of the line. */
MakeHtextLine: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Str;

 RestStr = Str;
 ThisStr = '';
 DO WHILE (LENGTH( RestStr) > 0)
    /* search HTEXT escape chars */
    p = VERIFY( RestStr, '&'HTEXT._EscapeChars, 'M');

    IF (p > 0) THEN
    DO
       /* better don't use positional parsing to handle p = 1 correctly */
       Chr = SUBSTR( RestStr, p, 1);
       PARSE VAR RestStr Next(Chr)RestStr;
       ThisStr = ThisStr''Next;

       IF (POS( Chr, HTEXT._EscapeChars) > 0) THEN
       DO
          /* double HTEXT escape chars */
          ThisStr = ThisStr''Chr''Chr;
       END;
       ELSE
       DO
          /* replace symbol */
          PARSE VAR RestStr Sym'.'After;
          ReplacedSymbol = ReplaceSymbol( Sym);
          IF (ReplacedSymbol \= '') THEN
          DO
             ThisStr = ThisStr''ReplacedSymbol;
             RestStr = After;
          END;
          ELSE
             ThisStr = ThisStr''Chr;
       END;

    END;
    ELSE
    DO
       /* append the entire rest */
       ThisStr = ThisStr''RestStr;
       RestStr = '';
    END;
 END;

 RETURN( ThisStr);

/* ------------------------------------------------------------------------- */
/* All TargetFile lines are set by this function. */
AddLine: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Next, fHeaderLine;

 IF (fHeaderLine = '') THEN
    fHeaderLine = FALSE;

 p = Pages.0;

 IF (DocState = 1) THEN
 DO
    IF (fHeaderLine) THEN
    DO
       Wrd1 = WORD( Next, 1);
       Rest = SUBWORD( Next, 2);
       SELECT
          WHEN (Wrd1 = '.an') THEN
             Pages.p._AnchorName = Rest;
          WHEN (Wrd1 = '.re') THEN
             Pages.p._ResId      = Rest;
          WHEN (Wrd1 = '.di') THEN
             Pages.p._Dimensions = Rest;
          WHEN (Wrd1 = '.in') THEN
          DO
             IF (LENGTH( Pages.p._IndexLines) > 0) THEN
                Pages.p._IndexLines = Pages.p._IndexLines''CrLf;
             Pages.p._IndexLines = Pages.p._IndexLines''Rest;
          END;
          WHEN (Wrd1 = '.hi') THEN
             Pages.p._fHide      = TRUE;
       OTHERWISE
          IF (LENGTH( Pages.p._HeaderLines) > 0) THEN
             Pages.p._HeaderLines = Pages.p._HeaderLines''CrLf;
          Pages.p._HeaderLines = Pages.p._HeaderLines''Next;
       END;
    END;

    ELSE
    DO
       Next = IndentLine( Next);
       IF (LENGTH( Pages.p._TextLines) > 0) THEN
          Pages.p._TextLines = Pages.p._TextLines''CrLf;
       Pages.p._TextLines = Pages.p._TextLines''Next;
    END;

 END;

 /* save line (global var) */
 LastLine = Next;

 RETURN( '');

/* ------------------------------------------------------------------------- */
RemoveLastLine: PROCEDURE EXPOSE (GlobalVars)

 p = Pages.0
 Text = Pages.p._TextLines;

 /* remove last line */
 lp = LASTPOS( CrLf, Text);
 IF (lp = 0) THEN
    Pages.p._TextLines = '';
 ELSE
    Pages.p._TextLines = LEFT( Text, lp - 1);

 /* set LastLine global var */
 lp = LASTPOS( CrLf, Pages.p._TextLines);
 IF (lp = 0) THEN
    LastLine = Pages.p._TextLines;
 ELSE
    LastLine = SUBSTR( Pages.p._TextLines, lp + LENGTH( CrLf));

 RETURN( '');

/* ------------------------------------------------------------------------- */
/* Indent line to current ListDepth, if not a tag or if not already indented */
IndentLine: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Next;

 IF (LEFT( Next, 1) \= '.' & WORDPOS( WORD( Next, 1), '- =') = 0) THEN
    Next = COPIES( ' ', ListIndent * ListDepth)''Next;

 RETURN( Next);

/* ------------------------------------------------------------------------- */
/* Start a new list item with a bullet char, indented to current ListDepth */
StartNewListItemLine: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Bullet;

 rcx = StartNewLine( OVERLAY( Bullet, COPIES( ' ', ListIndent * ListDepth),,
    ListIndent * (ListDepth - 1) + 1));

 RETURN( '');

/* ------------------------------------------------------------------------- */
/* Append strings either to the currently opened Link or to ThisLine */
AppendToLine: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Next;

 IF (fLinkOpen) THEN
    Link = Link''Next;
 ELSE
    ThisLine = ThisLine''Next;

 RETURN( '');

/* ------------------------------------------------------------------------- */
/* Terminate current line if non-empty */
FlushLine: PROCEDURE EXPOSE (GlobalVars)

 IF (LENGTH( ThisLine) > 0) THEN
 DO
    rcx = AddLine( ThisLine);
    ThisLine = '';
 END;

 RETURN( '');

/* ------------------------------------------------------------------------- */
/* Terminate current line and process the submitted string for the next line */
StartNewLine: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Next;

 IF (fLinkOpen) THEN
 DO
    SAY SourceFile'('LineCount'): Warning: Ignoring previous link due to missing :elink. tag.';
    fLinkOpen = FALSE;
 END;
 rcx = FlushLine();
 ThisLine = Next;

 RETURN( '');

/* ------------------------------------------------------------------------- */
/* Terminate current line and put the submitted string on an extra line */
AddExtraLine: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Next, fHeaderLine;

 IF (fHeaderLine = '') THEN
    fHeaderLine = FALSE;

 IF (fLinkOpen) THEN
 DO
    SAY SourceFile'('LineCount'): Warning: Ignoring previous link due to missing :elink. tag.';
    fLinkOpen = FALSE;
 END;
 rcx = FlushLine();
 rcx = AddLine( Next, fHeaderLine);

 RETURN( '');

/* ------------------------------------------------------------------------- */
/* Add to header */
AddHeaderLine: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Next;

 rcx = AddLine( Next, TRUE);

 RETURN( '');

/* ------------------------------------------------------------------------- */
/* Check if it's a link to a header with 'refid=' or 'res=' etc. that */
/* requires an ':elink.' tag, because it produces text. */
CheckLinkAttrList:  PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG TagAttr;

 TagAttr = SPACE( TagAttr, 1);
 Check = ' 'TRANSLATE( TagAttr)' ';
 SELECT
    WHEN (POS( ' REFTYPE=LAUNCH ', Check) > 0) THEN
    DO
       IF (fArtLinkOpen) THEN
          fLinkHasText = FALSE;
       ELSE
          fLinkHasText = TRUE;
    END
    WHEN (POS( ' REFTYPE=HD ', Check) = 0 & POS( ' REFTYPE=FN ', Check) = 0) THEN
       fLinkHasText = FALSE;
    WHEN (POS( ' REFID=', Check) = 0 & POS( ' RES=', Check) = 0) THEN
       fLinkHasText = FALSE;
    WHEN (POS( ' AUTO ', Check) > 0) THEN
       fLinkHasText = FALSE;
 OTHERWISE
    fLinkHasText = TRUE;
 END;

 IF \(fLinkHasText) THEN
 DO
    fLinkOpen = FALSE;
    Link = '';
 END;

 RETURN( fLinkHasText);

/* ------------------------------------------------------------------------- */
ProcessAttrList:  PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Type, TagAttr;

 TagAttr = SPACE( TagAttr, 1);
 ThisLink = '';
 Rest = STRIP( TagAttr);
 X = 0;
 Y = 0;
 Width  = 100;
 Height = 100;
 Bitmap = '';
 Text   = '';
 DO WHILE (Rest \= '')
    Rest = STRIP( Rest);
    p = VERIFY( Rest, ' =', 'M');
    IF (p = 0) THEN
       Chr = '';
    ELSE
       Chr = SUBSTR( Rest, p, 1);

    IF (Chr = '=') THEN
    DO
       PARSE VAR Rest IpfAttr'='Rest;
       IF (LEFT( Rest, 1) = "'") THEN
          PARSE VAR Rest "'"Val"'" Rest;
       ELSE
          PARSE VAR Rest Val Rest;
       Val = STRIP( Val);
    END;
    ELSE
    DO
       PARSE VAR Rest IpfAttr Rest;
       Val = '';
    END;

    /* determine HTEXT command etc. */
    Attr = ConvertIpfAttr( Type, IpfAttr);

    /* uppercase and spaces to underscores */
    AttrVal = TRANSLATE( TRANSLATE( Val), '_', ' ');
    SELECT
       WHEN (Attr = '') THEN
          /* ignored */
          NOP;
       WHEN (Attr = '.') THEN
          /* reference to an anchor */
          /* unused? */
          rcx = AppendToLine( '.'AttrVal' ');
       WHEN (Attr = '@') THEN
          /* hyperlink: link data */
       DO
          IF (RIGHT( Link, 4) = 'inf:') THEN
             rcx = AppendToLine( LinkDataMark''Val''LinkDataMark' ');
          ELSE
             rcx = AppendToLine( Val' ');
       END;
       WHEN (Attr = 'EXE') THEN
       DO
          /* exe link */
          IF (AttrVal = 'VIEW.EXE') THEN
             /*Link = '[inf:'SUBSTR( Link, 2)*/
             rcx = AppendToLine( 'inf:');
          ELSE IF (AttrVal = 'NETSCAPE.EXE') THEN
             /* no additional link data to add */
             NOP;
          ELSE
             /* ignore */
             SAY 'Warning: Link type for "'TagAttr'" unknown.';
       END;
       WHEN (Attr = '.an') THEN
          rcx = AddHeaderLine( Attr RefId2AnchorName( Val));
       WHEN (Attr = '.hi') THEN
          rcx = AddHeaderLine( Attr);
       WHEN (Attr = '.re') THEN
       DO
          rcx = SetResId( Val);
          rcx = AddHeaderLine( Attr Val);
       END;
       WHEN (Attr = 'X') THEN
          X = STRIP( Val, 'T', '%');
       WHEN (Attr = 'Y') THEN
          Y = STRIP( Val, 'T', '%');
       WHEN (Attr = 'WIDTH') THEN
          Width = STRIP( Val, 'T', '%');
       WHEN (Attr = 'HEIGHT') THEN
          Height = STRIP( Val, 'T', '%');
       WHEN (Attr = 'LINKRESID') THEN
       DO
          LinkResId = AttrVal;
          IF (fLinkOpen) THEN
             rcx = AppendToLine( '#'AttrVal' ');
       END;
       WHEN (Attr = 'LINKREFID') THEN
       DO
          LinkRefId = AttrVal;
          IF (fLinkOpen) THEN
             rcx = AppendToLine( '.'AttrVal' ');
       END;
       WHEN (Attr = 'INDEXID') THEN
          /* ignore, not supported yet */
          NOP;
       WHEN (Attr = 'INDEXREFID') THEN
          /* ignore, not supported yet */
          NOP;
       WHEN (Attr = '?') THEN
       DO
          SAY SourceFile'('LineCount'): Warning: Unknown attribute(s) in: 'TagAttr;
          rcx = AddExtraLine( '.. *** UNKNOWN ATTRIBUTE: 'IpfAttr', VALUE = 'Val);
       END;
       WHEN (Attr = '.bi') THEN
          Bitmap = Bitmap Val;
       WHEN (Attr = 'MARGIN') THEN
          rcx = AddExtraLine( '.'Lower( Type) Val)
       WHEN (Attr = 'TEXT') THEN
       DO
          IF (WORDPOS(Type, 'NOTE NT CAUTION WARNING') > 0) THEN
             Text = Val;
       END;
    OTHERWISE
       rcx = AddExtraLine( STRIP( Attr Val));
    END;
 END;

 /* add bitmap line to the output */
 IF (Bitmap \= '') THEN
    rcx = AddExtraLine( '.bi' STRIP( Bitmap));

 /* add notes to the output */
 SELECT
    /* single-par type, starting a new par */
    WHEN (Type = 'NOTE') THEN
    DO
       IF (Text = '') THEN
          Text = '[=Note]';
       rcx = StartNewLine( Text'  ');
    END;

    /* multi-par types, starting a new list */
    WHEN (Type = 'NT') THEN
    DO
       /* different from standard IPF style */
       IF (Text = '') THEN
          rcx = AddExtraLine( '.no');
       ELSE
          rcx = AddExtraLine( ".no text='"Text"'");
    END;

    WHEN (Type = 'CAUTION') THEN
    DO
       /* different from standard IPF style */
       IF (Text = '') THEN
          Text = '*Caution:*';
       rcx = AddExtraLine( ".no text='"Text"'");
    END;

    WHEN (Type = 'WARNING') THEN
    DO
       /* different from standard IPF style */
       IF (Text = '') THEN
          Text = '*Warning:*';
       rcx = AddExtraLine( ".no text='"Text"'");
    END;

 OTHERWISE
    NOP;
 END;

 /* handle non-percentage page dimension values */
 SELECT
    WHEN (TRANSLATE( X) = 'LEFT') THEN
       X = 0;
    WHEN (TRANSLATE( X) = 'RIGHT') THEN
       X = 100 - Width;
    WHEN (TRANSLATE( X) = 'CENTER') THEN
       X = 100 - Width % 2;
 OTHERWISE
    NOP;
 END;
 SELECT
    WHEN (TRANSLATE( Y) = 'BOTTOM') THEN
       Y = 0;
    WHEN (TRANSLATE( Y) = 'TOP') THEN
       Y = 100 - Height;
    WHEN (TRANSLATE( Y) = 'CENTER') THEN
       Y = 100 - Height % 2;
 OTHERWISE
    NOP;
 END;

 Dims = X Y Width Height;
 IF (Dims \= '0 0 100 100') THEN
 DO
    IF (DATATYPE( SPACE( Dims, 0)) \= 'NUM') THEN
    DO
       SAY SourceFile'('LineCount'): Warning: Unknown page dimension(s) in: 'Dims;
       IF (Type = 'HEADER') THEN
          rcx = AddHeaderLine( '.. *** UNKNOWN PAGE DIMENSION(S) IN: 'Dims);
    END;
    ELSE
    DO
       IF (Type = 'HEADER') THEN
          rcx = AddHeaderLine( '.di' Dims);
       ELSE
       DO
          IF (LinkResId \= '') THEN
          DO
             r = LinkResId;
             ResId.r._Dimensions = Dims;
          END;
          ELSE
          IF (LinkRefId \= '') THEN
          DO
             r = LinkRefId;
             RefId.r._Dimensions = Dims;
          END;
       END;
    END;

 END;

 RETURN( '');

/* ------------------------------------------------------------------------- */
/* For headers, links and index entries */
ConvertIpfAttr:  PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Type, IpfAttr;

 IpfAttr = TRANSLATE( IpfAttr);
 SELECT
    WHEN (IpfAttr = 'REFTYPE') THEN
       /* ignore, only reftype=hd or fn is known */
       Attr = '';
    WHEN (IpfAttr = 'RES') THEN
    DO
       IF (Type = 'LINK') THEN
          /* reference to an help id */
          Attr = 'LINKRESID';
       ELSE
          /* set an help id */
          Attr = '.re';
    END;
    WHEN (IpfAttr = 'OBJECT' & Type = 'LINK') THEN
       /* use Val */
       Attr = 'EXE';
    WHEN (IpfAttr = 'DATA' & Type = 'LINK') THEN
       /* use Val */
       Attr = '@';
    WHEN (IpfAttr = 'NAME' & Type = 'ARTWORK') THEN
       /* bitmap */
       Attr = '.bi';
    WHEN (WORDPOS( IpfAttr, 'RUNIN FIT LINKFILE') > 0 & Type = 'ARTWORK') THEN
       /* RUNIN is supported and default, others are ignored */
       Attr = '';
    WHEN (IpfAttr = 'ALIGN' & Type = 'ARTWORK') THEN
       Attr = '.bi';
    WHEN (IpfAttr = 'ALIGN' & Type = 'LINES') THEN
       Attr = '.li';
    WHEN (IpfAttr = 'MARGIN') THEN
       Attr = 'MARGIN';
    WHEN (WORDPOS( IpfAttr, 'ID NAME') > 0 & WORDPOS( Type, 'INDEX ARTWORK') = 0) THEN
       /* set an anchor */
       Attr = '.an';
    WHEN (IpfAttr = 'ID' & Type = 'INDEX') THEN
       /* used for referring in :i2. tags */
       Attr = 'INDEXID'
    WHEN (IpfAttr = 'REFID') THEN
    DO
       IF (Type = 'INDEX') THEN
          /* reference to an index anchor */
          Attr = 'INDEXREFID';
       ELSE IF (Type = 'LINK') THEN
          /* reference to an help id */
          Attr = 'LINKREFID';
       ELSE
          /* reference to an anchor */
          /* unused? */
          Attr = '.';
    END;
    WHEN (IpfAttr = 'CLEAR') THEN
       /* ignore, HTEXT adds this automatically for level 1 */
       Attr = '';
    WHEN (WORDPOS( IpfAttr, 'X VPX') > 0) THEN
       /* page dimension */
       Attr = 'X';
    WHEN (WORDPOS( IpfAttr, 'Y VPY') > 0) THEN
       /* page dimension */
       Attr = 'Y';
    WHEN (WORDPOS( IpfAttr, 'WIDTH VPCX') > 0) THEN
       /* page dimension */
       Attr = 'WIDTH';
    WHEN (WORDPOS( IpfAttr, 'HEIGHT VPCY') > 0) THEN
       /* page dimension */
       Attr = 'HEIGHT';
    WHEN (IpfAttr = 'GROUP') THEN
       /* ignore */
       Attr = '';
    WHEN (IpfAttr = 'AUTO') THEN
       /* ignore */
       Attr = '';
    WHEN (IpfAttr = 'CHILD') THEN
       /* ignore */
       Attr = '';
    WHEN (IpfAttr = 'VIEWPORT') THEN
       /* open new window, ignore */
       Attr = '';
    WHEN (IpfAttr = 'DEPENDENT') THEN
       /* ignore */
       Attr = '';
    WHEN (IpfAttr = 'SPLIT') THEN
       /* ignore and add .hide */
       Attr = '.hi';
    WHEN (IpfAttr = 'HIDE') THEN
       /* hide from toc */
       Attr = '.hi';
    WHEN (IpfAttr = 'OBJECT' & Type \= 'LINK') THEN
       /* ignore */
       Attr = '';
    WHEN (IpfAttr = 'TOC') THEN
       /* ignore */
       Attr = '';
    WHEN (IpfAttr = 'NOSEARCH') THEN
       /* ignore */
       Attr = '';
    WHEN (IpfAttr = 'NOPRINT') THEN
       /* ignore */
       Attr = '';
    WHEN (IpfAttr = 'SCROLL') THEN
       /* ignore */
       Attr = '';
    WHEN (IpfAttr = 'TITLEBAR') THEN
       /* ignore */
       Attr = '';
    WHEN (IpfAttr = 'RULES') THEN
       /* ignore */
       Attr = '';
    WHEN (IpfAttr = 'TEXT') THEN
       Attr = 'TEXT'
 OTHERWISE
    Attr = '?';
 END;
 RETURN( Attr);

/* ------------------------------------------------------------------------- */
/* Strip leading header numbers, previously inserted by HText's .HNUMBERING  */
StripHeaderNum: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG HTitle;

 HTitle = STRIP( HTitle);
 w1  = WORD( HTitle, 1);
 pw1 = WORDINDEX( HTitle, 1);
 pw2 = WORDINDEX( HTitle, 2);

 DO UNTIL (TRUE)
    /* HTitle with nums have at least 1 digit + 2 spaces prepended */
    IF (pw2 < 4) THEN
       LEAVE;

    /* header nums are separated by two spaces */
    Sep = SUBSTR( HTitle, pw2 - 2, 2);
    IF (Sep \= '  ') THEN
       LEAVE;

    /* parse header num */
    FirstDigit = LEFT( w1, 1);
    IF (LENGTH( w1) > 1) THEN
    DO
       RestNum    = SUBSTR( w1, 2, LENGTH( w1) - 2);
       LastDigit  = RIGHT( w1, 1);
    END;
    ELSE
    DO
       RestNum    = '';
       LastDigit  = '';
    END;

    Nums       = '0123456789'
    AlphaNums  = Nums''XRANGE( 'A', 'Z');

    /* check first digit of HTitle num */
    IF (VERIFY( FirstDigit, AlphaNums, 'M') = 0) THEN
       LEAVE;
    /* check last digit of HTitle num (header nums don't have a trailing dot) */
    IF (LastDigit \= '') THEN
       IF (DATATYPE( LastDigit) \= 'NUM') THEN
          LEAVE;
    /* check rest of HTitle num */
    IF (RestNum \= '') THEN
       IF (VERIFY( w1, Nums'.', 'M') = 0) THEN
          LEAVE;

    /* num found, so strip it */
    HTitle = SUBSTR( HTitle, pw2);
 END;

 RETURN( HTitle);

/* ------------------------------------------------------------------------- */
ProcessLink: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Link;
 PARSE VAR Link '['Url Text']';

 CheckUrl = TRANSLATE( WORD( TRANSLATE( Url, ' ', ':'), 1));

 LinkType = '';
 Bitmap   = '';
 SELECT
    WHEN (CheckUrl = 'INF') THEN
       LinkType = 'INF';
    WHEN (CheckUrl = 'MAILTO') THEN
       LinkType = 'EMAIL';
    WHEN (WORDPOS( CheckUrl, HTEXT._Urls) > 0) THEN
       LinkType = 'WEB';
 OTHERWISE
    NOP;
 END;

 SELECT

    WHEN (CheckUrl = 'INF') THEN
    DO
       /* ensure that Search is separated from Text, using HTEXT syntax */
       /* LinkData was enclosed with LinkDataMark by the attribute parser */
       PARSE VAR Link '[inf:'LinkData']';
       IF (LEFT( LinkData, LENGTH( LinkDataMark)) = LinkDataMark) THEN
          PARSE VAR LinkData (LinkDataMark)LinkData(LinkDataMark)Text;
       ELSE
          PARSE VAR LinkData LinkData Text;
       LinkData = STRIP( LinkData);
       Text     = STRIP( Text);
       /* handle enquoted inf Filename */
       SELECT
          WHEN LEFT( LinkData, 1) = '"' THEN
             PARSE VAR LinkData '"'Filename'"'Search;
          WHEN LEFT( LinkData, 1) = "'" THEN
             PARSE VAR LinkData "'"Filename"'"Search;
          OTHERWISE
             PARSE VAR LinkData Filename Search;
       END;
       Filename = STRIP( Filename);
       Search   = STRIP( Search);
       /* handle enquoted Search (no single quotes allowed in IPF) */
       IF LEFT( Search, 1) = '"' THEN
          PARSE VAR Search '"'Search'"' .;
       Search   = STRIP( Search);
       /* maybe (re-)add double quotes */
       EnqFilename = Filename;
       IF (POS( ' ', Filename) > 0) THEN
          EnqFilename = '"'Filename'"';
       EnqSearch   = Search;
       IF (POS( ' ', Search) > 0) THEN
          EnqSearch   = '"'Search'"';
       /* rebuild link */
       LinkString = STRIP( EnqFileName EnqSearch);
       SELECT
          WHEN (Text = '') THEN
             NOP;
          WHEN (Text = 'inf:'LinkString) THEN
             NOP;
          WHEN (Text = EnqSearch) THEN
             NOP;
          WHEN (Search = '') THEN
             LinkString = EnqFileName 'text="'Text'"';
       OTHERWISE
          /* with alternate Text, Search has to be enquoted */
          /* (or the text= attribute may be used)           */
          LinkString = STRIP( EnqFileName '"'Search'"' Text);
       END;
       Link = '[inf:'LinkString']';
    END;

    WHEN (CheckUrl = 'MAILTO') THEN
    DO
       /* remove Text if equal to Url part */
       IF (SUBSTR( Url, 8) = Text) THEN
          Link = '['Url']';
    END;

    WHEN (LEFT( URL, 1) = '.') THEN
       /* RefIds are resolved later, so a doubled title text */
       /* in an anchor link must be removed later as well    */
       NOP;

 OTHERWISE
    /* remove Text if equal to Url */
    IF (Url = Text) THEN
       Link = '['Url']';
 END;

 DO UNTIL (TRUE)

    IF (ThisArtWork = '') THEN
    DO
       /* only web, email and inf links have default bitmaps */
       IF (LinkType = '') THEN
          LEAVE;
    END;
    ELSE
    DO
       DefaultBitmap = GetDefaultBitmap( LinkType);
       /* parse ThisArtWork for bitmap filename */
       CheckPos = POS( ' NAME=', ' 'TRANSLATE( ThisArtWork));
       IF (CheckPos > 0) THEN
       DO
          BmpPos = CheckPos - 1 + 5;
          PARSE VAR ThisArtWork =(BmpPos)"'"Bitmap"'";
       END;
    END;

    IF (Bitmap = '') THEN
       LEAVE;

    IF (TRANSLATE( Bitmap) = TRANSLATE( DefaultBitmap)) THEN
    DO
       /* ignore bitmap */
       ThisArtWork = '';
       ThisArtLink = '';
       LEAVE;
    END;

    /* append bitmap name to link */
    PARSE VAR Link '['LinkData']';
    IF (POS( ' ', Bitmap)) THEN
       Bitmap = '"'Bitmap'"';
    Link = '['LinkData 'bitmap='Bitmap']';

 END;

 RETURN( Link);

/* ------------------------------------------------------------------------- */
GetDefaultBitmap: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG LinkType;


 IF \(Bitmap._fBitmapsChecked) THEN
    rcx = GetBitmaps();

 DefaultBitmap = '';
 SELECT
    WHEN (LinkType = 'WEB')   THEN DefaultBitmap = Bitmap._Web;
    WHEN (LinkType = 'EMAIL') THEN DefaultBitmap = Bitmap._Email;
    WHEN (LinkType = 'INF')   THEN DefaultBitmap = Bitmap._Inf;
 OTHERWISE NOP;
 END

 RETURN( DefaultBitmap);

/* ------------------------------------------------------------------------- */
/* check for bitmaps */
GetBitmaps: PROCEDURE EXPOSE (GlobalVars)

 Bitmap._Web   = GetBitmap( VALUE( 'WEBBMP',,   env));
 Bitmap._Email = GetBitmap( VALUE( 'EMAILBMP',, env));
 Bitmap._Inf   = GetBitmap( VALUE( 'INFBMP',,   env));
 Bitmap._fBitmapsChecked = TRUE;

 RETURN( '');

/* ------------------------------------------------------------------------- */
GetBitmap: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Filename;

 FoundFilename = '';
 CallDir       = GetCallDir();

 DO UNTIL (TRUE)
    IF (Filename = '') THEN
       LEAVE;

    /* search given path */
    IF (FileExist( Filename)) THEN
    DO
       FoundFilename = FileName;
       LEAVE;
    END;

    /* now search files without path */
    Filename = FILESPEC( 'N', Filename);

    /* search in call dir and subdir */
    CheckName = CallDir'\'Filename;
    IF (FileExist( CheckName)) THEN
    DO
       FoundFilename = CheckName;
       LEAVE;
    END;
    CheckName = CallDir'\bmp\'Filename;
    IF (FileExist( CheckName)) THEN
    DO
       FoundFilename = CheckName;
       LEAVE;
    END;
    CheckName = CallDir'\htext\'Filename;
    IF (FileExist( CheckName)) THEN
    DO
       FoundFilename = CheckName;
       LEAVE;
    END;

    /* search in environment vars */
    /* do not use SysSearchPath, it returns fully qualified pathnames */
    FoundFilename = SearchPath( 'INCLUDE', Filename);
    IF (FoundFilename \= '') THEN
       LEAVE;
    FoundFilename = SearchPath( 'HTINCLUDE', Filename);
    IF (FoundFilename \= '') THEN
       LEAVE;

 END;

 IF (FoundFilename = '') THEN
    FoundFilename = Filename;

 RETURN( FoundFilename);

/* ------------------------------------------------------------------------- */
/* This is called for ArtWorks either without an ArtLink */
/* or without a following text link                      */
ProcessArtWork: PROCEDURE EXPOSE (GlobalVars)

 SELECT
    WHEN (ThisArtWork = '') THEN
       NOP;

    /* no artlink or link followed, so add ThisArtWork to the output */
    WHEN (ThisArtLink = '') THEN
    DO
       rcx = ProcessAttrList( 'ARTWORK', ThisArtWork);

       /* reset global var */
       ThisArtWork = '';
    END;

 OTHERWISE
    /* artwork and artlink not followed by a link: add link with text=none */
    Link = '[';
    /* fLinkOpen makes ProcessAttrList write to the global var "Link" */
    fLinkOpen = TRUE;
    rcx = ProcessAttrList( 'LINK', ThisArtLink);
    fLinkOpen = FALSE;
    Link = STRIP( Link)']'

    BitmapOnlyLink = ProcessLink( Link);
    /* add text=none */
    PARSE VAR BitmapOnlyLink '['LinkData']';
    BitmapOnlyLink = '['LinkData 'text=none]';

    /* add link to the output */
    rcx = AppendToLine( BitmapOnlyLink);

    /* reset global vars */
    Link = '';
    ThisArtWork = '';
    ThisArtLink = '';
 END;

 RETURN( '');

/* ------------------------------------------------------------------------- */
ProcessIpfTag: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Tag, NextPart;
 PARSE VAR Tag TagWord TagAttr;
 TagWord = TRANSLATE( TagWord)
 p = Pages.0

 /* always end an open single-par note on a tag, except text attribute adding tags */
 IF (fSingleNoteOpen &,
     (LEFT( TagWord, 2) \= 'HP') &,
     (LEFT( TagWord, 3) \= 'EHP') &,
     (WORDPOS( TagWord, 'LINK ELINK FONT COLOR') = 0)) THEN
 DO
    IF (ListDepth = 0) THEN
       rcx = AddExtraLine( '');
    ELSE
       rcx = AddExtraLine( '.p');
    fSingleNoteOpen = FALSE;
 END;

 SELECT

    WHEN (TagWord = 'TITLE') THEN
    DO
       rcx = AddExtraLine( '.ti 'MakeHtextLine( NextPart));
       NextPart = '';  /* empty means: already processed */
    END;

    WHEN (TagWord = 'USERDOC') THEN
       DocState = 1;

    WHEN (TagWord = 'EUSERDOC') THEN
    DO
       rcx = AddExtraLine( '');
       DocState = 2;
       NextPart = '';  /* ignore possible rest */
    END;

    WHEN (TagWord = 'DOCPROF') THEN
    DO
       /* Recognize "toc =" only */
       Tag = ARG(1);
       HLevels = '';
       PARSE VAR Tag . Rest;
       PARSE VAR Rest 'toc='HLevels .;
       IF (HLevels \= '') THEN
          rcx = AddExtraLine( '.htoc 'HLevels);
    END;

    WHEN (WORDPOS( TagWord, 'H1 H2 H3 H4 H5 H6 FN') > 0) THEN
    DO

       p = Pages.0 + 1;
       Pages.0 = p;

       /* reset several page-specific vars and global vars */
       rcx = InitPage();

       wp = WORDPOS( TagWord, 'H1 H2 H3 H4 H5 H6 FN');
       Markers = '# = - . . . .';
       rcx = AddHeaderLine( '.. 'COPIES( WORD( Markers, wp), 75));
       IF (TagWord = 'FN') THEN
          rcx = AddHeaderLine( '.fn');
       ELSE
       DO
          Pages.p._Title = MakeHtextLine( StripHeaderNum( NextPart));
          rcx = AddHeaderLine( '.'RIGHT( TagWord, 1)' 'Pages.p._Title);
          rcx = AddHeaderLine( '.. 'COPIES( WORD( Markers, wp), 75));
       END;
       NextPart = '';  /* empty means: already processed */
       rcx = ProcessAttrList( 'HEADER', TagAttr);

       /* ensure that an anchor name exists */
       CheckAttrStr = ' 'TRANSLATE( TagAttr)' '
       IF (POS( ' ID=', CheckAttrStr) = 0) THEN
       DO
          ResPos = POS( ' RES=', CheckAttrStr)
          IF (ResPos > 0) THEN
          DO
             PARSE VAR CheckAttrStr ' RES='ResId .;
             AnchorName = RefId2AnchorName( '#'ResId);
             rcx = AddHeaderLine( '.an' AnchorName);
          END;
          /*
          ELSE
             SAY SourceFile'('LineCount'): Warning: Header specified without reference or resource id.';
          */
       END;

    END;

    WHEN (TagWord = 'EFN') THEN
       IF (LastLine \= '') THEN
          rcx = AddExtraLine( '');

    WHEN (TagWord = 'NOTE') THEN
    DO
       IF (LastLine \= '' & ListDepth = 0) THEN
          rcx = AddExtraLine( ''); /* add an empty line before */
       rcx = ProcessAttrList( TagWord, TagAttr);
       fSingleNoteOpen = TRUE;
    END;

    WHEN (WORDPOS( TagWord, 'NT CAUTION WARNING') > 0) THEN
    DO
       IF (LastLine \= '' & ListDepth = 0) THEN
          rcx = AddExtraLine( ''); /* add an empty line before */
       rcx = ProcessAttrList( TagWord, TagAttr);
       ListDepth = ListDepth + 1;
       rcx = StartNewListItemLine( '-');
    END;

    WHEN (TagWord = 'LINES') THEN
    DO
       Align = '';
       Str   = 'ALIGN=';
       Len   = LENGTH( Str);
       p = POS( Str, TRANSLATE( TagAttr));
       IF (p > 0) THEN
       DO
          p = p + Len
          PARSE VAR TagAttr =(p) Align .
       END;
       rcx = AddExtraLine( STRIP( '.li' Align));
       ListDepth = ListDepth + 1;
    END;

    WHEN (TagWord = 'TABLE') THEN
    DO
       rcx = AddExtraLine( '.ta' TagAttr);
       ListDepth = ListDepth + 1;
       fNewRow = FALSE;
    END;

    WHEN (TagWord = 'ROW') THEN
    DO
       fNewRow = TRUE;
    END;

    WHEN (TagWord = 'C') THEN
    DO
       IF \(fNewRow) THEN
          rcx = StartNewListItemLine( '-');
       ELSE
          rcx = StartNewListItemLine( '=');
       fNewRow = FALSE;
    END;

    WHEN (TagWord = 'LINK') THEN
    DO
       IF (fLinkOpen) THEN
       DO
          rcx = AddExtraLine( '.. *** Link ignored: 'Link);
          fLinkOpen = FALSE;
       END;
       IF (fArtLinkOpen) THEN
          /* store attribs in a global var, process ThisArtLink later */
          ThisArtLink = TagAttr;
       ELSE
       DO
          /* check if header is supported */
          fLinkHasText = CheckLinkAttrList( TagAttr);
          IF (fLinkHasText) THEN
          DO
             Link = '['
             fLinkOpen = TRUE;
          END;
          rcx = ProcessAttrList( TagWord, TagAttr);
       END;
    END;

    WHEN (TagWord = 'ELINK') THEN
    DO
       Link = STRIP( Link)']'
       fLinkOpen = FALSE;
       Link = ProcessLink( Link);
       rcx = AppendToLine( Link);
       Link = '';
       ThisArtWork = '';
       ThisArtLink = '';
    END;

    WHEN (TagWord = 'I1') THEN
    DO
       ThisIndex = MakeHtextLine( NextPart);
       NextPart = '';
       /* titles were added automatically, so skip if equal */
       IF (ThisIndex \= Pages.p._Title) THEN
       DO
          rcx = AddHeaderLine( '.in 'ThisIndex);
          rcx = ProcessAttrList( 'INDEX', TagAttr);
       END;
       /* store global var for addition to subindex tag I2 */
       LastIndex = ThisIndex;
    END;

    WHEN (TagWord = 'I2') THEN
    DO
       IF (LastIndex = '') THEN
          LastIndex = Pages.p._Title;
       rcx = AddHeaderLine( '.in 'LastIndex' - 'MakeHtextLine( NextPart));
       NextPart = '';
       rcx = ProcessAttrList( 'INDEX', TagAttr);
    END;

    WHEN (WORDPOS( TagWord, 'XMP CGRAPHIC') > 0) THEN
    DO
       rcx = AddExtraLine( '.fo off');
       fXmpOpen = TRUE;
    END;

    WHEN (WORDPOS( TagWord, 'EXMP ECGRAPHIC') > 0) THEN
    DO
       rcx = AddExtraLine( '.fo on');
       fXmpOpen = FALSE;
    END;

    WHEN (TagWord = 'HP1' | TagWord = 'EHP1') THEN
       rcx = AppendToLine( '#');

    WHEN (TagWord = 'HP2' | TagWord = 'EHP2') THEN
       rcx = AppendToLine( '*');

    WHEN (TagWord = 'HP3') THEN
       rcx = AppendToLine( '*#');

    WHEN (TagWord = 'EHP3') THEN
       rcx = AppendToLine( '#*');

    WHEN (TagWord = 'HP4') THEN
       rcx = AddExtraLine( '.at fc=blue');

    WHEN (TagWord = 'EHP4') THEN
       rcx = AddExtraLine( '.at fc=default');

    WHEN (TagWord = 'HP5' | TagWord = 'EHP5') THEN
       rcx = AppendToLine( '_');

    WHEN (TagWord = 'HP6') THEN
       rcx = AppendToLine( '_#');

    WHEN (TagWord = 'EHP6') THEN
       rcx = AppendToLine( '#_');

    WHEN (TagWord = 'HP7') THEN
       rcx = AppendToLine( '_*');

    WHEN (TagWord = 'EHP7') THEN
      rcx = AppendToLine( '*_');

    WHEN (TagWord = 'HP8') THEN
       rcx = AddExtraLine( '.at hilite');

    WHEN (TagWord = 'EHP8') THEN
      rcx = AddExtraLine( '.at fc=default');

    WHEN (TagWord = 'HP9') THEN
       rcx = AddExtraLine( '.at fc=pink');

    WHEN (TagWord = 'EHP9') THEN
       rcx = AddExtraLine( '.at fc=default');

    WHEN (WORDPOS( TagWord, 'SL OL UL DL PARML') > 0) THEN
    DO
       /* In HText, a description list is always replaced by a parameter */
       /* list, without any loss of functionality                        */
       IF (TagWord = 'DL' | TagWord = 'PARML') THEN
          Next = '.pl';
       ELSE
          Next = '.'Lower( TagWord);

       IF (TagWord = 'DL') THEN
       DO
          /* Maybe append default break option for description list */
          /* to behave like a non-breaking parameter list           */
          IF (POS( 'BREAK=', TRANSLATE( TagAttr)) = 0) THEN
          DO
             IF (TagAttr \= '') THEN
                TagAttr = TagAttr' ';
             TagAttr = TagAttr'break=none';
          END;
       END;
       IF (TagAttr \= '') THEN
          Next = Next' 'Lower( TagAttr);
       rcx = AddExtraLine( Next);
       ListDepth = ListDepth + 1;
    END;

    WHEN (WORDPOS( TagWord, 'ESL EOL EUL EDL EPARML ENT ECAUTION EWARNING' ||,
                            ' ETABLE ELINES') > 0) THEN
    DO
       /* before decreasing ListDepth: add end of previous list item */
       /* to the output (if any) with correct (previous) indent */
       rcx = FlushLine();
       IF (ListDepth > 0) THEN
          ListDepth = ListDepth - 1;
       ELSE
          SAY SourceFile'('LineCount'): Warning: No list to end, ListDepth is already 0.';
       IF (ListDepth = 0) THEN
       DO
          /* remove previous '.el' lines */
          DO FOREVER
             IF (LastLine = '.el') THEN
                rcx = RemoveLastLine();
             ELSE
                LEAVE;
          END;
          rcx = AddExtraLine( '');
       END;
       ELSE
          rcx = AddExtraLine( '.el');
    END;

    WHEN (WORDPOS( TagWord, 'LI DT DTHD PT') > 0) THEN
    DO
       IF (ListDepth < 1) THEN
       DO
          SAY SourceFile'('LineCount'): Warning: ListDepth is 0, changed to 1 to process current list item.';
          ListDepth = 1;
       END;
       rcx = StartNewListItemLine( '-');
    END;

    WHEN (WORDPOS( TagWord,  'DD DDHD PD') > 0) THEN
    DO
       IF (ListDepth < 1) THEN
       DO
          SAY SourceFile'('LineCount'): Warning: ListDepth is 0, changed to 1 to process current list item.';
          ListDepth = 1;
       END;
       rcx = StartNewListItemLine( '=');
    END;

    WHEN (TagWord = 'P') THEN
       SELECT
          /* for lists: don't end lists automatically by an empty line */
          WHEN (ListDepth > 0) THEN
             rcx = AddExtraLine( '.p');
          /* ignore multiple empty lines, but handles yet unprocessed ThisLine */
          WHEN (LastLine = '' & ThisLine = '') THEN
             /*rcx = AddExtraLine( '.p');*/
             NOP;
          /* allow to specify multiple empty lines by multiple :p. tags */
          WHEN (LastLine = '') THEN
             rcx = AddExtraLine( '.p');
       OTHERWISE
          /* an empty line is always better readable than a .p tag */
          rcx = AddExtraLine( '');
       END;

    WHEN (TagWord = 'LP') THEN
       /* not supported, just add a parbreak */
       rcx = AddExtraLine( '.p');

    WHEN (WORDPOS( TagWord, 'COLOR FONT') > 0) THEN
    DO
       IF ((POS( '=DEFAULT', TRANSLATE( TagAttr)) > 0) | (TagAttr = '')) THEN
       DO
          /* reset all attributes for any =DEFAULT string */
          IF ((LEFT( LastLine, 3) = '.at') & (POS( '=DEFAULT', TRANSLATE( LastLine)) > 0)) THEN
             rcx = RemoveLastLine();
          IF (LastLine \= '.at') THEN
             rcx = AddExtraLine( '.at');
       END;
       ELSE
          /* set the attributes (.at knows the attributes of :color. and :font.) */
          rcx = AddExtraLine( '.at' TagAttr);
    END;

    WHEN (TagWord = 'ARTWORK') THEN
       /* store attribs in a global var, process ThisArtWork later */
       ThisArtWork = TagAttr;

    WHEN (TagWord = 'ARTLINK') THEN
       fArtLinkOpen = TRUE;
    WHEN (TagWord = 'EARTLINK') THEN
       fArtLinkOpen = FALSE;

    WHEN (WORDPOS( TagWord, 'LM RM') > 0) THEN
       rcx = ProcessAttrList( TagWord, TagAttr);

 OTHERWISE
    /* Process unknown IPF tag */
    rcx = AddExtraLine( '.. *** NOT RESOLVED: :'Tag'.');
 END;

 RETURN( NextPart);

/* ------------------------------------------------------------------------- */
Lower: PROCEDURE

 Lower = 'abcdefghijklmnopqrstuvwxyz';
 Upper = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';

 PARSE ARG String;

 RETURN( TRANSLATE( String, Lower, Upper));

/* ------------------------------------------------------------------------- */
ReplaceSymbol: PROCEDURE EXPOSE (GlobalVars)
 Sym = ARG(1);

 IF (IPFC._fApsFileProcessed = FALSE) THEN
 DO
    /* read APS file into lines */
    l = 0;
    IPFC._ApsLine.0 = l;
    rcx = STREAM( IPFC._ApsFile, 'C', 'OPEN READ');
    IF (rcx \= 'READY:') THEN
    DO
       rc = ERROR.ACCESS_DENIED;
       SAY 'Error: APS file "'IPFC._ApsFile'" could not be opened.';
       SAY;
       'pause';
       EXIT( rc);
    END;

    DO WHILE (LINES( IPFC._ApsFile) > 0)
       l = l + 1;
       IPFC._ApsLine.0 = l;
       IPFC._ApsLine.l = LINEIN( IPFC._ApsFile);
    END;
    IPFC._fApsFileProcessed = TRUE;
    rcx = STREAM( IPFC._ApsFile, 'C', 'CLOSE');
 END;


 /* search Sym in APS lines */
 Char = ''
 DO l = 1 to IPFC._ApsLine.0
    PARSE VAR IPFC._ApsLine.l 1 ApsChar 2 '&'ApsSym'.';
    IF (Sym = ApsSym) THEN
    DO
       Char = ApsChar;
       /* double HyperText/2 escape chars */
       IF (POS( Char, HTEXT._EscapeChars) > 0) THEN
          Char = Char''Char;
       LEAVE;
    END;
 END;

 /* handle Sym as env var (set by a .nameit command) */
 IF (Char == '') THEN
   Char = '[='Sym']'

 RETURN( Char);

/* ------------------------------------------------------------------------- */
/* Replace numeric resource ids in links to anchor names */
ReplaceNumAnchorLinks: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG RestLines;

 TextLines = '';

 DO WHILE (LENGTH( RestLines) > 0)
    PARSE VAR RestLines Line(CrLf)RestLines;
    AnchorName = ''

    /* find anchor links */
    startp = 1
    DO FOREVER
       /* search next '[' or ']' char */
       p = VERIFY( Line, '[]', 'M', startp);
       IF (p = 0) THEN
          LEAVE;

       Char = SUBSTR( Line, p, 1);
       IF (p < LENGTH( Line)) THEN
          NextChar = SUBSTR( Line, p + 1, 1);
       ELSE
          NextChar = '';

       SELECT
          WHEN (Char = NextChar) THEN
          DO
             /* ignore doubled escape char */
             startp = p + 2;
             ITERATE;
          END;

          WHEN (WORDPOS( Char''NextChar, '[. [#') > 0) THEN
          DO
             LinkType = SUBSTR( Line, p + 1, 1);
             startp = p + 2;
             /* check for end of link */
             LinkEnd = POS( ']', Line, startp);
             IF (LinkEnd = 0) THEN
                ITERATE;  /* keep unmatched '[.' or '[#' */
             /* get link = string in between '[.' or '[#' and ']' */
             LinkData = SUBSTR( Line, startp, LinkEnd - startp);
             PARSE VAR LinkData LinkId LinkText;
             LinkId = STRIP( LinkId);
             LinkText  = STRIP( LinkText);
             IF (DATATYPE( LinkId) \= 'NUM') THEN
             DO
                startp = LinkEnd + 1;
                ITERATE;
             END

             /* find LinkId in either _ResId or _RefId array to get anchor name */
             AnchorName = '';
             IF (LinkType = '#') THEN
             DO p = 1 to Pages.0
                IF (Pages.p._ResId = LinkId) THEN
                DO
                   AnchorName = Pages.p._AnchorName;
                   AnchorTitle = Pages.p._Title;
                   LEAVE p;
                END;
             END;
             ELSE
             DO p = 1 to Pages.0
                IF (Pages.p._RefId = LinkId) THEN
                DO
                   AnchorName = Pages.p._AnchorName;
                   AnchorTitle = Pages.p._Title;
                   LEAVE p;
                END;
             END;

             IF (AnchorName = '') THEN
             DO
                IF (LinkType = '#') THEN
                   IdType = 'ResId';
                ELSE
                   IdType = 'RefId';
                SAY 'Warning: 'IdType' in link "'LinkData'" not replaced by an anchor name.';
                startp = LinkEnd + 1;
                ITERATE;
             END;

             /* replace res id link with anchor name link */
             /* LeftStr includes '[' and RightStr includes ']' */
             LeftStr  = SUBSTR( Line, 1, startp - 2);
             RightStr = SUBSTR( Line, LinkEnd);

             /* prepend '.', because also resource id links */
             /* are converted to anchor name links here     */
             IF (AnchorTitle = LinkText) THEN
                /* remove LinkText if equal to page title */
                AnchorLinkData = '.'AnchorName;
             ELSE
                AnchorLinkData = '.'STRIP( AnchorName LinkText);

             Line = LeftStr''AnchorLinkData;
             /* go after the ']' */
             startp = LENGTH( Line) + 1;
             Line = Line''RightStr
          END;
       OTHERWISE
          /* not an anchor link, so start at next char */
          startp = p + 1;
          ITERATE;
       END;
    END;

    IF (LENGTH( TextLines) > 0) THEN
       TextLines = TextLines''CrLf;
    TextLines = TextLines''Line;
 END;

 RETURN( TextLines);

/* ------------------------------------------------------------------------- */
/* Determine anchor name from header title. This is called to replace        */
/* numeric refids with anchor names for .an commands.                        */
RefId2AnchorName: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG RefId;
 /* if a ResId is submitted then it is preceeded by a '#' */

 p = Pages.0;
 /* replace a numeric RefId or a '#'ResId with a */
 /* unique name from header title                */
 IF (DATATYPE( RefId) = 'NUM' | LEFT( RefId, 1) = '#') THEN
 DO
    IF (Pages.p._Title = '' & DATATYPE( RefId) = 'NUM') THEN
       /* must use RefId for an empty title, as used in footnotes */
       Name = 'FN_'RefId;
    ELSE
       Name = Pages.p._Title;
 END;
 ELSE
    Name = RefId;

 AnchorName = MakeAnchorName( Name);

 /* store anchor name */
 Pages._AnchorList   = Pages._AnchorList AnchorName;
 Pages.p._AnchorName = AnchorName;
 /* store refid to replace it later in anchor links that use 'refid=' */
 Pages.p._RefId      = RefId;

 RETURN( AnchorName);

/* ------------------------------------------------------------------------- */
MakeAnchorName: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Name;

 AnchorName = Name;

 /* uppercase */
 Name = TRANSLATE( Name);
 /* replace all chars except numbers and uppercase letters with spaces */
 TabIn = XRANGE( '00'x, '2F'x)XRANGE( '3A'x, '40'x)XRANGE( '5B'x, 'FF'x);
 Name = TRANSLATE( Name,, TabIn);
 /* get only first 4 words */
 wlen = MIN( WORDS( Name), 4);
 Name = SUBWORD( Name, 1, wlen);
 /* spaces to underscore */
 Name = SPACE( Name,, '_');

 /* handle non-alphabetic name */
 IF (Name = '') THEN
    Name = 'NONAME'

 /* make it unique by maybe appending a number */
 Stem = Name;
 n = 0;
 DO FOREVER
    IF (WORDPOS( Name, Pages._AnchorList) > 0) THEN
    DO
       n = n + 1;
       Name = Stem'_'n;
    END;
    ELSE
    DO
       AnchorName = Name;
       LEAVE;
    END;
 END;

 RETURN( AnchorName);

/* ------------------------------------------------------------------------- */
SetResId: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG ResId;

 p = Pages.0;
 /* store resid to replace it later in links that use 'res=' */
 Pages.p._ResId = ResId;

 RETURN( '');

/* ------------------------------------------------------------------------- */
GetDimsForResId: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG r;

 Dims = '';
 SELECT
    WHEN (r = '') THEN NOP;
    WHEN (DATATYPE( r) \= 'NUM') THEN NOP;
    WHEN (SYMBOL( 'ResId.r._Dimensions') \= 'VAR') THEN NOP;
 OTHERWISE
    Dims = ResId.r._Dimensions;
 END;

 RETURN( Dims);

/* ------------------------------------------------------------------------- */
GetDimsForRefId: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG r;

 Dims = '';
 SELECT
    WHEN (r = '') THEN NOP;
    WHEN (SYMBOL( 'RefId.r._Dimensions') \= 'VAR') THEN NOP;
 OTHERWISE
    Dims = RefId.r._Dimensions;
 END;

 RETURN( Dims);

/* ------------------------------------------------------------------------- */
Halt:
 SAY;
 SAY 'Interrupted by user.';

 EXIT( ERROR.GEN_FAILURE);

/* ------------------------------------------------------------------------- */
ShowHelp: PROCEDURE EXPOSE (GlobalVars)

 PARSE SOURCE . . ThisFile;

 SAY;
 SAY Title;
 SAY;

 /* open file */
 rcx = STREAM( ThisFile, 'c', 'open read');

 /* skip header */
 DO i = 1 TO 3
    rcx = LINEIN( ThisFile);
 END;

 /* show help */
 DO WHILE (ThisLine \= ' */')
    ThisLine = LINEIN( Thisfile);
    SAY SUBSTR( ThisLine, 7);
 END;

 /* close file */
 rcx = STREAM( ThisFile, 'c', 'close');

 RETURN( '');

/* ------------------------------------------------------------------------- */
FileExist: PROCEDURE
 ARG FileName;

 RETURN( STREAM( Filename, 'C', 'QUERY EXISTS') \= '');

/* ------------------------------------------------------------------------- */
GetCalldir: PROCEDURE
 PARSE SOURCE . . CallName
 CallDir = FILESPEC( 'Drive', CallName)||FILESPEC( 'Path', CallName);
 RETURN( LEFT( CallDir, LENGTH( CallDir) - 1));

/* ------------------------------------------------------------------------- */
/* clone of SysSearchPath, which does not return fully qualified name */
SearchPath: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG EnvVar, Filename;

 FoundFilename = '';

 DO UNTIL (TRUE)
    EnvPath = VALUE( EnvVar,,env);

    DO WHILE (EnvPath \= '')
       PARSE VAR EnvPath ThisDir';'EnvPath;
       ThisFile = ThisDir'\'Filename;
       IF (FileExist( ThisFile)) THEN
       DO
          FoundFilename = ThisFile;
          LEAVE;
       END;
    END;
 END;

 RETURN( FoundFilename);

/* ------------------------------------------------------------------------- */
ChangeExtension: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Filename, NewExtension;

 /* prepend NewExtension with the leading dot if not specified */
 IF (LEFT( NewExtension, 1) \= '.') THEN
    NewExtension = '.'NewExtension;

 BaseName   = FILESPEC( 'N', Filename);
 BaseExtPos = LASTPOS( '.', BaseName);
 IF (BaseExtPos < 2) THEN
    NewName = FileName''NewExtension;
 ELSE
    NewName = SUBSTR( FileName, 1,,
                      LENGTH( Filename) - LENGTH( BaseName) + BaseExtPos - 1)''NewExtension;

 RETURN( NewName);

