MODULE MetaStore;
(* ========================================================================= *)
(*                                                                           *)
(*  Meta information storage module for the .NET to Gardens Point Component  *)
(*  Pascal Symbols tool.                                                     *)
(*      Copyright (c) Siu-Yuen Chan 2001.                                    *)
(*                                                                           *)
(*  This module defines various data structures and methods of a METASTORE,  *)
(*  which is for storing meta information read by the MetaParser module.     *)
(* ========================================================================= *)

IMPORT
    OH := ObjectHash,
    Error,
    RTS,
    ST := AscString;

CONST
    DefaultHashSize = 4000;

CONST
    (* the option of printing the assembly information *)
    NS = 1;             (* prints the namespaces information *)
    TP = 2;             (* prints the type information *)

CONST
    AssemblyExt* = ".dll";
    TypeLibExt*  = ".mcl";
    AnonArrHead  = "Arr";
    PointerHead  = "Ptr_";
    AnonArr      = "_arr";
    AnonRec      = "_rec";
    OrigCtor     = ".ctor";
    ReplCtor     = "init";
    Void*        = "Void";
    NULLSPACE*   = "(NullSpace)";     (* '(' precede all alphanumeric names *)
    DmyAssembly* = "GPCP";
    hlen         = LEN(AnonArrHead);

CONST
    PositiveSingleInfinity    = 2139095040;  (* 0x7F800000 *)
    NegativeSingleInfinity    = -8388608;    (* 0xFF800000 *)
    SingleNaN                 = -4194304;    (* 0xFFC00000 *)

CONST
    (* map from CP to .NET *)
    GpcpBoolean = "BOOLEAN";    DNetBoolean = "Boolean";    (* T01 *)
    GpcpSChar   = "SHORTCHAR";                              (* T02 *)
    GpcpChar    = "CHAR";       DNetChar    = "Char";
    GpcpDouble  = "REAL";       DNetDouble  = "Double";
    GpcpInt16   = "SHORTINT";   DNetInt16   = "Int16";
    GpcpInt32   = "INTEGER";    DNetInt32   = "Int32";
    GpcpInt64   = "LONGINT";    DNetInt64   = "Int64";
                                DNetIntPtr  = "IntPtr";
                                DNetObject  = "Object";
    GpcpSByte   = "BYTE";       DNetSByte   = "SByte";
    GpcpUByte   = "UBYTE";      DNetUByte   = "Byte";
    GpcpSingle  = "SHORTREAL";  DNetSingle  = "Single";
                                DNetString  = "String";
    GpcpTrue    = "TRUE";       DNetTrue    = "True";
    GpcpFalse   = "FALSE";      DNetFalse   = "False";

    (* not exist in CP, however ..., exist in many Microsoft dlls *)
 (* GpcpByte    = "BYTE";    *) DNetByte    = "Byte";
 (* GpcpUInt16  = "SHORTINT";*) DNetUInt16  = "UInt16";
 (* GpcpUInt32  = "INTEGER"; *) DNetUInt32  = "UInt32";
 (* GpcpUInt64  = "LONGINT"; *) DNetUInt64  = "UInt64";
                                DNetUIntPtr = "UIntPtr";
    (* exist in CP only *)
    GpcpSet     = "SET";
    GpcpAnyRec  = "ANYREC";
    GpcpAnyPtr  = "ANYPTR";
    GpcpCharArr = "POINTER TO ARRAY OF CHAR";

CONST
    (* possible type declarations *)
    Unknown*   = -1;
    Named*     = 1;
    Pointer*   = 2;

    (* possible value type declarations *)
    Enum*      = 4;
    Struct*    = 8;
    Primitive* = 16;
    
    (* possible reference type declarations *)
    RefType*   = 32;    (* all types above this are reference type *)
    Interface* = 32;
    Class*     = 64;
    Delegate*  = 128;
    Array*     = 256;
    Nested*    = 512;

CONST (* base-ordinals *)
    builtinTypeNum* = 16;
    notBs*  =  0;       unCertain* = 0;
    boolN*  =  1;       (* BOOLEAN *)
    sChrN*  =  2;       (* SHORTCHAR *)
    charN*  =  3;       (* CHAR *)
    byteN*  =  4;       (* BYTE *)
    sIntN*  =  5;       (* SHORTINT *)
    intN*   =  6;       (* INTEGER *)
    lIntN*  =  7;       (* LONGING *)
    sReaN*  =  8;       (* SHORTREAL *)
    realN*  =  9;       (* REAL *)
    setN*   = 10;       (* SET *)
    anyRec* = 11;       (* ANYREC *)
    anyPtr* = 12;       (* ANYPTR *)
    strN*   = 13;       (* STRING (ARRAY OF CHAR) *)
    sStrN*  = 14;       (* SHORTSTRING (ARRAY OF SHORTCHAR) *)
    uBytN*  = 15;       (* UBYTE *)
    metaN*  = 16;       (* META *)

CONST
    (* ======= Modifiers that can be obtained from meta information ======= *)
    (* ======= Not from what can be declared using C#               ======= *)
    (* Access modifiers, applies to classes, fields, interfaces, methods *)
    Vprivate*   = 0;             (* get_IsPrivate *)
    Vpublic*    = 1;             (* get_IsPublic *)
    Vreadonly*  = 2;             (* no such mode in .NET *)
    Vprotected* = 3;             (* get_IsFamily *)
    Vinternal*  = 4;             (* get_IsAssembly *)

    (* Record Attributes *)
    RabsBit = 0;                 (* bit 0 - abstract *)
    RlimBit = 1;                 (* bit 1 - limit *)
    RifaBit = 2;                 (* bit 2 - interface *)
    RnnaBit = 3;                 (* bit 3 - no NoArg constructor *)
    RvalBit = 4;                 (* bit 4 - value *)
    noAtt* = {};
    Rabstr* = {RabsBit};         (* get_IsAbstract *)
    Rlimit* = {RlimBit};         (* extend and allocate in defined module only *)
    Rextns* = {RabsBit,RlimBit}; (* NOT get_IsSealed *)
    RiFace* = {RifaBit};         (* get_IsInterface *)
    Rnnarg* = {RnnaBit};         (* class has no NoArg constructor *)
    RvalTp* = {RvalBit};         (* ValueType *)

    (* Field Attributes *)
    FconBit = 0;
    FstaBit = 1;
    Fconst* = {FconBit};         (* get_IsLiteral *)
    Fstat*  = {FstaBit};         (* get_IsStatic *)

    (* Method Attribute *)
    MnewBit*= 0;
    MabsBit*= 1;
    MempBit*= 2;
    Mcovar* = 3;                 (* ==> covariant return type *)
    Mfinal* = {};                (* get_IsFinal *)
    Mnew*   = {MnewBit};         (* NewSlot *)
    Mabstr* = {MabsBit};         (* get_IsAbstract *)
    Mempty* = {MempBit};
    MisExt* = {MabsBit,MempBit}; (* get_IsHideBySig *)
    Mmask*  = {MabsBit,MempBit};

    Mstatic* = TRUE;
    Mnonstatic* = FALSE;

CONST
    (* formal parameter attributes *)
    IsVal* = 0;                  (* Value parameter *)
    IsIn* = 1;                   (* IN parameter *)
    IsOut* = 2;                  (* OUT parameter *)
    IsVar* = 3;                  (* VAR parameter *)
    NotPar* = 4;

CONST
    (* ModulesName Types *)
    (* assembly name same as namespace name, and contains only one word,
       e.g. Accessibility.dll has only a namespace named Accessibility,
            and the module name should be:
                Accessibility 
            or
                Accessibility["[Accessibility]Accessibility"] *)
    SingleWord = 0;     

    (* assembly name same as namespace name, and contains multiple word,
       e.g. Microsoft.Win32.InterOp.dll has a namespace named Microsoft.Win32.InterOp,
            and the module name shoulle be:
                Microsoft_Win32_InterOp["[Microsoft.Win32.InterOp]Microsoft.Win32.InterOp"] *)
    MultipleWord = 1;

    (* assembly name different form namespace name, contains multiple word, and 
       with namespace name includes the entire assembly name
       e.g. Microsoft.Win32.InterOp.dll has a namespace named Microsoft.Win32.InterOp.Trident,
            and the module name shoulle be:
                Microsoft_Win32_InterOp_Trident["[Microsoft.Win32.InterOp]Microsoft.Win32.InterOp.Trident"] *)
    IncludeWord = 3;

    (* assembly name different from namespace name, contains multiple word, and
       with no relationship between assembly name and namespace name
       e.g. mscorlib.dll has a namespace named System.Reflection,
            and the module name should be:
                mscorlib_System_Reflection["[mscorlib]System.Reflection"] *)
    DifferentWord = 2;

CONST
    (* assembly version constant *)
    Major* = 0;
    Minor* = 1;
    Build* = 2;
    Revis* = 3;
    (* assembly strong name constant *)
    StrongNameLen= 16;               (* 16 hex digits *)
    B1StrongName = "SN=";
    B2StrongName = "PublicKeyToken=";

CONST
    IsNamespace  = TRUE;
    NotNamespace = FALSE;


TYPE
(*
    CharOpen* = POINTER TO ARRAY OF CHAR;
*)
    CharOpen* = ST.CharOpen;
    Attribute* = SET;
    
    (* this is NOT the object in [mscorlib]System.Object *)
    Object* = POINTER TO EXTENSIBLE 
        RECORD (OH.Object)
            name : CharOpen;       (* name of object *)
            fname: CharOpen;       (* fname of object *)
        END;

    AccessObject* = POINTER TO EXTENSIBLE
        RECORD (Object)
         (* name : CharOpen;  *)
         (* fname: CharOpen;  *)
            vmod : INTEGER;        (* visibility mode *)
            attr : Attribute;      (* attribute *)
        END;

    (* this list is ordered by the name of object *)
    OrderList* = POINTER TO EXTENSIBLE
        RECORD
            list   : OrderList;
            object : Object;
        END;

    FTraverser* = POINTER TO
        RECORD
            list : POINTER TO ARRAY OF Formal;
            curr : INTEGER;
        END;
    OTraverser* = POINTER TO
        RECORD
            list : OrderList;
            curr : OrderList;
        END;

    (* ===================================================================== *)

    MetaStructure = POINTER TO
        RECORD
            asmThis: Assembly;
            asmList: AssemblyList;
        END;

    Version* = POINTER TO ARRAY 4 OF SHORTINT;

    PublicKeyToken* = POINTER TO ARRAY 8 OF BYTE;

    Assembly* = POINTER TO 
        RECORD (Object)
         (* name : CharOpen   *)   (* only assembly name *)
         (* fname: CharOpen;  *)   (* including assembly name, ver & s/n *)
         (* ==== END OF Object ========== *)
            file : CharOpen;       (* file name of assembly (no path, no ext) *)
            vers : Version;        (* assembly version if any *)
            token: PublicKeyToken; (* public key token of assembly originator *)
            slist: OrderList;      (* ref to a list of local namespaces *)
            ktab : OH.HashTable;   (* for fast search of namespace and types *)
            snap : OrderList;      (* for a snapshot of all namespaces *)
        END;

    AssemblyList* = POINTER TO 
        RECORD (OrderList)
         (* list : OrderList; *)   (* links to other foreign Assembly objects *)
         (* object : Object; *)    (* namespace objects *)
        END;

    Namespace* = POINTER TO 
        RECORD (Object)
         (* name  : CharOpen; *)
         (* fname : CharOpen; *)   (* full hierarchical name *)
         (* ==== END OF Object ========== *)
            mord : INTEGER;        (* ModuleOrd for imported modules, 0 initially *)
            child : OrderList;
            asb   : Assembly;      (* the Assembly this namespace belongs *)
            isNS  : BOOLEAN;       (* already formed a namespace ? *)
            fnslist: SpaceList;    (* foreign assemblies *)
            tlist : OrderList;
            ktab : OH.HashTable;   (* for fast search of types *)
        END;

    SpaceList* = POINTER TO 
        RECORD (OrderList)
         (* list : OrderList; *)   (* links to other namespace objects *)
         (* object : Object; *)    (* namespace objects *)
        END;

    Type* = POINTER TO EXTENSIBLE
        RECORD (AccessObject)
         (* name : CharOpen;  *)
         (* fname: CharOpen;  *)   (* full qualified type name *)
         (* ==== END OF Object ==== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
            tord : INTEGER;        (* TypeOrd for writing sym file, 0 initially *)
            space: Namespace;      (* the Namespace this type belongs *)
            inhie: BOOLEAN;        (* type in declared type hierarchy, used by
                                      SymWriter *)
            anon : BOOLEAN;        (* whether type is anonymous *)
            aptr : PointerType;    (* back pointer to a anonymous PointerType(if there is one) pointing to this *)
        END;

    (* stores information parsed by SymReader, referenced by TypeList, not to be inserted into namespace tlist *)
    TempType* = POINTER TO EXTENSIBLE
        RECORD (Type)
         (* name : CharOpen;  *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
            dim  : INTEGER;         (* for storing information of   *)
            len  : INTEGER;         (* anonymous array with unknown *)
            elm  : Type;            (* element type                 *)
            cross: CrossRefs;
        END;

    (* will be inserted into namespace tlist, could later be replaced by PointerType, ArrayType or RecordType *)
    NamedType* = POINTER TO EXTENSIBLE
        RECORD (TempType)
         (* name : CharOpen;  *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
         (* dim  : INTEGER;   *)
         (* len  : INTEGER;   *)
         (* elm  : Type;      *)
         (* cross: CrossRefs; *)
         (* ==== END OF TempType ======== *)
        END;

    PointerType* = POINTER TO EXTENSIBLE
        RECORD (Type)
         (* name : CharOpen;  *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
            tgt  : Type;
        END;

    ArrayType* = POINTER TO EXTENSIBLE
        RECORD (Type)
         (* name : CharOpen   *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
            dim  : INTEGER;
            len  : INTEGER;
            elm  : Type;
        END;

    RecordType* = POINTER TO EXTENSIBLE
        RECORD (Type)
         (* name : CharOpen   *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
            base : Type;           (* base type if any *)
            impl : OrderList;      (* list of interfaces implementing *)
            dtype: Type;           (* declaring type if nested *)
            nlist: OrderList;      (* list of nested types *)
        END;

    ValueType* = POINTER TO EXTENSIBLE
        RECORD (RecordType)
         (* name : CharOpen   *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
         (* base : Type;      *)
         (* impl : OrderList; *)
         (* dtype: Type;      *)
         (* nlist: OrderList; *)
         (* ==== END OF RecordType ====== *)
            clist: ConstList;      (* list of class constants *)
        END;

    EnumType * = POINTER TO EXTENSIBLE
        RECORD (ValueType)
         (* name : CharOpen   *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
         (* base : Type;      *)
         (* impl : OrderList; *)
         (* dtype: Type;      *)
         (* nlist: OrderList; *)
         (* ==== END OF RecordType ====== *)
         (* clist: ConstList; *)
         (* ==== END OF ValueType ======= *)
            utype: CharOpen;       (* underlying type of field values *)
        END;

    (* includes: Boolean, Byte, SByte, Char, Double, Single, 
                 Int16, UInt16, Int32, UInt32, Int64, UInt64, Intptr, UIntPtr  *) 
    PrimType* = POINTER TO EXTENSIBLE
        RECORD (ValueType)
         (* name : CharOpen   *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
         (* base : Type;      *)
         (* impl : OrderList; *)
         (* dtype: Type;      *)
         (* nlist: OrderList; *)
         (* ==== END OF RecordType ====== *)
         (* clist: ConstList; *)
         (* ==== END OF ValueType ======= *)
            naCtor: BOOLEAN;       (* has no arg constructor *)
            vmlist: MethodList;    (* list of virtual & instance methods *)
            smlist: MethodList;    (* list of static methods *)
            sflist: FieldList;     (* list of static fields *)
        END;

    StrucType* = POINTER TO EXTENSIBLE
        RECORD (PrimType)
         (* name : CharOpen   *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
         (* base : Type;      *)
         (* impl : OrderList; *)
         (* clist: ConstList; *)
         (* nlist: OrderList; *)
         (* ==== END OF RecordType ====== *)
         (* dtype: Type;      *)
         (* ==== END OF ValueType ======= *)
         (* naCtor: BOOLEAN;  *)
         (* vmlist: MethodList; *)
         (* smlist: MethodList; *)
         (* sflist: FieldList;  *)
         (* ==== END OF PrimType ======= *)
            iflist: FieldList;     (* list of instance fields *)
        END;

    DelegType* = POINTER TO EXTENSIBLE
        RECORD (StrucType)
         (* name : CharOpen;  *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
         (* base : Type;      *)
         (* impl : OrderList; *)
         (* dtype: Type;      *)
         (* nlist: OrderList; *)
         (* ==== END OF RecordType ====== *)
         (* clist: ConstList; *)
         (* ==== END OF ValueType ======= *)
         (* naCtor: BOOLEAN;  *)
         (* vmlist: MethodList; *)
         (* smlist: MethodList; *)
         (* sflist: FieldList; *)
         (* ==== END OF PrimType ======= *)
         (* iflist: FieldList; *)  (* list of instance fields *)
         (* ==== END OF StrucType ======= *)
            isMul : BOOLEAN;       (* is multicast *)
        END;

    ClassType* = POINTER TO EXTENSIBLE 
        RECORD (StrucType)
         (* name : CharOpen   *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
         (* base : Type;      *)
         (* impl : OrderList; *)
         (* dtype: Type;      *)
         (* nlist: OrderList; *)
         (* ==== END OF RecordType ====== *)
         (* clist: ConstList; *)
         (* ==== END OF ValueType ======= *)
         (* naCtor: BOOLEAN;  *)
         (* vmlist: MethodList; *)
         (* smlist: MethodList; *)
         (* sflist: FieldList; *)
         (* ==== END OF PrimType ======= *)
         (* iflist: FieldList; *)
         (* ==== END OF StrucType ======= *)
            elist: EventList;      (* list of events *)
        END;

    (* A managed interface cannot contain data member or static member *)
    IntfcType* = POINTER TO EXTENSIBLE
        RECORD (RecordType)
         (* name : CharOpen   *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* tord : INTEGER;   *)
         (* space: Namespace; *)
         (* inhie: BOOLEAN;   *)
         (* anon : BOOLEAN;   *)
         (* aptr : PointerType; *)
         (* ==== END OF Type ============ *)
         (* base : Type;      *)
         (* impl : OrderList; *)
         (* dtype: Type;      *)
         (* nlist: OrderList; *)
         (* ==== END OF RecordType ====== *)
            vmlist: MethodList;    (* list of virtual & instance methods *)
        END;

    Method* = POINTER TO EXTENSIBLE
        RECORD (AccessObject)
         (* name : CharOpen   *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
            isCtor: BOOLEAN;       (* method is constructor *)
            isStat: BOOLEAN;       (* method is static, i.e. non-instance *)
            isVirt: BOOLEAN;       (* method is virtual *)
            isOVL: BOOLEAN;        (* method is overloaded *)
            oname: CharOpen;       (* method overload name(for GPCP) *)
            iname: CharOpen;       (* method invoke name(for .NET) *)
            formals: FormalList;   (* list of formals *)
            class: Type;           (* the class that this method belongs *)
        END;

    Function* = POINTER TO 
        RECORD (Method)
         (* name : CharOpen   *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* isCtor: BOOLEAN; *)    (* method is constructor *)
         (* isStat: BOOLEAN; *)    (* method is static, i.e. non-instance *)
         (* isVirt: BOOLEAN; *)    (* method is virtual *)
         (* isOVL: BOOLEAN;  *)    (* method is overloaded *)
         (* oname: CharOpen; *)    (* method overload name *)
         (* formals: FormalList; *)
         (* class: Type;     *)    (* the class that this method belongs *)
         (* ==== END OF Method ========== *)
            rtype: Type;
            ostd*: INTEGER;        (* un-resolved(outstanding) type name *)
            rtnPtr: BOOLEAN;       (* method returns a pointer to value type *)
        END;

    Event* = POINTER TO
        RECORD (Object)
         (* name  : CharOpen *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
            dtype : Type;          (* Declaring Type *)
            htype : Type;          (* Handling Type *)
            isMul : BOOLEAN;       (* is multicast *)
            addOn : Method;        (* "AddOn" method *)
            remOn : Method;        (* "RemoveOn" method *)
        END;

    Formal* = POINTER TO 
        RECORD (Object)
         (* name: CharOpen; *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
            type: Type;
            inout: INTEGER;        (* IN, OUT, ... *)
            fmlist: FormalList;    (* the formal list contains the formal *)
        END;

    Field* = POINTER TO EXTENSIBLE
        RECORD (AccessObject)
         (* name : CharOpen; *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
            type : Type;           (* type of the field *)
            isPtr: BOOLEAN;        (* field is a pointer to a value type *)
            class: Type;
        END;

    (* static literals *)
    Constant* = POINTER TO 
        RECORD (Field)
         (* name : CharOpen; *)
         (* fname: CharOpen;  *)
         (* ==== END OF Object ========== *)
         (* vmod : INTEGER;   *)
         (* attr : Attribute; *)
         (* ==== END OF AccessObject ==== *)
         (* type : Type;     *)
         (* isPtr: BOOLEAN   *)
            value: Literal;        (* for static string literal fields *)
        END;

    ConstList* = POINTER TO
        RECORD (OrderList)
         (* list: OrderList; *)    (* links to other constant objects declared
                                      in the same type *)
         (* object : Object; *)    (* constant objects *)
        END;

    EventList* = POINTER TO
        RECORD (OrderList)
         (* list: OrderList; *)    (* links to other event objects declared
                                      in the same type *)
         (* object : Object; *)    (* event objects *)
        END;

    FieldList* = POINTER TO 
        RECORD (OrderList)
         (* list: OrderList; *)    (* links to other field objects declared
                                      in the same type *)
         (* object : Object; *)    (* field objects *)
        END;

    MethodList* = POINTER TO
        RECORD (OrderList)
         (* list: OrderList; *)    (* links to other method objects declared
                                      in the same type *)
         (* object : Object; *)    (* method objects *)
        END;

    FormalList* = POINTER TO
        RECORD
            len : INTEGER;         (* number of formals in the formal list *)
            fmls*: POINTER TO ARRAY OF Formal;
            mth : Method;          (* the method that HAS this formal list *)
            sCode: CharOpen;       (* signature code for the method HAS this formal list*)
            ostd*: INTEGER;        (* number of name un-resolved(outstanding) formal types *)
        END;

    (* holding cross references for TempType and NamedType *)
    (* need to fix the refs when TempType  ==> NamedType,
                         or when NamedType ==> Concrete Type *)
    CrossRefs* = POINTER TO
        RECORD
            (* if the host type object of this list is an element type of an anonymous array (ARRAY OF ???),
               the list references to the Anonymous Array that has this element type *)
            arlist: ArList;
            (* if the host type object of this list is a basetype of RecordType(s),
               the list references to all RecordTypes that derive from this type *)
            bslist: BsList;
            (* if the host type object of this list is the interface type implement by RecordType(s),
               the list reference to all RecordTypes that implement this type *)
            iplist: IpList;
            (* if the host type object of this list is the type of record field(s),
               the list references to all fields that have this type *)
            fdlist: FdList;
            (* if the host type object of this list is the type of formal(s),
               the list references to all formals that have this type *)
            fmlist: FmList;
            (* if the host type object of this list is the return type of method(s),
               the list references to all methods that have this return type *)
            fnlist: FnList;
            (* if the host type object of this list is a target type of PointerType(s),
               the list references to all PointerTypes that point to this type *)
            ptlist: PtList;

        END;

    (* for fixing anonymous array types *)
    ArList = POINTER TO
        RECORD
            typ: Type;
            nxt: ArList;
        END;

    (* for fixing base types *)
    BsList = POINTER TO
        RECORD
            typ: RecordType;
            nxt: BsList;
        END;

    (* for fixing implement interface types *)
    IpList = POINTER TO
        RECORD
            typ: RecordType;
            nxt: IpList;
        END;

    (* for fixing field types *)
    FdList = POINTER TO
        RECORD
            fld: Field;
            nxt: FdList;
        END;

    (* for fixing formal types *)
    FmList = POINTER TO
        RECORD
            fml: Formal;
            nxt: FmList;
        END;

    (* for fixing method return types *)
    FnList = POINTER TO
        RECORD
            fn: Function;
            nxt: FnList;
        END;

    (* for fixing pointer target types *)
    PtList = POINTER TO
        RECORD
            typ: PointerType;
            nxt: PtList;
        END;



    Literal* = POINTER TO EXTENSIBLE RECORD END;
    BoolLiteral* = POINTER TO RECORD (Literal) value: BOOLEAN; END;
    CharLiteral* = POINTER TO RECORD (Literal) value: CHAR; END;
    SetLiteral*  = POINTER TO RECORD (Literal) value: SET; END;
    StrLiteral*  = POINTER TO RECORD (Literal) value: CharOpen; END;
    NumLiteral*  = POINTER TO EXTENSIBLE RECORD (Literal) END;
    ByteLiteral* = POINTER TO RECORD (NumLiteral) value: BYTE; END;
    IntLiteral*  = POINTER TO RECORD (NumLiteral) value: INTEGER; END;
    LIntLiteral* = POINTER TO RECORD (NumLiteral) value: LONGINT; END;
    SIntLiteral* = POINTER TO RECORD (NumLiteral) value: SHORTINT; END;
    FloatLiteral*= POINTER TO EXTENSIBLE RECORD (Literal) END;
    RealLiteral* = POINTER TO RECORD (FloatLiteral) value: REAL; END;
    SReaLiteral* = POINTER TO RECORD (FloatLiteral) value: SHORTREAL; END;


VAR
    meta        : MetaStructure;

    b1StrongName: CharOpen;
    b2StrongName: CharOpen;

    anonArr*    : CharOpen;
    anonRec*    : CharOpen;
    origCtor*   : CharOpen;
    replCtor*   : CharOpen;
    void*       : CharOpen;
    nullspace*  : CharOpen;
    gpcpBoolean, dnetBoolean : CharOpen;
(*
 *  gpcpByte   , dnetByte    : CharOpen;
 *)
    gpcpUByte  , dnetByte    : CharOpen;  (* new *)
    gpcpChar   , dnetChar    : CharOpen;
    gpcpSChar                : CharOpen;
    gpcpDouble , dnetDouble  : CharOpen;
    gpcpInt16  , dnetInt16   : CharOpen;
    gpcpInt32  , dnetInt32   : CharOpen;
    gpcpInt64  , dnetInt64   : CharOpen;
                 dnetIntPtr  : CharOpen;
                 dnetObject  : CharOpen;
    gpcpSByte  , dnetSByte   : CharOpen;
    gpcpSingle , dnetSingle  : CharOpen;
                 dnetString  : CharOpen;
    gpcpUInt16 , dnetUInt16  : CharOpen;
    gpcpUInt32 , dnetUInt32  : CharOpen;
    gpcpUInt64 , dnetUInt64  : CharOpen;
                 dnetUIntPtr : CharOpen;
    gpcpTrue   , dnetTrue    : CharOpen;
    gpcpFalse  , dnetFalse   : CharOpen;
    gpcpSet     : CharOpen;
    gpcpAnyRec  : CharOpen;
    gpcpAnyPtr  : CharOpen;
    gpcpCharArr : CharOpen;

    printOpt    : SET;

VAR
    HashSize: INTEGER;
    baseTypeArray- : ARRAY builtinTypeNum OF PrimType;
    dmyTyp-: Type;
    MethodNameMangling: BOOLEAN;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE ^ (ao: AccessObject) IsExported*(): BOOLEAN, NEW;
PROCEDURE ^ (ao: AccessObject) IsPublic*(): BOOLEAN, NEW;
PROCEDURE ^ (ao: AccessObject) IsPrivate*(): BOOLEAN, NEW;
PROCEDURE ^ (ao: AccessObject) IsProtected*(): BOOLEAN, NEW;
PROCEDURE ^ (ns: Namespace) InsertType2(tt: INTEGER; tname: CharOpen; ftname: CharOpen;
                                        bt: Type; dt: Type; il: OrderList): Type, NEW;
PROCEDURE ^ NewType(tt: INTEGER): Type;
PROCEDURE ^ (typ: Type) Initialize(name: CharOpen; fname: CharOpen;
                                   space: Namespace), NEW, EXTENSIBLE;
PROCEDURE ^ (typ: Type) NotAnonymous*(), NEW;
PROCEDURE ^ (prim: PrimType) MarkPrimitiveTypeOrd*(), NEW;
PROCEDURE ^ (mth: Method) EncryptSignature*(), NEW;
PROCEDURE ^ (fl: FormalList) CreateSigCode*(fulrtyp: CharOpen), NEW;
PROCEDURE ^ (fl: FormalList) IsNoArg*(): BOOLEAN, NEW;
PROCEDURE ^ FixReferences*(tpTemp: Type; tpDesc: Type);

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE SetHashSize*(sz: INTEGER);
BEGIN
    HashSize := sz;
END SetHashSize;


PROCEDURE SetMethodNameManglingOn*();
BEGIN
    MethodNameMangling := TRUE;
END SetMethodNameManglingOn;


PROCEDURE SetMethodNameManglingOff*();
BEGIN
    MethodNameMangling := FALSE;
END SetMethodNameManglingOff;


PROCEDURE WithoutMethodNameMangling*(): BOOLEAN;
BEGIN
    RETURN (MethodNameMangling = FALSE);
END WithoutMethodNameMangling;


PROCEDURE IsOneWordName*(asbname: CharOpen; nsname: CharOpen): BOOLEAN;
(* both assembly name and namespace name contains no separators ('.' and '_') inside *)
BEGIN
    RETURN ((ST.StrChr(nsname,'.') = ST.NotExist) & (ST.StrChr(asbname,'_') = ST.NotExist));
END IsOneWordName;


PROCEDURE IsSameWordName*(asbname: CharOpen; nsname: CharOpen): BOOLEAN;
(* assembly name is identical to namespace name
   (excluding separators inside the names) *)
BEGIN
    RETURN (asbname^ = ST.StrSubChr(nsname,'.','_')^);
END IsSameWordName;


PROCEDURE IsInclWordName*(asbname: CharOpen; nsname: CharOpen): BOOLEAN;
(* Returns TRUE if asbname is include inside nsname (excluding separators) *)
VAR
    head: CharOpen;
    idx: INTEGER;
BEGIN
    idx := LEN(asbname) - 1;
    head := ST.SubStr(nsname, 0, idx-1);
    RETURN ( (asbname^ = ST.StrSubChr(head,'.','_')^) &
             ((nsname[idx] = '.') OR (nsname[idx] = '_')));
END IsInclWordName;


PROCEDURE MakeModuleName*(asbname: CharOpen; nsname: CharOpen): CharOpen;
VAR
    onewordname: BOOLEAN;
    samewordname: BOOLEAN;
    inclwordname: BOOLEAN;
    modname: CharOpen;
    tail: CharOpen;
BEGIN
    onewordname := IsOneWordName(asbname, nsname);
    samewordname := IsSameWordName(asbname, nsname);
    inclwordname := IsInclWordName(asbname, nsname);

    IF samewordname THEN
        (* e.g. assembly: Accessibility.dll, namespace: Accessibility *)
        (*      assembly: Microsoft_Win32_InterOp.dll, namespace: Microsoft.Win32.InterOp *)
        (* cannot be null namespace here *)
        modname := ST.StrCatChr(asbname,'_');
    ELSE
        IF inclwordname THEN
            IF ST.StrCmp(asbname,ST.NullString) = ST.Equal THEN
                (* dummy primitive types have a NullString asbname *)
                modname := ST.NullString;
            ELSE
                (* e.g. assembly: Microsoft.VisualBasic.dll, namespace:Microsoft.VisualBasic.Helpers *)
                tail := ST.StrSubChr(ST.SubStr(nsname, LEN(asbname), LEN(nsname)-1), '.', '_');
                modname := ST.StrCat(asbname, ST.ToChrOpen("__"));
                modname := ST.StrCat(modname, tail);
            END; (* IF *)
        ELSE
            IF onewordname THEN
                IF nsname^ = NULLSPACE THEN
                    (* e.g. assembly: RTS.dll, namespace: nothing *)
                    modname := asbname;
                ELSE
                    (* e.g. assembly: IEHost.dll, namespace: IIEHost *)
                    modname := ST.StrCatChr(asbname,'_');
                    modname := ST.StrCat(modname, ST.StrSubChr(nsname,'.','_'));
                END; (* IF *)
            ELSE
                IF nsname^ = NULLSPACE THEN
                    (* e.g. assembly: CustomMarshalers.dll, namespace: nothing *)
                    modname := asbname;
                ELSE
                    (* e.g. assembly: CustomMarshalers.dll, namespace: System.Runtime.InteropServices.CustomMarshalers *)
                    (* e.g. assembly: Microsoft.Vsa.Vb.CodeDOMProcessor, namespace: Microsoft.Vsa.Vb.CodeDOM *)
                    (* e.g. assembly: Base, namespace: BaseN *)
                    modname := ST.StrCatChr(asbname,'_');
                    modname := ST.StrCat(modname, ST.StrSubChr(nsname,'.','_'));
                END; (* IF *)
            END; (* IF *)
        END; (* IF *)
    END; (* IF *)

    RETURN modname;
END MakeModuleName;


PROCEDURE MakeCharLiteral*(c: CHAR): CharLiteral;
VAR
    lit: CharLiteral;
BEGIN
    NEW(lit); lit.value := c; RETURN lit;
END MakeCharLiteral;


PROCEDURE MakeBoolLiteral*(b: BOOLEAN): BoolLiteral;
VAR
    lit: BoolLiteral;
BEGIN
    NEW(lit); lit.value := b; RETURN lit;
END MakeBoolLiteral;


PROCEDURE MakeByteLiteral*(b: BYTE): ByteLiteral;
VAR
    lit: ByteLiteral;
BEGIN
    NEW(lit); lit.value := b; RETURN lit;
END MakeByteLiteral;


PROCEDURE MakeIntLiteral*(i: INTEGER): IntLiteral;
VAR
    lit: IntLiteral;
BEGIN
    NEW(lit); lit.value := i; RETURN lit;
END MakeIntLiteral;


PROCEDURE MakeLIntLiteral*(l: LONGINT): LIntLiteral;
VAR
    lit: LIntLiteral;
BEGIN
    NEW(lit); lit.value := l; RETURN lit;
END MakeLIntLiteral;


PROCEDURE MakeRealLiteral*(r: REAL): RealLiteral;
VAR
    lit: RealLiteral;
BEGIN
    NEW(lit); lit.value := r; RETURN lit;
END MakeRealLiteral;


PROCEDURE MakeSetLiteral*(s: SET): SetLiteral;
VAR
    lit: SetLiteral;
BEGIN
    NEW(lit); lit.value := s; RETURN lit;
END MakeSetLiteral;


PROCEDURE MakeSIntLiteral*(s: SHORTINT): SIntLiteral;
VAR
    lit: SIntLiteral;
BEGIN
    NEW(lit); lit.value := s; RETURN lit;
END MakeSIntLiteral;


PROCEDURE MakeSReaLiteral*(s: SHORTREAL): SReaLiteral;
VAR
    lit: SReaLiteral;
BEGIN
    NEW(lit); lit.value := s; RETURN lit;
END MakeSReaLiteral;


PROCEDURE MakeStrLiteral*(s: CharOpen): StrLiteral;
VAR
    lit: StrLiteral;
BEGIN
    NEW(lit); lit.value := s; RETURN lit;
END MakeStrLiteral;


PROCEDURE CreateLiteral*(IN val: CharOpen; typ: CharOpen): Literal;
VAR
    rc : BOOLEAN;
    res: INTEGER;
    lit: Literal;
    bo : BOOLEAN;
    by : BYTE;
    in : INTEGER;
    li : LONGINT;
    re : REAL;
    si : SHORTINT;
    sr : SHORTREAL;
    st : CharOpen;
    len: INTEGER;
BEGIN
    (* currently handle literals of the following .net types *)
    (* ?,Boolean,?,Byte,?,Char,?,Double,?,Int16,?,Int32,
       ?,Int64,?,IntPtr,?,SByte,?,Single,?,String,?,UInt16,?,UInt32,?,UInt64,?,UIntPtr,? *)
    lit := NIL;

    CASE ST.StrCmp(typ, dnetInt32) OF
      ST.Equal  :
        RTS.StrToInt(val, in, rc);
        lit := MakeIntLiteral(in);
    | ST.Less   : (* ?,Boolean,?,Byte,?,Char,?,Double,?,Int16,? *)
        CASE ST.StrCmp(typ, dnetChar) OF
          ST.Equal  :
            lit := MakeCharLiteral(val[0]);
        | ST.Less   : (* ?,Boolean,?,Byte,? *)
            CASE ST.StrCmp(typ, dnetBoolean) OF
              ST.Equal  :
                RTS.StrToBool(val, bo, rc);
                lit := MakeBoolLiteral(bo);
            | ST.Less   : (* ? *)
                ASSERT(FALSE);
            ELSE (* ?,Byte,? *)
                CASE ST.StrCmp(typ, dnetByte) OF
                  ST.Equal  :
                    RTS.StrToUByte(val, by, rc);
                    lit := MakeByteLiteral(by);
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ? *)
                    ASSERT(FALSE);
                END; (* CASE *)
            END; (* CASE *)
        ELSE (* ?,Double,?,Int16,? *)
            CASE ST.StrCmp(typ, dnetDouble) OF
              ST.Equal  :
                RTS.StrToReal(val, re, rc);
                lit := MakeRealLiteral(re);
            | ST.Less   : (* ? *)
                ASSERT(FALSE);
            ELSE (* ?,Int16,? *)
                CASE ST.StrCmp(typ, dnetInt16) OF
                  ST.Equal  :
                    RTS.StrToShort(val, si, rc);
                    lit := MakeSIntLiteral(si);
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ? *)
                    ASSERT(FALSE);
                END; (* CASE *)
            END; (* CASE *)
        END; (* CASE *)
    ELSE (* ?,Int64,?,IntPtr,?,SByte,?,Single,?,String,?,UInt16,?,UInt32,?,UInt64,?,UIntPtr,? *)
        CASE ST.StrCmp(typ, dnetString) OF
          ST.Equal  :
            len := LEN(val);
            IF len > 0 THEN
                st := val;
            ELSE
                st := ST.NullString;
            END; (* IF *)
            lit := MakeStrLiteral(st);
        | ST.Less   : (* ?,Int64,?,IntPtr,?,SByte,?,Single,? *)
            CASE ST.StrCmp(typ, dnetSByte) OF
              ST.Equal  :
                RTS.StrToByte(val, by, rc);
                lit := MakeByteLiteral(by);
            | ST.Less   : (* ?,Int64,?,IntPtr,? *)
                CASE ST.StrCmp(typ, dnetInt64) OF
                  ST.Equal  :
                    RTS.StrToLong(val, li, rc);
                    lit := MakeLIntLiteral(li);
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ?,IntPtr,? *)
                    CASE ST.StrCmp(typ, dnetIntPtr) OF
                      ST.Equal  :
                        ASSERT(FALSE);
                    | ST.Less   : (* ? *)
                        ASSERT(FALSE);
                    ELSE (* ? *)
                        ASSERT(FALSE);
                    END; (* CASE *)
                END; (* CASE *)
            ELSE (* ?,Single,? *)
                CASE ST.StrCmp(typ, dnetSingle) OF
                  ST.Equal  :
                    RTS.StrToSReal(val, sr, rc);
                    lit := MakeSReaLiteral(sr);
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ? *)
                    ASSERT(FALSE);
                END; (* CASE *)
            END; (* IF *)
        ELSE (* ?,UInt16,?,UInt32,?,UInt64,?,UIntPtr, *)
            CASE ST.StrCmp(typ, dnetUInt32) OF
              ST.Equal  :
                RTS.StrToUInt(val, in, rc);
                lit := MakeIntLiteral(in);
            | ST.Less   : (* ?,UInt16,? *)
                CASE ST.StrCmp(typ, dnetUInt16) OF
                  ST.Equal  :
                    RTS.StrToUShort(val, si, rc);
                    lit := MakeSIntLiteral(si);
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ? *)
                    ASSERT(FALSE);
                END; (* CASE *)
            ELSE (* ?,UInt64,?,UIntPtr,? *)
                CASE ST.StrCmp(typ, dnetUInt64) OF
                  ST.Equal  :
                    RTS.StrToULong(val, li, rc);
                    lit := MakeLIntLiteral(li);
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ?,UIntPtr,? *)
                    CASE ST.StrCmp(typ, dnetUIntPtr) OF
                      ST.Equal  :
                        ASSERT(FALSE);
                    | ST.Less   : (* ? *)
                        ASSERT(FALSE);
                    ELSE (* ? *)
                        ASSERT(FALSE);
                    END; (* CASE *)
                END; (* CASE *)
            END; (* CASE *)
        END; (* CASE *)
    END; (* CASE *)

    RETURN lit;
END CreateLiteral;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE SplitTypeName*(IN fulltypename: CharOpen;
                        OUT fasbname: CharOpen;
                        OUT asbname: CharOpen;
                        OUT ftname: CharOpen);
(* For PDC version, the fulltypename is in the form of:
 * e.g. "System.Int32, mscorlib.dll"
 * or   "System.Configuration.Internal.QueryCell, mngdist, Ver=0.0.0.0, Loc="""
 * For beta 1 version, the fulltypename is in the form of:
 * e.g. "System.Int32, mscorlib"
 * or   "System.Configuration.Internal.QueryCell, mngdist, Ver=0.0.0.0, Loc="""
 * special case:
 * e.g. "Microsoft.ComServices.Admin.ICatalogCollection, Microsoft.COMServices"
 *
 * returns assembly name (asbname), full type name (ftname), type name (tname).
 *)
VAR
    idx: INTEGER;
    str: CharOpen;
BEGIN
    (* separate full type name and full assembly name *)
    idx := ST.StrChr(fulltypename,' ');
    ASSERT(idx # ST.NotExist);

    fasbname := ST.SubStr(fulltypename, idx+1, LEN(fulltypename)-1);
    idx := ST.StrChr(fasbname,',');
    IF idx # ST.NotExist THEN
        (* long form of assembly name *)
        asbname := ST.SubStr(fasbname, 0, idx-1);

        (* prepare for obtain ftname *)
        idx := ST.StrChr(fulltypename,',');
    ELSE
        (* short form of assembly name *)
        idx := ST.StrRChr(fasbname,'.');
        IF idx # ST.NotExist THEN
            (* either PDC version, or special case *)
            str := ST.SubStr(fasbname, idx, LEN(fasbname)-1);
            IF (str^ = AssemblyExt) OR (str^ = TypeLibExt) THEN
                (* PDC version, get rid of ".dll", or ".mcl" *)
                asbname := ST.SubStr(fasbname, 0, idx-1);
                fasbname := asbname;
            ELSE
                asbname := fasbname;
            END; (* IF *)
        ELSE
            (* beta 1 version *)
            asbname := fasbname;
        END; (* IF *)

        (* prepare for obtain ftname *)
        idx := ST.StrRChr(fulltypename,',');
    END; (* IF *)
    ASSERT(idx # ST.NotExist);
    ftname := ST.SubStr(fulltypename, 0, idx-1);
END SplitTypeName;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (o: Object) GetName*(): CharOpen, NEW, EXTENSIBLE;
BEGIN
    RETURN o.name;
END GetName;


PROCEDURE (o: Object) GetFullName*(): CharOpen, NEW, EXTENSIBLE;
BEGIN
    RETURN o.fname;
END GetFullName;


PROCEDURE (o: Object) SetFullName*(fname: CharOpen), NEW, EXTENSIBLE;
BEGIN
    o.fname := fname;
END SetFullName;


PROCEDURE (o: Object) GetRawFullName*(): CharOpen, NEW, EXTENSIBLE;
BEGIN
    RETURN o.fname;
END GetRawFullName;


PROCEDURE (o: Object) SetName*(name: CharOpen), NEW, EXTENSIBLE;
BEGIN
    o.name := name;
END SetName;


PROCEDURE (o: Object) InsertType*(tt: INTEGER; name: CharOpen;
                                  bt: Type; dt: Type; il: OrderList
                                 ): Type, NEW, EXTENSIBLE;
BEGIN
    RETURN NIL;
END InsertType;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (tv: FTraverser) Initialize*(fl: FormalList), NEW;
BEGIN
    tv.list := fl.fmls;
    tv.curr := 0;
END Initialize;


PROCEDURE (tv: FTraverser) GetNextFormal*(): Formal, NEW;
BEGIN
    IF tv.curr < LEN(tv.list) THEN
        INC(tv.curr);
        RETURN(tv.list[tv.curr-1](Formal));
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetNextFormal;


PROCEDURE (tv: OTraverser) Initialize*(list: OrderList), NEW;
BEGIN
    tv.list := list;
    tv.curr := list;
END Initialize;


PROCEDURE (tv: OTraverser) GetNextAssembly*(): Assembly, NEW;
BEGIN
    IF tv.curr.list # NIL THEN
        tv.curr := tv.curr.list;
        RETURN(tv.curr.object(Assembly));
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetNextAssembly;


PROCEDURE (tv: OTraverser) GetNextNamespace*(): Namespace, NEW;
BEGIN
    IF tv.curr.list # NIL THEN
        tv.curr := tv.curr.list;
        RETURN(tv.curr.object(Namespace));
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetNextNamespace;


PROCEDURE (tv: OTraverser) GetNextType*(): Type, NEW;
BEGIN
    IF tv.curr.list # NIL THEN
        tv.curr := tv.curr.list;
        RETURN(tv.curr.object(Type));
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetNextType;


PROCEDURE (tv: OTraverser) GetNextMethod*(): Method, NEW;
BEGIN
    IF tv.curr.list # NIL THEN
        tv.curr := tv.curr.list;
        RETURN(tv.curr.object(Method));
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetNextMethod;


PROCEDURE (tv: OTraverser) GetNextConstant*(): Constant, NEW;
BEGIN
    IF tv.curr.list # NIL THEN
        tv.curr := tv.curr.list;
        RETURN(tv.curr.object(Constant));
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetNextConstant;


PROCEDURE (tv: OTraverser) GetNextField*(): Field, NEW;
BEGIN
    IF tv.curr.list # NIL THEN
        tv.curr := tv.curr.list;
        RETURN(tv.curr.object(Field));
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetNextField;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (ol: OrderList) ASearch*(name: CharOpen; OUT ret: OrderList): BOOLEAN, NEW;
(* Search for an assembly name in a OrderList. 
 * IF object of same name already in the list,
 *   ret reference to where the object name is matched, 
 *   and return TRUE,
 * ELSE
 *   and return FALSE.
 *)
VAR
    cmpres: BOOLEAN;
    capName: CharOpen;
    capAsb: CharOpen;
BEGIN
    IF ol.list = NIL THEN ret := ol; RETURN FALSE; END;
    ret := ol;
    capName := ST.ToCap(name);
    REPEAT
        capAsb := ST.ToCap(ret.list.object.name);
        cmpres := (capAsb^ = capName^);
        IF ~cmpres THEN ret := ret.list; END;
    UNTIL cmpres OR (ret.list = NIL);
    IF cmpres THEN
        ret := ret.list;
        RETURN TRUE;
    ELSE
        RETURN FALSE;
    END; (* IF *)
END ASearch;


PROCEDURE (ol: OrderList) ISearch*(name: CharOpen; OUT ret: OrderList): BOOLEAN, NEW;
(* Search for a suitable place to insert a new object to the list.
 * IF object of same name already in the list,
 *   ret reference to where the object name is matched, 
 *   and return TRUE,
 * ELSE
 *   ret reference to where the object name should be inserted after, 
 *   and return FALSE.
 *)
VAR
    cmpres: INTEGER;
BEGIN
    IF ol.list = NIL THEN ret := ol; RETURN FALSE; END;
    ret := ol;
    REPEAT
        cmpres := ST.StrCmp(ret.list.object.name, name);
        IF cmpres = ST.Less THEN ret := ret.list; END;
    UNTIL (cmpres # ST.Less) OR (ret.list = NIL);
    IF cmpres = ST.Equal THEN
        ret := ret.list;
        RETURN TRUE;
    ELSE
        RETURN FALSE;
    END; (* IF *)
END ISearch;


PROCEDURE (ol: OrderList) MSearch*(name: CharOpen; oname: CharOpen; OUT ret: OrderList): BOOLEAN, NEW;
(* use for search method names *)
(* Search for a suitable place to insert a new method to the list.
 * IF method of same name already in the list,
 *   ret reference to where the method name is matched, 
 *   and return TRUE,
 * ELSE
 *   ret reference to where the method name should be inserted after, 
 *   and return FALSE.
 *)
VAR
    cmpres1: INTEGER;
    cmpres2: INTEGER;
BEGIN
    IF ol.list = NIL THEN ret := ol; RETURN FALSE; END;
    (* try to find the occurence of the method name *)
    ret := ol;
    REPEAT
        cmpres1 := ST.StrCmp(ret.list.object.name, name);
        IF cmpres1 = ST.Less THEN ret := ret.list; END;
    UNTIL (cmpres1 # ST.Less) OR (ret.list = NIL);

    IF cmpres1 = ST.Equal THEN
        (* now need to find the existence of overload method name *)
        REPEAT
            cmpres1 := ST.StrCmp(ret.list.object.name, name);
            cmpres2 := ST.StrCmp(ret.list.object(Method).oname, oname);
            IF (cmpres1 = ST.Equal) & (cmpres2 = ST.Less) THEN ret := ret.list; END;
        UNTIL (cmpres1 # ST.Equal) OR (cmpres2 # ST.Less) OR (ret.list = NIL) OR (ret.list.object(Method).oname = NIL);

        IF cmpres2 = ST.Equal THEN
            ret := ret.list;
            RETURN TRUE;
        ELSE
            RETURN FALSE;
        END; (* IF *)
    ELSE
        RETURN FALSE;
    END; (* IF *)
END MSearch;


PROCEDURE (ol: OrderList) IFSearch*(fname: CharOpen; OUT ret: OrderList): BOOLEAN, NEW;
(* Search for a suitable place to insert a new object to the list.
 * IF object of same fname already in the list,
 *   ret reference to where the object fname is matched, 
 *   and return TRUE,
 * ELSE
 *   ret reference to where the object fname should be inserted after, 
 *   and return FALSE.
 *)
VAR
    cmpres: INTEGER;
BEGIN
    IF ol.list = NIL THEN ret := ol; RETURN FALSE; END;
    ret := ol;
    REPEAT
        cmpres := ST.StrCmp(ret.list.object.fname, fname);
        IF cmpres = ST.Less THEN ret := ret.list; END;
    UNTIL (cmpres # ST.Less) OR (ret.list = NIL);
    IF cmpres = ST.Equal THEN
        ret := ret.list;
        RETURN TRUE;
    ELSE
        RETURN FALSE;
    END; (* IF *)
END IFSearch;


PROCEDURE (ol: OrderList) Insert*(o: Object), NEW;
(* insert after 'ol' - the current list node *)
VAR
    temp: OrderList;
BEGIN
    NEW(temp); temp.object := o; temp.list := ol.list; ol.list := temp;
END Insert;


PROCEDURE (ol: OrderList) Replace*(o: Object), NEW;
(* insert after 'ol' - the current list node *)
VAR
    temp: OrderList;
BEGIN
    ol.object := o;
END Replace;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE GetAllAssemblies*(): OrderList;
BEGIN
    RETURN meta.asmList;
END GetAllAssemblies;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE NewNamespace(name: CharOpen; fname: CharOpen;
                       asb: Assembly; isNS: BOOLEAN): Namespace;
VAR
    ns: Namespace;
BEGIN
    NEW(ns);
    ns.name := name;
    ns.mord := 0;
    ns.fname := fname;
    ns.isNS := isNS;
    ns.child := NIL;
    NEW(ns.fnslist); ns.fnslist.list := NIL; ns.fnslist.object := NIL;
    IF isNS THEN
        ns.asb := asb;
        NEW(ns.tlist); ns.tlist.list := NIL; ns.tlist.object := NIL;
    ELSE
        ns.asb := NIL;
        ns.tlist := NIL;
    END; (* IF *)
    NEW(ns.ktab); ns.ktab.InitObjectHash(HashSize);
    RETURN ns;
END NewNamespace;


PROCEDURE (ns: Namespace) Dumped*(): BOOLEAN, NEW;
BEGIN
    RETURN ns.mord # unCertain;
END Dumped;


PROCEDURE (ns: Namespace) GetModuleOrd*(): INTEGER, NEW;
BEGIN
    RETURN ns.mord;
END GetModuleOrd;


PROCEDURE (ns: Namespace) SetModuleOrd*(mord: INTEGER), NEW;
BEGIN
    ns.mord := mord;
END SetModuleOrd;


PROCEDURE (ns: Namespace) ClearModuleOrd*(), NEW;
BEGIN
    ns.mord := unCertain;
END ClearModuleOrd;


PROCEDURE (ns: Namespace) GetAssembly*(): Assembly, NEW;
BEGIN
    RETURN ns.asb;
END GetAssembly;


PROCEDURE (ns: Namespace) GetAssemblyFile*(): CharOpen, NEW;
BEGIN
    RETURN ns.asb.file;
END GetAssemblyFile;


PROCEDURE (ns: Namespace) GetAssemblyName*(): CharOpen, NEW;
BEGIN
    RETURN ns.asb.name;
END GetAssemblyName;


PROCEDURE (ns: Namespace) GetName*(): CharOpen;
BEGIN
    RETURN ns.fname;
END GetName;


PROCEDURE (ns: Namespace) GetTypes*(): OrderList, NEW;
BEGIN
    RETURN ns.tlist;
END GetTypes;


PROCEDURE (ns: Namespace) GetType*(tname: CharOpen): Type, NEW;
VAR
    lhead, inspos: OrderList;
BEGIN
    lhead := ns.tlist;
    IF lhead.ISearch(tname, inspos) THEN
        RETURN inspos.object(Type);
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetType;


PROCEDURE (ns: Namespace) HasForeignSpaces*(): BOOLEAN, NEW;
BEGIN
    RETURN ns.fnslist.list # NIL;
END HasForeignSpaces;


PROCEDURE (ns: Namespace) GetForeignSpaces*(): OrderList, NEW;
BEGIN
    RETURN ns.fnslist;
END GetForeignSpaces;


PROCEDURE (ns: Namespace) GetExistType*(ftname: CharOpen): Type, NEW;
VAR
    key: INTEGER;
BEGIN
    IF (ns.ktab.IsNamedObjExist(ftname, key)) THEN
        RETURN(ns.ktab.KeyToObject(key)(Type));
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetExistType;


PROCEDURE (ns: Namespace) AddType*(typ: Type): Type, NEW;
(* Returns NIL if type is successfully added, or if a type of same name
   is already existed, it returns the existing type *)
VAR
    tname: CharOpen;
    ftname: CharOpen;
    key: INTEGER;
    lhead, inspos: OrderList;
    exist: Type;
BEGIN
    tname := typ.GetName();
    ftname := typ.GetFullName();
    IF ~ns.ktab.IsNamedObjExist(ftname, key) THEN
        (* type not exist *)
        lhead := ns.tlist;
        ASSERT(~lhead.ISearch(tname, inspos));
        inspos.Insert(typ);
        ns.ktab.enterObj(key, typ);
        RETURN NIL;
    ELSE
        (* type already exist *)
        exist := ns.ktab.KeyToObject(key)(Type);
        RETURN exist;
    END; (* IF *)
END AddType;


PROCEDURE ExtractDim(tname: CharOpen; VAR dim: INTEGER; VAR digit: INTEGER);
(* Returns 0 if tname not match "AnonArrHead<dim>ElmType" *)
VAR dim2, idx: INTEGER;
BEGIN
    idx := LEN(AnonArrHead); dim2 := ORD(tname[idx]) - ORD('0');
    WHILE (dim2 >= 0) & (dim2 <= 9) DO
        INC(digit); dim := dim * 10 + dim2;
        INC(idx); dim2 := ORD(tname[idx]) - ORD('0');
    END; (* WHILE *)
END ExtractDim;


PROCEDURE IsNamedTypeArray(tname: CharOpen; VAR dim: INTEGER; VAR digit: INTEGER): BOOLEAN;
VAR head, anonarrhead: CharOpen;
    i: INTEGER;
BEGIN
    head := ST.SubStr(tname, 0, hlen-1);
    anonarrhead := ST.ToChrOpen(AnonArrHead);
    FOR i := 0 TO hlen-1 DO
        IF head[i] # anonarrhead[i] THEN RETURN FALSE; END;
    END; (* FOR *)
    (* assume dimension of an array is less than 9 *)
    ExtractDim(tname, dim, digit);
    RETURN (dim > 0);
END IsNamedTypeArray;


PROCEDURE (ns: Namespace) InsertPointer*(tname: CharOpen;
                                        ftname: CharOpen;
                                        target: Type): PointerType, NEW;
(* tname - type name, 
 * ftname - full type name (with namespace preceding)
 *)
VAR
    typ: PointerType;
    key: INTEGER;
    lhead, inspos: OrderList;
    exist: Type;
    short, long, nsname: CharOpen;
    dim: INTEGER;
    digit: INTEGER;
    i: INTEGER;
BEGIN
    dim := 0; digit := 0;
    IF IsNamedTypeArray(tname, dim, digit) THEN
        short := ST.SubStr(tname, hlen+digit, LEN(tname)-1);
        nsname := ST.SubStr(ftname, 0, LEN(ftname)-LEN(tname)-1);
        FOR i := 0 TO dim-1 DO
            short := ST.StrCat(short, ST.ToChrOpen("[]"));
        END; (* FOR *)
        long := ST.StrCat(nsname, short);
    ELSE
        short := tname;
        long :=ftname;
    END; (* IF *)
    IF ~ns.ktab.IsNamedObjExist(long, key) THEN
        lhead := ns.tlist;
        ASSERT(~lhead.ISearch(short, inspos));

        typ := NewType(Pointer)(PointerType);
        typ.tgt := target;
        typ.Initialize(short, long, ns);

        inspos.Insert(typ);
        ns.ktab.enterObj(key, typ);
    ELSE
        exist := ns.ktab.KeyToObject(key)(Type);
        WITH exist: PointerType DO
            typ := exist;
        | exist: NamedType DO
            (* a dummy NamedType is there, replace it *)
            lhead := ns.tlist;
            ASSERT(lhead.ISearch(short, inspos));

            typ := NewType(Pointer)(PointerType);
            typ.tgt := target;
            typ.Initialize(short, long, ns);

            FixReferences(exist, typ);
            inspos.Replace(typ);
            ns.ktab.enterObj(key, typ);
        ELSE
            ASSERT(FALSE); typ := NIL;
        END; (* IF *)
    END; (* IF *)
    RETURN typ;
END InsertPointer;


PROCEDURE (ns: Namespace) InsertValuePointer*(target: Type): PointerType, NEW;
VAR
    typ: PointerType;
    ptname: CharOpen;
    pftname: CharOpen;
    idx: INTEGER;
BEGIN
    IF RvalBit IN target.attr THEN
        ptname := ST.StrCat(ST.ToChrOpen(PointerHead), target.name);
        idx := ST.StrRChr(target.fname, '.');
        pftname := ST.StrCat(ST.SubStr(target.fname,0,idx), ptname);
        typ := ns.InsertPointer(ptname, pftname, target);
        typ.vmod := target.vmod;
        RETURN typ;
    ELSE
        (* currently handles only POINTER TO ValueType(.NET)  *)
        ASSERT(FALSE); RETURN NIL;
    END; (* IF *)
END InsertValuePointer;


PROCEDURE (ns: Namespace) InsertArray*(tname: CharOpen;
                                      ftname: CharOpen;
                                      dim: INTEGER;
                                      len: INTEGER;
                                      elm: Type): ArrayType, NEW;
(* called by SymReader_ParseArrayType *)
(* tname - type name, 
 * ftname - full type name (with namespace preceding)
 *)
VAR
    typ: ArrayType;
    exist: Type;
    key: INTEGER;
    lhead, inspos: OrderList;
BEGIN
    IF ~ns.ktab.IsNamedObjExist(ftname, key) THEN
        lhead := ns.tlist;
        ASSERT(~lhead.ISearch(tname, inspos));
    
        typ := NewType(Array)(ArrayType);
        typ.Initialize(tname, ftname, ns);
        typ.dim := dim;
        typ.len := len;
        typ.elm := elm;

        inspos.Insert(typ);
        ns.ktab.enterObj(key, typ);
    ELSE
        exist := ns.ktab.KeyToObject(key)(Type);
        WITH exist: ArrayType DO
            typ := exist;
        | exist: NamedType DO
            (* a dummy NamedType is there, replace it *)
            lhead := ns.tlist;
            ASSERT(lhead.ISearch(tname, inspos));
    
            typ := NewType(Array)(ArrayType);
            typ.Initialize(tname, ftname, ns);
            typ.dim := dim;
            typ.len := len;
            typ.elm := elm;

            FixReferences(exist, typ);
            inspos.Replace(typ);
            ns.ktab.enterObj(key, typ);
        ELSE
            ASSERT(FALSE); typ := NIL;
        END; (* IF *)
    END; (* IF *)
    RETURN typ;
END InsertArray;


PROCEDURE (ns: Namespace) InsertNamedType*(tname: CharOpen; ftname: CharOpen): Type, NEW;
(* As a result, type other from NamedType could be returned if PointerType, ArrayType
   or RecordType of same name are found exist *)
VAR
    ntyp: NamedType;   (* new NamedType *)
    key: INTEGER;
    lhead, inspos: OrderList;
    short, long, nsname: CharOpen;
    dim: INTEGER;
    digit: INTEGER;
    i: INTEGER;
BEGIN
    dim := 0; digit := 0;
    IF IsNamedTypeArray(tname, dim, digit) THEN
        short := ST.SubStr(tname, hlen+digit, LEN(tname)-1);
        nsname := ST.SubStr(ftname, 0, LEN(ftname)-LEN(tname)-1);
        FOR i := 0 TO dim-1 DO
            short := ST.StrCat(short, ST.ToChrOpen("[]"));
        END; (* FOR *)
        long := ST.StrCat(nsname, short);
    ELSE
        short := tname;
        long :=ftname;
    END; (* IF *)
    IF ~ns.ktab.IsNamedObjExist(long, key) THEN
        lhead := ns.tlist;
        ASSERT(~lhead.ISearch(short, inspos));
    
        NEW(ntyp); ntyp.cross := NIL;
        ntyp.Initialize(short, long, ns);
        inspos.Insert(ntyp);
        ns.ktab.enterObj(key, ntyp);
        RETURN ntyp;
    ELSE
        RETURN ns.ktab.KeyToObject(key)(Type);
    END; (* IF *)
END InsertNamedType;


PROCEDURE (ns: Namespace) InsertRecord*(tname: CharOpen;
                                       ftname: CharOpen;
                                       tt: INTEGER): RecordType, NEW;
VAR
    typ: RecordType;
    key: INTEGER;
    lhead, inspos: OrderList;
    exist: Type;
BEGIN
    IF ~ns.ktab.IsNamedObjExist(ftname, key) THEN
        lhead := ns.tlist;
        ASSERT(~lhead.ISearch(tname, inspos));
    
        typ := NewType(tt)(RecordType);
        typ.Initialize(tname, ftname, ns);
        IF tt = Primitive THEN
            typ(PrimType).MarkPrimitiveTypeOrd();
        END; (* IF *)

        inspos.Insert(typ);
        ns.ktab.enterObj(key, typ);
    ELSE
        exist := ns.ktab.KeyToObject(key)(Type);
        WITH exist: RecordType DO
            typ := exist;
        | exist: NamedType DO
            (* a dummy NamedType is there, replace it *)
            lhead := ns.tlist;
            ASSERT(lhead.ISearch(tname, inspos));

            typ := NewType(tt)(RecordType);
            typ.Initialize(tname, ftname, ns);

            FixReferences(exist, typ);
            inspos.Replace(typ);
            ns.ktab.enterObj(key, typ);
        ELSE
            ASSERT(FALSE); typ := NIL;
        END; (* IF *)
    END; (* IF *)
    RETURN typ;
END InsertRecord;


PROCEDURE (ns: Namespace) InsertNestedRecord(tname: CharOpen;
                                             ftname: CharOpen;
                                             tt: INTEGER; dt: Type): RecordType, NEW;
(* Pre: the delegate type is not in the namespace.
 *      the declaring type is already in the namespace.
 *)
VAR
    fuldtyp: CharOpen;  (* name of declaring type *)
    idx, key: INTEGER;
    inspos, lhead: OrderList;
    typ: RecordType;     (* nested type *)
    exist: Type;
BEGIN
    fuldtyp := dt.fname;
    IF ns.ktab.IsNamedObjExist(fuldtyp, key) THEN
        dt := ns.ktab.KeyToObject(key)(Type);

        IF ~ns.ktab.IsNamedObjExist(ftname, key) THEN
            typ := NewType(tt MOD Nested)(RecordType);
            typ.Initialize(tname, ftname, ns);
            typ.dtype := dt;

            (* insert into current namespace's tlist *)
            lhead := ns.tlist;
            ASSERT(~lhead.ISearch(tname, inspos));
            inspos.Insert(typ);

            (* insert into declaring type's nlist *)
            WITH dt: PointerType DO
                lhead := dt.tgt(ValueType).nlist;
            | dt: ValueType DO
                lhead := dt(ValueType).nlist;
            ELSE
                ASSERT(FALSE); lhead := NIL;
            END; (* WITH *)
            ASSERT(~lhead.ISearch(tname, inspos));
            inspos.Insert(typ);
            ns.ktab.enterObj(key, typ);
        ELSE
            (* named nested record already exist *)
            exist := ns.ktab.KeyToObject(key)(Type);
            WITH exist: RecordType DO
                typ := exist;

                (* insert into declaring type's nlist *)
                WITH dt: PointerType DO
                    lhead := dt.tgt(ValueType).nlist;
                | dt: ValueType DO
                    lhead := dt(ValueType).nlist;
                ELSE
                    ASSERT(FALSE); lhead := NIL;
                END; (* WITH *)
                IF ~lhead.ISearch(tname, inspos) THEN
                    inspos.Insert(typ);
                END; (* IF *)
            | exist: NamedType DO
                (* a dummy NamedType is there, replace it *)
                typ := NewType(tt MOD Nested)(RecordType);
                typ.Initialize(tname, ftname, ns);
                typ.dtype := dt;

                FixReferences(exist, typ);

                (* Replace into current namespace's tlist *)
                lhead := ns.tlist;
                ASSERT(lhead.ISearch(tname, inspos));
                inspos.Replace(typ);

                (* Replace or insert into declaring type's nlist *)
                WITH dt: PointerType DO
                    lhead := dt.tgt(ValueType).nlist;
                | dt: ValueType DO
                    lhead := dt(ValueType).nlist;
                ELSE
                    ASSERT(FALSE); lhead := NIL;
                END; (* WITH *)
                IF lhead.ISearch(tname, inspos) THEN
                    inspos.Replace(typ);
                ELSE
                    inspos.Insert(typ);
                END; (* IF *)
                ns.ktab.enterObj(key, typ);
            ELSE
                ASSERT(FALSE); typ := NIL;
            END; (* WITH *)
        END; (* IF *)
    ELSE
        (* the declaring type has not been parsed *)
        ASSERT(FALSE); typ := NIL;
    END; (* IF *)
    RETURN typ;
END InsertNestedRecord;


PROCEDURE (ns: Namespace) InsertNested(tname: CharOpen;
                                       ftname: CharOpen;
                                       tt: INTEGER; dt: Type): PointerType, NEW;
(* Pre: the delegate type is not in the namespace.
 *      the declaring type is already in the namespace.
 *)
VAR
    long, short: CharOpen;
    fuldtyp: CharOpen;  (* name of declaring type *)
    idx, key: INTEGER;
    inspos, lhead: OrderList;
    target: RecordType;    (* nested type *)
    typ: PointerType;     (* nested type *)
    exist: Type;
BEGIN
    (* make a anonymous record for the nested pointer type *)
    short := ST.StrCat(tname, anonRec);
    long := ST.StrCat(ftname, anonRec);
    target := ns.InsertNestedRecord(short, long, tt, dt);
    target.anon := TRUE;

    fuldtyp := dt.fname;
    IF ns.ktab.IsNamedObjExist(fuldtyp, key) THEN
        (* if declaring type of the nested type is already exist, should always be *)
        dt := ns.ktab.KeyToObject(key)(Type);

        IF ~ns.ktab.IsNamedObjExist(ftname, key) THEN
            (* nested type hasn't been inserted before *)
            typ := NewType(Pointer)(PointerType);
            typ.tgt := target;
            typ.Initialize(tname, ftname, ns);

            (* insert into current namespace's tlist *)
            lhead := ns.tlist;
            ASSERT(~lhead.ISearch(tname, inspos));
            inspos.Insert(typ);

            (* insert into declaring type's nlist *)
            WITH dt: PointerType DO
                lhead := dt.tgt(ValueType).nlist;
            | dt: ValueType DO
                lhead := dt(ValueType).nlist;
            ELSE
                ASSERT(FALSE); lhead := NIL;
            END; (* WITH *)
            ASSERT(~lhead.ISearch(tname, inspos));
            inspos.Insert(typ);
            ns.ktab.enterObj(key, typ);
        ELSE
            (* nested type is already exist *)
            exist := ns.ktab.KeyToObject(key)(Type);
            WITH exist: PointerType DO
                typ := exist;

                (* insert into declaring type's nlist *)
                WITH dt: PointerType DO
                    lhead := dt.tgt(ValueType).nlist;
                | dt: ValueType DO
                    lhead := dt(ValueType).nlist;
                ELSE
                    ASSERT(FALSE); lhead := NIL;
                END; (* WITH *)
                IF ~lhead.ISearch(tname, inspos) THEN
                    inspos.Insert(typ);
                END; (* IF *)

            | exist: NamedType DO
                (* a dummy NamedType is there, replace it *)
                typ := NewType(Pointer)(PointerType);
                typ.tgt := target;
                typ.Initialize(tname, ftname, ns);

                FixReferences(exist, typ);

                (* Replace into current namespace's tlist *)
                lhead := ns.tlist;
                ASSERT(lhead.ISearch(tname, inspos));
                inspos.Replace(typ);

                (* Replace or insert into declaring type's nlist *)
                WITH dt: PointerType DO
                    lhead := dt.tgt(ValueType).nlist;
                | dt: ValueType DO
                    lhead := dt(ValueType).nlist;
                ELSE
                    ASSERT(FALSE); lhead := NIL;
                END; (* WITH *)
                IF lhead.ISearch(tname, inspos) THEN
                    (* found it in declaring type's nlist, replace it *)
                    inspos.Replace(typ);
                ELSE
                    inspos.Insert(typ);
                END; (* IF *)
                ns.ktab.enterObj(key, typ);
            ELSE
                ASSERT(FALSE); typ := NIL;
            END; (* WITH *)
        END; (* IF *)
    ELSE
        (* the declaring type has not been parsed *)
        ASSERT(FALSE); typ := NIL;
    END; (* IF *)
    RETURN typ;
END InsertNested;


PROCEDURE (ns: Namespace) InsertArrayType(tt: INTEGER; tname: CharOpen;
                                          fname: CharOpen): Type, NEW;
(* tname e.g. Object[], fname e.g. System.Object[] *)
(* tname e.g. Object[][], fname e.g. System.Object[][] *)
(* tname e.g. Object[,], fname e.g. System.Object[,] *)
(* Pre: tname and fname are null terminated character string *)
VAR
    dim, i: INTEGER;
    bidx, eidx, key: INTEGER;
    llen, slen: INTEGER;
    long, short: CharOpen;
    inspos, lhead: OrderList;
    etyp, typ: Type;
BEGIN
    slen := LEN(tname); llen := LEN(fname);
    bidx := ST.StrChr(tname, '[');
    IF ST.StrChr(tname,',') = ST.NotExist THEN
        dim := (slen - bidx) DIV 2;
        short := ST.SubStr(tname, 0, slen-2-2*dim);
        long := ST.SubStr(fname, 0, llen-2-2*dim);
    ELSE
        eidx := ST.StrRChr(tname, ']');
        dim := eidx - bidx;
        short := ST.SubStr(tname, 0, slen-3-dim);
        long := ST.SubStr(fname, 0, llen-3-dim);
    END; (* IF *)

    (* insert the element type then dimensions of array type *)
    typ := NIL; etyp := NIL; i := 0;
    LOOP
        IF i = 0 THEN
            tt := tt MOD Array;
            typ := ns.InsertType2(tt, short, long, NIL, NIL, NIL);
        ELSE
            typ := ns.InsertArray(short, long, i, 0, etyp);
            typ.anon := TRUE;
        END; (* IF *)

        etyp := typ;
        IF i >= dim THEN EXIT END;
        i := i + 1;
        short := ST.StrCat(short, anonArr);
        long := ST.StrCat(long, anonArr);
    END; (* LOOP *)

    RETURN typ;
END InsertArrayType;


PROCEDURE (ns: Namespace) InsertType2(tt: INTEGER; tname: CharOpen; ftname: CharOpen;
                                      bt: Type; dt: Type; il: OrderList): Type, NEW;
VAR
    short, long: CharOpen;
    idx, len: INTEGER;
    inspos, lhead: OrderList;
    key: INTEGER;
    typ: Type;
    asbname: CharOpen;
    target: Type;
BEGIN
    (* determine whether it is a Nested type *)
    IF tt >= Nested THEN
        (* it is a Nested type *)
        IF (tt MOD Nested) >= Interface THEN
            typ := ns.InsertNested(tname, ftname, tt, dt);
        ELSE
            typ := ns.InsertNestedRecord(tname, ftname, tt, dt);
        END; (* IF *)
        RETURN typ;
    END; (* IF *)

    (* not a Nested type, determine whether it is an Array type *)
    IF tt >= Array THEN
        (* it is an Array type *)
        target := ns.InsertArrayType(tt, tname, ftname);

        (* Make sure same pointer to array type hasn't been inserted as NamedType by SymReader (Arr1....) *)

        typ := ns.InsertPointer(tname, ftname, target);
        RETURN typ;
    END; (* IF *)

    (* not a Nested type, not an Array type *)
    IF tt >= RefType THEN
        short := ST.StrCat(tname, anonRec);
        long := ST.StrCat(ftname, anonRec);
        target := ns.InsertRecord(short, long, tt);
        target.anon := TRUE;
        typ := ns.InsertPointer(tname, ftname, target);
    ELSE
        typ := ns.InsertRecord(tname, ftname, tt);
        target := typ;
    END; (* IF *)

    WITH target: RecordType DO
        WITH target: ValueType DO
            IF (bt # NIL) & (target.base = NIL) THEN
                target.base := bt;
            ELSE
            END; (* IF *)
        ELSE
        END; (* WITH *)

        IF (il # NIL) & (target.impl = NIL) THEN target.impl := il; END;
    ELSE
    END; (* WITH *)

    RETURN typ;
END InsertType2;


PROCEDURE (ns: Namespace) InsertType*(tt: INTEGER; qftname: CharOpen;
                                      bt: Type; dt: Type; il: OrderList): Type;
(* 'qftname' specifies the qualified full type name, it is in the form of:
 * e.g. "System.Int32, mscorlib.dll"
 * 'btname' specifies the base type name of the inserting type.
 * tt is the type of the inserting type, e.g. ClassType/PrimType/..
 * btt is the type of the base type of the inserting type.
 *)
VAR
    idx, len: INTEGER;
    ftname, tname: CharOpen;
    long, short: CharOpen;
    inspos, lhead: OrderList;
    key: INTEGER;
    typ: Type;
    asbname: CharOpen;
    fasbname: CharOpen;
    target: Type;
    rslt: Type;
BEGIN
    SplitTypeName(qftname, fasbname, asbname, ftname);    (* fasbname & asbname is not used *)
    idx := ST.StrRChr(ftname, '.');
    IF idx # ST.NotExist THEN
        tname := ST.SubStr(ftname,idx+1,LEN(ftname)-1);
    ELSE
        tname := ftname;
    END; (* IF *)

    IF ~ns.ktab.IsNamedObjExist(ftname, key) THEN                                       (* if it is anonymous array type, its name may already exist *)
        RETURN ns.InsertType2(tt, tname, ftname, bt, dt, il);
    ELSE
        rslt := ns.ktab.KeyToObject(key)(Type);
        IF ~(rslt IS ArrayType) & ~(rslt IS PointerType) & ~(rslt IS RecordType) THEN
            (* it is NamedType that already exist *)
            IF tt = Named THEN 
                (* also a named type is being inserted *)
                RETURN rslt;
            ELSE
                (* replace the existing NamedType with a concrete type *)
                RETURN ns.InsertType2(tt, tname, ftname, bt, dt, il);
            END; (* IF *)
        ELSE
            (* it is concrete type that already exist *)
            IF bt # NIL THEN
                WITH rslt: PointerType DO
                    target := rslt.tgt;
                    WITH target: RecordType DO
                        IF target.base = NIL THEN target.base := bt; END;
                    ELSE
                    END; (* WITH *)
                | rslt: RecordType DO
                    IF rslt.base = NIL THEN rslt.base := bt; END;
                ELSE
                END; (* WITH *)
            END; (* IF *)
            RETURN rslt;

        END; (* IF *)
    END; (* IF *)
END InsertType;


PROCEDURE (ns: Namespace) InsertEnumType*(tt: INTEGER; qftname: CharOpen;
                                          bt: Type; dt: Type; utype: CharOpen): EnumType, NEW;
VAR
    enum: EnumType;
BEGIN
    enum := ns.InsertType(tt, qftname, bt, dt, NIL)(EnumType);
    enum.utype := utype;
    RETURN enum;
END InsertEnumType;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE GetAssemblyByName*(asbname: CharOpen): Assembly;
VAR
    lhead, inspos: OrderList;
    asb: Assembly;
BEGIN
    lhead := meta.asmList;
    IF lhead.ASearch(asbname, inspos) THEN
        RETURN inspos.object(Assembly);
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetAssemblyByName;

PROCEDURE NewAssembly*(fname: CharOpen; name: CharOpen;
                       file: CharOpen): Assembly;
(* fname is in the form:
 *   "mscorlib, Ver=2000.14.1812.10, SN=03689116d3a4ae33"
 * name is the filename of assembly without extension:
 *   "mscorlib"
 *)
VAR
    idx: INTEGER;
    asb: Assembly;
BEGIN
    NEW(asb);
    asb.fname := fname;
    asb.name := name;
    asb.file := file;
    NEW(asb.slist); asb.slist.list := NIL; asb.slist.object := NIL;
    NEW(asb.ktab); asb.ktab.InitObjectHash(HashSize);
    NEW(asb.snap); asb.snap.list := NIL; asb.snap.object := NIL;
    RETURN asb;
END NewAssembly;


PROCEDURE InsertAssembly*(fname: CharOpen; file: CharOpen): Assembly;
(* fname composes of <assemblyname>, <version>, <location>|<culture>, <strongname>; can be:
 * (1) '<assemblyname>, <version>, <location>|<culture>, <strongname>' OR
 * (2) '<assemblyname>, <version>, <strongname>' OR
 * (3) '<assemblyname>, <version>' OR
 * (4) '<assemblyname>'
 *     assemblyname is defined as: (a) fname: name1...namei.dll, OR
 *                                 (b) fname: name1.dll, OR
 *                                 (c) fname: name1...namei, OR
 *                                 (d) fname: name1
 * for Pre-beta2
 *     version is defined as:    'Ver=<major>.<minor>.<build>.<revision>'
 *     location is defined as:   'Loc="???"'
 *     strongname is defined as: 'SN=<PublicKeyToken>'
 * for beta2
 *     version is defined as:    'Version=<major>.<minor>.<build>.<revision>'
 *     culture is defined as:    'Culture=<culture>'
 *     strongname is defined as: 'PublicKeyToken=<PublicKeyToken>'
 *                     could be: 'PublicKeyToken=null'
 *
 * file can be:
 * (1) name1...namei OR
 * (2) name1
 *)
CONST
    NullStrongName = "null";
VAR
    asb: Assembly;
    lhead, inspos: OrderList;
    isfirst: BOOLEAN;
    name: CharOpen;
    str: CharOpen;
    idx1: INTEGER;
    idx2: INTEGER;
    vdot: INTEGER; (* position of the dot after a version part *)
    ver: CharOpen;
    version: Version;
    sn: CharOpen;
    token: PublicKeyToken;
    i: INTEGER;
    rc: BOOLEAN;
    procVer: BOOLEAN;
    procTok: BOOLEAN;
BEGIN
    version := NIL; token := NIL; procVer := FALSE; procTok := FALSE; str := ST.NullString;
    idx1 := ST.StrChr(fname, ',');
    IF idx1 # ST.NotExist THEN
        (* AssemblyName processing: case (1), (2) or (3) *)
        name := ST.SubStr(fname,0,idx1-1);
        procVer := TRUE;
    ELSE
        (* AssemblyName processing: case (4) *)
        name := ST.SubStr(fname,0,LEN(fname)-1);
    END; (* IF *)

    (* process the assembly name *)
    idx2 := ST.StrRChr(name, '.');
    IF idx2 # ST.NotExist THEN
        (* case (a), (b) or (c) *)
        str := ST.SubStr(name,idx2,LEN(name)-1);
        IF (str^ = AssemblyExt) OR (str^ = TypeLibExt) THEN
            (* case (a) or (b) *)
            name := ST.StrSubChr(ST.SubStr(name,0,idx2-1),'.','_');
            (* name = name1_.._namei *) 
        ELSE
            (* case (c) *)
            name := ST.StrSubChr(name,'.','_');
            (* name = name1_.._namei *) 
        END; (* IF *)
    ELSE
        (* case (d) *)
        (* name = name1 *)
    END; (* IF *)

    IF procVer THEN
        (* process the assembly version *)
        ASSERT((fname[idx1+2]='V') & (fname[idx1+3]='e') & (fname[idx1+4]='r'));
        IF (fname[idx1+5]='=') THEN
            (* Pre-beta2 version *)
            str := ST.SubStr(fname,idx1+6,LEN(fname)-1);
        ELSE
            (* beta2 version *)
            ASSERT((fname[idx1+5]='s') & (fname[idx1+6]='i') & (fname[idx1+7]='o') & (fname[idx1+8]='n') & (fname[idx1+9]='='));
            str := ST.SubStr(fname,idx1+10,LEN(fname)-1);
        END; (* IF *)
        NEW(version);
        (* get the major, minor and build part of version *)
        FOR i := 0 TO 2 DO
            vdot := ST.StrChr(str,'.');
            ASSERT(vdot # ST.NotExist);
            ver := ST.SubStr(str,0,vdot-1);
            RTS.StrToUShort(ver, version[i], rc);
            str := ST.SubStr(str,vdot+1,LEN(str)-1);
        END; (* FOR *)
        (* get the revision part of version *)
        idx1 := ST.StrChr(str, ',');
        IF idx1 # ST.NotExist THEN
            (* case (1) or (2) *)
            ver := ST.SubStr(str,0,idx1-1);
            RTS.StrToUShort(ver, version[Revis], rc);
            procTok := TRUE;
        ELSE
            (* case (3) *)
            ver := ST.SubStr(str,0,LEN(str)-1);
            RTS.StrToUShort(ver, version[Revis], rc);
        END; (* IF *)
    END; (* IF *)

    IF procTok THEN
        (* process the assembly strong name *)
        idx1 := ST.StrStr(str, b1StrongName);
        IF idx1 # ST.NotExist THEN
            sn := ST.SubStr(str,idx1+LEN(b1StrongName)-1,idx1+LEN(b1StrongName)-1+StrongNameLen-1);
            NEW(token);
        ELSE
            idx1 := ST.StrStr(str, b2StrongName);
            ASSERT(idx1 # ST.NotExist);
            sn := ST.SubStr(str,idx1+LEN(b2StrongName)-1,idx1+LEN(b2StrongName)-1+StrongNameLen-1);
            IF ST.StrCmp(sn, ST.ToChrOpen(NullStrongName)) # ST.Equal THEN NEW(token) END;
        END; (* IF *)
        IF token # NIL THEN
            FOR i := 0 TO StrongNameLen-1 BY 2 DO
                str := ST.SubStr(sn, i, i+1);
                RTS.HexStrToUByte(str, token[i DIV 2], rc);
            END; (* FOR *)
        END; (* IF *)
    END; (* IF *)
    str := name;
    lhead := meta.asmList;
    IF lhead.ISearch(str, inspos) THEN
        (* assembly with same name found *)
        IF inspos.object(Assembly).vers = NIL THEN
            inspos.object(Assembly).vers := version;
        ELSE
            (* should compare the versions to make sure they are identical, or issue exception *)
            IF (version # NIL) &
               ~((version[0] = inspos.object(Assembly).vers[0]) &
                 (version[1] = inspos.object(Assembly).vers[1]) &
                 (version[2] = inspos.object(Assembly).vers[2]) &
                 (version[3] = inspos.object(Assembly).vers[3])) THEN
                Error.WriteString("!!!Error!!! - InsertAssembly - assemblies with same name but different version already exist"); Error.WriteLn;
                Error.WriteString("                             - assembly : " + name^); Error.WriteLn;
                Error.WriteString("                             - version parsing : ");
                Error.WriteInt(version[0],1); Error.Write('.'); Error.WriteInt(version[1],1); Error.Write('.');
                Error.WriteInt(version[2],1); Error.Write('.'); Error.WriteInt(version[3],1); Error.WriteLn;
                version := inspos.object(Assembly).vers;
                Error.WriteString("                             - version existing : ");
                Error.WriteInt(version[0],1); Error.Write('.'); Error.WriteInt(version[1],1); Error.Write('.');
                Error.WriteInt(version[2],1); Error.Write('.'); Error.WriteInt(version[3],1); Error.WriteLn;
                ASSERT(FALSE); RETURN NIL;
            END; (* IF *)
        END; (* IF *)
        IF inspos.object(Assembly).token = NIL THEN
            inspos.object(Assembly).token := token;
        ELSE
            (* should compare the public key tokens to make sure they are identical, or issue exception *)
            IF (token # NIL) &
               ~((token[0] = inspos.object(Assembly).token[0]) &
                 (token[1] = inspos.object(Assembly).token[1]) &
                 (token[2] = inspos.object(Assembly).token[2]) &
                 (token[3] = inspos.object(Assembly).token[3]) &
                 (token[4] = inspos.object(Assembly).token[4]) &
                 (token[5] = inspos.object(Assembly).token[5]) &
                 (token[6] = inspos.object(Assembly).token[6]) &
                 (token[7] = inspos.object(Assembly).token[7])) THEN
                Error.WriteString("!!!Error!!! - InsertAssembly - assemblies with same name but different public key token already exist"); Error.WriteLn;
                Error.WriteString("                             - assembly : " + name^); Error.WriteLn;
                Error.WriteString("                             - public key token parsing : ");
                Error.WriteInt(token[0],1); Error.Write('.'); Error.WriteInt(token[1],1); Error.Write('.');
                Error.WriteInt(token[2],1); Error.Write('.'); Error.WriteInt(token[3],1); Error.Write('.');
                Error.WriteInt(token[4],1); Error.Write('.'); Error.WriteInt(token[5],1); Error.Write('.');
                Error.WriteInt(token[6],1); Error.Write('.'); Error.WriteInt(token[7],1); Error.WriteLn;
                token := inspos.object(Assembly).token;
                Error.WriteString("                             - public key token existing : ");
                Error.WriteInt(token[0],1); Error.Write('.'); Error.WriteInt(token[1],1); Error.Write('.');
                Error.WriteInt(token[2],1); Error.Write('.'); Error.WriteInt(token[3],1); Error.Write('.');
                Error.WriteInt(token[4],1); Error.Write('.'); Error.WriteInt(token[5],1); Error.Write('.');
                Error.WriteInt(token[6],1); Error.Write('.'); Error.WriteInt(token[7],1); Error.WriteLn;
                ASSERT(FALSE); RETURN NIL;
            END; (* IF *)
        END;
        RETURN inspos.object(Assembly);
    ELSE
        (* insert a new assembly *)
        asb := NewAssembly(fname, name, file);
        asb.vers := version; asb.token := token;
        isfirst := FALSE; IF meta.asmList.list = NIL THEN isfirst := TRUE END;
        inspos.Insert(asb);
    END; (* IF *)
    (* it may not be the first after other assemblies inserted *)
    IF isfirst THEN meta.asmThis := asb; END;
    RETURN asb;
END InsertAssembly;


PROCEDURE (a: Assembly) GetFileName*(): CharOpen, NEW;
BEGIN
    RETURN a.file;
END GetFileName;


PROCEDURE (a: Assembly) GetNamespaces*(): OrderList, NEW;
(* Pre: SnapShot has been done *)
BEGIN
    RETURN a.snap;
END GetNamespaces;


PROCEDURE (a: Assembly) GetVersion*(): Version, NEW;
BEGIN
    RETURN a.vers;
END GetVersion;


PROCEDURE (a: Assembly) SetVersion*(version: Version), NEW;
BEGIN
    IF a.vers # NIL THEN
        IF ~((version[0] = a.vers[0]) & (version[1] = a.vers[1]) &
             (version[2] = a.vers[2]) & (version[3] = a.vers[3])) THEN
            Error.WriteString("!!!Error!!! - SetVersion - assemblies with same name but different version already exist"); Error.WriteLn;
            Error.WriteString("                         - assembly : " + a.name^); Error.WriteLn;
            Error.WriteString("                         - version parsing : ");
            Error.WriteInt(version[0],1); Error.Write('.'); Error.WriteInt(version[1],1); Error.Write('.');
            Error.WriteInt(version[2],1); Error.Write('.'); Error.WriteInt(version[3],1); Error.WriteLn;
            Error.WriteString("                         - version existing : ");
            Error.WriteInt(a.vers[0],1); Error.Write('.'); Error.WriteInt(a.vers[1],1); Error.Write('.');
            Error.WriteInt(a.vers[2],1); Error.Write('.'); Error.WriteInt(a.vers[3],1); Error.WriteLn;
            ASSERT(FALSE);
        END; (* IF *)
    ELSE
        a.vers := version;
    END; (* IF *)
END SetVersion;


PROCEDURE (a: Assembly) GetPublicKeyToken*(): PublicKeyToken, NEW;
BEGIN
    RETURN a.token;
END GetPublicKeyToken;


PROCEDURE (a: Assembly) SetPublicKeyToken*(token: PublicKeyToken), NEW;
BEGIN
    IF a.token # NIL THEN
        IF ~((token[0] = a.token[0]) & (token[1] = a.token[1]) &
             (token[2] = a.token[2]) & (token[3] = a.token[3]) &
             (token[4] = a.token[4]) & (token[5] = a.token[5]) &
             (token[6] = a.token[6]) & (token[7] = a.token[7])) THEN
            Error.WriteString("!!!Error!!! - SetPublicKeyToken - assemblies with same name but different public key token already exist"); Error.WriteLn;
            Error.WriteString("                                - assembly : " + a.name^); Error.WriteLn;
            Error.WriteString("                                - public key token parsing : ");
            Error.WriteInt(token[0],1); Error.Write('.'); Error.WriteInt(token[1],1); Error.Write('.');
            Error.WriteInt(token[2],1); Error.Write('.'); Error.WriteInt(token[3],1); Error.Write('.');
            Error.WriteInt(token[4],1); Error.Write('.'); Error.WriteInt(token[5],1); Error.Write('.');
            Error.WriteInt(token[6],1); Error.Write('.'); Error.WriteInt(token[7],1); Error.WriteLn;
            Error.WriteString("                                - public key token existing : ");
            Error.WriteInt(a.token[0],1); Error.Write('.'); Error.WriteInt(a.token[1],1); Error.Write('.');
            Error.WriteInt(a.token[2],1); Error.Write('.'); Error.WriteInt(a.token[3],1); Error.Write('.');
            Error.WriteInt(a.token[4],1); Error.Write('.'); Error.WriteInt(a.token[5],1); Error.Write('.');
            Error.WriteInt(a.token[6],1); Error.Write('.'); Error.WriteInt(a.token[7],1); Error.WriteLn;
            ASSERT(FALSE);
        END; (* IF *)
    ELSE
        a.token := token;
    END; (* IF *)
END SetPublicKeyToken;


PROCEDURE (a: Assembly) GetNamespace*(nsname: CharOpen): Namespace, NEW;
VAR
    key: INTEGER;
BEGIN
    IF a.ktab.IsNamedObjExist(nsname, key) THEN
        RETURN a.ktab.KeyToObject(key)(Namespace);
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetNamespace;


PROCEDURE (a: Assembly) InsertNamespace*(nsname: CharOpen): Namespace, NEW;
VAR
    head, front, tail: CharOpen;
    idx, len: INTEGER;
    inspos, lhead: OrderList;
    ns : Namespace;
    found: BOOLEAN;
    key: INTEGER;
BEGIN
    len := LEN(nsname);
    front := ST.NullString;
    tail := nsname;

    (* add a dummy NullSpace if the namespace is null *)
    IF LEN(tail) = 1 THEN tail := ST.ToChrOpen(NULLSPACE); END;

    IF ~a.ktab.IsNamedObjExist(tail, key) THEN
        lhead := a.slist;
        idx := ST.StrChr(tail, '.');
        WHILE idx # ST.NotExist DO
            head := ST.SubStr(tail, 0, idx-1);
            front := ST.StrCatChr(front,'.');
            front := ST.StrCat(front,head);
            IF ~lhead.ISearch(head, inspos) THEN
                ns := NewNamespace(head, ST.SubStr(front,1,LEN(front)-1), a, NotNamespace);
                inspos.Insert(ns);
            ELSE
                ns := inspos.object(Namespace);
            END; (* IF *)
            tail := ST.SubStr(tail, idx+1, len-1);
            len := len - idx - 1;
            idx := ST.StrChr(tail, '.');
            IF (len > 0) & (ns.child = NIL) THEN
                NEW(ns.child); ns.child.list := NIL; ns.child.object := NIL;
            END; (* IF *)
            lhead := ns.child;
        END; (* WHILE *)
        IF ~lhead.ISearch(tail, inspos) THEN
            front := ST.StrCatChr(front,'.');
            front := ST.StrCat(front,tail);
            ns := NewNamespace(tail, ST.SubStr(front,1,LEN(front)-1), a, IsNamespace);
            inspos.Insert(ns);
        ELSE
            ns := inspos.object(Namespace);
            ns.isNS := TRUE;
            ns.asb := a;
            NEW(ns.tlist); ns.tlist.list := NIL; ns.tlist.object := NIL;
        END; (* IF *)
        a.ktab.enterObj(key, ns);
    ELSE
        (* namespace already exist *)
        ns := a.ktab.KeyToObject(key)(Namespace);
    END; (* IF *)
    RETURN ns;
END InsertNamespace;


PROCEDURE (a: Assembly) NamespaceCount*(): INTEGER, NEW;
VAR
    cursor: OrderList;
    cnt: INTEGER;
BEGIN
    cursor := a.snap.list; cnt := 0;
    WHILE cursor # NIL DO
        INC(cnt); cursor := cursor.list;
    END; (* WHILE *)
    RETURN cnt;
END NamespaceCount;


    PROCEDURE ^ TraverseSide(l: OrderList; VAR r: OrderList);

    PROCEDURE TraverseDown(l: OrderList; VAR r: OrderList);
    VAR
        ns: Namespace;
        tv: OTraverser;
        fasb: Assembly;
    BEGIN
        IF l.object # NIL THEN
            ns := l.object(Namespace);

            IF ns.isNS THEN
                (* add this namespace to the assembly's ns snapshot list *)
                NEW(r.list); r := r.list;
                r.object := ns; r.list := NIL;
            END; (* IF *)

            IF ns.child # NIL THEN
                TraverseDown(ns.child.list, r);
                TraverseSide(ns.child.list, r);
            END; (* IF *)
        END; (* IF *)
    END TraverseDown;

    PROCEDURE TraverseSide(l: OrderList; VAR r: OrderList);
    BEGIN
        IF l.list # NIL THEN
            TraverseDown(l.list, r);
            TraverseSide(l.list, r);
        END; (* IF *)
    END TraverseSide;

PROCEDURE (a: Assembly) SnapShot*(), NEW;
(* Create a list of namespaces (MODULEs) *)
VAR
    snapptr: OrderList;

BEGIN
    IF a.slist.list # NIL THEN
        snapptr := a.snap;
        TraverseDown(a.slist.list, snapptr);
        TraverseSide(a.slist.list, snapptr);
    END; (* IF *)
END SnapShot;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (ao: AccessObject) GetVisibility*(): INTEGER, NEW;
BEGIN
    RETURN ao.vmod;
END GetVisibility;


PROCEDURE (ao: AccessObject) SetVisibility*(vmod: INTEGER), NEW, EXTENSIBLE;
BEGIN
    ao.vmod := vmod;
END SetVisibility;


PROCEDURE SetTypeVisibility(typ: Type; vmod: INTEGER);
BEGIN
    CASE typ.vmod OF
      Vprivate: typ.vmod := vmod;
    | Vinternal: typ.vmod := vmod;
    | Vprotected: 
        CASE vmod OF
          Vpublic: typ.vmod := Vpublic;
        ELSE
        END; (* CASE *)
    | Vpublic:
    ELSE
    END; (* CASE *)
END SetTypeVisibility;


PROCEDURE (typ: Type) SetVisibility*(vmod: INTEGER), EXTENSIBLE;
BEGIN
    SetTypeVisibility(typ, vmod);
END SetVisibility;

PROCEDURE (typ: PointerType) SetVisibility*(vmod: INTEGER);
VAR
    elm,tgt: Type;
BEGIN
    tgt := typ.tgt;

    WITH tgt: ArrayType DO
        elm := tgt.elm;
        CASE elm.vmod OF
          Vprivate: typ.vmod := Vprivate;
        | Vpublic: typ.vmod := vmod;
        | Vprotected:
            CASE vmod OF
              Vprivate: typ.vmod := Vprivate;
            | Vpublic: typ.vmod := Vprotected;
            | Vprotected: typ.vmod := Vprotected;
            | Vinternal: typ.vmod := Vinternal;
            ELSE
            END; (* CASE *)
        | Vinternal: typ.vmod := Vinternal;
        ELSE
        END; (* CASE *)
    ELSE
        SetTypeVisibility(typ, vmod);
    END; (* WITH *)
    IF tgt.anon THEN tgt.vmod := Vprivate; END;
END SetVisibility;


PROCEDURE (ao: AccessObject) InclAttributes*(attr: Attribute), NEW;
VAR
    i: INTEGER;
BEGIN
    FOR i := 0 TO 31 DO
        IF i IN attr THEN INCL(ao.attr, i); END;
    END; (* FOR *)
END InclAttributes;


PROCEDURE (ao: AccessObject) ExclAttributes*(attr: Attribute), NEW;
VAR
    i: INTEGER;
BEGIN
    FOR i := 0 TO 31 DO
        IF i IN attr THEN EXCL(ao.attr, i); END;
    END; (* FOR *)
END ExclAttributes;


PROCEDURE (ao: AccessObject) IsExported*(): BOOLEAN, NEW;
BEGIN
    RETURN (ao.vmod = Vpublic) OR (ao.vmod = Vprotected);
END IsExported;

PROCEDURE (ao: AccessObject) IsPrivate*(): BOOLEAN, NEW;
BEGIN
    RETURN (ao.vmod = Vprivate);
END IsPrivate;

PROCEDURE (ao: AccessObject) IsPublic*(): BOOLEAN, NEW;
BEGIN
    RETURN (ao.vmod = Vpublic);
END IsPublic;

PROCEDURE (ao: AccessObject) IsProtected*(): BOOLEAN, NEW;
BEGIN
    RETURN (ao.vmod = Vprotected);
END IsProtected;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE MakeDummyPrimitive*(tord: INTEGER): PrimType;
(* Dummy Primitive will be: e.g.
 *     []System.BOOLEAN
 *     []System.CHAR
 *     []System.BYTE
 *     []System.INTEGER
 *     []System.LONGINT
 *     []System.REAL
 * but with "GPCP" as the dummy assembly file name.
 *)
CONST
    SysSpace = "System";
VAR
    asb: Assembly;
    ns: Namespace;
    prim: PrimType;

    PROCEDURE OrdToName(tord: INTEGER): CharOpen;
    BEGIN
        CASE tord OF
          notBs :  (* not Base *)
                   ASSERT(FALSE); RETURN NIL;
        | boolN :  (* BOOLEAN *)
                   RETURN gpcpBoolean;
        | sChrN :  (* SHORTCHAR *)
                   RETURN gpcpSChar;
        | charN :  (* CHAR *)
                   RETURN gpcpChar;
        | byteN :  (* BYTE *)
                   RETURN gpcpSByte;
        | sIntN :  (* SHORTINT *)
                   RETURN gpcpInt16;
        | intN  :  (* INTEGER *)
                   RETURN gpcpInt32;
        | lIntN :  (* LONGING *)
                   RETURN gpcpInt64;
        | sReaN :  (* SHORTREAL *)
                   RETURN gpcpSingle;
        | realN :  (* REAL *)
                   RETURN gpcpDouble;
        | setN  :  (* SET *)
                   RETURN gpcpSet;
        | anyRec:  (* ANYREC *)
                   RETURN gpcpAnyRec;
        | anyPtr:  (* ANYPTR *)
                   RETURN gpcpAnyPtr;
        | strN  :  (* STRING (ARRAY OF CHAR) *)
                   RETURN gpcpCharArr;
        | sStrN :  (* SHORTSTRING (ARRAY OF SHORTCHAR) *)
                   ASSERT(FALSE); RETURN NIL;
        | uBytN :  (* UBYTE *)
                   RETURN gpcpUByte;
        | metaN :  (* META *)
                   ASSERT(FALSE); RETURN NIL;
        ELSE
                   ASSERT(FALSE); RETURN NIL;
        END; (* CASE *)
    END OrdToName;

BEGIN
    asb := InsertAssembly(ST.NullString, ST.ToChrOpen(DmyAssembly));
    ns := asb.InsertNamespace(ST.ToChrOpen(SysSpace));
    NEW(prim); 
    prim.name := OrdToName(tord);
    prim.fname := ST.StrCat(ST.StrCatChr(ST.ToChrOpen(SysSpace), '.'), prim.name);
    prim.vmod := Vpublic;
    prim.tord := tord;
    prim.space := ns;
    prim.inhie := FALSE;
    prim.anon := FALSE;
    prim.base := NIL;
    prim.impl := NIL;
    prim.dtype := NIL;
    prim.clist := NIL;
    prim.sflist := NIL;
    prim.vmlist := NIL;
    prim.smlist := NIL;
    baseTypeArray[prim.tord] := prim;
    RETURN prim;
END MakeDummyPrimitive;


PROCEDURE NewType(tt: INTEGER): Type;
VAR
    t: Type;
    a: ArrayType;
    c: ClassType;
    d: DelegType;
    e: EnumType;
    i: IntfcType;
    p: PrimType;
    r: PointerType;
    s: StrucType;
BEGIN
    CASE tt OF
      Array    : 
                 NEW(a); a.dim := 0; a.len := 0; a.elm := NIL; t := a;
    | Class    : 
                 NEW(c); c.base := NIL; c.impl := NIL; c.dtype := NIL; c.naCtor := FALSE; t := c;
                 NEW(c.clist);  c.clist.list := NIL;  c.clist.object := NIL;
                 NEW(c.vmlist); c.vmlist.list := NIL; c.vmlist.object := NIL;
                 NEW(c.smlist); c.smlist.list := NIL; c.smlist.object := NIL;
                 NEW(c.sflist); c.sflist.list := NIL; c.sflist.object := NIL;
                 NEW(c.iflist); c.iflist.list := NIL; c.iflist.object := NIL;
                 NEW(c.elist);  c.elist.list := NIL;  c.elist.object := NIL;
                 NEW(c.nlist);  c.nlist.list := NIL;  c.nlist.object := NIL;
    | Delegate :
                 NEW(d); d.base := NIL; d.impl := NIL; d.dtype := NIL; d.naCtor := FALSE; t := d;
                 NEW(d.clist); d.clist.list := NIL; d.clist.object := NIL;
                 NEW(d.nlist);  d.nlist.list := NIL;  d.nlist.object := NIL;
                 NEW(d.vmlist); d.vmlist.list := NIL; d.vmlist.object := NIL;
                 NEW(d.smlist); d.smlist.list := NIL; d.smlist.object := NIL;
                 NEW(d.sflist); d.sflist.list := NIL; d.sflist.object := NIL;
                 NEW(d.iflist); d.iflist.list := NIL; d.iflist.object := NIL;
                 d.isMul := FALSE;
    | Interface: 
                 NEW(i); i.base := NIL; i.impl := NIL; i.dtype := NIL; t := i;
                 NEW(i.vmlist); i.vmlist.list := NIL; i.vmlist.object := NIL;
                 NEW(i.nlist);  i.nlist.list := NIL;  i.nlist.object := NIL;
    | Enum     : 
                 NEW(e); e.base := NIL; e.impl := NIL; e.dtype := NIL; t := e;
                 NEW(e.clist); e.clist.list := NIL; e.clist.object := NIL;
                 NEW(e.nlist);  e.nlist.list := NIL;  e.nlist.object := NIL;
    | Pointer  :
                 NEW(r); r.tgt := NIL; t := r;
    | Primitive: 
                 NEW(p); p.base := NIL; p.impl := NIL; p.dtype := NIL; p.naCtor := FALSE; t := p;
                 NEW(p.clist); p.clist.list := NIL; p.clist.object := NIL;
                 NEW(p.nlist);  p.nlist.list := NIL;  p.nlist.object := NIL;
                 NEW(p.vmlist); p.vmlist.list := NIL; p.vmlist.object := NIL;
                 NEW(p.smlist); p.smlist.list := NIL; p.smlist.object := NIL;
                 NEW(p.sflist); p.sflist.list := NIL; p.sflist.object := NIL;
    | Struct   : 
                 NEW(s); s.base := NIL; s.impl := NIL; s.dtype := NIL; s.naCtor := FALSE; t := s;
                 NEW(s.clist); s.clist.list := NIL; s.clist.object := NIL;
                 NEW(s.nlist);  s.nlist.list := NIL;  s.nlist.object := NIL;
                 NEW(s.vmlist); s.vmlist.list := NIL; s.vmlist.object := NIL;
                 NEW(s.smlist); s.smlist.list := NIL; s.smlist.object := NIL;
                 NEW(s.sflist); s.sflist.list := NIL; s.sflist.object := NIL;
                 NEW(s.iflist); s.iflist.list := NIL; s.iflist.object := NIL;
    ELSE
        t := NIL;
    END; (* IF *)
    t.vmod := Vprivate;
    t.inhie := FALSE;
    t.anon := FALSE;
    t.aptr := NIL;
    RETURN(t);
END NewType;


PROCEDURE (typ: Type) Initialize(name: CharOpen; fname: CharOpen;
                                  space: Namespace), NEW, EXTENSIBLE;
BEGIN
    typ.name := name;
    typ.fname := fname;
    typ.space := space;
    typ.tord := unCertain;
    WITH typ : PointerType DO
        IF typ.tgt.name = NIL THEN
            IF typ.tgt IS ArrayType THEN
                typ.tgt.name := ST.StrCat(typ.name, anonArr);
                typ.tgt.fname := ST.StrCat(typ.fname, anonArr);
            ELSIF typ.tgt IS RecordType THEN
                typ.tgt.name := ST.StrCat(typ.name,  anonRec);
                typ.tgt.fname := ST.StrCat(typ.fname,  anonRec);
            ELSE
            END; (* IF *)
            typ.tgt.anon := TRUE;
        END; (* IF *)
        typ.tgt.space := space;
        typ.tgt.tord := unCertain;
    ELSE
    END; (* IF *)
END Initialize;


PROCEDURE (typ: Type) InsertField*(IN fname: CharOpen; 
                                    IN ftyp: Type;
                                    IN fvalue: Literal;
                                    static: BOOLEAN;
                                    isPtr: BOOLEAN): Field, NEW, EXTENSIBLE;
(* to be redefined in its sub-class *)
BEGIN
    RETURN NIL;
END InsertField;


PROCEDURE (typ: Type) InsertConstructor*(IN mname: CharOpen; 
                                         IN rtyp: Type;
                                         fl: FormalList): Function, NEW, EXTENSIBLE;
(* to be redefined in its sub-class *)
BEGIN
    ASSERT(FALSE);
    RETURN NIL;
END InsertConstructor;


PROCEDURE (typ: Type) InsertMethod*(IN mname: CharOpen; 
                                     IN rt: Type;
                                     static: BOOLEAN;
                                     virtual: BOOLEAN;
                                     rtnPtr: BOOLEAN;           (* method returns a pointer to ValueType *)
                                     fl: FormalList): Method, NEW, EXTENSIBLE;
(* to be redefined in its sub-class *)
BEGIN
    RETURN NIL;
END InsertMethod;


PROCEDURE (typ: Type) GetEventList*(): OrderList, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: ClassType DO
            RETURN target.elist;
        ELSE
            RETURN NIL;
        END; (* WITH *)
    | typ: ClassType DO
        RETURN typ.elist;
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetEventList;


PROCEDURE (tv: OTraverser) GetNextEvent*(): Event, NEW;
BEGIN
    IF tv.curr.list # NIL THEN
        tv.curr := tv.curr.list;
        RETURN(tv.curr.object(Event));
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetNextEvent;


PROCEDURE (typ: Type) HasEvents*(): BOOLEAN, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: ClassType DO
            RETURN (target.elist.list # NIL);
        ELSE
            RETURN FALSE;
        END; (* WITH *)
    | typ: ClassType DO
        RETURN (typ.elist.list # NIL);
    ELSE
        RETURN FALSE;
    END; (* WITH *)
END HasEvents;


PROCEDURE (type: Type) InsertEvent*(IN ename: CharOpen; 
                                    multi: BOOLEAN;
                                    dt: Type;
                                    ht: Type;
                                    amtd: Method;
                                    rmtd: Method): Event, NEW, EXTENSIBLE;
(* to be redefined in its sub-class *)
BEGIN
    RETURN NIL;
END InsertEvent;


PROCEDURE (typ: Type) Dumped*(): BOOLEAN, NEW;
BEGIN
    RETURN typ.tord # unCertain;
END Dumped;


PROCEDURE (typ: Type) GetTypeOrd*(): INTEGER, NEW;
BEGIN
    RETURN typ.tord;
END GetTypeOrd;


PROCEDURE (typ: Type) SetTypeOrd*(tord: INTEGER), NEW;
BEGIN
    typ.tord := tord;
END SetTypeOrd;


PROCEDURE (typ: Type) ClearTypeOrd*(), NEW;
BEGIN
    typ.tord := unCertain;
END ClearTypeOrd;


PROCEDURE (typ: Type) GetAssembly*(): Assembly, NEW;
BEGIN
    RETURN typ.space.asb;
END GetAssembly;


PROCEDURE (typ: Type) GetAssemblyName*(): CharOpen, NEW;
BEGIN
    IF typ.space # NIL THEN
        RETURN typ.space.asb.name;
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetAssemblyName;


PROCEDURE (typ: Type) GetNamespaceName*(): CharOpen, NEW;
BEGIN
    RETURN typ.space.fname;
END GetNamespaceName;


PROCEDURE (typ: Type) GetNamespace*(): Namespace, NEW;
BEGIN
    RETURN typ.space;
END GetNamespace;


PROCEDURE (typ: Type) SetNamespace*(ns: Namespace), NEW;
BEGIN
    typ.space := ns;
END SetNamespace;


PROCEDURE (typ: Type) IsAbstract*(): BOOLEAN, NEW;
BEGIN
    RETURN (RabsBit IN typ.attr) & ~(RlimBit IN typ.attr);
END IsAbstract;


PROCEDURE (typ: Type) IsExtensible*(): BOOLEAN, NEW;
BEGIN
    RETURN (RabsBit IN typ.attr) & (RlimBit IN typ.attr);
END IsExtensible;


PROCEDURE (typ: Type) IsValueType*(): BOOLEAN, NEW;
BEGIN
    RETURN (RvalBit IN typ.attr)
END IsValueType;


PROCEDURE (typ: Type) NotAnonymous*(), NEW;
BEGIN
    typ.anon := FALSE;
END NotAnonymous;


PROCEDURE (typ: Type) SetAnonymous*(), NEW;
BEGIN
    typ.anon := TRUE;
END SetAnonymous;


PROCEDURE (typ: Type) IsAnonymous*(): BOOLEAN, NEW;
BEGIN
    RETURN typ.anon;
END IsAnonymous;


PROCEDURE (typ: Type) GetAnonymousPointerType*(): PointerType, NEW;
BEGIN
    RETURN typ.aptr;
END GetAnonymousPointerType;


PROCEDURE (typ: Type) IsInterface*(): BOOLEAN, NEW;
BEGIN
    WITH typ: PointerType DO
        RETURN typ.tgt IS IntfcType;
    ELSE
        RETURN typ IS IntfcType;
    END; (* WITH *)
END IsInterface;

PROCEDURE (typ: Type) IsInHierarchy*(): BOOLEAN, NEW;
BEGIN
    RETURN typ.inhie;
END IsInHierarchy;


PROCEDURE (typ: Type) SetInHierarchy*(), NEW;
VAR
    target: Type;
BEGIN
    typ.inhie := TRUE;

    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: RecordType DO
            IF target.base # NIL THEN
                target.base.SetInHierarchy();
            END; (* IF *)
        ELSE
        END; (* WITH *)
    | typ: RecordType DO
        IF typ.base # NIL THEN
            typ.base.SetInHierarchy();
        END; (* IF *)
    ELSE
    END; (* WITH *)

END SetInHierarchy;


PROCEDURE (typ: Type) ClearInHierarchy*(), NEW;
VAR
    target: Type;
BEGIN
    typ.inhie := FALSE;

    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: RecordType DO
            IF target.base # NIL THEN
                target.base.ClearInHierarchy();
            END; (* IF *)
        ELSE
        END; (* WITH *)
    | typ: RecordType DO
        IF typ.base # NIL THEN
            typ.base.ClearInHierarchy();
        END; (* IF *)
    ELSE
    END; (* WITH *)

END ClearInHierarchy;


PROCEDURE (typ: Type) GetBaseType*(): Type, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: RecordType DO
            RETURN target.base;
        ELSE
            RETURN NIL;
        END; (* WITH *)
    | typ: RecordType DO
        RETURN typ.base;
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetBaseType;


PROCEDURE (typ: Type) SetBaseType*(base: Type), NEW;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: RecordType DO
            target.base := base;
        ELSE
            ASSERT(FALSE);
        END; (* WITH *)
    | typ: RecordType DO
        typ.base := base;
    ELSE
        ASSERT(FALSE);
    END; (* IF *)
END SetBaseType;


PROCEDURE (typ: Type) IsInterfacePtr*(): BOOLEAN, NEW;
BEGIN
    WITH typ: PointerType DO
        RETURN typ.tgt IS IntfcType;
    ELSE
        RETURN FALSE;
    END; (* IF *)
END IsInterfacePtr;


PROCEDURE (typ: Type) GetInterfaces*(): OrderList, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: RecordType DO
            RETURN target.impl;
        ELSE
            RETURN NIL;
        END; (* WITH *)
    | typ: RecordType DO
        RETURN typ.impl;
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetInterfaces;


PROCEDURE (typ: Type) HasImplInterfaces*(): BOOLEAN, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: RecordType DO
            RETURN (target.impl # NIL) & (target.impl.list # NIL);
        ELSE
            RETURN FALSE;
        END; (* WITH *)
    | typ: RecordType DO
        RETURN (typ.impl # NIL) & (typ.impl.list # NIL);
    ELSE
        RETURN FALSE;
    END; (* IF *)
END HasImplInterfaces;


PROCEDURE (typ: Type) GetConstants*(): OrderList, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: ValueType DO
            RETURN target.clist;
        ELSE
            RETURN NIL;
        END; (* WITH *)
    | typ: ValueType DO
        RETURN typ.clist;
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetConstants;


PROCEDURE (typ: Type) HasConstants*(): BOOLEAN, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: ValueType DO
            RETURN (target.clist.list # NIL);
        ELSE
            RETURN FALSE;
        END; (* WITH *)
    | typ: ValueType DO
        RETURN (typ.clist.list # NIL);
    ELSE
        RETURN FALSE;
    END; (* WITH *)
END HasConstants;


PROCEDURE (typ: Type) GetEvents*(): OrderList, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: ClassType DO
            RETURN target.elist;
        ELSE
            RETURN NIL;
        END; (* WITH *)
    | typ: ClassType DO
        RETURN typ.elist;
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetEvents;


PROCEDURE (typ: Type) GetInstanceFields*(): OrderList, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: StrucType DO
            RETURN target.iflist;
        ELSE
            RETURN NIL;
        END; (* WITH *)
    | typ: StrucType DO
        RETURN typ.iflist;
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetInstanceFields;


PROCEDURE (typ: StrucType) GetInstanceField*(fname: CharOpen): Field;
VAR
    lhead, inspos: OrderList;
    str: CharOpen;
BEGIN
    lhead := typ.iflist;
    str := fname;
    IF lhead.ISearch(str, inspos) THEN
        RETURN inspos.object(Field);
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetInstanceField;


PROCEDURE (typ: Type) GetInstanceField*(fname: CharOpen): Field, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        RETURN target.GetInstanceField(fname);
    ELSE
        RETURN NIL;
    END; (* WITH *)
END GetInstanceField;


PROCEDURE (typ: Type) HasInstanceFields*(): BOOLEAN, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: StrucType DO
            RETURN target.iflist.list # NIL;
        ELSE
            RETURN FALSE;
        END; (* WITH *)
    | typ: StrucType DO
        RETURN typ.iflist.list # NIL;
    ELSE
        RETURN FALSE;
    END; (* WITH *)
END HasInstanceFields;


PROCEDURE (typ: Type) GetStaticFields*(): OrderList, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: PrimType DO
            RETURN target.sflist;
        ELSE
            RETURN NIL;
        END; (* WITH *)
    | typ: PrimType DO
        RETURN typ.sflist;
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetStaticFields;


PROCEDURE (typ: PrimType) GetStaticField*(fname: CharOpen): Field;
VAR
    lhead, inspos: OrderList;
BEGIN
    lhead := typ.sflist;
    IF lhead.ISearch(fname, inspos) THEN
        RETURN inspos.object(Field);
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetStaticField;


PROCEDURE (typ: Type) GetStaticField*(fname: CharOpen): Field, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        RETURN target.GetStaticField(fname);
    ELSE
        RETURN NIL;
    END; (* WITH *)
END GetStaticField;


PROCEDURE (typ: Type) HasStaticFields*(): BOOLEAN, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: PrimType DO
            RETURN target.sflist.list # NIL;
        ELSE
            RETURN FALSE;
        END; (* WITH *)
    | typ: PrimType DO
        RETURN typ.sflist.list # NIL;
    ELSE
        RETURN FALSE;
    END; (* WITH *)
END HasStaticFields;


PROCEDURE (typ: Type) GetNestedTypes*(): OrderList, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: RecordType DO
            RETURN target.nlist;
        ELSE
            RETURN NIL;
        END; (* WITH *)
    | typ: RecordType DO
        RETURN typ.nlist;
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetNestedTypes;


PROCEDURE (typ: Type) HasNestedTypes*(): BOOLEAN, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: RecordType DO
            RETURN (target.nlist.list # NIL);
        ELSE
            RETURN FALSE;
        END; (* WITH *)
    | typ: RecordType DO
        RETURN (typ.nlist.list # NIL);
    ELSE
        RETURN FALSE;
    END; (* WITH *)
END HasNestedTypes;


PROCEDURE (typ: Type) HasNoArgConstructor*(): BOOLEAN, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: StrucType DO
            RETURN target.naCtor;
        ELSE
            RETURN FALSE;
        END; (* WITH *)
    | typ: StrucType DO
        RETURN typ.naCtor;
    ELSE
        RETURN FALSE;
    END; (* WITH *)
END HasNoArgConstructor;


PROCEDURE (typ: Type) SetHasNoArgConstructor*(), NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: StrucType DO
            target.naCtor := TRUE;
        ELSE
        END; (* WITH *)
    | typ: StrucType DO
        typ.naCtor := TRUE;
    ELSE
    END; (* WITH *)
END SetHasNoArgConstructor;


PROCEDURE (typ: Type) HasStatic*(): BOOLEAN, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: RecordType DO
            WITH target: ValueType DO
                WITH target: PrimType DO
                    RETURN ((target.smlist.list # NIL) OR
                            (target.sflist.list # NIL) OR
                            (target.clist.list # NIL));
                ELSE
                    RETURN (target.clist.list # NIL);
                    (* EnumType *)
                END; (* WITH *)
            ELSE
                (* IntfcType *)
                RETURN FALSE;
            END; (* WITH *)
        ELSE
            (* ArrayType *)
            RETURN FALSE;
        END; (* WITH *)
    | typ: RecordType DO
        WITH typ: ValueType DO
            WITH typ: PrimType DO
                RETURN ((typ.smlist.list # NIL) OR
                        (typ.sflist.list # NIL) OR
                        (typ.clist.list # NIL));
            ELSE
                (* EnumType *)
                RETURN (typ.clist.list # NIL);
            END; (* WITH *)
        ELSE
            (* IntfcType *)
            RETURN FALSE;
        END; (* WITH *)
    ELSE
        (* ArrayType *)
        RETURN FALSE;
    END; (* WITH *)
END HasStatic;


PROCEDURE (typ: Type) GetStaticMethods*(): OrderList, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: PrimType DO
            RETURN target.smlist;
        ELSE
            RETURN NIL;
        END; (* WITH *)
    | typ: PrimType DO
        RETURN typ.smlist;
    ELSE
        RETURN NIL;
    END; (* WITH *)
END GetStaticMethods;


PROCEDURE (typ: PrimType) GetStaticMethod*(mname: CharOpen): Method;
VAR
    lhead, inspos: OrderList;
BEGIN
    lhead := typ.smlist;
    IF lhead.ISearch(mname, inspos) THEN
        RETURN inspos.object(Method);
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetStaticMethod;


PROCEDURE (typ: Type) GetStaticMethod*(mname: CharOpen): Method, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        RETURN target.GetStaticMethod(mname);
    ELSE
        RETURN NIL;
    END; (* WITH *)
END GetStaticMethod;


PROCEDURE (typ: Type) HasStaticMethods*(): BOOLEAN, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: PrimType DO
            RETURN (target.smlist.list # NIL);
        ELSE
            RETURN FALSE;
        END; (* WITH *)
    | typ: PrimType DO
        RETURN (typ.smlist.list # NIL);
    ELSE
        RETURN FALSE;
    END; (* WITH *)
END HasStaticMethods;


PROCEDURE (typ: Type) GetVirtualMethods*(): OrderList, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: IntfcType DO
            RETURN target.vmlist;
        | target: PrimType DO
            RETURN target.vmlist;
        ELSE
            RETURN NIL;
        END; (* WITH *)
    | typ: IntfcType DO
        RETURN typ.vmlist;
    | typ: PrimType DO
        RETURN typ.vmlist;
    ELSE
        RETURN NIL;
    END; (* WITH *)
END GetVirtualMethods;


PROCEDURE (typ: Type) IsDelegate*(): BOOLEAN, NEW;
BEGIN
    WITH typ: PointerType DO
        RETURN typ.tgt IS DelegType;
    ELSE
        RETURN typ IS DelegType;
    END; (* WITH *)
END IsDelegate;


PROCEDURE (typ: Type) SetMulticast*(), NEW;
(* typ is a Delegate type *)
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: DelegType DO
            target.isMul := TRUE;
        ELSE
            ASSERT(FALSE);
        END; (* WITH *)
    | typ: DelegType DO
        typ.isMul := TRUE;
    ELSE
        ASSERT(FALSE);
    END; (* WITH *)
END SetMulticast;


PROCEDURE (typ: Type) IsMulticast*(): BOOLEAN, NEW;
(* typ is a Delegate type *)
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: DelegType DO
            RETURN target.isMul;
        ELSE
            ASSERT(FALSE);
        END; (* WITH *)
    | typ: DelegType DO
        RETURN typ.isMul;
    ELSE
        ASSERT(FALSE);
    END; (* WITH *)
    RETURN FALSE;
END IsMulticast;


PROCEDURE (typ: Type) GetInvokeMethod*(): Method, NEW;
(* typ is a Delegate type *)
VAR
    target: Type;
    lhead, inspos: OrderList;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: DelegType DO
            lhead := target.vmlist;
            IF lhead.ISearch(ST.ToChrOpen("Invoke"), inspos) THEN
                RETURN inspos.object(Method);
            ELSE
                ASSERT(FALSE); RETURN NIL;
            END; (* IF *)
        ELSE
            ASSERT(FALSE); RETURN NIL;
        END; (* WITH *)
    | typ: DelegType DO
        lhead := typ.vmlist;
        IF lhead.ISearch(ST.ToChrOpen("Invoke"), inspos) THEN
            RETURN inspos.object(Method);
        ELSE
            ASSERT(FALSE); RETURN NIL;
        END; (* IF *)
    ELSE
        ASSERT(FALSE); RETURN NIL;
    END; (* WITH *)
END GetInvokeMethod;


PROCEDURE (typ: IntfcType) GetVirtualMethod*(mname: CharOpen): Method;
VAR
    lhead, inspos: OrderList;
BEGIN
    lhead := typ.vmlist;
    IF lhead.ISearch(mname, inspos) THEN
        RETURN inspos.object(Method);
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetVirtualMethod;


PROCEDURE (typ: PrimType) GetVirtualMethod*(mname: CharOpen): Method;
VAR
    lhead, inspos: OrderList;
BEGIN
    lhead := typ.vmlist;
    IF lhead.ISearch(mname, inspos) THEN
        RETURN inspos.object(Method);
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetVirtualMethod;


PROCEDURE (typ: Type) GetVirtualMethod*(mname: CharOpen): Method, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        RETURN target.GetVirtualMethod(mname);
    ELSE
        RETURN NIL;
    END; (* WITH *)
END GetVirtualMethod;


PROCEDURE (typ: Type) HasVirtualMethods*(): BOOLEAN, NEW, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: IntfcType DO
            RETURN target.vmlist.list # NIL;
        | target: PrimType DO
            RETURN target.vmlist.list # NIL;
        ELSE
            RETURN FALSE;
        END; (* WITH *)
    | typ: IntfcType DO
        RETURN typ.vmlist.list # NIL;
    | typ: PrimType DO
        RETURN typ.vmlist.list # NIL;
    ELSE
        RETURN FALSE;
    END; (* WITH *)
END HasVirtualMethods;


PROCEDURE (typ: PointerType) IsArrayPointer*(): BOOLEAN, NEW;
BEGIN
    RETURN typ.tgt IS ArrayType;
END IsArrayPointer;


PROCEDURE (typ: Type) IsNested*(): BOOLEAN, NEW;
VAR
    target: Type;
BEGIN
    WITH typ: PointerType DO
        target := typ.tgt;
        WITH target: RecordType DO
            RETURN target.dtype # NIL;
        ELSE
            RETURN FALSE;
        END; (* WITH *)
    | typ: RecordType DO
        RETURN typ.dtype # NIL;
    ELSE
        RETURN FALSE;
    END; (* WITH *)
END IsNested;


PROCEDURE (typ: Type) IsTempType*(): BOOLEAN, NEW;
BEGIN
    WITH typ: TempType DO
        WITH typ: NamedType DO
            RETURN FALSE;
        ELSE
            RETURN TRUE;
        END; (* WITH *)
    ELSE
        RETURN FALSE;
    END; (* WITH *)
END IsTempType;


PROCEDURE (typ: Type) IsDummyType*(): BOOLEAN, NEW;
BEGIN
    WITH typ: PointerType DO
        RETURN FALSE;
    | typ: ArrayType DO
        RETURN FALSE;
    | typ: RecordType DO
        RETURN FALSE;
    | typ: NamedType DO
        RETURN FALSE;
    | typ: TempType DO
        RETURN FALSE;
    ELSE
        IF ST.StrCmp(typ.name, dmyTyp.name) = ST.Equal THEN
            RETURN TRUE;
        ELSE
            RETURN FALSE;
        END; (* IF *)
    END; (* WITH *)
END IsDummyType;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (typ: NamedType) GetName*(): CharOpen;
VAR
    dim: INTEGER;
    tname: CharOpen;
    sdim: ARRAY 4 OF CHAR;
BEGIN
    IF ST.StrRChr(typ.name, ']') = ST.NotExist THEN
        (* non-array NamedType *)
        RETURN typ.name;
    ELSE
        (* array NamedType *)
        dim := 1;
        tname := ST.SubStr(typ.name, 0, LEN(typ.name)-4);
        WHILE ST.StrRChr(tname, ']') # ST.NotExist DO
            INC(dim);
            tname := ST.SubStr(tname, 0, LEN(tname)-4);
        END; (* WHILE *)
        RTS.IntToStr(dim, sdim);
        tname := ST.StrCat(ST.ToChrOpen(sdim), tname);
        RETURN ST.StrCat(ST.ToChrOpen(AnonArrHead), tname);
    END; (* IF *)
END GetName;


PROCEDURE (typ: NamedType) GetFullName*(): CharOpen;
BEGIN
    IF ST.StrRChr(typ.fname, ']') = ST.NotExist THEN
        (* non-array NamedType *)
        RETURN typ.name;
    ELSE
        (* array NamedType *)
        RETURN ST.StrCat(ST.StrCatChr(typ.space.fname, '.'), typ.GetName());
    END; (* IF *)
END GetFullName;


PROCEDURE (typ: PointerType) GetName*(): CharOpen;
VAR
    dim: INTEGER;
    tname: CharOpen;
    sdim: ARRAY 4 OF CHAR;
    tgt: Type;
    idx: INTEGER;
BEGIN
    tgt := typ.tgt;
    WITH tgt: ArrayType DO
        NEW(tname,2);
        idx := ST.StrRChr(tgt.name, '_');
        RTS.IntToStr(tgt.dim, tname);
        tname := ST.StrCat(tname, ST.SubStr(tgt.name, 0, idx-1));
        RETURN ST.StrCat(ST.ToChrOpen(AnonArrHead), tname);
    ELSE
        (* could be non-array type, or POINTER TO ARRAY type 
           but array target is not built at the moment *)
        IF ST.StrRChr(typ.name, ']') = ST.NotExist THEN
            (* non-array type *)
            RETURN typ.name;
        ELSE
            (* possible array type *)
            dim := 1;
            tname := ST.SubStr(typ.name, 0, LEN(typ.name)-4);
            WHILE ST.StrRChr(tname, ']') # ST.NotExist DO
                INC(dim);
                tname := ST.SubStr(tname, 0, LEN(tname)-4);
            END; (* WHILE *)
            RTS.IntToStr(dim, sdim);
            tname := ST.StrCat(ST.ToChrOpen(sdim), tname);
            RETURN ST.StrCat(ST.ToChrOpen(AnonArrHead), tname);
        END; (* IF *)
    END; (* WITH *)
END GetName;


PROCEDURE (typ: PointerType) GetFullName*(): CharOpen;
BEGIN
    IF ~(typ.tgt IS ArrayType) & (ST.StrRChr(typ.fname, ']') = ST.NotExist) THEN
        (* non-array PointerType *)
        RETURN typ.name;
    ELSE
        (* array PointerType *)
        RETURN ST.StrCat(ST.StrCatChr(typ.space.fname, '.'), typ.GetName());
    END; (* IF *)
END GetFullName;


PROCEDURE (typ: PointerType) GetTarget*(): Type, NEW;
BEGIN
    RETURN typ.tgt;
END GetTarget;


PROCEDURE (typ: PointerType) SetTarget*(tgt: Type), NEW;
BEGIN
    typ.tgt := tgt;
END SetTarget;


PROCEDURE MakeAnonymousPointerType*(target: Type): PointerType;
VAR
    typ: PointerType;
BEGIN
    typ := NewType(Pointer)(PointerType);
    typ.name := ST.ToChrOpen("");
    typ.fname := ST.ToChrOpen("");
    typ.space := NIL;
    typ.vmod := Vprivate;
    typ.tord := unCertain;
    typ.anon := TRUE;
    typ.tgt := target;
    typ.tgt.aptr := typ;
    RETURN typ;
END MakeAnonymousPointerType;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (typ: RecordType) MakeMethod*(IN mname: CharOpen;
                                        static: BOOLEAN;
                                        rtype: Type;
                                        fl: FormalList): Method, NEW;
VAR
    mth: Method;
    fn: Function;
BEGIN
    IF rtype = NIL THEN
        NEW(mth);
    ELSE
        NEW(fn); fn.rtype := rtype; fn.ostd := 0; fn.rtnPtr := FALSE; mth := fn;
    END; (* IF *)
    mth.name := mname;
    mth.isCtor := FALSE;
    mth.isStat := static;
    mth.isVirt := FALSE;
    mth.formals := fl; fl.mth := mth;
    mth.class := typ;

    RETURN mth;
END MakeMethod;


PROCEDURE (typ: RecordType) AddMethod*(IN new: Method): Method, NEW;
VAR
    lhead, lhead2: OrderList;
    inspos, inspos2: OrderList;
    old: Method;
    newovlname: CharOpen;
BEGIN
    (* determine which method list the method should go into *)
    IF new.isStat THEN
        WITH typ: PrimType DO
            lhead := typ.smlist;
        ELSE
            (* shouldn't have static method in RecordType other from
               PrimType, StrucType and ClassType *)
            lhead := NIL; ASSERT(FALSE);
        END; (* WITH *)
    ELSE
        WITH typ: PrimType DO
            lhead := typ.vmlist;
        | typ: IntfcType DO
            lhead := typ.vmlist;
        ELSE
            (* shouldn't have Instance/Virtual method in RecordType other from
               PrimType, StrucType, ClassType and IntfcType *)
            lhead := NIL; ASSERT(FALSE);
        END; (* WITH *)
    END; (* IF *)

    (* put the method into the class method list *)
    IF lhead.ISearch(new.name, inspos) THEN
        (* method with same name found *)
        IF (inspos.object(Method).formals.sCode = NIL) THEN
            (* old was inserted from SymReader *)
            IF (new.formals.sCode = NIL) THEN
                (* new is also inserted from SymReader *)
                old := inspos.object(Method);
                RETURN old;
            ELSE
                (* new is inserted from MetaParser, so replace old *)
                inspos.Replace(new);
                RETURN new;
            END; (* IF *)
        ELSE
            (* old was inserted from MetaParser *)
            IF new.formals.sCode = NIL THEN
                (* new is inserted from SymReader,
                   and has no method overload ==> no sigCode present in symbol file *)
                (* discard the new one, use the old one inserted from MetaParser *)
                old := inspos.object(Method);
                RETURN old;
            ELSE
                IF inspos.object(Method).formals.sCode^ = new.formals.sCode^ THEN
                    (* Identical method has already been inserted before *)
                    old := inspos.object(Method);
                    (* use the new formal list, as it has more updated type information on the formals *)
                    old.formals := new.formals;
                    old.formals.mth := old;
                    RETURN old;
                ELSE
                    (* an overload method found *)
                    new.isOVL := TRUE;
                    IF ~inspos.object(Method).isOVL THEN
                        (* the method with same name found is also an overload method, but not declared as,
                           so encrypt its signature *)
                        inspos.object(Method).isOVL := TRUE;
                        inspos.object(Method).EncryptSignature();
                    ELSE
                        (* the method with same name found is already declared as an overload method *)
                    END; (* IF *)

                    (* search the method list for the overload method *)
                    newovlname := ST.StrCatChr(new.name,'_');
                    newovlname := ST.StrCat(newovlname, new.formals.sCode);
                    IF lhead.MSearch(new.name, newovlname, inspos) THEN
                        (* method with same overload name found *)
                        old := inspos.object(Method);
                        (* use the new formal list, as it has more updated type information on the formals *)
                        old.formals := new.formals;
                        old.formals.mth := old;
                        RETURN old;
                    ELSE
                        inspos.Insert(new);
                        IF MethodNameMangling = FALSE THEN new.oname := newovlname; END;
                    END; (* IF *)
                END; (* IF *)
            END; (* IF *)
        END; (* IF *)
    ELSE
        (* search the other method list for name mangling *)
        WITH typ: PrimType DO
            IF lhead = typ.vmlist THEN
                lhead2 := typ.smlist;
            ELSE
                lhead2 := typ.vmlist;
            END; (* WITH *)
        ELSE
            (* Only PrimType can have static method *)
            lhead2 := NIL;
        END; (* IF *)
        IF (lhead2 # NIL) & lhead2.ISearch(new.name, inspos2) THEN
            (* method with same name found in the other method list *)
            new.isOVL := TRUE;
            IF ~inspos2.object(Method).isOVL THEN
                (* the method with same name found in the other method list is also
                   an overload method, so encrypt its signature if it has not been done *)
                inspos2.object(Method).isOVL := TRUE;
                inspos2.object(Method).EncryptSignature();
            END; (* IF *)
        ELSE
            (* either type has no smlist, or no method with same name found in both smlist and vmlist *)
        END; (* IF *)
        inspos.Insert(new);
        IF MethodNameMangling = FALSE THEN
            newovlname := ST.StrCatChr(new.name,'_');
            newovlname := ST.StrCat(newovlname, new.formals.sCode);
            new.oname := newovlname;
        END;
    END; (* IF *)
    RETURN new;
END AddMethod;


PROCEDURE (typ: RecordType) InsertMethod*(IN mname: CharOpen; 
                                        IN rtype: Type;
                                        static: BOOLEAN;
                                        virtual: BOOLEAN;
                                        rtnPtr: BOOLEAN;                (* method returns a pointer to ValueType *)
                                        fl: FormalList): Method;
(* although RecordType method, types other from StrucType and IntfcType are ruled out *)
VAR
    mth: Method;
    rt : Type;
    lhead, inspos: OrderList;
BEGIN
    IF rtype.name^ = Void THEN
        rt := NIL;
    ELSE
        rt := rtype;
    END; (* IF *)
    mth := typ.MakeMethod(mname, static, rt, fl);

    IF (rtype # NIL) & (rtype.space # typ.space) THEN
        (* if the return type is in foreign space *)
        lhead := typ.space.fnslist;
        IF ~lhead.IFSearch(rtype.space.fname, inspos) THEN
            inspos.Insert(rtype.space);
        END; (* IF *)
    END; (* IF *)

    mth := typ.AddMethod(mth);
    IF ~static THEN mth.isVirt := virtual END;

    RETURN mth;
END InsertMethod;


PROCEDURE (typ: PointerType) InsertMethod*(IN mname: CharOpen; 
                                           IN rtype: Type;
                                           static: BOOLEAN;
                                           virtual: BOOLEAN;
                                           rtnPtr: BOOLEAN;             (* method returns a pointer to ValueType *)
                                           fl: FormalList): Method;
VAR
    mth: Method;
    target: Type;
BEGIN
    mth := NIL;
    target := typ.tgt;
    WITH target: RecordType DO
        mth := target.InsertMethod(mname, rtype, static, virtual, rtnPtr, fl);
        mth.class := typ;
    ELSE
    END; (* WITH *)
    RETURN mth;
END InsertMethod;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (class: ClassType) InsertEvent*(IN ename: CharOpen; 
                                         multi: BOOLEAN;
                                         dt: Type;
                                         ht: Type;
                                         amtd: Method;
                                         rmtd: Method): Event, EXTENSIBLE;
VAR
    evt: Event;
    inspos: OrderList;
BEGIN
    NEW(evt);
    evt.name := ename;
    evt.isMul := multi;
    evt.dtype := dt;
    evt.htype := ht;
    evt.addOn := amtd;
    evt.remOn := rmtd;

    (* put the event into the class event list *)
    IF ~class.elist.ISearch(evt.name, inspos) THEN
        inspos.Insert(evt);
    ELSE
        ASSERT(FALSE);
    END; (* IF *)

    RETURN evt;
END InsertEvent;


PROCEDURE (class: PointerType) InsertEvent*(IN ename: CharOpen; 
                                          multi: BOOLEAN;
                                          dt: Type;
                                          ht: Type;
                                          amtd: Method;
                                          rmtd: Method): Event, EXTENSIBLE;
VAR
    target: Type;
BEGIN
    target := class.tgt;
    WITH target: ClassType DO
        RETURN target.InsertEvent(ename, multi, dt, ht, amtd, rmtd);
    ELSE
        RETURN NIL;
    END; (* WITH *)
END InsertEvent;


PROCEDURE (typ: ValueType) MakeConstant*(IN cname: CharOpen;
                                         IN ctyp: Type;
                                         IN fvalue: Literal): Constant, NEW;
VAR
    const: Constant;
BEGIN
    NEW(const);
    const.name := cname;
    const.type := ctyp; const.value := fvalue;
    const.class := typ;

    RETURN const;
END MakeConstant;


PROCEDURE (typ: ValueType) MakeField*(IN fname: CharOpen;
                                      ftyp: Type;
                                      isPtr: BOOLEAN): Field, NEW;
VAR
    field: Field;
BEGIN
    NEW(field);
    field.name := fname;
    field.type := ftyp; field.isPtr := isPtr;
    field.class := typ;

    RETURN field;
END MakeField;


PROCEDURE (typ: ValueType) AddField*(field: Field; static: BOOLEAN): BOOLEAN, NEW;
VAR
    olist: OrderList;
    inspos: OrderList;
BEGIN
    IF field IS Constant THEN
        olist := typ.clist;
    ELSE
        IF static THEN
            (* Static field variable *)
            WITH typ: PrimType DO
                olist := typ.sflist;
            ELSE  (* nothing else here ??? *)
                olist := NIL; ASSERT(FALSE);
                RETURN FALSE;
            END; (* WITH *)
        ELSE
            (* Instance field variable *)
            WITH typ: StrucType DO
                olist := typ.iflist;
            ELSE (* PrimType *)
                RETURN FALSE;
            END; (* WITH *)
        END; (* IF *)
    END; (* IF *)

    (* put the field into the class field list *)
    IF ~olist.ISearch(field.name, inspos) THEN
        inspos.Insert(field);
    ELSE
    END; (* IF *)
    RETURN TRUE;
END AddField;


PROCEDURE (typ: ValueType) InsertField*(IN fname: CharOpen; 
                                        IN ftyp: Type;
                                        IN fvalue: Literal;
                                        static: BOOLEAN; 
                                        isPtr: BOOLEAN): Field, EXTENSIBLE;
VAR
    field: Field;
    lhead, inspos: OrderList;
BEGIN
    (* Is it a field constant or a field variable *)
    IF fvalue = NIL THEN
        field := typ.MakeField(fname, ftyp, isPtr);
    ELSE
        field := typ.MakeConstant(fname, ftyp, fvalue);
    END; (* IF *)

    IF ftyp.space # typ.space THEN
        (* if the field type is in foreign space *)
        lhead := typ.space.fnslist;
        IF ~lhead.IFSearch(ftyp.space.fname, inspos) THEN
            inspos.Insert(ftyp.space);
        END; (* IF *)
    END; (* IF *)

    IF typ.AddField(field, static) THEN
        RETURN field;
    ELSE
        RETURN NIL;
    END; (* IF *)
END InsertField;


PROCEDURE (typ: PointerType) InsertField*(IN fname: CharOpen; 
                                        IN ftyp: Type;
                                        IN fvalue: Literal;
                                        static: BOOLEAN;
                                        isPtr: BOOLEAN): Field, EXTENSIBLE;
VAR
    fld: Field;
    target: Type;
BEGIN
    fld := NIL;
    target := typ.tgt;
    WITH target: ValueType DO
        fld := target.InsertField(fname, ftyp, fvalue, static, isPtr);
        fld.class := typ;
    ELSE
        ASSERT(FALSE);
    END; (* IF *)
    RETURN fld;
END InsertField;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (prim: PrimType) ConvTypeName*(): CharOpen, NEW;
BEGIN
    CASE prim.tord OF
      boolN : RETURN gpcpBoolean;
    | sChrN : RETURN gpcpSChar;  (* SHORTCHAR *)
    | charN : RETURN gpcpChar;
    | byteN : RETURN gpcpSByte;
    | uBytN : RETURN gpcpUByte;
    | sIntN : RETURN gpcpInt16;
    | intN  : RETURN gpcpInt32;
    | lIntN : RETURN gpcpInt64;
    | sReaN : RETURN gpcpSingle;
    | realN : RETURN gpcpDouble;
(*
    | setN  : RETURN = 10;       (* SET *)
    | anyRec: RETURN = 11;       (* ANYREC *)
    | anyPtr: RETURN = 12;       (* ANYPTR *)
    | strN  : RETURN = 13;       (* STRING (ARRAY OF CHAR) *)
    | sStrN : RETURN = 14;       (* SHORTSTRING (ARRAY OF SHORTCHAR) *)
*)
    ELSE
        RETURN ST.NullString;
    END; (* CASE *)
END ConvTypeName;


PROCEDURE (prim: PrimType) MarkPrimitiveTypeOrd*(), NEW;
(* currently handle conversion of the following .net types *)
(* ?,Boolean,?,Byte,?,Char,?,Double,?,Int16,?,Int32,
   ?,Int64,?,IntPtr,?,SByte,?,Single,?,UInt16,?,UInt32,?,UInt64,? *)
VAR
    tname: CharOpen;
BEGIN
    tname := prim.name;
    CASE ST.StrCmp(tname, dnetInt32) OF
      ST.Equal  : prim.tord := intN;
    | ST.Less   : (* ?,Boolean,?,Byte,?,Char,?,Double,?,Int16,? *)
        CASE ST.StrCmp(tname, dnetChar) OF
          ST.Equal  : prim.tord := charN;
        | ST.Less   : (* ?,Boolean,?,Byte,? *)
            CASE ST.StrCmp(tname, dnetBoolean) OF
              ST.Equal  : prim.tord := boolN;
            | ST.Less   : (* ? *)
                ASSERT(FALSE);
            ELSE (* ?,Byte,? *)
                CASE ST.StrCmp(tname, dnetByte) OF
                  ST.Equal  : prim.tord := uBytN;  
(*
 *                ST.Equal  : (* prim.tord := byteN; *) 
 * <<== wrong, .NET mscorlib_System.Byte does not map to GPCP BYTE
 *                            prim.tord := unCertain;
 *)
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ? *)
                    ASSERT(FALSE);
                END; (* CASE *)
            END; (* CASE *)
        ELSE (* ?,Double,?,Int16,? *)
            CASE ST.StrCmp(tname, dnetDouble) OF
              ST.Equal  : prim.tord := realN;
            | ST.Less   : (* ? *)
                ASSERT(FALSE);
            ELSE (* ?,Int16,? *)
                CASE ST.StrCmp(tname, dnetInt16) OF
                  ST.Equal  : prim.tord := sIntN;
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ? *)
                    ASSERT(FALSE);
                END; (* CASE *)
            END; (* CASE *)
        END; (* CASE *)
    ELSE (* ?,Int64,?,IntPtr,?,SByte,?,Single,?,UInt16,?,UInt32,?,UInt64,?,UIntPtr *)
        CASE ST.StrCmp(tname, dnetSingle) OF
          ST.Equal  : prim.tord := sReaN;
        | ST.Less   : (* ?,Int64,?,IntPtr,?,SByte,? *)
            CASE ST.StrCmp(tname, dnetIntPtr) OF
              ST.Equal  : prim.tord := notBs;
            | ST.Less   : (* ?,Int64,? *)
                CASE ST.StrCmp(tname, dnetInt64) OF
                  ST.Equal  : prim.tord := lIntN;
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ? *)
                    ASSERT(FALSE);
                END; (* CASE *)
            ELSE (* ?,SByte,? *)
                CASE ST.StrCmp(tname, dnetSByte) OF
                  ST.Equal  : prim.tord := byteN;
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ? *)
                    ASSERT(FALSE);
                END; (* CASE *)
            END; (* IF *)
        ELSE (* ?,UInt16,?,UInt32,?,UInt64,?,UIntPtr,? *)
            CASE ST.StrCmp(tname, dnetUInt32) OF
              ST.Equal  : (* prim.tord := intN; *) (* <<== wrong, .NET mscorlib_System.UInt32 does not map to GPCP INTEGER *)
                          prim.tord := unCertain;
            | ST.Less   : (* ?,UInt16,? *)
                CASE ST.StrCmp(tname, dnetUInt16) OF
                  ST.Equal  : (* prim.tord := sIntN; *)  (* <<== wrong, .NET mscorlib_System.UInt16 does not map to GPCP SHORTINT *)
                              prim.tord := unCertain;
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ? *)
                    ASSERT(FALSE);
                END; (* CASE *)
            ELSE (* ?,UInt64,?,UIntPtr,? *)
                CASE ST.StrCmp(tname, dnetUInt64) OF
                  ST.Equal  : (* prim.tord := lIntN; *)  (* <<== wrong, .NET mscorlib_System.UInt64 does not map to GPCP LONGINT *)
                              prim.tord := unCertain;
                | ST.Less   : (* ? *)
                    ASSERT(FALSE);
                ELSE (* ?,UIntPtr,? *)
                    CASE ST.StrCmp(tname, dnetUIntPtr) OF
                      ST.Equal  : prim.tord := notBs;
                    | ST.Less   : (* ? *)
                        ASSERT(FALSE);
                    ELSE (* ? *)
                        ASSERT(FALSE);
                    END; (* CASE *)
                END; (* CASE *)
            END; (* CASE *)
        END; (* CASE *)
    END; (* CASE *)
    baseTypeArray[prim.tord] := prim;
END MarkPrimitiveTypeOrd;


PROCEDURE (struc: PrimType) InsertConstructor*(IN cname: CharOpen; 
                                                IN rtyp: Type;
                                                fl: FormalList): Function;
VAR
    fn: Function;
    old: Function;
    inspos, lhead: OrderList;
    newovlname: CharOpen;
BEGIN
    (* constructor needs to be a function as required by GPCP *)
    NEW(fn);
    fn.rtnPtr := FALSE;
    fn.rtype := rtyp;
    IF fl.IsNoArg() THEN
        struc.naCtor := TRUE;
    END; (* IF *)
    (* cctor is already eliminated in MetaParser_InsertConstructor() *)
    ASSERT (cname^ = OrigCtor);
    (* replace the name of constructor as required by GPCP *)
    fn.name := replCtor;
    fn.iname := origCtor;
    (* change the virtual constructor to static as required by GPCP *)
    fn.class := struc; fn.isCtor := TRUE; fn.isStat := TRUE; fn.isVirt := FALSE; fn.formals := fl; fl.mth := fn;
    (* put the function into the static method list *)
    lhead := struc.smlist;

    IF lhead.ISearch(fn.name, inspos) THEN
        (* constructor with same name found *)
      IF (inspos.object(Method).formals.sCode = NIL) THEN
          (* old was inserted from SymReader *)
          IF (fl.sCode = NIL) THEN
              (* new is also inserted from SymReader *)
              old := inspos.object(Function);
              RETURN old;
          ELSE
              (* new is inserted from MetaParser, so replace old *)
              inspos.Replace(fn);
              RETURN fn;
          END; (* IF *)
      ELSE
        IF inspos.object(Method).formals.sCode^ = fl.sCode^ THEN
            (* Identical constructor has already been inserted before *)
            old := inspos.object(Function);
            (* use the new formal list, as it has more updated type information on the formals *)
            old.formals := fl;
            old.formals.mth := old;
            RETURN old;
        ELSE
            (* an overload constructor found *)
            fn.isOVL := TRUE;
            IF ~inspos.object(Function).isOVL THEN
                (* the constructor with same name found is also an overload constructor,
                   so encrypt its signature if it has not been done *)
                inspos.object(Function).isOVL := TRUE;
                inspos.object(Function).EncryptSignature();
            ELSE
                (* the constructor with same name found is already declared as an overload constructor *)
            END; (* IF *)
            (* re-search the static method list for the overload constructor *)
            newovlname := ST.StrCatChr(fn.name,'_');
            newovlname := ST.StrCat(newovlname, fl.sCode);
            IF lhead.MSearch(fn.name, newovlname, inspos) THEN
                (* constructor with same overload name found *)
                old := inspos.object(Function);
                (* use the new formal list, as it has more updated type information on the formals *)
                old.formals := fl;
                old.formals.mth := old;
                RETURN old;
            ELSE
                inspos.Insert(fn);
            END; (* IF *)
        END; (* IF *)
      END;
    ELSE
        inspos.Insert(fn);
    END; (* IF *)
    RETURN fn;
END InsertConstructor;


PROCEDURE (typ: PointerType) InsertConstructor*(IN cname: CharOpen; 
                                                IN rtyp: Type;
                                                fl: FormalList): Function;
VAR
    ctor: Function;
    target: Type;
BEGIN
    ctor := NIL;
    target := typ.tgt;
    WITH target: PrimType DO
        ctor := target.InsertConstructor(cname, rtyp, fl);
        ctor.class := typ;
    ELSE
        ASSERT(FALSE);
    END; (* WITH *)
    RETURN ctor;
END InsertConstructor;


PROCEDURE (typ: EnumType) GetUnderlyingType*(): CharOpen, NEW;
BEGIN
    RETURN(typ.utype);
END GetUnderlyingType;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (f: Field) GetType*(): Type, NEW;
BEGIN
    RETURN f.type;
END GetType;


PROCEDURE (f: Field) GetDeclaringType*(): Type, NEW;
BEGIN
    RETURN f.class;
END GetDeclaringType;


PROCEDURE (f: Field) SetDeclaringType*(typ: Type), NEW;
BEGIN
    f.class := typ;
END SetDeclaringType;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (lit: Literal) IsBool*(): BOOLEAN, NEW;
BEGIN
    RETURN lit IS BoolLiteral;
END IsBool;


PROCEDURE (lit: Literal) IsByte*(): BOOLEAN, NEW;
BEGIN
    RETURN lit IS ByteLiteral;
END IsByte;


PROCEDURE (lit: Literal) IsChar*(): BOOLEAN, NEW;
BEGIN
    RETURN lit IS CharLiteral;
END IsChar;


PROCEDURE (lit: Literal) IsInt*(): BOOLEAN, NEW;
BEGIN
    RETURN lit IS IntLiteral;
END IsInt;


PROCEDURE (lit: Literal) IsLInt*(): BOOLEAN, NEW;
BEGIN
    RETURN lit IS LIntLiteral;
END IsLInt;


PROCEDURE (lit: Literal) IsSInt*(): BOOLEAN, NEW;
BEGIN
    RETURN lit IS SIntLiteral;
END IsSInt;


PROCEDURE (lit: Literal) IsReal*(): BOOLEAN, NEW;
BEGIN
    RETURN lit IS RealLiteral;
END IsReal;


PROCEDURE (lit: Literal) IsSet*(): BOOLEAN, NEW;
BEGIN
    RETURN lit IS SetLiteral;
END IsSet;


PROCEDURE (lit: Literal) IsSReal*(): BOOLEAN, NEW;
BEGIN
    RETURN lit IS SReaLiteral;
END IsSReal;


PROCEDURE (lit: Literal) IsString*(): BOOLEAN, NEW;
BEGIN
    RETURN lit IS StrLiteral;
END IsString;


PROCEDURE (lit: BoolLiteral) GetValue*(): BOOLEAN, NEW;
BEGIN
    RETURN lit.value;
END GetValue;


PROCEDURE (lit: CharLiteral) GetValue*(): CHAR, NEW;
BEGIN
    RETURN lit.value;
END GetValue;


PROCEDURE (lit: SetLiteral) GetValue*(): SET, NEW;
BEGIN
    RETURN lit.value;
END GetValue;


PROCEDURE (lit: StrLiteral) GetValue*(): CharOpen, NEW;
BEGIN
    RETURN lit.value;
END GetValue;


PROCEDURE (lit: NumLiteral) GetValue*(): LONGINT, NEW;
VAR
    value: LONGINT;
BEGIN
    WITH lit: ByteLiteral DO
        value := lit.value;
        RETURN value;
    |    lit: IntLiteral DO
        value := lit.value;
        RETURN value;
    |    lit: LIntLiteral DO
        RETURN lit.value;
    |    lit: SIntLiteral DO
        value := lit.value;
        RETURN value;
    ELSE
        ASSERT(FALSE); RETURN 0;
    END; (* WITH *)
END GetValue;


PROCEDURE (lit: FloatLiteral) GetValue*(): REAL, NEW;
VAR
    value: REAL;
BEGIN
    WITH lit: RealLiteral DO
        value := lit.value;
        RETURN value;
    |    lit: SReaLiteral DO
        value := lit.value;
        RETURN value;
    ELSE
        ASSERT(FALSE); RETURN 0.0;
    END; (* WITH *)
END GetValue;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (const: Constant) IsChar*(): BOOLEAN, NEW;
BEGIN
    RETURN (const.value IS CharLiteral);
END IsChar;


PROCEDURE (const: Constant) IsString*(): BOOLEAN, NEW;
BEGIN
    RETURN (const.value IS StrLiteral);
END IsString;


PROCEDURE (const: Constant) GetValue*(): Literal, NEW;
BEGIN
    RETURN const.value;
END GetValue;


PROCEDURE (const: Constant) GetCharValue*(): CharOpen, NEW;
VAR
    lit: Literal;
    val: ARRAY 32 OF CHAR;
BEGIN
    lit := const.value;
    WITH
     lit: BoolLiteral DO
        IF lit.value THEN RETURN gpcpTrue ELSE RETURN gpcpFalse END;
    | lit: ByteLiteral DO
        RTS.IntToStr(lit.value,val); RETURN ST.ToChrOpen(val);
    | lit: CharLiteral DO
        val[0] := lit.value; val[1] := 0X; RETURN ST.ToChrOpen(val);
    | lit: IntLiteral DO
        RTS.IntToStr(lit.value,val); RETURN ST.ToChrOpen(val);
    | lit: LIntLiteral DO
        RTS.LongToStr(lit.value,val); RETURN ST.ToChrOpen(val);
    | lit: SIntLiteral DO
        RTS.IntToStr(lit.value,val); RETURN ST.ToChrOpen(val);
    | lit: RealLiteral DO
        RTS.RealToStr(lit.value,val); RETURN ST.ToChrOpen(val);
    | lit: SReaLiteral DO
        RTS.SRealToStr(lit.value,val); RETURN ST.ToChrOpen(val);
    | lit: StrLiteral DO
        RETURN lit.value;
    ELSE
    END; (* WITH *)
    RETURN NIL;
END GetCharValue;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (arr: ArrayType) GetDimension*(): INTEGER, NEW;
BEGIN
    RETURN arr.dim;
END GetDimension;


PROCEDURE (arr: ArrayType) SetDimension*(dim: INTEGER), NEW;
BEGIN
    arr.dim := dim;
END SetDimension;


PROCEDURE (arr: ArrayType) GetElement*(): Type, NEW;
BEGIN
    RETURN arr.elm;
END GetElement;


PROCEDURE (arr: ArrayType) GetLength*(): INTEGER, NEW;
BEGIN
    RETURN arr.len;
END GetLength;


PROCEDURE (arr: ArrayType) SetLength*(len: INTEGER), NEW;
BEGIN
    len := len;
END SetLength;


PROCEDURE (arr: ArrayType) GetName*(): CharOpen;
VAR
    idx: INTEGER;
    tname: CharOpen;
BEGIN
    RETURN(arr.name);
END GetName;


PROCEDURE (arr: ArrayType) GetLongName*(): CharOpen, NEW;
VAR
    idx: INTEGER;
    tname: CharOpen;
    modname: CharOpen;
BEGIN
    NEW(tname,2);
    idx := ST.StrChr(arr.name, '[');
    RTS.IntToStr(arr.dim, tname);
    tname := ST.StrCat(tname, ST.SubStr(arr.name, 0, idx-1));
    tname := ST.StrCat(ST.ToChrOpen(AnonArrHead), tname);
    modname := ST.StrCatChr(arr.space.asb.name, '_');
    modname := ST.StrCat(modname, arr.space.fname);
    modname := ST.StrCatChr(modname, '_');
    tname := ST.StrCat(modname, tname);
    RETURN tname;
END GetLongName;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (fld: Field) SetType*(typ: Type), NEW;
BEGIN
    fld.type := typ;
END SetType;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (mth: Method) GetFormals*(): FormalList, NEW;
BEGIN
    RETURN mth.formals;
END GetFormals;


PROCEDURE (mth: Method) GetFormal*(fname: CharOpen): Formal, NEW;
(* get the method formal that has the 'fname' *)
VAR
    i: INTEGER;
    fl: FormalList;
BEGIN
    fl := mth.formals;
    i := 0;
    WHILE (i < fl.len) & (fname^ # fl.fmls[i].name^) DO INC(i); END;
    IF i < fl.len THEN
        RETURN fl.fmls[i];
    ELSE
        RETURN NIL;
    END; (* IF *)
END GetFormal;


PROCEDURE (mth: Method) GetFormalP*(pos: INTEGER): Formal, NEW;
(* get the method formal that is in the position 'pos' *)
VAR
    fl: FormalList;
BEGIN
    fl := mth.formals;
    IF fl.len <= pos THEN RETURN NIL END;
    RETURN fl.fmls[pos];
END GetFormalP;


PROCEDURE (mth: Method) GetFullName*(): CharOpen;
VAR
    name: CharOpen;
BEGIN
    name := ST.StrCatChr(mth.class.GetFullName(),'.');
    RETURN ST.StrCat(name, mth.GetName());
END GetFullName;


PROCEDURE (mth: Method) GetName*(): CharOpen;
BEGIN
    IF MethodNameMangling = FALSE THEN
        RETURN mth.name;
    ELSE
        IF mth.isOVL THEN
            RETURN mth.oname;
        ELSE
            RETURN mth.name;
        END; (* IF *)
    END; (* IF *)
END GetName;


PROCEDURE (mth: Method) GetAssembly*(): Assembly, NEW;
BEGIN
    RETURN mth.class.space.asb;
END GetAssembly;


PROCEDURE (mth: Method) GetNamespace*(): Namespace, NEW;
BEGIN
    RETURN mth.class.space;
END GetNamespace;


PROCEDURE (mth: Method) GetDeclaringType*(): Type, NEW;
BEGIN
    RETURN mth.class;
END GetDeclaringType;


PROCEDURE (mth: Method) SetDeclaringType*(typ: Type), NEW;
BEGIN
    mth.class := typ;
END SetDeclaringType;


PROCEDURE (mth: Method) GetInvokeName*(): CharOpen, NEW;
BEGIN
    RETURN mth.iname;
END GetInvokeName;


PROCEDURE (mth: Method) SetInvokeName*(iname: CharOpen), NEW;
BEGIN
    mth.iname := iname;
END SetInvokeName;


PROCEDURE (mth: Method) IsAbstract*(): BOOLEAN, NEW;
BEGIN
    RETURN (MabsBit IN mth.attr) & ~(MempBit IN mth.attr);
END IsAbstract;


PROCEDURE (mth: Method) IsVirtual*(): BOOLEAN, NEW;
BEGIN
    RETURN mth.isVirt;
END IsVirtual;


PROCEDURE (mth: Method) IsConstructor*(): BOOLEAN, NEW;
BEGIN
    RETURN mth.isCtor;
END IsConstructor;


PROCEDURE (mth: Method) SetConstructor*(), NEW;
BEGIN
    mth.isCtor := TRUE;
END SetConstructor;


PROCEDURE (mth: Method) IsExtensible*(): BOOLEAN, NEW;
BEGIN
    RETURN (MabsBit IN mth.attr) & (MempBit IN mth.attr);
END IsExtensible;


PROCEDURE (mth: Method) IsNew*(): BOOLEAN, NEW;
BEGIN
    RETURN (MnewBit IN mth.attr);
END IsNew;


PROCEDURE (mth: Method) IsOverload*(): BOOLEAN, NEW;
BEGIN
    RETURN mth.isOVL;
END IsOverload;


PROCEDURE (mth: Method) SetOverload*(ovlname: CharOpen), NEW;
BEGIN
    mth.isOVL := TRUE;
    mth.oname := ovlname;
    IF ~mth.isCtor THEN
        mth.formals.sCode := ST.SubStr(mth.oname, LEN(mth.name), LEN(mth.oname)-1);
        mth.iname := mth.name;
    ELSE
        mth.formals.sCode := ST.SubStr(mth.oname, LEN(replCtor), LEN(mth.oname)-1);
        mth.iname := origCtor;
    END; (* IF *)
END SetOverload;


PROCEDURE (mth: Method) FixSigCode*(), NEW;
VAR
    fultyp: CharOpen;
BEGIN
    IF MethodNameMangling = FALSE THEN
        WITH mth: Function DO
            fultyp := mth.rtype.fname;
        ELSE
            fultyp := void;
        END; (* WITH *)
        mth.formals.CreateSigCode(fultyp);
    END; (* IF *)
END FixSigCode;


PROCEDURE (mth: Method) EncryptSignature*(), NEW;
VAR
    code: CharOpen;
    iname: CharOpen;  (* .NET invoke name *)
BEGIN
    code := ST.StrCat(ST.ToChrOpen("_"), mth.formals.sCode);
    mth.oname := ST.StrCat(mth.name, code);
    IF mth.isCtor THEN iname := origCtor; ELSE iname := mth.name; END;
    mth.iname := iname;
END EncryptSignature;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (fun: Function) GetReturnType*(): Type, NEW;
BEGIN
        RETURN fun.rtype;
END GetReturnType;


PROCEDURE (fun: Function) SetReturnType*(rtyp: Type), NEW;
VAR
    otype: Type;
BEGIN
    otype := fun.rtype;
    WITH otype: TempType DO
        WITH otype: NamedType DO
        ELSE
            DEC(fun.ostd);
        END; (* WITH *)
    ELSE
    END; (* WITH *)
    fun.rtype := rtyp;
END SetReturnType;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE CreateInterfaceList*(): OrderList;
VAR
    il: OrderList;
BEGIN
    NEW(il); il.object := NIL; il.list := NIL;
    RETURN il;
END CreateInterfaceList;


PROCEDURE (il: OrderList) AddInterface*(intfc: Type), NEW;
(* Add an IntfcType object to an interface list *)
VAR
    tname: CharOpen;
    inspos: OrderList;
BEGIN
    tname := intfc.GetName();
    IF ~il.ISearch(tname,inspos) THEN
        (* interface not exist *)
        inspos.Insert(intfc);
    ELSE
        (* interface already exist *)
    END; (* IF *)
END AddInterface;


PROCEDURE (typ: RecordType) AddInterface*(intfc: Type), NEW;
BEGIN
    IF typ.impl= NIL THEN
        typ.impl := CreateInterfaceList();
    END; (* IF *)
    typ.impl.AddInterface(intfc);
END AddInterface;


PROCEDURE (typ: PointerType) AddInterface*(intfc: Type), NEW;
VAR
    target: Type;
BEGIN
    target := typ.tgt;
    WITH target: RecordType DO
        target.AddInterface(intfc);
    ELSE
        ASSERT(FALSE);
    END; (* IF *)
END AddInterface;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE CreateFormalList*(len: INTEGER): FormalList;
VAR
    fl: FormalList;
BEGIN
    NEW(fl);
    fl.len := len;
    fl.ostd := 0;
    IF len > 0 THEN NEW(fl.fmls, len) END;
    RETURN fl;
END CreateFormalList;


PROCEDURE (fl: FormalList) IsNoArg*(): BOOLEAN, NEW;
BEGIN
    RETURN (fl.len = 0);
END IsNoArg;


PROCEDURE MakeFormal*(IN fmlname: CharOpen;
                      ft: Type;
                      inout: INTEGER): Formal;
VAR
    fml: Formal;
BEGIN
    NEW(fml);
    fml.name := fmlname;
    fml.inout:= inout;
    ASSERT(ft # NIL);
    fml.type := ft;
    RETURN fml;
END MakeFormal;


PROCEDURE (fl: FormalList) AddFormal*(IN fml: Formal;
                                      pos: INTEGER), NEW;
BEGIN
    fl.fmls[pos] := fml;
    fml.fmlist := fl;
END AddFormal;


PROCEDURE (fl: FormalList) InsertFormal*(IN fmlname: CharOpen;
                                         ft: Type;
                                         inout: INTEGER;
                                         pos: INTEGER;
                                         mthcls: Type): Formal, NEW;
VAR
    fml: Formal;
    lhead, inspos: OrderList;
BEGIN
    fml := MakeFormal(fmlname, ft, inout);
    fl.fmls[pos] := fml;
    fml.fmlist := fl;
    IF ft.space # mthcls.space THEN
        (* if the field type is in foreign space *)
        lhead := mthcls.space.fnslist;
        IF ~lhead.IFSearch(ft.space.fname, inspos) THEN
            inspos.Insert(ft.space);
        END; (* IF *)
    END; (* IF *)
    RETURN fml;
END InsertFormal;


PROCEDURE Signature2Code(ftype: CharOpen): CharOpen;
CONST
    MaxCode = 33;
    (* return code for CLS primitive types: *)
    SysBoolean= 'B';    SysSByte  = 'b';
    SysChar   = 'C';    SysDouble = 'R'; (* REAL *)
    SysInt16  = 'i';    SysInt32  = 'I';
    SysInt64  = 'L';    SysObject = 'O';
    SysSingle = 'f';    SysString = 'S';
    (* return code for non CLS primitive types: *)
    SysByte   = 'c';    SysUInt16 = 'x';
    SysUInt32 = 'y';    SysUInt64 = 'z';
    SysIntPtr = 'p';    SysUIntPtr = 'q';
    (* return code for other class object types: *)
    Other  = 'o';
    (* return code for Array types: *)
    Array  = 'A'; (* A? - where ? is type code of array element *)
VAR
    rtnstr: CharOpen;
    curr: INTEGER;
    idx: INTEGER;
BEGIN
    NEW(rtnstr,MaxCode); curr := 0;

    (* Add Array code if ftype is an array type *)
    idx := ST.StrRChr(ftype,'[');
    IF idx # ST.NotExist THEN
        ftype := ST.SubStr(ftype,0,idx-1);
        rtnstr[curr] := Array;
        INC(curr);
    END; (* IF *)

    rtnstr[curr] := Other; rtnstr[curr+1] := 0X;
    idx := ST.StrRChr(ftype, '.');
    ftype := ST.SubStr(ftype, idx+1, LEN(ftype));

    (* first check for CP primitive types *)
    (* ?,Boolean,?,Char,?,Double,?,Int16,?,Int32,?,Int64,?,Object,?,SByte,?,Single,?,String,? *)
    CASE ST.StrCmp(ftype, dnetInt32) OF
      ST.Equal  : rtnstr[curr] := SysInt32;
    | ST.Less   : (* ?,Boolean,?,Char,?,Double,?,Int16,? *)
        CASE ST.StrCmp(ftype, dnetChar) OF
          ST.Equal  : rtnstr[curr] := SysChar;
        | ST.Less   : (* ?,Boolean,? *)
          CASE ST.StrCmp(ftype, dnetBoolean) OF
            ST.Equal  : rtnstr[curr] := SysBoolean;
          | ST.Less   : (* ? *)
          ELSE (* ? *)
          END; (* CASE *)
        ELSE (* ?,Double,?,Int16,? *)
            CASE ST.StrCmp(ftype, dnetDouble) OF
              ST.Equal  : rtnstr[curr] := SysDouble;
            | ST.Less   : (* ? *)
            ELSE (* ?,Int16,? *)
                CASE ST.StrCmp(ftype, dnetInt16) OF
                  ST.Equal  : rtnstr[curr] := SysInt16;
                | ST.Less   : (* ? *)
                ELSE (* ? *)
                END; (* CASE *)
            END; (* CASE *)
        END; (* CASE *)
    ELSE (* ?,Int64,?,Object,?,SByte,?,Single,?,String,? *)
        CASE ST.StrCmp(ftype, dnetSByte) OF
          ST.Equal  : rtnstr[curr] := SysSByte;
        | ST.Less   : (* ?,Int64,?,Object,? *)
          CASE ST.StrCmp(ftype, dnetObject) OF
            ST.Equal  : rtnstr[curr] := SysObject;
          | ST.Less   : (* ?,Int64,? *)
              CASE ST.StrCmp(ftype, dnetInt64) OF
                ST.Equal  : rtnstr[curr] := SysInt64;
              | ST.Less   : (* ? *)
              ELSE (* ? *)
              END; (* CASE *)
          ELSE (* ? *)
          END; (* CASE *)
        ELSE (* ?,Single,?,String,? *)
            CASE ST.StrCmp(ftype, dnetString) OF
              ST.Equal  : rtnstr[curr] := SysString;
            | ST.Less   : (* ?,Single,? *)
                CASE ST.StrCmp(ftype, dnetSingle) OF
                  ST.Equal  : rtnstr[curr] := SysSingle;
                | ST.Less   : (* ? *)
                ELSE (* ? *)
                END; (* CASE *)
            ELSE (* ? *)
            END; (* CASE *)
        END; (* CASE *)
    END; (* CASE *)

    (* then check for non-CP primitive types *)
    (* ?,Byte,?,IntPtr,?,UInt16,?,UInt32,?,UInt64,?,UIntPtr,? *)
    IF rtnstr[curr] = Other THEN
        CASE ST.StrCmp(ftype, dnetUInt32) OF
          ST.Equal  : rtnstr[curr] := SysUInt32;
        | ST.Less   : (* ?,Byte,?,IntPtr,?,UInt16,? *)
            CASE ST.StrCmp(ftype, dnetIntPtr) OF
              ST.Equal  : rtnstr[curr] := SysIntPtr;
            | ST.Less   : (* ?,Byte,? *)
                CASE ST.StrCmp(ftype, dnetByte) OF
                  ST.Equal  : rtnstr[curr] := SysByte;
                | ST.Less   : (* ? *)
                ELSE (* ? *)
                END; (* CASE *)
            ELSE (* ?,UInt16,? *)
                CASE ST.StrCmp(ftype, dnetUInt16) OF
                  ST.Equal  : rtnstr[curr] := SysUInt16;
                | ST.Less   : (* ? *)
                ELSE (* ? *)
                END; (* CASE *)
            END; (* CASE *)
        ELSE (* ?,UInt64,?,UIntPtr,? *)
            CASE ST.StrCmp(ftype, dnetUInt64) OF
              ST.Equal  : rtnstr[curr] := SysUInt64;
            | ST.Less   : (* ? *)
            ELSE (* ?,UIntPtr,? *)
                CASE ST.StrCmp(ftype, dnetUIntPtr) OF
                  ST.Equal  : rtnstr[curr] := SysUIntPtr;
                | ST.Less   : (* ? *)
                ELSE (* ? *)
                END; (* CASE *)
            END; (* CASE *)
        END; (* CASE *)
    END;

    RETURN ST.ToChrOpen(rtnstr);
END Signature2Code;


PROCEDURE HashSignature(fl: FormalList; fulrtyp: CharOpen; str: CharOpen; isvoid: BOOLEAN ): CharOpen;
VAR
    pos, tot,sum, idx, len, hash: INTEGER;
    fulftyp: CharOpen;
    hs: CharOpen;
BEGIN [UNCHECKED_ARITHMETIC]
    tot := 0;
    FOR pos := 0 TO fl.len-1 DO
        fulftyp := fl.fmls[pos].type.fname;
        len := LEN(fulftyp$);
        sum := 0;
        FOR idx := 0 TO len-1 DO
            INC(sum, sum);
            IF sum < 0 THEN INC(sum) END;
            (* when pos=0, pos+0 cause a zero sum , and pos+1 cause hash
               collison when overload of "P(a:T1):T2" and "P(a:T2):T1" *)
            INC(sum, (pos+2)*ORD(fulftyp[idx]));
        END; (* FOR *)
        INC(tot,sum);
    END; (* FOR *)

    IF ~isvoid THEN
        fulftyp := fulrtyp;
        len := LEN(fulftyp$);
        sum := 0;
        FOR idx := 0 TO len-1 DO
            INC(sum, sum);
            IF sum < 0 THEN INC(sum) END;
            INC(sum, ORD(fulftyp[idx]));
        END; (* FOR *)
        INC (tot,sum);
    END; (* IF *)

    hash := tot MOD 4099;
    NEW(hs,5); RTS.IntToStr(hash, hs);
    hs := ST.StrCat(ST.ToChrOpen("_"), ST.ToChrOpen(hs));
    RETURN ST.StrCat(str,hs);
END HashSignature;


PROCEDURE (fl: FormalList) CreateSigCode*(fulrtyp: CharOpen), NEW;
CONST
    MaxCode = 33;
    (* return code for other class object types: *)
    Other  = 'o';
VAR
    code: CharOpen;
    i, len: INTEGER;
    isvoid: BOOLEAN;
BEGIN
    NEW(code, MaxCode);
    isvoid := (fulrtyp^ = Void);
    IF isvoid THEN
        RTS.IntToStr(fl.len, code);
    ELSE
        RTS.IntToStr(fl.len+1, code);
    END; (* IF *)
    code := ST.ToChrOpen(code);
    FOR i := 0 TO fl.len-1 DO
        code := ST.StrCat(code, Signature2Code(fl.fmls[i].type.fname));
    END; (* FOR *)
    IF ~isvoid THEN
        code := ST.StrCat(code, Signature2Code(fulrtyp));
    END; (* IF *)
    fl.sCode := code;
    IF ST.StrChr(code, Other) # ST.NotExist THEN
        fl.sCode := HashSignature(fl, fulrtyp, code, isvoid);
    END; (* IF *)
END CreateSigCode;


PROCEDURE (fl: FormalList) Length*(): INTEGER, NEW;
BEGIN
    RETURN fl.len;
END Length;


PROCEDURE (fl: FormalList) GetSigCode*(): CharOpen, NEW;
BEGIN
    RETURN fl.sCode;
END GetSigCode;


PROCEDURE (f: Formal) GetMethod*(): Method, NEW;
BEGIN
    RETURN f.fmlist.mth;
END GetMethod;


PROCEDURE (f: Formal) GetType*(): Type, NEW;
BEGIN
    RETURN f.type;
END GetType;


PROCEDURE (f: Formal) SetType*(typ: Type; countOutstanding: BOOLEAN), NEW;
VAR
    otype: Type;
BEGIN
    IF countOutstanding THEN
        otype := f.type;
        IF otype.IsDummyType() THEN
            IF (~typ.IsDummyType()) & (~typ.IsTempType()) THEN
                DEC(f.fmlist.ostd);
            END; (* IF *)
        ELSE
            WITH otype: TempType DO
                WITH otype: NamedType DO
                ELSE
                    IF (~typ.IsDummyType()) & (~typ.IsTempType()) THEN
                        DEC(f.fmlist.ostd);
                    END; (* IF *)
                END; (* WITH *)
            ELSE
            END; (* WITH *)
        END; (* IF *)
    END; (* IF *)
    f.type := typ;
END SetType;


PROCEDURE (f: Formal) GetParameterMode*(): INTEGER, NEW;
BEGIN
    RETURN f.inout;
END GetParameterMode;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (evt: Event) GetHandlerType*(): Type, NEW;
BEGIN
    RETURN evt.htype;
END GetHandlerType;


PROCEDURE (evt: Event) GetInvokeMethod*(): Method, NEW;
VAR
    htype: Type;
    target: Type;
    lhead, inspos: OrderList;
BEGIN
    htype := evt.htype;
    WITH htype: PointerType DO
        target := htype.tgt;
        lhead := target(PrimType).vmlist;
        IF lhead.ISearch(ST.ToChrOpen("Invoke"), inspos) THEN
            RETURN inspos.object(Method);
        ELSE
            ASSERT(FALSE); RETURN NIL;
        END; (* IF *)
    ELSE
        ASSERT(FALSE); RETURN NIL;
    END; (* WITH *)
END GetInvokeMethod;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE GetNamespaceByName*(asbname: CharOpen;
                              nsname: CharOpen): Namespace;
VAR
    asb: Assembly;
BEGIN
    asb := GetAssemblyByName(asbname);
    IF asb = NIL THEN RETURN NIL END;
    RETURN asb.GetNamespace(nsname);
END GetNamespaceByName;


PROCEDURE GetTypeByName*(asbname: CharOpen;
                         nsname: CharOpen;
                         tname: CharOpen): Type;
VAR
    ns: Namespace;
BEGIN
    ns := GetNamespaceByName(asbname, nsname);
    IF ns = NIL THEN RETURN NIL END;
    RETURN ns.GetType(tname);
END GetTypeByName;


PROCEDURE GetMethodByName*(asbname: CharOpen;
                           nsname: CharOpen;
                           tname: CharOpen;
                           mname: CharOpen): Method;
VAR
    typ: Type;
    mth: Method;
BEGIN
    typ := GetTypeByName(asbname, nsname, tname);
    IF typ = NIL THEN RETURN NIL END;
    mth := typ.GetVirtualMethod(mname);
    IF mth = NIL THEN
        RETURN typ.GetStaticMethod(mname);
    ELSE
        RETURN mth;
    END; (* IF *)
END GetMethodByName;


PROCEDURE GetFormalByName*(asbname: CharOpen;
                    nsname: CharOpen;
                    tname: CharOpen;
                    mname: CharOpen;
                    fname: CharOpen): Formal;
VAR
    mth: Method;
    fml: Formal;
BEGIN
    mth := GetMethodByName(asbname, nsname, tname, mname);
    IF mth = NIL THEN RETURN NIL END;
    fml := mth.GetFormal(fname);
    RETURN fml
END GetFormalByName;


PROCEDURE GetFormalByPosition*(asbname: CharOpen;
                    nsname: CharOpen;
                    tname: CharOpen;
                    mname: CharOpen;
                    pos: INTEGER): Formal;
VAR
    mth: Method;
BEGIN
    mth := GetMethodByName(asbname, nsname, tname, mname);
    IF mth = NIL THEN RETURN NIL END;
    RETURN mth.GetFormalP(pos);
END GetFormalByPosition;


PROCEDURE GetFunctionRtnType*(asbname: CharOpen;
                    nsname: CharOpen;
                    tname: CharOpen;
                    mname: CharOpen): Type;
VAR
    fun: Function;
    typ: Type;
BEGIN
    fun := GetMethodByName(asbname, nsname, tname, mname)(Function);
    IF fun = NIL THEN RETURN NIL END;
    typ := fun.GetReturnType();
    RETURN typ;
END GetFunctionRtnType;


(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)

PROCEDURE (arr: TempType) SetDimension*(dim: INTEGER), NEW;
BEGIN
    arr.dim := dim;
END SetDimension;


PROCEDURE (arr: TempType) SetLength*(len: INTEGER), NEW;
BEGIN
    len := len;
END SetLength;


PROCEDURE (typ: TempType) AddAnonymousArrayType*(atyp: TempType), NEW;
VAR
    temp: ArList;
BEGIN
    temp := typ.cross.arlist;
    WHILE temp.nxt # NIL DO temp := temp.nxt; END;
    NEW(temp.nxt); temp.nxt.typ := atyp; temp.nxt.nxt := NIL;
END AddAnonymousArrayType;


PROCEDURE (typ: TempType) AddDeriveRecordType*(rtyp: RecordType), NEW;
VAR
    temp: BsList;
BEGIN
    temp := typ.cross.bslist; WHILE temp.nxt # NIL DO temp := temp.nxt; END;
    NEW(temp.nxt); temp.nxt.typ := rtyp; temp.nxt.nxt := NIL;
END AddDeriveRecordType;


PROCEDURE (typ: TempType) AddImplRecordType*(rtyp: RecordType), NEW;
VAR
    temp: IpList;
BEGIN
    temp := typ.cross.iplist;
    WHILE temp.nxt # NIL DO temp := temp.nxt; END;
    NEW(temp.nxt); temp.nxt.typ := rtyp; temp.nxt.nxt := NIL;
END AddImplRecordType;


PROCEDURE (typ: TempType) AddReferenceField*(fld: Field), NEW;
VAR
    temp: FdList;
BEGIN
    temp := typ.cross.fdlist; WHILE temp.nxt # NIL DO temp := temp.nxt; END;
    NEW(temp.nxt); temp.nxt.fld := fld; temp.nxt.nxt := NIL;
END AddReferenceField;


PROCEDURE (typ: TempType) AddReferenceFormal*(fml: Formal), NEW;
VAR
    temp: FmList;
BEGIN
    temp := typ.cross.fmlist; WHILE temp.nxt # NIL DO temp := temp.nxt; END;
    NEW(temp.nxt); temp.nxt.fml := fml; temp.nxt.nxt := NIL;
END AddReferenceFormal;


PROCEDURE (typ: TempType) AddReferenceFunction*(fn: Function), NEW;
VAR
    temp: FnList;
BEGIN
    temp := typ.cross.fnlist; WHILE temp.nxt # NIL DO temp := temp.nxt; END;
    NEW(temp.nxt); temp.nxt.fn := fn; temp.nxt.nxt := NIL;
END AddReferenceFunction;


PROCEDURE (typ: TempType) AddSrcPointerType*(ptyp: PointerType), NEW;
VAR
    temp: PtList;
BEGIN
    temp := typ.cross.ptlist; WHILE temp.nxt # NIL DO temp := temp.nxt; END;
    NEW(temp.nxt); temp.nxt.typ := ptyp; temp.nxt.nxt := NIL;
END AddSrcPointerType;


PROCEDURE (typ: TempType) GetFirstPTCrossRef*(): PointerType, NEW;
BEGIN
    RETURN typ.cross.ptlist.nxt.typ;
END GetFirstPTCrossRef;


PROCEDURE (typ: TempType) GetNonAnonymousPTCrossRef*(): PointerType, NEW;
VAR
    temp: PtList;
BEGIN
    temp := typ.cross.ptlist;
    WHILE (temp.nxt # NIL) & temp.nxt.typ.anon DO
        temp := temp.nxt;
    END; (* WHILE *)
    RETURN temp.nxt.typ;
END GetNonAnonymousPTCrossRef;


PROCEDURE NewTempType*(): TempType;
VAR
    typ: TempType;
BEGIN
    NEW(typ); typ.anon := FALSE; typ.space := NIL;
    NEW(typ.cross);
    NEW(typ.cross.arlist); typ.cross.arlist.typ := NIL; typ.cross.arlist.nxt := NIL;
    NEW(typ.cross.bslist); typ.cross.bslist.typ := NIL; typ.cross.bslist.nxt := NIL;
    NEW(typ.cross.iplist); typ.cross.iplist.typ := NIL; typ.cross.iplist.nxt := NIL;
    NEW(typ.cross.fdlist); typ.cross.fdlist.fld := NIL; typ.cross.fdlist.nxt := NIL;
    NEW(typ.cross.fmlist); typ.cross.fmlist.fml := NIL; typ.cross.fmlist.nxt := NIL;
    NEW(typ.cross.fnlist); typ.cross.fnlist.fn  := NIL; typ.cross.fnlist.nxt := NIL;
    NEW(typ.cross.ptlist); typ.cross.ptlist.typ := NIL; typ.cross.ptlist.nxt := NIL;
    RETURN typ;
END NewTempType;


PROCEDURE FixReferences*(tpTemp: Type; tpDesc: Type);
(* tpTemp could be TempType or NamedType *)
VAR
    arCur: ArList;
    bsCur: BsList;
    ipCur: IpList;
    fdCur: FdList;
    fmCur: FmList;
    fnCur: FnList;
    ptCur: PtList;
    fml: Formal;
    oostd: INTEGER;
    mth: Method;
    fn: Method;
    rtype: Type;
    class: Type;
    tgt: Type;
    tname: ST.CharOpen;
    ftname: ST.CharOpen;
    ns: Namespace;
    atype: Type;
BEGIN
    (* fixing anonymous array type references *)
    arCur := tpTemp(TempType).cross.arlist;
    WHILE arCur.nxt # NIL DO
        atype := arCur.nxt.typ;
        WITH atype: TempType DO
            (* Insert the anonymous array type here *)
            ns := tpDesc.space;
            tname := ST.StrCat(tpDesc.name, anonArr);
            ftname := ST.StrCat(ST.StrCatChr(ns.GetName(), '.'),tname);
            arCur.nxt.typ := ns.InsertArray(tname, ftname, atype.dim, atype.len, tpDesc);
        ELSE
        END; (* WITH *)
        arCur := arCur.nxt;
    END; (* WHILE *) 

    (* fixing base type references *)
    bsCur := tpTemp(TempType).cross.bslist;
    WHILE bsCur.nxt # NIL DO
        bsCur.nxt.typ.SetBaseType(tpDesc);
        bsCur := bsCur.nxt;
    END; (* WHILE *) 

    (* fixing interface implementing type references *)
    ipCur := tpTemp(TempType).cross.iplist;
    WHILE ipCur.nxt # NIL DO
        ipCur.nxt.typ.AddInterface(tpDesc);
        ipCur := ipCur.nxt;
    END; (* WHILE *) 

    (* fixing field type references *)
    fdCur := tpTemp(TempType).cross.fdlist;
    WHILE fdCur.nxt # NIL DO
        fdCur.nxt.fld.SetType(tpDesc);
        fdCur := fdCur.nxt;
    END; (* WHILE *) 

    (* fixing formal type references *)
    fmCur := tpTemp(TempType).cross.fmlist;
    WHILE fmCur.nxt # NIL DO
        IF MethodNameMangling = FALSE THEN
            fml := fmCur.nxt.fml;
            oostd := fml.fmlist.ostd;
            fml.SetType(tpDesc, TRUE);
            IF (oostd > 0) & (fml.fmlist.ostd = 0) THEN
                (* there are old outstanding unsolved formal types, and now is all solved *)
                mth := fml.fmlist.mth;
                fn := mth;
                class := mth.class;
                WITH fn: Function DO
                    (* outstanding unsolved formal types have been all solved, 
                       for a function, now need to check its outstanding unsolved return type *)
                    IF fn.ostd = 0 THEN
                        rtype := fn.rtype;
                        WITH rtype: TempType DO
                        ELSE
                            fn.FixSigCode();
                            WITH class: PointerType DO
                                tgt := class.tgt;
                                WITH tgt: RecordType DO
                                    mth := tgt.AddMethod(mth);
                                ELSE
                                    ASSERT(FALSE);
                                END; (* WITH *)
                            |  class: RecordType DO
                                mth := class.AddMethod(mth);
                            ELSE
                                ASSERT(FALSE);
                            END; (* WITH *)
                        END; (* WITH *)
                    END; (* IF *)
                ELSE
                    mth.FixSigCode();
                    WITH class: PointerType DO
                        tgt := class.tgt;
                        WITH tgt: RecordType DO
                            mth := tgt.AddMethod(mth);
                        ELSE
                            ASSERT(FALSE);
                        END; (* WITH *)
                    |  class: RecordType DO
                        mth := class.AddMethod(mth);
                    ELSE
                        ASSERT(FALSE);
                    END; (* WITH *)
                END; (* WITH *)
            END; (* IF *)
        ELSE
            fmCur.nxt.fml.SetType(tpDesc, TRUE);
        END; (* IF *)
        fmCur := fmCur.nxt;
    END; (* WHILE *) 

    fnCur := tpTemp(TempType).cross.fnlist;
    WHILE fnCur.nxt # NIL DO
        IF MethodNameMangling = FALSE THEN
            mth := fnCur.nxt.fn;
            fn := mth;
            class := mth.class;
            WITH fn: Function DO
                oostd := fn.ostd;
                fn.SetReturnType(tpDesc);
                IF (oostd > 0) & (fn.ostd = 0) THEN
                    (* outstanding unsolved return type has been solved, need to check its outstanding unsolved formal types *)
                    IF fn.formals.ostd = 0 THEN
                        fn.FixSigCode();
                        WITH class: PointerType DO
                            tgt := class.tgt;
                            WITH tgt: RecordType DO
                                mth := tgt.AddMethod(mth);
                            ELSE
                                ASSERT(FALSE);
                            END; (* WITH *)
                        | class: RecordType DO
                            mth := class.AddMethod(mth);
                        ELSE
                            ASSERT(FALSE);
                        END; (* WITH *)
                    END; (* IF *)
                END; (* IF *)
            ELSE
                ASSERT(FALSE);
            END; (* WITH *)
        ELSE
            fnCur.nxt.fn.SetReturnType(tpDesc);
        END; (* IF *)
        fnCur := fnCur.nxt;
    END; (* WHILE *) 

    ptCur := tpTemp(TempType).cross.ptlist;
    WHILE ptCur.nxt # NIL DO
        ptCur.nxt.typ.SetTarget(tpDesc);
        ptCur := ptCur.nxt;
    END; (* WHILE *) 

    WITH tpDesc: NamedType DO
        (* maintain the cross references, as a NamedType could be turned to 
           PointerType, ArrayType or RecordType later *)
        tpDesc.cross := tpTemp(TempType).cross;
    ELSE
    END; (* WITH *)
END FixReferences;

(* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< *)


BEGIN
    NEW(meta);
    NEW(meta.asmList); meta.asmList.list := NIL; meta.asmList.object := NIL;
    NEW(dmyTyp); dmyTyp.Initialize(ST.ToChrOpen("DummyType"), ST.ToChrOpen("DummyType"), NIL);

    b1StrongName:= ST.ToChrOpen(B1StrongName);
    b2StrongName:= ST.ToChrOpen(B2StrongName);

    anonArr     := ST.ToChrOpen(AnonArr);
    anonRec     := ST.ToChrOpen(AnonRec);
    origCtor    := ST.ToChrOpen(OrigCtor);
    replCtor    := ST.ToChrOpen(ReplCtor);
    void        := ST.ToChrOpen(Void);
    nullspace   := ST.ToChrOpen(NULLSPACE);

    gpcpBoolean := ST.ToChrOpen(GpcpBoolean);
 (*
  * gpcpByte    := ST.ToChrOpen(GpcpByte);
  *)
    gpcpChar    := ST.ToChrOpen(GpcpChar);
    gpcpSChar   := ST.ToChrOpen(GpcpSChar);
    gpcpDouble  := ST.ToChrOpen(GpcpDouble);
    gpcpInt16   := ST.ToChrOpen(GpcpInt16);
    gpcpInt32   := ST.ToChrOpen(GpcpInt32);
    gpcpInt64   := ST.ToChrOpen(GpcpInt64);
    gpcpSByte   := ST.ToChrOpen(GpcpSByte);
    gpcpUByte   := ST.ToChrOpen(GpcpUByte);	(* new *)
    gpcpSingle  := ST.ToChrOpen(GpcpSingle);
 (* gpcpUInt16  := ST.ToChrOpen(GpcpUInt16); *)
 (* gpcpUInt32  := ST.ToChrOpen(GpcpUInt32); *)
 (* gpcpUInt64  := ST.ToChrOpen(GpcpUInt64); *)
    gpcpTrue    := ST.ToChrOpen(GpcpTrue);
    gpcpFalse   := ST.ToChrOpen(GpcpFalse);
    gpcpSet     := ST.ToChrOpen(GpcpSet);
    gpcpAnyRec  := ST.ToChrOpen(GpcpAnyRec);
    gpcpAnyPtr  := ST.ToChrOpen(GpcpAnyPtr);
    gpcpCharArr := ST.ToChrOpen(GpcpCharArr);

    dnetBoolean := ST.ToChrOpen(DNetBoolean);
    dnetByte    := ST.ToChrOpen(DNetByte);
    dnetChar    := ST.ToChrOpen(DNetChar);
    dnetDouble  := ST.ToChrOpen(DNetDouble);
    dnetInt16   := ST.ToChrOpen(DNetInt16);
    dnetInt32   := ST.ToChrOpen(DNetInt32);
    dnetInt64   := ST.ToChrOpen(DNetInt64);
    dnetIntPtr  := ST.ToChrOpen(DNetIntPtr);
    dnetObject  := ST.ToChrOpen(DNetObject);
    dnetSByte   := ST.ToChrOpen(DNetSByte);
    dnetSingle  := ST.ToChrOpen(DNetSingle);
    dnetString  := ST.ToChrOpen(DNetString);
    dnetUInt16  := ST.ToChrOpen(DNetUInt16);
    dnetUInt32  := ST.ToChrOpen(DNetUInt32);
    dnetUInt64  := ST.ToChrOpen(DNetUInt64);
    dnetUIntPtr := ST.ToChrOpen(DNetUIntPtr);
    dnetTrue    := ST.ToChrOpen(DNetTrue);
    dnetFalse   := ST.ToChrOpen(DNetFalse);

    HashSize := DefaultHashSize;

END MetaStore.
