MODULE MetaParser;
(* ========================================================================= *)
(*                                                                           *)
(*  Meta information parsing module for the .NET to Gardens Point Component  *)
(*  Pascal Symbols tool.                                                     *)
(*      Copyright (c) Siu-Yuen Chan 2001.                                    *)
(*                                                                           *)
(*  This module parses meta information inside .NET assembly files.  The     *)
(*  meta information read will then be stored into the METASTORE defined     *)
(*  by the MetaStore module.                                                 *)
(* ========================================================================= *)

IMPORT
    RTS,
    Sys := mscorlib_System,
    Ref := mscorlib_System_Reflection,
    (* ================================ *)
    ASCII,
    Error,
    GPFiles,
    GF := GPBinFiles,
    (* ================================ *)
    MS := MetaStore,
    ST := AscString;


CONST
    SystemRoot = "SystemRoot";
    ProgramFiles = "ProgramFiles";

    Beta1DefaultPath* = "\Microsoft.Net\Framework\v1.0.2204\";
    Beta2DefaultPath* = "\Microsoft.Net\Framework\v1.0.2914\";
    RC1DefaultPath*   = "\Microsoft.NET\Framework\v1.0.3328\";
    RC3DefaultPath*   = "\Microsoft.Net\Framework\v1.0.3512\";
    Full1DefaultPath* = "\Microsoft.Net\Framework\v1.0.3705\";

    Beta2InteropPath* = "\Microsoft.Net\Primary Interop Assemblies\";
    RC1InteropPath*   = "\Microsoft.Net\Primary Interop Assemblies\";
    RC3InteropPath*   = "\Microsoft.Net\Primary Interop Assemblies\";
    Full1InteropPath* = "\Microsoft.Net\Primary Interop Assemblies\";

    VerDefault* = 0;
    VerBeta1*   = 1;
    VerBeta2*   = 2;
    VerRC1*     = 3;
    VerRC3*     = 4;
    VerFull1*   = 5;

    MaxAssemblies* = 128;
    WithMember = TRUE;
    WithoutMember = FALSE;

CONST
    NewSlot = Ref.MethodAttributes.NewSlot;

TYPE
    CharOpen* = ST.CharOpen;

    AssemblyHistory = POINTER TO ARRAY OF CharOpen;

    Parser* = POINTER TO
        RECORD
            asbPath*: CharOpen;         (* "C:\WINNT\Microsoft.NET\Framework\v1.0.2914\" *)
            asbFile*: CharOpen;         (* "mscorlib" *)
            asb: Ref.Assembly;
            ao: MS.Assembly;
        END;

VAR
    newslot: INTEGER;
    warn: BOOLEAN;
    nonPub: BOOLEAN;
    CurrentPath: CharOpen;
    DefaultPath*: CharOpen;
    InteropPath*: CharOpen;
    History: AssemblyHistory;
    AsbCount: INTEGER;
    typISerializable: Sys.Type;
    typDelegate: Sys.Type;
    typMulticastDelegate: Sys.Type;
    typObject: Sys.Type;
    typComObject: Sys.Type;


PROCEDURE ^ (p: Parser) InsertMethod(IN mthi: Ref.MethodInfo; tp: MS.Type): MS.Method, NEW;
PROCEDURE ^ (p: Parser) InsertType(typ: Sys.Type; ns: MS.Namespace; withmember: BOOLEAN): MS.Type, NEW;
PROCEDURE ^ (p:Parser) PreWalkType(typ: Sys.Type), NEW;


PROCEDURE SetVersion*(version: INTEGER);
CONST
    mscorlib = "mscorlib.dll";
VAR
    prefixP: ST.CharOpen;
    prefixS: ST.CharOpen;
    corlib : ST.CharOpen;
BEGIN
    prefixS := ST.ToChrOpen(Sys.Environment.GetEnvironmentVariable(SystemRoot).ToCharArray());
    prefixP := Sys.Environment.GetEnvironmentVariable(ProgramFiles).ToCharArray();
    CASE version OF
      VerBeta1   : (* set default to Beta1 *)
        DefaultPath := ST.StrCat(prefixS, ST.ToChrOpen(Beta1DefaultPath));
        InteropPath := ST.NullString;
    | VerBeta2   : (* set default to Beta2 *)
        DefaultPath := ST.StrCat(prefixS, ST.ToChrOpen(Beta2DefaultPath));
        InteropPath := ST.StrCat(prefixP, ST.ToChrOpen(Beta2InteropPath));
    | VerRC1     : (* set default to RC1 *)
        DefaultPath := ST.StrCat(prefixS, ST.ToChrOpen(RC1DefaultPath));
        InteropPath := ST.StrCat(prefixP, ST.ToChrOpen(RC1InteropPath));
    | VerRC3     : (* set default to RC3 *)
        DefaultPath := ST.StrCat(prefixS, ST.ToChrOpen(RC3DefaultPath));
        InteropPath := ST.StrCat(prefixP, ST.ToChrOpen(RC3InteropPath));
    | VerFull1   : (* set default to Full Version 1.0 *)
        DefaultPath := ST.StrCat(prefixS, ST.ToChrOpen(Full1DefaultPath));
        InteropPath := ST.StrCat(prefixP, ST.ToChrOpen(Full1InteropPath));
    | VerDefault :
        DefaultPath := ST.StrCat(prefixS, ST.ToChrOpen(Full1DefaultPath));
        corlib := ST.StrCat(DefaultPath, ST.ToChrOpen(mscorlib));
        IF GPFiles.exists(corlib) THEN
            (* default is Full Version 1.0 *)
            InteropPath := ST.StrCat(prefixP, ST.ToChrOpen(Full1InteropPath));
        ELSE
            DefaultPath := ST.StrCat(prefixS, ST.ToChrOpen(RC3DefaultPath));
            corlib := ST.StrCat(DefaultPath, ST.ToChrOpen(mscorlib));
            IF GPFiles.exists(corlib) THEN
                (* default is RC3 *)
                InteropPath := ST.StrCat(prefixP, ST.ToChrOpen(RC3InteropPath));
            ELSE
                DefaultPath := ST.StrCat(prefixS, ST.ToChrOpen(RC1DefaultPath));
                corlib := ST.StrCat(DefaultPath, ST.ToChrOpen(mscorlib));
                IF GPFiles.exists(corlib) THEN
                    (* default is RC1 *)
                    InteropPath := ST.StrCat(prefixP, ST.ToChrOpen(RC1InteropPath));
                ELSE
                    DefaultPath := ST.StrCat(prefixS, ST.ToChrOpen(Beta2DefaultPath));
                    corlib := ST.StrCat(DefaultPath, ST.ToChrOpen(mscorlib));
                    IF GPFiles.exists(corlib) THEN
                        (* default is Beta2 *)
                        InteropPath := ST.StrCat(prefixP, ST.ToChrOpen(Beta2InteropPath));
                    ELSE
                        DefaultPath := ST.StrCat(prefixS, ST.ToChrOpen(Beta1DefaultPath));
                        corlib := ST.StrCat(DefaultPath, ST.ToChrOpen(mscorlib));
                        IF GPFiles.exists(corlib) THEN
                            (* default is Beta1 *)
                            InteropPath := ST.NullString;
                        ELSE
                            (* other .NET versions not supported by N2CPS *)
                            Error.WriteString("Can't find supported .NET version"); Error.WriteLn;
                            ASSERT(FALSE);
                        END; (* IF *)
                    END; (* IF *)
                END; (* IF *)
            END; (* IF *)
        END; (* IF *)
    ELSE
        ASSERT(FALSE);
    END; (* CASE *)
END SetVersion;


PROCEDURE SetNonPublic*();
BEGIN
    nonPub := TRUE;
END SetNonPublic;


PROCEDURE WarningOff*();
BEGIN
    warn := FALSE;
END WarningOff;


PROCEDURE WarningOn*();
BEGIN
    warn := TRUE;
END WarningOn;


PROCEDURE IsWarningOn*(): BOOLEAN;
BEGIN
    RETURN warn;
END IsWarningOn;


PROCEDURE Warning(msg: ARRAY OF CHAR);
BEGIN
    IF warn THEN
        Error.WriteString(msg); Error.WriteLn;
    END; (* IF *)
END Warning;


PROCEDURE IndentWarning(msg: ARRAY OF CHAR);
BEGIN
    IF warn THEN
        Error.Write(ASCII.HT); Error.Write(ASCII.HT); Error.WriteString(msg); Error.WriteLn;
    END; (* IF *)
END IndentWarning;

(* ========================================================================== *)

PROCEDURE Initialize();
VAR
    temp: INTEGER;
BEGIN
    temp := NewSlot; newslot := 0;
    WHILE temp > 1 DO temp := temp DIV 2; INC(newslot); END;
    CurrentPath := ST.NullString;
    NEW(History, MaxAssemblies);
    AsbCount := 0;
    nonPub := FALSE;
    typISerializable := Sys.Type.GetType("System.Runtime.Serialization.ISerializable");
    typDelegate := Sys.Type.GetType("System.Delegate");
    typMulticastDelegate := Sys.Type.GetType("System.MulticastDelegate");
    typObject := Sys.Type.GetType("System.Object");
    typComObject := Sys.Type.GetType("System.__ComObject");
END Initialize;

(* ========================================================================== *)

PROCEDURE GetFieldVisibility(fldi: Ref.FieldInfo): INTEGER;
VAR
    vis: INTEGER;
BEGIN
    IF fldi.get_IsFamily() THEN
        vis := MS.Vprotected;
    ELSIF fldi.get_IsPrivate() THEN
        vis := MS.Vprivate;
    ELSIF fldi.get_IsPublic() THEN
        vis := MS.Vpublic;
    ELSE
        vis := MS.Vinternal;
    END; (* IF *)
    RETURN vis;
END GetFieldVisibility;


PROCEDURE SetFieldVisibility(fld: MS.Field; fldi: Ref.FieldInfo);
BEGIN
    IF fldi.get_IsFamily() THEN
        fld.SetVisibility(MS.Vprotected);
    ELSIF fldi.get_IsPrivate() THEN
        fld.SetVisibility(MS.Vprivate);
    ELSIF fldi.get_IsPublic() THEN
        fld.SetVisibility(MS.Vpublic);
    ELSE
        fld.SetVisibility(MS.Vinternal);
    END; (* IF *)
END SetFieldVisibility;


PROCEDURE SetFieldAttribute(fld: MS.Field; fldi: Ref.FieldInfo);
BEGIN
    IF fldi.get_IsStatic() THEN fld.InclAttributes(MS.Fstat) END;
    IF fldi.get_IsLiteral() THEN fld.InclAttributes(MS.Fconst) END;
END SetFieldAttribute;

(* ========================================================================== *)

PROCEDURE GetMethodVisibility(mthi: Ref.MethodBase): INTEGER;
VAR
    vis: INTEGER;
BEGIN
    IF mthi.get_IsFamily() THEN
         vis := MS.Vprotected;
    ELSIF mthi.get_IsPrivate() THEN
        vis := MS.Vprivate;
    ELSIF mthi.get_IsPublic() THEN
        vis := MS.Vpublic;
    ELSE
        vis := MS.Vinternal;
    END; (* IF *)
    RETURN vis;
END GetMethodVisibility;


PROCEDURE SetMethodVisibility(mth: MS.Method; mthi: Ref.MethodBase);
BEGIN
    IF mthi.get_IsFamily() THEN
        mth.SetVisibility(MS.Vprotected);
    ELSIF mthi.get_IsPrivate() THEN
        mth.SetVisibility(MS.Vprivate);
    ELSIF mthi.get_IsPublic() THEN
        mth.SetVisibility(MS.Vpublic);
    ELSE
        mth.SetVisibility(MS.Vinternal);
    END; (* IF *)
END SetMethodVisibility;


PROCEDURE SetMethodAttribute(mth: MS.Method; mthi: Ref.MethodBase);
    PROCEDURE IsNew(m: Ref.MethodBase): BOOLEAN;
    VAR
        maset: SET;
    BEGIN
        maset := BITS(m.get_Attributes());
        RETURN (newslot IN maset);
    END IsNew;
BEGIN
    IF mth.IsVirtual() THEN
        (* virtual method *)
        IF IsNew(mthi) THEN
            mth.InclAttributes(MS.Mnew);
        END; (* IF *)
        IF mthi.get_IsHideBySig() THEN
            mth.InclAttributes(MS.MisExt);
        END; (* IF *)
        IF mthi.get_IsAbstract() THEN
            mth.ExclAttributes(MS.MisExt);
            mth.InclAttributes(MS.Mabstr);
        END; (* IF *)
        IF mthi.get_IsFinal() THEN
            mth.ExclAttributes(MS.MisExt);
            mth.InclAttributes(MS.Mfinal);
        END; (* IF *)
    ELSE
        (* instance method (or static method, which is don't care) *)
        mth.InclAttributes(MS.Mnew);

        mth.ExclAttributes(MS.MisExt);
        mth.InclAttributes(MS.Mfinal);
    END; (* IF *)
END SetMethodAttribute;

(* ========================================================================== *)

PROCEDURE IsNested(t: Sys.Type): BOOLEAN;
BEGIN
    RETURN (t.get_IsNestedFamily() OR
            t.get_IsNestedAssembly() OR
            t.get_IsNestedPrivate() OR
            t.get_IsNestedPublic() OR
            t.get_IsNestedFamORAssem() OR
            t.get_IsNestedFamANDAssem());
END IsNested;


PROCEDURE GetTypeVisibility(typ: Sys.Type): INTEGER;
VAR
    vis: INTEGER;
BEGIN
    IF typ.get_IsPublic() OR typ.get_IsNestedPublic() THEN
        vis := MS.Vpublic;
    ELSIF typ.get_IsNestedFamily() THEN
        vis := MS.Vprotected;
    ELSE
        vis := MS.Vprivate;
    END; (* IF *)
    RETURN vis;
END GetTypeVisibility;


PROCEDURE SetTypeVisibility(tp: MS.Type; typ: Sys.Type);
VAR
    dtyp: Sys.Type;
BEGIN
    IF IsNested(typ) THEN
        dtyp := typ.get_DeclaringType();
        IF typ.get_IsNestedPublic() THEN
            IF dtyp.get_IsPublic() THEN
                tp.SetVisibility(MS.Vpublic)
            ELSE
                tp.SetVisibility(MS.Vprivate)
            END; (* IF *)
        ELSIF typ.get_IsNestedFamily() THEN
            IF dtyp.get_IsPublic() THEN
                tp.SetVisibility(MS.Vprotected);
            ELSE
                tp.SetVisibility(MS.Vprivate)
            END; (* IF *)
        ELSE
            tp.SetVisibility(MS.Vprivate)
        END; (* IF *)
    ELSE
        IF typ.get_IsPublic() THEN
            tp.SetVisibility(MS.Vpublic)
        ELSE
            tp.SetVisibility(MS.Vprivate)
        END; (* IF *)
    END; (* IF *)
END SetTypeVisibility;


PROCEDURE SetTypeAttribute(tp: MS.Type; typ: Sys.Type);
VAR
    tgt: MS.Type;
BEGIN
    IF tp IS MS.PointerType THEN tgt := tp(MS.PointerType).GetTarget(); ELSE tgt := NIL; END;
    IF typ.get_IsAbstract() THEN
        tp.InclAttributes(MS.Rabstr);
        IF tgt # NIL THEN tgt.InclAttributes(MS.Rabstr); END;
    ELSIF ~typ.get_IsSealed() THEN
        tp.InclAttributes(MS.Rextns);
        IF tgt # NIL THEN tgt.InclAttributes(MS.Rextns); END;
    END; (* IF *)
(*
    IF typ.get_IsValueType() OR typ.get_IsUnmanagedValueType() THEN
        tp.InclAttributes(MS.RvalTp);
    ELSIF typ.get_IsInterface() THEN
        IF tgt # NIL THEN tgt.InclAttributes(MS.RiFace); END;
    END; (* IF *)
*)
    IF typ.get_IsClass() THEN
        (* do nothing here *)
    ELSIF typ.get_IsInterface() THEN
        IF tgt # NIL THEN tgt.InclAttributes(MS.RiFace); END;
    ELSE
        (* all sort of ValueTypes here *)
        tp.InclAttributes(MS.RvalTp);
    END; (* IF *)
END SetTypeAttribute;

(* ========================================================================== *)

PROCEDURE IsDelegate(typ: Sys.Type): BOOLEAN;
(* Pre: typ IsClass and typ inherits System.Object *)
VAR t: Sys.Type;
BEGIN
    t := typ;
    WHILE ~(t.Equals(typObject)) DO
        IF t.Equals(typDelegate) THEN
            (* we are not interesting in "System.Delegate" or "System.MulticastDelegate",
               but their subclasses *)
            RETURN ~(typ.Equals(typDelegate) OR typ.Equals(typMulticastDelegate));
        END; (* IF *)
        t := t.get_BaseType();
    END; (* WHILE *)
    RETURN FALSE;
END IsDelegate;


PROCEDURE IsMulticastDelegate(typ: Sys.Type): BOOLEAN;
(* Pre: typ inherits System.Delegate *)
VAR t: Sys.Type;
BEGIN
    t := typ;
    WHILE ~(t.Equals(typDelegate)) DO
        IF t.Equals(typMulticastDelegate) THEN
            (* we are not interesting in "System.MulticastDelegate", but its subclasses *)
            RETURN ~typ.Equals(typMulticastDelegate);
        END; (* IF *)
        t := t.get_BaseType();
    END; (* WHILE *)
    RETURN FALSE;
END IsMulticastDelegate;

(* ========================================================================== *)

PROCEDURE TypeConvert(t: Sys.Type): INTEGER;
(* If t is:
 *    Enum      , return code 4;
 *    Structure , return code 8;
 *    Primitive , return code 16;
 *    Interface , return code 32;
 *    Class     , return code 64;
 *    Delegate  , return code 128;
 *    Array     , return code + 256;
 *    Nested    , return code + 512;
 *)
VAR
    tt: INTEGER;
    sysarr: Sys.Type;
BEGIN
    IF t = NIL THEN RETURN MS.Unknown END;
    tt := 0;

    IF IsNested(t) THEN tt := tt + MS.Nested; END;

    IF t.get_IsArray() THEN
        sysarr := Sys.Type.GetType("System.Array");
        IF t # sysarr THEN
            tt := tt + MS.Array;
            t := t.GetElementType();
            WHILE t.get_IsArray() DO t := t.GetElementType() END;
        END; (* IF *)
    END; (* IF *)

    IF t.get_IsClass() THEN
        IF IsDelegate(t) THEN RETURN tt + MS.Delegate; END;
        RETURN tt + MS.Class;
    ELSIF t.get_IsInterface() THEN
        RETURN tt + MS.Interface;
    ELSE
        IF t.get_IsEnum() THEN
            RETURN tt + MS.Enum;
        ELSIF t.get_IsPrimitive() THEN
            RETURN tt + MS.Primitive;
        ELSE                    (* Struct Type *)
            RETURN tt + MS.Struct;
        END; (* IF *)
    END; (* IF *)
END TypeConvert;

(* ========================================================================== *)

PROCEDURE GetAssemblyRealName*(asbname: CharOpen): CharOpen;
VAR
    name: CharOpen;
    fullname: CharOpen;
    asb: Ref.Assembly;
    realname: CharOpen;
    idx: INTEGER;
    len: INTEGER;
BEGIN (* GetAssemblyRealName *)
    idx := ST.StrChr(asbname,',');
    IF idx # ST.NotExist THEN
        name := ST.SubStr(asbname, 0, idx-1);
    ELSE
        name := asbname;
    END; (* IF *)

    len := LEN(name);
    IF name[len-2] = '_' THEN name := ST.SubStr(name, 0, len-3); END;

    (* try <asbname.dll> at current path *)
    fullname := ST.StrCat(CurrentPath, name);
    fullname := ST.StrCat(fullname, ST.ToChrOpen(MS.AssemblyExt));
    IF GPFiles.exists(fullname) THEN
        asb := Ref.Assembly.LoadFrom(Sys.String.init(ST.TrimNull(fullname)));
    ELSE
        (* try <asbname.dll> at Default path *)
        fullname := ST.StrCat(DefaultPath, name);
        fullname := ST.StrCat(fullname, ST.ToChrOpen(MS.AssemblyExt));
        IF GPFiles.exists(fullname) THEN
            asb := Ref.Assembly.LoadFrom(Sys.String.init(ST.TrimNull(fullname)));
        ELSE
            (* try <asbname.dll> at Interop path *)
            fullname := ST.StrCat(InteropPath, name);
            fullname := ST.StrCat(fullname, ST.ToChrOpen(MS.AssemblyExt));
            IF GPFiles.exists(fullname) THEN
                asb := Ref.Assembly.LoadFrom(Sys.String.init(ST.TrimNull(fullname)));
            ELSE
                (* try <asbname.mcl> at current path *)
                fullname := ST.StrCat(CurrentPath, name);
                fullname := ST.StrCat(fullname, ST.ToChrOpen(MS.TypeLibExt));
                IF GPFiles.exists(fullname) THEN
                    asb := Ref.Assembly.LoadFrom(Sys.String.init(ST.TrimNull(fullname)));
                ELSE
                    (* try <asbname.mcl> at Default path *)
                    fullname := ST.StrCat(DefaultPath, name);
                    fullname := ST.StrCat(fullname, ST.ToChrOpen(MS.TypeLibExt));
                    IF GPFiles.exists(fullname) THEN
                        asb := Ref.Assembly.LoadFrom(Sys.String.init(ST.TrimNull(fullname)));
                    ELSE
                        (* Should give warning message in here or not? *)
                        asb := NIL;
                    END; (* IF *)
                END; (* IF *)
            END; (* IF *)
        END; (* IF *)
    END; (* IF *)

    IF asb # NIL THEN
        realname := ST.ToChrOpen(asb.get_FullName().ToCharArray());
        idx := ST.StrChr(realname,',');
        realname := ST.SubStr(realname,0,idx-1);
    ELSE
        (* can't find the assembly, assume the input asbname is the real name *)
        realname := name;
    END; (* IF *)
    RETURN realname;
END GetAssemblyRealName;


PROCEDURE (p: Parser) GetFullQualifiedName(VAR t: Sys.Type): CharOpen, NEW;
(* Returns the qualified full type name, including the Assembly name. 
 * It is in the form of:
 * e.g. "System.Int32, mscorlib.dll"
 *)
VAR
    qftname: CharOpen;
    ftname: CharOpen;
    asbname: CharOpen;
    idx: INTEGER;
    elmtyp: Sys.Type;
    elmname: CharOpen;
BEGIN
    IF t = NIL THEN RETURN ST.NullString; END;
    IF t.get_IsPointer() THEN
        (* put the '*' into the AssemblyQualifiedName *)
        qftname := ST.ToChrOpen(t.GetElementType().get_AssemblyQualifiedName().ToCharArray());
        idx := ST.StrChr(qftname, ' ');
        asbname := ST.SubStr(qftname, idx+1, LEN(qftname)-1);
        asbname := GetAssemblyRealName(asbname);
        ftname := ST.StrCat(ST.ToChrOpen(t.ToString().ToCharArray()), ST.ToChrOpen(", "));
        qftname := ST.StrCat(ftname, asbname);
    ELSIF t.get_IsByRef() THEN
        qftname := ST.ToChrOpen(t.GetElementType().get_AssemblyQualifiedName().ToCharArray());

        idx := ST.StrChr(qftname, ' ');
        ftname := ST.SubStr(qftname, 0, idx-1);   (* include the coma *)
        asbname := ST.SubStr(qftname, idx+1, LEN(qftname)-1);
        asbname := GetAssemblyRealName(asbname);
        ftname := ST.StrCatChr(ftname, ' ');
        qftname := ST.StrCat(ftname, asbname);

    ELSIF t.get_IsArray() THEN
        elmtyp := t.GetElementType();
        WHILE (elmtyp # NIL) & (elmtyp.get_IsArray()) DO elmtyp := elmtyp.GetElementType() END;

        IF elmtyp # NIL THEN
            (* really array type *)
            elmname := ST.ToChrOpen(elmtyp.get_AssemblyQualifiedName().ToCharArray());
            idx := ST.StrChr(elmname, ' ');
            asbname := ST.SubStr(elmname, idx+1, LEN(elmname)-1);
            asbname := GetAssemblyRealName(asbname);

            qftname := ST.ToChrOpen(t.get_AssemblyQualifiedName().ToCharArray());
            idx := ST.StrChr(qftname, ' ');
            ftname := ST.SubStr(qftname, 0, idx);
            qftname := ST.StrCat(ftname, asbname);
        ELSE
            (* not really Array type, but [mscorlib]System.Array *)
            qftname := ST.ToChrOpen(t.get_AssemblyQualifiedName().ToCharArray());

            idx := ST.StrChr(qftname, ' ');
            ftname := ST.SubStr(qftname, 0, idx-1);   (* include the coma *)
            asbname := ST.SubStr(qftname, idx+1, LEN(qftname)-1);
            asbname := GetAssemblyRealName(asbname);
            ftname := ST.StrCatChr(ftname, ' ');
            qftname := ST.StrCat(ftname, asbname);
        END; (* IF *)
    ELSE
        qftname := ST.ToChrOpen(t.get_AssemblyQualifiedName().ToCharArray());

        idx := ST.StrChr(qftname, ' ');
        ftname := ST.SubStr(qftname, 0, idx-1);   (* include the coma *)
        asbname := ST.SubStr(qftname, idx+1, LEN(qftname)-1);
        asbname := GetAssemblyRealName(asbname);
        ftname := ST.StrCatChr(ftname, ' ');
        qftname := ST.StrCat(ftname, asbname);
    END; (* IF *)
    RETURN qftname;
END GetFullQualifiedName;

(* ========================================================================== *)

PROCEDURE (p: Parser) InsertTypeSpace(qualtyp: CharOpen): MS.Namespace, NEW;
(* qualified full type name can be either in the form of:
 * (1) stdole.IPictureDisp, stdole, Ver=2.0.0.0, Loc="", SN=03689116d3a4ae33
 * or in the form of:
 * (2) System.Void, mscorlib
 *)
VAR
    idx: INTEGER;
    asbfile: CharOpen;    (* assembly file name, e.g. stdole *)
    fulasb: CharOpen;     (* full assembly name, e.g. stdole, Ver=2.0.0.0, Loc="", SN=03689116da4ae33 *)
    fultyp: CharOpen;     (* full type name,     e.g. stdole.IPictureDisp *)
    nsname: CharOpen;     (* namespace name,     e.g. stdole *)
    ao: MS.Assembly;
    ns: MS.Namespace;
BEGIN
    MS.SplitTypeName(qualtyp, fulasb, asbfile, fultyp);
    idx := ST.StrRChr(fultyp,'.');
    nsname := ST.SubStr(qualtyp, 0, idx-1);

    IF p.asbFile^ # asbfile^ THEN
        ao := MS.InsertAssembly(fulasb, asbfile);
        ns := ao.InsertNamespace(nsname);
    ELSE
        ns := p.ao.InsertNamespace(nsname);
    END; (* IF *)
    RETURN ns;
END InsertTypeSpace;

(* ========================================================================== *)

PROCEDURE (p: Parser) FieldWarning(fldi: Ref.FieldInfo; msg: ARRAY OF CHAR), NEW;
BEGIN
    Warning("!!! WARNING !!! - " + msg);
    IndentWarning("Assembly: " + p.asbFile^); 
    IndentWarning("Class : " + ST.ToChrOpen(fldi.get_ReflectedType().get_FullName().ToCharArray())^);
    IndentWarning("Field: " + ST.ToChrOpen(fldi.get_Name().ToCharArray())^);
END FieldWarning;


PROCEDURE (p: Parser) SkipFieldWarning(fldi: Ref.FieldInfo; msg: ARRAY OF CHAR), NEW;
BEGIN
    p.FieldWarning(fldi, msg);
    IndentWarning("field skipped");
END SkipFieldWarning;


PROCEDURE (p: Parser) SkipFieldTypeWarning(fldi: Ref.FieldInfo; oftyp: Sys.Type; msg: ARRAY OF CHAR), NEW;
BEGIN
    p.FieldWarning(fldi, msg);
    IndentWarning("Type  : " + ST.ToChrOpen(oftyp.get_FullName().ToCharArray())^);
    IndentWarning("field skipped");
END SkipFieldTypeWarning;


PROCEDURE (p: Parser) PreWalkField(typ: Sys.Type; IN fldi: Ref.FieldInfo), NEW;
VAR
    fldname: CharOpen;
    ftyp: Sys.Type;
    qualftyp: CharOpen;
    fvalue: CharOpen;
    fv: MS.Literal;
    idx: INTEGER;
    utname: CharOpen;
    isPtr: BOOLEAN;
    ns: MS.Namespace;
BEGIN (* MetaParser_Parser_PreWalkField *)
    IF fldi.get_IsPrivate() THEN RETURN END;
    fv := NIL;
    fvalue := ST.NullString;
    utname := ST.NullString;
    ftyp := fldi.get_FieldType();
    IF ftyp = NIL THEN
        p.SkipFieldWarning(fldi, "Field type missing");
        RETURN;
    END; (* IF *)

    IF ftyp.get_IsPointer() THEN ftyp := ftyp.GetElementType(); END;
    qualftyp := p.GetFullQualifiedName(ftyp);
    ns := p.InsertTypeSpace(qualftyp);
END PreWalkField;

(* ========================================================================== *)

    PROCEDURE TryCatchFieldValue(fldi: Ref.FieldInfo; typ: Sys.Type): Sys.Object;
    VAR fval: Sys.Object;
    BEGIN
        fval := fldi.GetValue(typ);
        RETURN fval;

        RESCUE(x);
        Error.WriteString(RTS.getStr(x)); Error.WriteLn;
        RETURN NIL;
    END TryCatchFieldValue;


PROCEDURE (p: Parser) InsertField(typ: Sys.Type; IN fldi: Ref.FieldInfo; tp: MS.Type), NEW;
VAR
    fldname: CharOpen;     (* field name *)
    ftyp: Sys.Type;        (* field type *)
    qualftyp: CharOpen;    (* qualified field type name *)
    fvalue: CharOpen;      (* field value if constant*)
    fv: MS.Literal;        (* MetaStore field value *)
    utname: CharOpen;      (* underlying type name if Enum type *)
    fld: MS.Field;         (* MetaStore field descriptor *)
    isPtr: BOOLEAN;        (* whether field is a pointer to value type *)
    static: BOOLEAN;       (* whether field is static *)
    idx: INTEGER;
    ftns: MS.Namespace;    (* namespace of field type *)
    ft: MS.Type;           (* MetaStore field type *)
    tempt: MS.Type;
    oftyp: Sys.Type;
    skip: BOOLEAN;
    fval: Sys.Object;
    visField: INTEGER;
    vis: INTEGER;
BEGIN (* MetaParser_Parser_InsertField *)
    IF fldi.get_IsPrivate() THEN RETURN END;
    fvalue := ST.Null32String;
    fv := NIL;
    utname := ST.NullString;
    skip := FALSE;
    visField := GetFieldVisibility(fldi);
    static := fldi.get_IsStatic();
    fldname := ST.ToChrOpen(fldi.get_Name().ToCharArray());
    ftyp := fldi.get_FieldType();
    IF ftyp = NIL THEN
        p.SkipFieldWarning(fldi, "Field type missing");
        RETURN;
    END; (* IF *)

    IF ftyp.get_IsPointer() THEN
        isPtr := TRUE;
        oftyp := ftyp;
        ftyp := ftyp.GetElementType();

        IF ftyp.get_IsPointer() THEN
            p.SkipFieldTypeWarning(fldi, oftyp, "Assembly contains field type of POINTER TO POINTER");
            RETURN;
        ELSIF ftyp.get_IsArray() THEN
            p.SkipFieldTypeWarning(fldi, oftyp, "Assembly contains field type of POINTER TO POINTER to ARRAY");
            RETURN;
        ELSIF ftyp.get_IsInterface() THEN
            p.SkipFieldTypeWarning(fldi, oftyp, "Assembly contains field type of POINTER TO POINTER to INTERFACE");
            RETURN;
        ELSIF ftyp.get_IsClass() THEN
            p.SkipFieldTypeWarning(fldi, oftyp, "Assembly contains field type of POINTER TO POINTER to RECORD");
            RETURN;
        ELSIF ftyp.get_IsEnum() THEN
            p.SkipFieldTypeWarning(fldi, oftyp, "Assembly contains field type of POINTER TO enumeration");
            RETURN;
        ELSIF ftyp.get_IsPrimitive() THEN
            p.SkipFieldTypeWarning(fldi, oftyp, "Assembly contains field type of POINTER TO primitive");
            RETURN;
        END; (* IF *)

        vis := GetTypeVisibility(ftyp);
        CASE visField OF
          MS.Vpublic:     IF vis = MS.Vprivate THEN
                              p.SkipFieldTypeWarning(fldi, oftyp, "Public field has a Private field type");
                              RETURN;
                          END; (* IF *)
        | MS.Vprotected:  IF vis = MS.Vprivate THEN
                              p.SkipFieldTypeWarning(fldi, oftyp, "Protected field has a Private field type");
                              RETURN;
                          END; (* IF *)
        ELSE
        END; (* CASE *)
    ELSE
        isPtr := FALSE;
        IF fldi.get_IsLiteral() THEN
            fval := TryCatchFieldValue(fldi, typ);
            IF fval = NIL THEN
                p.SkipFieldWarning(fldi, "Failed to get value of a static literal field");
                RETURN;
            END; (* IF *)

            RTS.ObjToStr(fval, fvalue);
            IF LEN(fvalue) > 0 THEN
                fvalue := ST.ToChrOpen(fvalue);
            ELSE
                fvalue := ST.NullString;
            END; (* IF *)
        ELSE
            vis := GetTypeVisibility(ftyp);
            CASE visField OF
              MS.Vpublic:     IF vis = MS.Vprivate THEN
                                  p.SkipFieldTypeWarning(fldi, ftyp, "Public field has a Private field type");
                                  RETURN;
                              END; (* IF *)
            | MS.Vprotected:  IF vis = MS.Vprivate THEN
                                  p.SkipFieldTypeWarning(fldi, ftyp, "Protected field has a Private field type");
                                  RETURN;
                              END; (* IF *)
            ELSE
            END; (* CASE *)
        END; (* IF *)
    END; (* IF *)

    qualftyp := p.GetFullQualifiedName(ftyp);
    ftns := p.InsertTypeSpace(qualftyp);
    ft := p.InsertType(ftyp, ftns, WithoutMember);

    IF tp IS MS.EnumType THEN    (* the type holds the field is an enumeration type *)
        IF fldi.get_IsSpecialName() THEN    (* value__ *)
            (* do we need to do anything with this special value__ field *)
        ELSE
            utname := ST.ToChrOpen(Sys.Enum.GetUnderlyingType(typ).ToString().ToCharArray());
            idx := ST.StrRChr(utname, '.');
            IF idx # ST.NotExist THEN utname := ST.SubStr(utname, idx+1, LEN(utname)); END;
            fv := MS.CreateLiteral(fvalue, utname);
            fld := tp.InsertField(fldname, ft, fv, static, isPtr);
            IF fld # NIL THEN
                SetFieldVisibility(fld, fldi); SetFieldAttribute(fld, fldi);
            END; (* IF *)
        END; (* IF *)
    ELSE
        IF (ftyp.get_IsEnum() & fldi.get_IsStatic()) THEN    (* field type is enumeration *)
            utname := ST.ToChrOpen(Sys.Enum.GetUnderlyingType(ftyp).ToString().ToCharArray());
            idx := ST.StrRChr(utname, '.');
            IF idx # ST.NotExist THEN utname := ST.SubStr(utname, idx+1, LEN(utname)); END;
            fv := MS.CreateLiteral(fvalue, utname);
        ELSIF fldi.get_IsLiteral() THEN    (* field type is literal *)
            utname := ST.ToChrOpen(ftyp.ToString().ToCharArray());
            idx := ST.StrRChr(utname,'.');
            IF idx # ST.NotExist THEN utname := ST.SubStr(utname, idx+1, LEN(utname)); END;
            fv := MS.CreateLiteral(fvalue, utname);
        ELSE
            IF isPtr THEN
                (* create a dummy pointer to the value type for exporting *)
                ft := ftns.InsertValuePointer(ft);
            END; (* IF *)
        END; (* IF *)
        fld := tp.InsertField(fldname, ft, fv, static, isPtr);
        IF fld # NIL THEN
            SetFieldVisibility(fld, fldi); SetFieldAttribute(fld, fldi);
        END; (* IF *)
    END; (* IF *)
END InsertField;

(* ========================================================================== *)

PROCEDURE LocateInterfaceMethod(mi: Ref.MethodBase; t: Sys.Type): Ref.MethodInfo;
VAR
    bf: Ref.BindingFlags;
    memarr: Ref.Arr1MemberInfo;
    mem: Ref.MemberInfo;
    i: INTEGER;
    mt: Ref.MemberTypes;
    str,str2: CharOpen;
BEGIN
    bf := Ref.BindingFlags.Static + 
          Ref.BindingFlags.Instance +
          Ref.BindingFlags.Public + 
          Ref.BindingFlags.NonPublic +          (* include private and family *)
          Ref.BindingFlags.DeclaredOnly;
    memarr := t.GetMembers(bf);
    FOR i := 0 TO LEN(memarr)-1 DO
        mem := memarr[i];
        mt := mem.get_MemberType();
        IF mt = Ref.MemberTypes.Method THEN
            str := ST.ToChrOpen(t.get_FullName().ToCharArray());
            str := ST.StrCatChr(str, '.');
            str := ST.StrCat(str, ST.ToChrOpen(mem.get_Name().ToCharArray()));
            str2 := ST.ToChrOpen(mi.get_Name().ToCharArray());
            IF str^ =  str2^ THEN
                RETURN mem(Ref.MethodInfo);
            END; (* IF *)
        END; (* IF *)
    END; (* FOR *)
    RETURN NIL;
END LocateInterfaceMethod;

(* ========================================================================== *)

PROCEDURE GetParameterFromInterface(mi: Ref.MethodBase): Ref.Arr1ParameterInfo;
VAR
    t: Sys.Type;
    itarr: Sys.Arr1Type;
    mth: Ref.MethodInfo;
    it: Sys.Type;
    i: INTEGER;
BEGIN
    t := mi.get_ReflectedType();
    itarr := t.GetInterfaces();
    FOR i := 0 TO LEN(itarr)-1 DO
        it := itarr[i];
        mth := LocateInterfaceMethod(mi, it);
        IF mth # NIL THEN
            RETURN mth.GetParameters();
        ELSE
        END; (* IF *)
    END; (* FOR *)
    RETURN NIL;
END GetParameterFromInterface;

(* ========================================================================== *)

PROCEDURE (p: Parser) FormalWarning(mb: Ref.MethodBase; fmlname: ARRAY OF CHAR; msg: ARRAY OF CHAR), NEW;
BEGIN
    Warning("!!! Warning !!! - " + msg);
    IndentWarning("Assembly: " + p.asbFile^);
    IndentWarning("Class : " + ST.ToChrOpen(mb.get_ReflectedType().get_FullName().ToCharArray())^);
    IndentWarning("Method: " + ST.ToChrOpen(mb.get_Name().ToCharArray())^);
    IndentWarning("Formal named: " + fmlname);
END FormalWarning;


PROCEDURE (p: Parser) FormalTypeWarning(mb: Ref.MethodBase; oftyp: Sys.Type; fmlname: ARRAY OF CHAR; msg: ARRAY OF CHAR), NEW;
BEGIN
    p.FormalWarning(mb, fmlname, msg);
    IndentWarning("Formal type name: " + ST.ToChrOpen(oftyp.get_FullName().ToCharArray())^);
END FormalTypeWarning;


PROCEDURE (p: Parser) SkipFormalWarning(mb: Ref.MethodBase; fmlname: ARRAY OF CHAR; msg: ARRAY OF CHAR), NEW;
BEGIN
    p.FormalWarning(mb, fmlname, msg);
    IndentWarning("Method skipped");
END SkipFormalWarning;


PROCEDURE (p: Parser) SkipFormalTypeWarning(mb: Ref.MethodBase; oftyp: Sys.Type; fmlname: ARRAY OF CHAR; msg: ARRAY OF CHAR), NEW;
BEGIN
    p.FormalTypeWarning(mb, oftyp, fmlname, msg);
    IndentWarning("Method skipped");
END SkipFormalTypeWarning;


PROCEDURE (p: Parser) PreWalkFormals(IN mb: Ref.MethodBase), NEW;
VAR
    pari: Ref.ParameterInfo;
    piarr: Ref.Arr1ParameterInfo;
    i, len: INTEGER;
    qualftyp: CharOpen;
    ftyp: Sys.Type;
    ns: MS.Namespace;
BEGIN
    piarr := mb.GetParameters();
    len := LEN(piarr);

    FOR i := 0 TO len-1 DO
        pari := piarr[i];

        ftyp := pari.get_ParameterType();
        IF ftyp # NIL THEN
            IF ftyp.get_IsByRef() OR ftyp.get_IsPointer() THEN
                (* IsByRef '&', IsPointer '*' *)
                ftyp := ftyp.GetElementType();
            END; (* IF *)

            qualftyp := p.GetFullQualifiedName(ftyp);
            ns := p.InsertTypeSpace(qualftyp);
        ELSE
            Warning("!!!! Warning !!!! a null formal type detected");
            IndentWarning("Assembly: " + p.asbFile^);
            IndentWarning("Class : " + ST.ToChrOpen(mb.get_ReflectedType().get_FullName().ToCharArray())^);
            IndentWarning("Method: " + ST.ToChrOpen(mb.get_Name().ToCharArray())^);
        END; (* IF *)
    END; (* FOR *)
END PreWalkFormals;

(* ========================================================================== *)

PROCEDURE (p: Parser) InsertFormals(IN mthcls: MS.Type; IN mb: Ref.MethodBase; visMethod: INTEGER): MS.FormalList, NEW;
VAR
    pari: Ref.ParameterInfo;
    piarr: Ref.Arr1ParameterInfo;
    ipiarr: Ref.Arr1ParameterInfo;
    i, len: INTEGER;
    fl: MS.FormalList;
    fmlname: CharOpen;
    qualftyp: CharOpen;
    utname: CharOpen;
    ftyp: Sys.Type;
    oftyp: Sys.Type;
    pos: INTEGER;
    ppass: INTEGER;
    str: Sys.String;
    fml: MS.Formal;
    ft: MS.Type;
    tempt: MS.Type;
    ftns: MS.Namespace;
    argno: ARRAY 10 OF CHAR;
    isPtr: BOOLEAN;
    skip: BOOLEAN;
    vis: INTEGER;
BEGIN
    piarr := mb.GetParameters();
    len := LEN(piarr);

    fl := MS.CreateFormalList(len);
    skip := FALSE;
    FOR i := 0 TO len-1 DO
        pari := piarr[i];
        ppass := MS.IsVal;
        isPtr := FALSE;
        utname := ST.NullString;

        IF pari.get_IsIn() THEN
            ppass := MS.IsIn;
        ELSIF pari.get_IsOut() THEN
            ppass := MS.IsOut;
        ELSIF pari.get_IsRetval() THEN
        END; (* IF *)

        str := pari.get_Name();
        IF str # NIL THEN
            fmlname := str.ToCharArray();
            IF LEN(fmlname) = 0 THEN
                ipiarr := GetParameterFromInterface(mb);
                IF ipiarr # NIL THEN
                    pari := ipiarr[i];
                    fmlname := ST.ToChrOpen(pari.get_Name().ToCharArray());
                ELSE
                    RTS.IntToStr(i+1, argno);
                    fmlname := ST.StrCat(ST.ToChrOpen("A_"), ST.ToChrOpen(argno));
                    p.FormalWarning(mb, fmlname, "Formal has zero length parameter name, name assigned");
                END; (* IF *)
            ELSE
                fmlname := ST.ToChrOpen(fmlname);
            END; (* IF *)
        ELSE
            ipiarr := GetParameterFromInterface(mb);
            IF ipiarr # NIL THEN
                pari := ipiarr[i];
                fmlname := ST.ToChrOpen(pari.get_Name().ToCharArray());
            ELSE
                RTS.IntToStr(i+1, argno);
                fmlname := ST.StrCat(ST.ToChrOpen("A_"), ST.ToChrOpen(argno));
                p.FormalWarning(mb, fmlname, "Formal has NIL parameter name, name assigned");
            END; (* IF *)
        END; (* IF *)

        pos := pari.get_Position();
        ftyp := pari.get_ParameterType();

        IF ftyp # NIL THEN
            IF ftyp.get_IsByRef() OR ftyp.get_IsPointer() THEN
                (* IsByRef '&', IsPointer '*' *)
                ppass := MS.IsVar;
                oftyp := ftyp;
                ftyp := ftyp.GetElementType();

                (* GPCP cannot handle VAR pointer to primitive type *)
                IF ftyp.get_IsPointer() THEN
                    ftyp := ftyp.GetElementType();
                    IF ftyp.get_IsPointer() THEN
                        p.SkipFormalTypeWarning(mb, oftyp, fmlname, "Assembly contains method of formal type POINTER TO POINTER");
                        RETURN NIL;
                    ELSIF ftyp.get_IsArray() THEN
                        p.SkipFormalTypeWarning(mb, oftyp, fmlname, "Assembly contains method of formal type POINTER TO POINTER to ARRAY");
                        RETURN NIL;
                    ELSIF ftyp.get_IsInterface() THEN
                        p.SkipFormalTypeWarning(mb, oftyp, fmlname, "Assembly contains method of formal type POINTER TO POINTER to INTERFACE");
                        RETURN NIL;
                    ELSIF ftyp.get_IsClass() THEN
                        p.SkipFormalTypeWarning(mb, oftyp, fmlname, "Assembly contains method of formal type POINTER TO POINTER to RECORD");
                        RETURN NIL;
                    ELSIF ftyp.get_IsEnum() THEN
                        p.SkipFormalTypeWarning(mb, oftyp, fmlname, "Assembly contains method of formal type POINTER TO enumeration");
                        RETURN NIL;
                    ELSIF ftyp.get_IsPrimitive() THEN
                        p.SkipFormalTypeWarning(mb, oftyp, fmlname, "Assembly contains method of formal type POINTER TO primitive");
                        RETURN NIL;
                    END; (* IF *)
                    isPtr := TRUE;
                END; (* IF *)
            ELSIF ftyp.get_IsPrimitive() THEN
                (* primitive type cannot be passed as IN in GPCP *)
                IF ppass = MS.IsIn THEN ppass := MS.IsVal; END;
            ELSE
            END; (* IF *)

            vis := GetTypeVisibility(ftyp);
            CASE visMethod OF
              MS.Vpublic:     IF vis = MS.Vprivate THEN
                                  p.SkipFormalTypeWarning(mb, ftyp, fmlname, "Public method has a formal with Private type");
                                  RETURN NIL;
                              END; (* IF *)
            | MS.Vprotected:  IF vis = MS.Vprivate THEN
                                  p.SkipFormalTypeWarning(mb, ftyp, fmlname, "Protected method has a formal with Private type");
                                  RETURN NIL;
                              END; (* IF *)
            | MS.Vinternal:   IF vis = MS.Vprivate THEN
                                  p.SkipFormalTypeWarning(mb, ftyp, fmlname, "Assembly method has a formal with Private type");
                                  RETURN NIL;
                              END; (* IF *)
            ELSE
            END; (* CASE *)

            qualftyp := p.GetFullQualifiedName(ftyp);
            ftns := p.InsertTypeSpace(qualftyp);
            ft := p.InsertType(ftyp, ftns, WithoutMember);
            IF isPtr THEN    (* this is a (VAR formalname: POINTER TO value type) *)
                (* create a dummy pointer to the value type for exporting *)
                ft := ftns.InsertValuePointer(ft);
            END; (* IF *)

            fml := fl.InsertFormal(fmlname, ft ,ppass, pos, mthcls); 
        ELSE
            p.SkipFormalWarning(mb, fmlname, "A null formal type detected");
            RETURN NIL
        END; (* IF *)
    END; (* FOR *)
    RETURN fl;
END InsertFormals;

(* ========================================================================== *)

PROCEDURE (p: Parser) PreWalkConstructor(typ: Sys.Type; IN mb: Ref.MethodBase), NEW;
BEGIN
    (* we don't need to translate static constructor as it is impossible
       to call this method *)
    IF mb.get_IsPrivate() THEN RETURN END;
    IF mb.get_IsStatic() THEN RETURN; END;
    p.PreWalkFormals(mb);
END PreWalkConstructor;

(* ========================================================================== *)

PROCEDURE (p: Parser) InsertConstructor(typ: Sys.Type; IN mb: Ref.MethodBase; tp: MS.Type), NEW;
VAR
    cname: CharOpen;    (* constructor name *)
    qualrtyp: CharOpen; (* return type *)
    mth: MS.Method;     (* method ref *)
    attr: MS.Attribute;
    fl: MS.FormalList;
    visConst: INTEGER;
BEGIN
    IF mb.get_IsPrivate() THEN RETURN END;
    (* we don't need to translate static constructor as it is impossible
       to call this method *)
    IF mb.get_IsStatic() THEN RETURN; END;
    visConst := GetMethodVisibility(mb);
    cname := ST.ToChrOpen(mb.get_Name().ToCharArray());

    (* the type of the constructor is its return type *)
    qualrtyp := p.GetFullQualifiedName(typ);
    fl := p.InsertFormals(tp, mb, visConst);
    IF fl = NIL THEN RETURN; END;
    fl.CreateSigCode(tp.GetRawFullName());

    mth := tp.InsertConstructor(cname, tp, fl);
    IF mth.IsOverload() THEN mth.EncryptSignature(); END;
    SetMethodVisibility(mth, mb);
    SetMethodAttribute(mth, mb);
END InsertConstructor;

(* ========================================================================== *)

PROCEDURE (p: Parser) PreWalkMethod(IN mthi: Ref.MethodInfo), NEW;
VAR
    mthname: CharOpen;  (* method name *)
    qualrtyp: CharOpen; (* return type *)
    utname: CharOpen;   (* underlying type *)
    rtyp: Sys.Type;     (* return type ref *)
    mth: MS.Method;     (* method ref *)
    rt: MS.Type;        (* function return type *)
    mb: Ref.MethodBase;
    rtnPtr: BOOLEAN;
    fl: MS.FormalList;
    ns: MS.Namespace;
BEGIN (* MetaParser_Parser_PreWalkMethod *)
    IF mthi.get_IsPrivate() THEN RETURN END;
    utname := ST.NullString;
    mthname := ST.ToChrOpen(mthi.get_Name().ToCharArray());
    rtyp := mthi.get_ReturnType();
    IF rtyp # NIL THEN
        IF rtyp.get_IsPointer() THEN rtyp := rtyp.GetElementType(); END;
        qualrtyp := p.GetFullQualifiedName(rtyp);
        ns := p.InsertTypeSpace(qualrtyp);
        (* going to build the formal parameter list for the method *)
        mb := mthi; p.PreWalkFormals(mb);
    ELSE
        (* return type refers to a private class *)
        Warning("WARNING !!! WARNING !!! WARNING !!! null return type detected");
        IndentWarning("Assembly: " + p.asbFile^);
        IndentWarning("Class : " + ST.ToChrOpen(mthi.get_ReflectedType().get_FullName().ToCharArray())^);
        IndentWarning("Method: " + ST.ToChrOpen(mthi.get_Name().ToCharArray())^);
        (* how can the ildasm display the private classes ??? *)
        RETURN;
    END; (* IF *)
END PreWalkMethod;

(* ========================================================================== *)

PROCEDURE (p: Parser) MethodWarning(mthi: Ref.MethodInfo; msg: ARRAY OF CHAR), NEW;
BEGIN
    Warning("!!! WARNING !!! - " + msg);
    IndentWarning("Assembly: " + p.asbFile^);
    IndentWarning("Class : " + ST.ToChrOpen(mthi.get_ReflectedType().get_FullName().ToCharArray())^);
    IndentWarning("Method: " + ST.ToChrOpen(mthi.get_Name().ToCharArray())^);
END MethodWarning;


PROCEDURE (p: Parser) SkipMethodWarning(mthi: Ref.MethodInfo; msg: ARRAY OF CHAR), NEW;
BEGIN
    p.MethodWarning(mthi, msg);
    IndentWarning("Method skipped");
END SkipMethodWarning;


PROCEDURE (p: Parser) SkipMethodTypeWarning(mthi: Ref.MethodInfo; rtyp: Sys.Type; msg: ARRAY OF CHAR), NEW;
BEGIN
    p.MethodWarning(mthi, msg);
    IndentWarning("Method return type name: " + ST.ToChrOpen(rtyp.get_FullName().ToCharArray())^);
    IndentWarning("Method skipped");
END SkipMethodTypeWarning;


PROCEDURE (p: Parser) InsertMethod(IN mthi: Ref.MethodInfo; tp: MS.Type): MS.Method, NEW;
VAR
    mthname: CharOpen;  (* method name *)
    rtname: CharOpen;
    fulrtyp: CharOpen;
    qualrtyp: CharOpen; (* return type *)
    utname: CharOpen;   (* underlying type *)
    rtyp: Sys.Type;     (* return type ref *)
    mth: MS.Method;     (* method ref *)
    rt: MS.Type;        (* function return type *)
    mb: Ref.MethodBase;
    rtnPtr: BOOLEAN;
    fl: MS.FormalList;
    static: BOOLEAN;
    virtual: BOOLEAN;
    rtns: MS.Namespace;
    visMethod: INTEGER;
    vis: INTEGER;
BEGIN (* MetaParser_Parser_InsertMethod *)
    IF mthi.get_IsPrivate() THEN RETURN NIL END;
    utname := ST.NullString;
    static := mthi.get_IsStatic();
    virtual := mthi.get_IsVirtual();
    visMethod := GetMethodVisibility(mthi);
    mthname := ST.ToChrOpen(mthi.get_Name().ToCharArray());
    rtyp := mthi.get_ReturnType();
    IF rtyp # NIL THEN
        IF rtyp.get_IsEnum() THEN
            utname := Sys.Enum.GetUnderlyingType(rtyp).ToString().ToCharArray();
        END; (* IF *)

        IF rtyp.get_IsPointer() THEN
            rtyp := rtyp.GetElementType();

            IF rtyp.get_IsPointer() THEN
                p.SkipMethodTypeWarning(mthi, rtyp, "Assembly contains method with a Return type of POINTER TO POINTER");
                RETURN NIL;
            ELSIF rtyp.get_IsArray() THEN
                p.SkipMethodTypeWarning(mthi, rtyp, "Assembly contains method with a Return type of POINTER TO POINTER TO ARRAY");
                RETURN NIL;
            ELSIF rtyp.get_IsInterface() THEN
                p.SkipMethodTypeWarning(mthi, rtyp, "Assembly contains method with a Return type of POINTER TO POINTER TO INTERFACE");
                RETURN NIL;
            ELSIF rtyp.get_IsClass() THEN
                p.SkipMethodTypeWarning(mthi, rtyp, "Assembly contains method with a Return type of POINTER TO POINTER TO RECORD");
                RETURN NIL;
            ELSIF rtyp.get_IsEnum() THEN
                p.SkipMethodTypeWarning(mthi, rtyp, "Assembly contains method with a Return type of POINTER TO ENUM");
            ELSIF rtyp.get_IsPrimitive() THEN
                p.SkipMethodTypeWarning(mthi, rtyp, "Assembly contains method with a Return type of POINTER TO primitive");
                RETURN NIL;
            END; (* IF *)

            rtnPtr := TRUE;
        ELSE
            rtnPtr := FALSE;
        END; (* IF *)

        vis := GetTypeVisibility(rtyp);
        CASE visMethod OF
          MS.Vpublic:     IF vis = MS.Vprivate THEN
                              p.SkipMethodTypeWarning(mthi, rtyp, "Public method has a Private Return type");
                              RETURN NIL;
                          END; (* IF *)
        | MS.Vprotected:  IF vis = MS.Vprivate THEN
                              p.SkipMethodTypeWarning(mthi, rtyp, "Protected method has a Private Return type");
                              RETURN NIL;
                          END; (* IF *)
        ELSE
        END; (* CASE *)

        qualrtyp := p.GetFullQualifiedName(rtyp);
        rtns := p.InsertTypeSpace(qualrtyp);
        rt := p.InsertType(rtyp, rtns, WithoutMember);

        IF rtnPtr THEN
            (* create a dummy pointer to the value type for exporting *)
            rt := rtns.InsertValuePointer(rt);
        END; (* IF *)

        (* going to build the formal parameter list for the method *)
        mb := mthi;
        fl := p.InsertFormals(tp, mb, visMethod);
        IF fl = NIL THEN RETURN NIL; END;

        fulrtyp := rt.GetRawFullName();
        rtname := rt.GetName();
        IF rtname^ # MS.Void THEN
            fl.CreateSigCode(fulrtyp);
        ELSE
            fl.CreateSigCode(rtname);
        END; (* IF *)

        mth := tp.InsertMethod(mthname, rt, static, virtual, rtnPtr, fl);
        IF ~mthi.get_IsPrivate() & ~mthi.get_IsAssembly() THEN
            WITH mth: MS.Function DO
                rt.SetVisibility(MS.Vpublic);
            ELSE
            END; (* WITH *)
        END; (* IF *)

        IF mth.IsOverload() THEN mth.EncryptSignature(); END;
        SetMethodVisibility(mth, mthi);
        SetMethodAttribute(mth, mthi);
        RETURN mth;
    ELSE
        (* return type refers to a private class *)
        p.SkipMethodWarning(mthi, "Null return type detected");
        (* how can the ildasm display the private classes ??? *)
        RETURN NIL;
    END; (* IF *)
END InsertMethod;

(* ========================================================================== *)

PROCEDURE (p: Parser) PreWalkEvent(IN evti: Ref.EventInfo), NEW;
VAR
    evtname: CharOpen;
    dtyp: Sys.Type;
    qualdtyp: CharOpen;
    htyp: Sys.Type;
    qualhtyp: CharOpen;
    ns: MS.Namespace;
    mthi: Ref.MethodInfo;
BEGIN
    evtname := ST.ToChrOpen(evti.get_Name().ToCharArray());

    dtyp := evti.get_DeclaringType();
    qualdtyp := p.GetFullQualifiedName(dtyp);
    ns := p.InsertTypeSpace(qualdtyp);

    htyp := evti.get_EventHandlerType();
    qualhtyp := p.GetFullQualifiedName(htyp);
    ns := p.InsertTypeSpace(qualhtyp);

    mthi := evti.GetAddMethod(TRUE);
    p.PreWalkMethod(mthi);

    mthi := evti.GetRemoveMethod(TRUE);
    p.PreWalkMethod(mthi);
END PreWalkEvent;

(* ========================================================================== *)

PROCEDURE (p: Parser) SkipEventTypeWarning(evti: Ref.EventInfo; htyp: Sys.Type; msg: ARRAY OF CHAR), NEW;
BEGIN
    Warning("!!! WARNING !!! - " + msg);
    IndentWarning("Assembly: " + p.asbFile^); 
    IndentWarning("Class : " + ST.ToChrOpen(evti.get_ReflectedType().get_FullName().ToCharArray())^);
    IndentWarning("Event: " + ST.ToChrOpen(evti.get_Name().ToCharArray())^);
    IndentWarning("HandlerType : " + ST.ToChrOpen(htyp.get_FullName().ToCharArray())^);
    IndentWarning("event skipped");
END SkipEventTypeWarning;


PROCEDURE (p: Parser) InsertEvent(IN evti: Ref.EventInfo; tp: MS.Type), NEW;
VAR
    evtname : CharOpen;       (* event name *)
    dtyp    : Sys.Type;       (* type which declare the event *)
    qualdtyp: CharOpen;       (* qualified full name of dtyp *)
    dtns    : MS.Namespace;   (* namespace of dtyp *)
    dt      : MS.Type;        (* type which declare the event *)
    htyp    : Sys.Type;       (* event handler type of event *)
    qualhtyp: CharOpen;       (* qualified full name of htyp *)
    htns    : MS.Namespace;   (* namespace of htype *)
    ht      : MS.Type;        (* event handler type of event *)
    mthi    : Ref.MethodInfo; (* temporary *)
    amth    : MS.Method;      (* method for adding handler to the event's delegate list (+=) *)
    rmth    : MS.Method;      (* method for removing handler from the event's delegate list (-=) *)
    evt     : MS.Event;       (* event *)
    vis     : INTEGER;
BEGIN
    evtname := ST.ToChrOpen(evti.get_Name().ToCharArray());

    dtyp := evti.get_DeclaringType();
    qualdtyp := p.GetFullQualifiedName(dtyp);
    dtns := p.InsertTypeSpace(qualdtyp);
    dt := p.InsertType(dtyp, dtns, WithoutMember);

    htyp := evti.get_EventHandlerType();
    vis := GetTypeVisibility(htyp);
    IF vis = MS.Vprivate THEN
        p.SkipEventTypeWarning(evti, htyp, "Event has a Private handler type");
        RETURN;
    END; (* IF *)

    qualhtyp := p.GetFullQualifiedName(htyp);
    htns := p.InsertTypeSpace(qualhtyp);
    ht := p.InsertType(htyp, htns, WithoutMember);

    mthi := evti.GetAddMethod(TRUE);
    amth := p.InsertMethod(mthi, tp);

    mthi := evti.GetRemoveMethod(TRUE);
    rmth := p.InsertMethod(mthi, tp);

    evt := tp.InsertEvent(evtname, evti.get_IsMulticast(), dt, ht, amth, rmth);
END InsertEvent;

(* ========================================================================== *)

PROCEDURE (p: Parser) PreWalkMember(typ: Sys.Type), NEW;
VAR
    bf: Ref.BindingFlags;
    miarr: Ref.Arr1MemberInfo;
    i: INTEGER;
    mi: Ref.MemberInfo;
    mt: Ref.MemberTypes;
    ntyp: Sys.Type;
    nt: MS.Type;
BEGIN
    bf := Ref.BindingFlags.Static + 
          Ref.BindingFlags.Instance +
          Ref.BindingFlags.Public + 
          Ref.BindingFlags.NonPublic +          (* include private and family *)
          Ref.BindingFlags.DeclaredOnly;
    miarr := typ.GetMembers(bf);
    FOR i := 0 TO LEN(miarr)-1 DO
        mi := miarr[i];
        mt := mi.get_MemberType();
        CASE mt OF
          Ref.MemberTypes.Constructor:          (* 0x01 *)
            p.PreWalkConstructor(typ, mi(Ref.MethodBase));
        | Ref.MemberTypes.Event:                (* 0x02 *)
            p.PreWalkEvent(mi(Ref.EventInfo));
        | Ref.MemberTypes.Field:                (* 0x04 *)
            p.PreWalkField(typ, mi(Ref.FieldInfo));
        | Ref.MemberTypes.Method:               (* 0x08 *)
            p.PreWalkMethod(mi(Ref.MethodInfo));
        | Ref.MemberTypes.Property:             (* 0x10 *)
        | Ref.MemberTypes.TypeInfo:             (* 0x20 *)
        | Ref.MemberTypes.Custom:               (* 0x40 *)
        | Ref.MemberTypes.NestedType:           (* 0x80 *)
            ntyp := mi(Sys.Type);
            p.PreWalkType(ntyp);
        | Ref.MemberTypes.All:                  (* 0xBF *)
        ELSE
        END; (* CASE *)
    END; (* FOR *)
END PreWalkMember;

(* ========================================================================== *)

PROCEDURE (p: Parser) InsertMember(typ: Sys.Type; tp: MS.Type; ns: MS.Namespace), NEW;
VAR
    miarr: Ref.Arr1MemberInfo;
    i: INTEGER;
    mi: Ref.MemberInfo;
    bf: Ref.BindingFlags;
    mt: Ref.MemberTypes;
    mth: MS.Method;
    nt: MS.Type;
    ntyp: Sys.Type;
BEGIN
    bf := Ref.BindingFlags.Static + 
          Ref.BindingFlags.Instance +
          Ref.BindingFlags.Public + 
          Ref.BindingFlags.NonPublic +          (* include private and family *)
          Ref.BindingFlags.DeclaredOnly;
    miarr := typ.GetMembers(bf);
    FOR i := 0 TO LEN(miarr)-1 DO
        mi := miarr[i];
        mt := mi.get_MemberType();

        CASE mt OF
          Ref.MemberTypes.Constructor:          (* 0x01 *)
            IF ~mi(Ref.MethodBase).get_IsPrivate() THEN
                p.InsertConstructor(typ, mi(Ref.MethodBase), tp);
            END; (* IF *)
        | Ref.MemberTypes.Event:                (* 0x02 *)
            p.InsertEvent(mi(Ref.EventInfo), tp);
        | Ref.MemberTypes.Field:                (* 0x04 *)
            IF ~mi(Ref.FieldInfo).get_IsPrivate() THEN
                p.InsertField(typ, mi(Ref.FieldInfo), tp);
            END; (* IF *)
        | Ref.MemberTypes.Method:               (* 0x08 *)
            IF ~mi(Ref.MethodInfo).get_IsPrivate() THEN
                mth := p.InsertMethod(mi(Ref.MethodInfo), tp(MS.Type));
            END; (* IF *)
        | Ref.MemberTypes.Property:             (* 0x10 *)
            (* nothing to handle for property, its method is already
               handled in Method case *)
        | Ref.MemberTypes.TypeInfo:             (* 0x20 *)
        | Ref.MemberTypes.Custom:               (* 0x40 *)
        | Ref.MemberTypes.NestedType:           (* 0x80 *)
            ntyp := mi(Sys.Type);
            IF ntyp.get_IsPublic() OR ntyp.get_IsNestedPublic()
                                   OR ntyp.get_IsNestedFamily() THEN
                nt := p.InsertType(ntyp, ns, WithMember);
            END; (* IF *)
        | Ref.MemberTypes.All:                  (* 0xBF *)
        ELSE                                    (* Error *)
        END; (* CASE *)
    END; (* FOR *)
END InsertMember;

(* ========================================================================== *)

PROCEDURE (p:Parser) PreWalkType(typ: Sys.Type), NEW;
(* ns is a MS.Namespace object *)
VAR
    btyp: Sys.Type;
    qualbtyp: CharOpen;
    attr : MS.Attribute;
    ns: MS.Namespace;
BEGIN
    btyp := typ.get_BaseType();
    IF btyp # NIL THEN
        qualbtyp := p.GetFullQualifiedName(btyp);
        ns := p.InsertTypeSpace(qualbtyp);
    END; (* IF *)
    p.PreWalkMember(typ);
END PreWalkType;

(* ========================================================================== *)

PROCEDURE (p: Parser) InsertType(typ: Sys.Type; ns: MS.Namespace; withmember: BOOLEAN): MS.Type, NEW;
(* ns is a MS.Namespace object *)
VAR
    class: MS.PointerType;
    deleg: MS.PointerType;
    intfc: MS.PointerType;
    struc: MS.StrucType;
    enum : MS.EnumType;
    prim : MS.PrimType;
    btyp : Sys.Type;
    btns : MS.Namespace;
    bt: MS.Type;
    qualbtyp: CharOpen;
    qualtyp: CharOpen;
    dtyp: Sys.Type;
    dt: MS.Type;
    etyp: Sys.Type;
    et: MS.Type;
    tarr: Sys.Arr1Type;
    ilist: MS.OrderList;
    ityp : Sys.Type;
    it: MS.Type;
    i: INTEGER;
    itns: MS.Namespace;
    qualityp: CharOpen;
    fasbname: CharOpen;
    asbname: CharOpen;
    ftname: CharOpen;
    key: INTEGER;
    ttyp: MS.Type;
BEGIN
    tarr := NIL; ilist := NIL;
    qualtyp := p.GetFullQualifiedName(typ);

    MS.SplitTypeName(qualtyp, fasbname, asbname, ftname); 
    ttyp := ns.GetExistType(ftname);
    IF (ttyp # NIL) & ~(ttyp IS MS.NamedType) & ~withmember THEN RETURN ttyp; END;

    IF ~typ.get_IsArray() THEN
        btyp := typ.get_BaseType();
        tarr := typ.GetInterfaces();
    ELSE
        (* Array types *)
        etyp := typ.GetElementType();
        IF etyp = NIL THEN
            (* [mscorlib]System.Array *)
            btyp := typ.get_BaseType();
        ELSE
            (* real array type *)
            btyp := NIL;
            et := p.InsertType(etyp, ns, WithoutMember);
        END; (* IF *)
    END; (* IF *)

    IF btyp # NIL THEN
        IF btyp.get_IsCOMObject() THEN
            (* base type is a COM object *)
            IF btyp.Equals(typComObject) THEN
                (* base type is mscorlib.System.__ComObject *)
                btyp := typObject;
            END;
        END;

        qualbtyp := p.GetFullQualifiedName(btyp);
        btns := p.InsertTypeSpace(qualbtyp);
        bt := p.InsertType(btyp, btns, WithoutMember);
    ELSE
        bt := NIL;
    END; (* IF *)

    IF (tarr # NIL) & (LEN(tarr) > 0) THEN
        ilist := MS.CreateInterfaceList();
        FOR i := 0 TO LEN(tarr)-1 DO
            ityp := tarr[i];
            IF ~ityp.Equals(typISerializable) THEN
                qualityp := p.GetFullQualifiedName(ityp);
                itns := p.InsertTypeSpace(qualityp);
                it := p.InsertType(ityp, itns, WithoutMember);
                IF it.IsInterfacePtr() THEN
                    ilist.AddInterface(it);
                ELSE
                    ASSERT(FALSE);
                END; (* IF *)
            END; (* IF *)
        END; (* FOR *)
    END; (* IF *)

    IF IsNested(typ) THEN
        dtyp := typ.get_DeclaringType();
        dt := p.InsertType(dtyp, ns, WithoutMember);
    ELSE
        dt := NIL;
    END; (* IF *)

    IF typ.get_IsClass() THEN
        IF IsDelegate(typ) THEN 
            deleg := ns.InsertType(TypeConvert(typ), qualtyp, bt, dt, ilist)(MS.PointerType);
            SetTypeVisibility(deleg, typ);
            IF ~(deleg.GetTarget() IS MS.ArrayType) THEN
                SetTypeAttribute(deleg, typ);
            END; (* IF *)
            IF IsMulticastDelegate(typ) THEN
                deleg.SetMulticast();
            END; (* IF *)
            IF withmember THEN
                p.InsertMember(typ, deleg, ns);
            END; (* IF *)
            RETURN deleg;
        ELSE
            class := ns.InsertType(TypeConvert(typ), qualtyp, bt, dt, ilist)(MS.PointerType);
            SetTypeVisibility(class, typ);
            IF ~(class.GetTarget() IS MS.ArrayType) THEN
                SetTypeAttribute(class, typ);
            END; (* IF *)
            IF withmember THEN
                p.InsertMember(typ, class, ns);
            END; (* IF *)
            RETURN class;
        END; (* IF *)
    ELSIF typ.get_IsInterface() THEN
        intfc := ns.InsertType(TypeConvert(typ), qualtyp, bt, dt, ilist)(MS.PointerType);
        SetTypeVisibility(intfc, typ);
        SetTypeAttribute(intfc, typ);
        IF withmember THEN
            p.InsertMember(typ, intfc, ns);
        END; (* IF *)
        RETURN intfc;
    ELSE
        IF typ.get_IsEnum() THEN
            enum := ns.InsertEnumType(TypeConvert(typ), qualtyp, bt, dt,
                    ST.ToChrOpen(Sys.Enum.GetUnderlyingType(typ).ToString().ToCharArray()));
            SetTypeVisibility(enum, typ);
            SetTypeAttribute(enum, typ);
            IF withmember THEN
                p.InsertMember(typ, enum, ns);
            END; (* IF *)
            RETURN enum;
        ELSIF typ.get_IsPrimitive() THEN
            prim := ns.InsertType(TypeConvert(typ), qualtyp, bt, dt, ilist)(MS.PrimType);
            SetTypeVisibility(prim, typ);
            SetTypeAttribute(prim, typ);
            IF withmember THEN
                p.InsertMember(typ, prim, ns);
            END; (* IF *)
            RETURN prim;
        ELSE                    (* Struct Type *)
            struc := ns.InsertType(TypeConvert(typ), qualtyp, bt, dt, ilist)(MS.StrucType);
            SetTypeVisibility(struc, typ);
            SetTypeAttribute(struc, typ);
            IF withmember THEN
                p.InsertMember(typ, struc, ns);
            END; (* IF *)
            RETURN struc;
        END; (* IF *)
    END; (* IF *)
    RETURN NIL;
END InsertType;

(* ========================================================================== *)

PROCEDURE NewParser*(): Parser;
VAR
    p: Parser;
BEGIN
    NEW(p); p.asbPath := NIL; p.asbFile := NIL;
    RETURN p;
END NewParser;

(* ========================================================================== *)

PROCEDURE (p: Parser) GetAsmPath*(): CharOpen, NEW;
BEGIN
    RETURN p.asbPath;
END GetAsmPath;

(* ========================================================================== *)

    PROCEDURE TryCatchLoadAssembly(fullname: CharOpen): Ref.Assembly;
    VAR asb: Ref.Assembly;
    BEGIN
        asb := Ref.Assembly.LoadFrom(Sys.String.init(ST.TrimNull(fullname)));
        RETURN asb;

        RESCUE(x);
        Error.WriteString(RTS.getStr(x)); Error.WriteLn;
        RETURN NIL;
    END TryCatchLoadAssembly;


PROCEDURE (p: Parser) LoadAssembly*(fullname: CharOpen): BOOLEAN, NEW;
VAR
    idx1, idx2 : INTEGER;
    asb: Ref.Assembly;
BEGIN
    idx1 := ST.StrRChr(fullname, '\');
    idx2 := ST.StrRChr(fullname, '.');
(*  IF p.asbFile = NIL THEN *)
        (* store the file path of the assembly *)
        p.asbPath := ST.SubStr(fullname, 0, idx1);

        IF p.asbPath = ST.NullString THEN
            p.asbPath := ST.StrCatChr(ST.ToChrOpen(GPFiles.CurrentDirectory()), '\');
            p.asbFile := ST.SubStr(fullname, 0, idx2-1);
            fullname := ST.StrCat(p.asbPath, fullname);
        ELSE
            p.asbFile := ST.SubStr(fullname, idx1+1, idx2-1);
        END; (* IF *)

        CurrentPath := p.asbPath;
(*  END; (* IF *) *)
    asb := TryCatchLoadAssembly(fullname);
    p.asb := asb;
    RETURN asb # NIL;
END LoadAssembly;

(* ========================================================================== *)

PROCEDURE (p: Parser) PreWalkAssembly*(), NEW;
VAR
    tarr: Sys.Arr1Type;
    i: INTEGER;
    typ: Sys.Type;
    str: Sys.String;
    ns: MS.Namespace;
    tp: MS.Type;
BEGIN
    (* get an array of classes inside the DLL *)
    IF nonPub THEN
        tarr := p.asb.GetTypes();
    ELSE
        tarr := p.asb.GetExportedTypes();
    END; (* IF *)
    FOR i := 0 TO LEN(tarr)-1 DO
        typ := tarr[i];
        IF typ # NIL THEN p.PreWalkType(typ); END;
    END; (* FOR *)
END PreWalkAssembly;

(* ========================================================================== *)

PROCEDURE (p: Parser) InAssemblyHistory(): BOOLEAN, NEW;
VAR i: INTEGER;
BEGIN
    FOR i := 0 TO AsbCount-1 DO
        IF (History[i] = NIL) THEN
            RETURN FALSE;
        ELSE
            IF (ST.StrCmp(History[i], ST.StrCat(p.asbPath, p.asbFile)) = ST.Equal) THEN
                RETURN TRUE;
            END; (* IF *)
        END; (* IF *)
    END; (* FOR *)
    RETURN FALSE;
END InAssemblyHistory;


PROCEDURE (p: Parser) ParseAssembly*(): MS.Assembly, NEW;
(* Remark:
 * GetExportedTypes() fails on mscorlib.dll if the mscorlib is not 
 * LoadFrom("C:\WINNT\Microsoft.NET\Framework\v1.0.3512\mscorlib.dll").
 * Exception is:
 *    Unhandled Exception: System.TypeLoadException: Could not load type
 *    System.Object from assembly mscorlib, Version=1.0.3300.0, Culture=neutral,
 *    PublicKeyToken=b77a5c561934e089 because the parent does not exist.
 *    at System.Reflection.Assembly.GetExportedTypes()
 *)
VAR
    ao: MS.Assembly;
    tarr: Sys.Arr1Type;
    i: INTEGER;
    typ: Sys.Type;
    str: Sys.String;
    ns: MS.Namespace;
    tp: MS.Type;
BEGIN
    (* either insert an assembly, or retrieve reference to an existing assembly *)
    p.ao := MS.InsertAssembly(ST.ToChrOpen(p.asb.get_FullName().ToCharArray()), p.asbFile);

    IF ~p.InAssemblyHistory() THEN
        (* get an array of classes inside the DLL *)
        IF nonPub THEN
            tarr := p.asb.GetTypes();
        ELSE
            tarr := p.asb.GetExportedTypes();
        END; (* IF *)

        FOR i := 0 TO LEN(tarr)-1 DO
            typ := tarr[i];
            IF typ # NIL THEN
                str := typ.get_Namespace();
                IF str # NIL THEN
                    ns := p.ao.InsertNamespace(ST.ToChrOpen(str.ToCharArray()));
                ELSE
                    ns := p.ao.InsertNamespace(ST.NullString);
                END; (* IF *)
                tp := p.InsertType(typ, ns, WithMember);
            END; (* IF *)
        END; (* FOR *)

        History[AsbCount] := ST.StrCat(p.asbPath, p.asbFile);
        INC(AsbCount);
    ELSE
    END; (* IF *)

    RETURN p.ao;
END ParseAssembly;

(* ========================================================================== *)

BEGIN
    Initialize();
END MetaParser.


