module Mines;

/*
The game 'MacMines' in Concurrent Clean.

Run the program using the "No Console" option (Application options).

To generate an application for this program the memory of the Clean
0.8 application should be set to at least 1800K.
The linker needs an additional 500K of free memory inside or outside
the Clean 0.8 application.
To launch the generated application 1100K of free memory is needed
outside the Clean 0.8 application.
*/

import	StdClass;
import StdInt, StdMisc, StdBool, StdString, StdArray;
import deltaEventIO, deltaMenu, deltaTimer, deltaDialog, deltaSystem;
import MinesBest, Help;


    

::	* Mines	:== (Minefield, NrVisible, Pebbles, NrMines, Dimension, Time, MinesBest, Font, RandomSeed);
::	NrMines		:== Int;
::	NrVisible	:== Int;

::	* IO :==  IOState Mines;

::	DrawSpotFunction :== Spot ->  Position -> DrawFunction ;

 
     

	MinesID :== 1;
		NewGameID	:== 10;
		HelpID		:== 11;
		BestTimesID	:== 12;
		QuitID		:== 13;
	SkillID :== 2;
		EasyID	 :== 20;
		InterID  :== 21;
		HardID	 :== 22;
		CustomID :== 23;

	SkillItem id t key f :== MenuRadioItem id t (Key key) Able f;


	SkillDialogID :== 1;
		WidthID   :== 11;
		HeightID  :== 12;
		NrMinesID :== 13;
		OkID	  :== 14;
		CancelID  :== 15;

	OverDlogID :== 2;
		NameID   :== 21;
		OverOKID :== 22;

	TimerID	:== 1;

	MinesWindow skill_dim
		:== FixedWindow WindowID (0,0) "Mines"
				(WindowPictDomain skill_dim) DrawGame
				[GoAway Quit, Mouse Able PlayMines];

	WindowID :== 1;
	BestWdID :== 2;

	HelpFile		:== "MinesHelp";
	HiScoresFile	:== "mineshi";

	NoMods	:== (False,False, False, False);

    
InitialiseRandomSeed	:: Mines IO -> (Mines, IO);
InitialiseRandomSeed (_, mv, ps, nr, dm, tm, (files, bt), ft, _) io
	=	((mineField, mv, ps, nr, dm, tm, (files, bt), ft, newSeed), io`);
	where {
		(mineField, newSeed)
			=	SowMines nr dm seed;
		((hours,minutes,seconds), io`)
			=	GetCurrentTime io;
		seed
			=	seconds + minutes * 60 + hours * 3600 + 1
	};


Start	:: * World -> * World;
Start world =  CloseEvents events` (closefiles (WriteHiScores hifile best`) world``);
	where {
	(mf,vm,ps,nr,dm,tm,best`,f,seed)=: mines;
	(mines,events`)            =: PlayMinesGame best events;
	(hifile, best)             =: ReadHiScores HiScoresFile files;
	(events,world``)           =: OpenEvents world`;
	(files, world` )           =: openfiles world;
	};

PlayMinesGame	:: MinesBest EVENTS -> (Mines,EVENTS);
PlayMinesGame (files,best) events
	=  StartIO [ menu , window, about, time] initstate [InitialiseRandomSeed] events;
		where {
		initstate  =: SetMines EasyMines EasyDim (files`,best) 0;

		menu=: MenuSystem [

				PullDownMenu MinesID "Mines" Able [
					MenuItem NewGameID "New Game" (Key 'N') Able NewGame,
					MenuItem BestTimesID "Best Times" (Key 'B') Able ShowBest,
					MenuSeparator,
					MenuItem QuitID "Quit" (Key 'Q') Able Quit],

				PullDownMenu SkillID "Skill" Able [
					MenuRadioItems EasyID [
						SkillItem EasyID "Easy" 'E' (SetGame EasyMines EasyDim),
						SkillItem InterID "Intermediate" 'I' (SetGame InterMines InterDim),
						SkillItem HardID "Hard" 'H' (SetGame HardMines HardDim),
						SkillItem CustomID "Custom..." 'C' Custom ]]];

		window=: WindowSystem [MinesWindow EasyDim];
		
		about =: DialogSystem [about_dialog];
		(about_dialog,files`)=: MakeAboutDialog "Mines" HelpFile files Help;

		time  =: TimerSystem [Timer TimerID Unable TicksPerSecond Timing];
		};

SetMines	:: Int Dimension MinesBest RandomSeed -> Mines;
SetMines nrmines dim best seed
	= 	(mineField, 0, [], nrmines, dim, Off, best, times, newSeed);
		where {
		(mineField, newSeed)
			=	SowMines nrmines dim seed;
		(bt, times)=: SelectFont "Times" ["BoldStyle"] 12;
		};

Help	:: Mines IO -> (Mines, IO);
Help (mf, mv, ps, nr, dm, tm, (files, bt), ft, seed) io_state
	= 	((mf, mv, ps, nr, dm, tm, (files`, bt), ft, seed), io_state`);
		where {
		(files`,io_state`)=: ShowHelp HelpFile files io_state;
		};

Custom	:: Mines IO -> (Mines, IO);
Custom mines=:(mf, mv, pbls, nrms, (col, row), tm, bt, ft, seed) io_state
	= 	OpenModalDialog dialog mines io_state;
		where {
		dialog=: CommandDialog SkillDialogID "Custom" [] OkID [
					DynamicText 1 Left (Pixel wid) "Width (1-30):",
					EditText WidthID (RightTo 1) (Pixel 40) 1 (toString col),
					DynamicText 3 Left (Pixel wid) "Height (2-17):",
					EditText HeightID (RightTo 3) (Pixel 40) 1 (toString row),
					DynamicText 5 Left (Pixel wid) "Mines:",
					EditText NrMinesID (RightTo 5) (Pixel 40) 1 (toString total),
					DialogButton CancelID Center "Cancel" Able Cancel,
					DialogButton OkID (RightTo CancelID) "OK" Able OK];
		total 			 =:  Length_new pbls  + nrms;
		wid				 =: FontStringWidth "Height (2-17): " dfont;
		(b,dfont)		 =: SelectFont font style size;
		(font,style,size)=: DefaultFont;
		};

OK	:: DialogInfo Mines IO -> (Mines, IO);
OK info (oldminef,oldvis,oldpebbles,oldnr,olddim,oldtime,oldbest,oldfont, seed) io_state
	= 	SetGame nr_mines dim mines` (CloseActiveDialog io_state);
		where {
		width	=: Between 1 30               (TextToNumber width_text    0);
		height	=: Between 2 17               (TextToNumber height_text   0);
		nr_mines=: Between 1 (width * height) (TextToNumber nr_mines_text 0);
		mines`	=: ([], 0, [], nr_mines, dim, oldtime, oldbest, oldfont, seed);
		width_text   =: GetEditText WidthID   info;
		height_text  =: GetEditText HeightID  info;
		nr_mines_text=: GetEditText NrMinesID info;
		dim=:(width, height);
		};

Cancel	:: DialogInfo Mines IO -> (Mines, IO);
Cancel dialog mines io_state =  (mines, CloseActiveDialog io_state);


/*	The menu definition:
*/

Quit	:: Mines IO -> (Mines, IO);
Quit mines io_state =  (mines, QuitIO io_state);

NewGame	:: Mines IO -> (Mines, IO);
NewGame (oldminef, vismines, pebbles, oldnr, dim, time, best, font, seed) io_state
	= 	DrawInWindowFrame WindowID Erase_and_DrawGame mines` io_state`;
		where {
		mines`	 =: SetMines nr_mines dim best seed;
		nr_mines =:  Length_new pebbles  + oldnr;
		io_state`=: ChangeIOState [ChangeUpdateFunction WindowID DrawGame,
								  ActivateWindow WindowID,
								  EnableMouse WindowID] io_state;
		};

ShowBest	:: Mines IO -> (Mines, IO);
ShowBest state io
	= 	(state, OpenWindows [FixedWindow BestWdID (40,40) "Hall of Fame"
							             ((0,0),size) UpdateBest []] io);
		where {
		size=: (BestX,BestY);
		};

UpdateBest	:: UpdateArea Mines -> (Mines, [DrawFunction]);
UpdateBest area mines=:(mf, nv, pbls, nm, dim, time, (files, best), font, seed)
	= 	(mines, [ShowBestTimes best]);
							 
SetGame	:: Int Dimension Mines IO -> (Mines, IO);
SetGame nr_mines dim mines=: ((oldminef, vismines, pebbles, oldnr, olddim, time, best, font, seed)) io_state= 	DrawInWindowFrame WindowID Erase_and_DrawGame mines` io_state`;
		where {
		mines1	 =: SetMines nr_mines dim best seed;
		io_state1=: ChangeUpdateFunction WindowID NoUpdates io_state;
		(mines`,io_state2)=: ChangePictureDomain WindowID (WindowPictDomain dim) mines1 io_state1;
		io_state`=: ChangeIOState [ EnableMouse WindowID,
								   ActivateWindow WindowID,
								   ChangeUpdateFunction WindowID DrawGame] io_state2;
		};

NoUpdates	:: UpdateArea Mines -> (Mines, [DrawFunction]);
NoUpdates area mines =  (mines, []);

/*	The mouse definition:
*/

PlayMines	:: MouseState Mines IO -> (Mines, IO);
PlayMines (p,ButtonUp,m) mines io_state =  (mines, io_state);
PlayMines (p,ButtonStillDown,m) mines io_state =  (mines, io_state);
PlayMines (pos, buttondown, ShiftOnly) mines=: ((minefield, nr_visible, pebbles, nr_mines, dim, time, best, font, seed)) io_state= 	PutOrGetPebble (MousePositionToPosition pos dim) mines io_state;
PlayMines (pos, buttondown, NoMods) mines=: ((minefield, nr_visible, pebbles, nr_mines, dim, time, best, font, seed)) io_state= 	StepCautiously (MousePositionToPosition pos dim) mines io_state;
PlayMines mouse mines io_state =  (mines, io_state);

MousePositionToPosition	:: Point Dimension -> Position;
MousePositionToPosition (x,y) (col,row)
		| x <  size * col  && y <  size * row = 	(inc (x / size), inc (y / size));
	= 	(0, 0);
	where {
	size=:SizeArea;
		
	};

PutOrGetPebble	:: Position Mines IO -> (Mines, IO);
PutOrGetPebble (0, 0) mines io_state =  (mines, io_state);
PutOrGetPebble pos mines=:(mf, nr_visible, pebbles, 0, dim, time, best, font, seed) io_state
		| Unique pos pebbles= 	(mines, io_state);
	= 	GetPebble pos mines io_state;
PutOrGetPebble pos mines=:(mf, nrvis, pebbles, nrmines, dim, Running time, best, font, seed) io_state
		| Unique pos pebbles= 	PutPebble pos mines io_state;
	= 	GetPebble pos mines io_state;
PutOrGetPebble pos mines=:(mf, nr_visible, pebbles, nr_mines, dim, Off, best, font, seed) io_state
	= 	PutPebble pos mines` io_state`;
		where {
		mines`	 =: (mf, nr_visible, pebbles, nr_mines, dim, Running 0, best, font, seed);
		io_state`=: EnableTimer TimerID io_state;
		};

PutPebble	:: Position Mines IO -> (Mines, IO);
PutPebble pos mines=:(minefield, nr_visible, pebbles, nr_mines, dim, time, best, font, seed) io_state
		| not (InvisibleSpot (GetSpot pos minefield))= 	(mines, io_state);
	= 	FinalSpotRevealed mines` io_state`;
		where {
		mines`	 =: (minefield, inc nr_visible, [pos : pebbles], dec nr_mines, dim, time, best, font, seed);
		io_state`=: DrawInWindow WindowID [DrawPebble pos] io_state;
		};

GetPebble	:: Position Mines IO -> (Mines, IO);
GetPebble pos (mf, nr_visible, pebbles, nr_mines, dim, time, best, font, seed) io_state
	= 	(mines`, io_state`);
		where {
		mines`	 =: (mf, dec nr_visible, RemovePebble pos pebbles, nr_mines`, dim, time, best, font, seed);
		io_state`=: DrawInWindow WindowID [DrawEmptyArea pos,DrawNrMines font nr_mines` dim] io_state;
		nr_mines`=: inc nr_mines;
		};

StepCautiously	:: Position Mines IO -> (Mines, IO);
StepCautiously (0,0) mines io_state =  (mines, io_state);
StepCautiously pos mines=: ((minefield, nr_vis, pebbles, nr_mines, dim, time=:Running t, best, font, seed)) io_state							| MineSpot spot= 	(mines`, end_game);
								| not (InvisibleSpot spot)= 	(mines,  io_state);
		| NulSpot spot= 	FinalSpotRevealed safe_mines rev_io_state;
	= 	FinalSpotRevealed spot_mines spot_io_state;
		where {
		spot	 =: GetSpot pos minefield;

		end_game =: ChangeIOState [DrawInWindow WindowID draw_final_game,
								  ChangeUpdateFunction WindowID DrawFinalGame,
								  DisableTimer TimerID,
								  DisableMouse WindowID] io_state;
		(mines`, draw_final_game)=: DrawFinalGame [] mines;

		spot_mines=: (spot_minefield,one_more_visible,one_pebble_less,one_more_mine,dim,time,best,font, seed);
		one_more_visible=: nr_vis + (1 + (nr_pebbles_less - nr_pebbles));
		one_more_mine	=: nr_mines + (nr_pebbles - nr_pebbles_less);
		one_pebble_less	=: RemovePebble pos pebbles;
		spot_io_state	=: DrawInWindow WindowID [DrawSpot spot` pos] io_state;
		(spot`, spot_minefield)=: RevealSpot pos minefield;

		safe_mines	=: (safe_minefield, more_visible, less_pebbles, more_mines, dim, time, best, font, seed);
		more_visible=: nr_vis + (nr_revealed_spots - (nr_pebbles - nr_less_pebbles));
		more_mines	=:  nr_pebbles + nr_mines  - nr_less_pebbles;
		nr_pebbles	=: Length_new pebbles;
		(less_pebbles, safe_minefield, nr_revealed_spots, rev_io_state)=: 
							RevealSafeSpots dim pos pebbles minefield io_state;
		nr_pebbles_less=:(Length_new one_pebble_less);
		nr_less_pebbles=:(Length_new less_pebbles);
		};
StepCautiously pos mines=:(minefield, nr_visible, pebbles, nr_mines, dim, Off, best, font, seed) io_state
	= 	StepCautiously pos mines` io_state`;
		where {
		mines`	 =: (minefield, nr_visible, pebbles, nr_mines, dim, Running 0, best, font, seed);
		io_state`=: EnableTimer TimerID io_state;
		};

FinalSpotRevealed	:: Mines IO -> (Mines, IO);
FinalSpotRevealed mines=: ((minefield, nr_visible, pebbles, nr_mines, dim=: (col, row), time, best, font, seed)) io_state| nr_mines == 0 && nr_visible ==  col * row  = 	CheckForBestTime mines` io_state`;
	= 	(mines, io_state``);
		where {
		(mines`, draw_final_game)=: DrawFinalGame [] mines;
		io_state` =: ChangeIOState [DrawInWindow WindowID draw_final_game,
									ChangeUpdateFunction WindowID DrawFinalGame,
									DisableTimer TimerID,
									DisableMouse WindowID] io_state;
		io_state``=: DrawInWindow WindowID [DrawNrMines font nr_mines dim] io_state;
		};

DrawFinalGame	:: UpdateArea Mines -> (Mines, [DrawFunction]);
DrawFinalGame upd_area mines=:(minefield, nr_visible, pebbles, nr_mines, dim, time, best, font, seed)
	= 	(mines, [SetFont	 font,
				 DrawGrid					dim,
				 DrawNrMines font nr_mines	dim,
				 DrawTime	 font (GetTime time) dim :
				 Concat (MapMinefield minefield DrawAnySpot)
						(MapMinefield minefield (DrawCorrectnessPebble pebbles))]);

CheckForBestTime	:: Mines IO -> (Mines, IO);
CheckForBestTime state=:(mf, nv, pbls, nm, dim, time, (files, best), ft, seed) io
	| ItsABestTime (Length_new pbls) dim time best = 	OpenModalDialog dialog state io;
	= 	(state, io);
		where {
		dialog=: CommandDialog OverDlogID "Game Over" [] OverOKID [st1,st2,et,ok];
		st1	=: StaticText 1 Left "Game Over with a new best time!";
		st2	=: StaticText 2 Left  "Your name:";
		et	=: EditText NameID (RightTo 2) (MM 40.0) 1 "";
		ok	=: DialogButton OverOKID Center "OK" Able OverOK;
		};

OverOK	:: DialogInfo Mines IO -> (Mines, IO);
OverOK info state=:(mf, nv, pbls, nrm, dim, time, (files, best), ft, seed) io
	| name == "" = 	(state, io`);
	= 	((mf, nv, pbls, nrm, dim, time, (files, best`), ft, seed), io``);
		where {
		best`=: AddBestTime (LimitString 16 name) (Length_new pbls) dim time best;
		io`` =: DrawInWindow BestWdID [ShowBestTimes best`] io`;
		io`  =: CloseActiveDialog io;
		name =: GetEditText NameID info;
		};

/*	The update definition:
*/

DrawGame	:: UpdateArea Mines -> (Mines, [DrawFunction]);
DrawGame upd_area mines=:(minefield, nr_visible, pebbles, nr_mines, dim, time, best, font, seed)
	= 	(mines, [SetFont	 					font,
				 DrawGrid						dim,
				 DrawNrMines font nr_mines		dim,
				 DrawTime font (GetTime time)	dim :
				 	Concat (MapMinefield minefield DrawSpot) (Map pebbles DrawPebble)]);

// Erase_and_DrawGame	:: UpdateArea Mines -> (Mines, [DrawFunction]);
Erase_and_DrawGame [area:rest] mines =  (mines`, [EraseRectangle area : rest`]);
		where {
		(mines`, rest`)=: Erase_and_DrawGame rest mines;
		};
Erase_and_DrawGame area mines =  DrawGame area mines;

GetTime	:: Time -> Int;
GetTime (Running time)	=  time;
GetTime Off				=  0;


/*	The timer definition:
*/


Timing	:: TimerState Mines IO -> (Mines, IO);
Timing passed mines=: ((minefield, nr_visible, pebbles, nr_mines, dim, Running time, best, font, seed)) io_state= 	(mines`, io_state`);
		where {
		mines`	 =: (minefield, nr_visible, pebbles, nr_mines, dim, Running now, best, font, seed);
		now		 =: time + passed;
		io_state`=: DrawInWindow WindowID [DrawTime font now dim] io_state;
		};
Timing passed mines io_state
	= 	(mines, io_state);


/*	Reveal all newly discovered spots:
*/

RevealSafeSpots	:: Dimension Position Pebbles Minefield IO
	->	(Pebbles, Minefield, Int, !IO);
RevealSafeSpots dim pos pebbles minefield io_state
	= 	(pebbles``, minefield``, AreaLength revealed_spots, io_state``);
		where {
		(pebbles``, minefield``, revealed_spots, io_state``)=:
			RevealAreas dim pos Compass pebbles` minefield` [[(pos,spot)]] io_state`;
		(spot, minefield`)=: RevealSpot pos minefield;
		pebbles`		  =: RemovePebble pos pebbles;
		io_state`		  =: DrawInWindow WindowID [DrawSpot spot pos] io_state;
		};

RevealAreas	:: Dimension Position [Vector] Pebbles Minefield [[(Position, Spot)]] IO
	->	(Pebbles, Minefield, [[(Position, Spot)]], !IO);
RevealAreas dim pos [v : vs] pebbles minefield done io_state
	= 	RevealAreas dim pos vs pebbles` minefield` done` io_state`;
		where {
		pos`=: TranslatePoint v pos;
		(pebbles`, minefield`, done`, io_state`)=:
			RevealArea dim pos` v pebbles minefield done io_state;
		};
RevealAreas dim pos [] pebbles minefield done io_state =  (pebbles, minefield, done, io_state);

RevealArea	:: Dimension Position Vector Pebbles Minefield [[(Position, Spot)]] IO
	->	(Pebbles, Minefield, [[(Position, Spot)]], !IO);
RevealArea dim pos v pebbles minefield done io_state
							| not (InMinefield dim pos)
																	|| not (UniqueArea pos done)
																	|| not (InvisibleSpot spot`)= 	(pebbles, minefield, done, io_state);
		| NulSpot spot= 	RevealAreas dim pos vs pebbles` minefield` done` io_state`;
	= 	(pebbles`, minefield`, done`, io_state`);
		where {
		done`			  =: AddArea pos spot done;
		pebbles`		  =: RemovePebble pos pebbles;
		(spot, minefield`)=: RevealSpot pos minefield;
		io_state`		  =: DrawInWindow WindowID [DrawSpot spot pos] io_state;
		vs	 =: NextCompass v;
		spot`=: GetSpot pos minefield;
		};

     

	Compass :== [(-1,-1), (0,-1), (1,-1), (1,0), (1,1), (0,1), (-1,1), (-1,0)];


    

NextCompass	:: Vector -> [Vector];
NextCompass v=:(0,y) =  [v,(y,0),(Minus y,0),(1,y),(-1,y)];
NextCompass v=:(x,0) =  [v,(0,x),(0,Minus x),(x,1),(x,-1)];
NextCompass v=:(x,y) =  [v,(x,0),(0,y),(x,Minus y),(Minus x,y)];

InMinefield	:: Dimension Position -> Bool;
InMinefield dim (0,y) =  False;
InMinefield dim (x,0) =  False;
InMinefield dim=:(col, row) (x,y)
		| x <= col= 	y <= row;
	= 	False;

AddArea	:: Position Spot [[(Position, Spot)]] -> [[(Position, Spot)]];
AddArea pos=:(x,y) spot [col_areas=:[area=:((x`,y`), spot`) : areas_x] : areas]
			| x < x`= 	[[(pos, spot)], col_areas : areas];
		| x == x`= 	[AddColArea pos spot col_areas : areas];
	= 	[col_areas : AddArea pos spot areas];
AddArea pos spot [[] : areas] =  AddArea pos spot areas;
AddArea pos spot [] =  [[(pos, spot)]];

AddColArea	:: Position Spot [(Position, Spot)] -> [(Position, Spot)];
AddColArea pos=:(x,y) spot [area=:((x`,y`), spot`) : areas]
		| y < y`= 	[(pos, spot), area : areas];
					| y == y`= 	[area : areas];
	= 	[area : AddColArea pos spot areas];
AddColArea pos spot [] =  [(pos, spot)];

AreaLength	:: [[x]] -> Int;
AreaLength [list : lists] =   Length_new list  +  AreaLength lists ;
AreaLength [] =  0;


/*	Utilities:
*/

WindowSize	:: PictureDomain -> (Int,Int);
WindowSize (min, max) =  max;

TextToNumber	:: String Int -> Int;
TextToNumber "" n =  n	;
TextToNumber s	n =  
		if is_digit number 0;
		where {
		number		 =: TextToNumber (s % (1, dec (size s))) ( 10 * n  + d);
		(is_digit, d)=: Digit (s.[0]);
		};

Digit	:: Char -> (Bool, Int);
Digit '0'  =  (True, 0)   ; 	  Digit '1' =  (True, 1)	;
Digit '2'  =  (True, 2)   ; 	  Digit '3' =  (True, 3)	;
Digit '4'  =  (True, 4)   ; 	  Digit '5' =  (True, 5)	;
Digit '6'  =  (True, 6)   ; 	  Digit '7' =  (True, 7)	;
Digit '8'  =  (True, 8)   ; 	  Digit '9' =  (True, 9)	;
Digit c	=  (False,0)	;

Between	:: Int Int Int -> Int;
Between min max n
			| n < min= 	min;
			| n > max= 	max;
	= 	n;

Length_new	:: [x] -> Int;
Length_new [] =  0	;
Length_new [x : xs] =  inc (Length_new xs);

Concat	:: [x] [x] -> [x];
Concat [x : xs] ys =  [x : Concat xs ys];
Concat [] ys =  ys;

Unique	:: Position [Position] -> Bool;
Unique mine [] =  True;
Unique mine=:(x,y) [(x`,y`) : mines]
		| x == x` && y == y`= 	False;
	= 	Unique mine mines;

UniqueArea	:: Position [[(Position, Spot)]] -> Bool;
UniqueArea mine=:(x,y) [col_areas=:[area=:((x`,y`), spot) : areas_x] : areas]
								| x < x`= 	True;
		| x == x`= 	UniqueColArea mine col_areas;
	= 	UniqueArea mine areas;
UniqueArea mine [[] : areas] =  UniqueArea mine areas;
UniqueArea mine [] =  True;

UniqueColArea	:: Position [(Position, Spot)] -> Bool;
UniqueColArea mine=:(x,y) [((x`,y`), spot) : areas]
		| y < y`= 	True;
		| y == y`= 	False;
	= 	UniqueColArea mine areas;
UniqueColArea mine [] =  True;


/*	Common Map:
*/

Map	:: [x] (x -> y) -> [y];
Map [x : xs] f =  [f x : Map xs f];
Map [] f =  [];

/*	Uncommon Map:
*/

MapMinefield	:: Minefield DrawSpotFunction -> [DrawFunction];
MapMinefield minefield f =  MapMinefieldToDrawFunctions minefield (1,1) f;

MapMinefieldToDrawFunctions	:: Minefield Position DrawSpotFunction -> [DrawFunction];
MapMinefieldToDrawFunctions [col_mines : minefield] pos f
	= 	MapColMinefieldToDrawFunctions col_mines minefield pos f;
MapMinefieldToDrawFunctions [] pos f =  [];

MapColMinefieldToDrawFunctions	:: [Spot] Minefield Position DrawSpotFunction -> [DrawFunction];
MapColMinefieldToDrawFunctions [spot : spots] minefield pos=:(col, row) f
	= 	[f spot pos : MapColMinefieldToDrawFunctions spots minefield (col, inc row) f];
MapColMinefieldToDrawFunctions [] minefield (col, row) f
	= 	MapMinefieldToDrawFunctions minefield (inc col, 1) f;
