/**
 *  Classroom Scheduler
 *  Copyright (C) 2004 Colin Archibald, Ph.D.
 *  https://sourceforge.net/projects/cr-scheduler/
 *
 *  Licensed under the Academic Free License version 2.0
 */

package panels;

import java.awt.*;
import java.awt.event.*;
import java.awt.print.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.table.*;

import resources.*;
import application.*;

/**
 * A panel on the tabbed pane to allow the user to look at the schedule
 * in different formats.
 */

public class ViewSchedulePanel extends JPanel implements Observer {
    private Schedule schedule = Schedule.getSchedule();
    
    // Table view
    private JTable schedTable;
    private ScheduleTableModel scheduleTableModel;
    private JScrollPane tableScrollPane;
    
    // Time chart view
    private ViewGraphSched viewGraphSched = new ViewGraphSched(null);
    private JScrollPane chartScrollPane;
    
    // CardLayout to flip between them
    private JPanel leftPanel;
    private CardLayout cardLayout;
    
    private ButtonGroup tableOrChart;
    private JRadioButton tableButton, chartButton;
    
    private ButtonGroup whatToShow;
    private JRadioButton showAll, showRoom, showProfessor;
    
    private JComboBox roomCB, professorCB;
    private JButton printView, printAll;
    
    private GridBagConstraints gbc = startingGBC();
    private TitledBorder titledBorder;
    private Border border = BorderFactory.createLoweredBevelBorder();
    
    
    public ViewSchedulePanel() {
        makeWidgets();
        doTheLayout();
        addToolTips();
        addListeners();
    }

    public void update(Observable o, Object arg) {
        if (tableButton.isSelected())
            scheduleTableModel.fireTableDataChanged();  // update the table
        
        // update the comboboxes
        TableColumn profColumn = schedTable.getColumnModel().getColumn(3);
        
        professorCB.removeAllItems();
        Iterator it = schedule.getProfessors().iterator();
        while(it.hasNext())
            professorCB.addItem(it.next());
        
        roomCB.removeAllItems();
        it = schedule.getClassrooms().iterator();
        while(it.hasNext())
            roomCB.addItem(it.next());
        
    }
    
    private void makeWidgets() {
        scheduleTableModel = new ScheduleTableModel();
        TableSorter sorter = new TableSorter(scheduleTableModel);
        schedTable = new JTable(sorter);
        sorter.addMouseListenerToHeaderInTable(schedTable);
        
        schedTable.setPreferredScrollableViewportSize(new Dimension(450, 700));
        
        tableScrollPane = new JScrollPane(schedTable,
        ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,
        ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        
        chartScrollPane = new JScrollPane(viewGraphSched,
        ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,
        ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        
        chartScrollPane.setPreferredSize(new Dimension(450, 700));
        
        tableOrChart = new ButtonGroup();
        tableButton = new JRadioButton("Table", true);
        chartButton = new JRadioButton("Time Chart", false);
        
        tableOrChart.add(tableButton);
        tableOrChart.add(chartButton);
        
        whatToShow = new ButtonGroup();
        showAll = new JRadioButton("All", true);
        showRoom = new JRadioButton("Room", false);
        showProfessor = new JRadioButton("Professor", false);
        whatToShow.add(showAll);
        whatToShow.add(showRoom);
        whatToShow.add(showProfessor);
        
        roomCB = new JComboBox();
        professorCB = new JComboBox();
        
        printView = new JButton(" Print This View ");
        printAll = new JButton(" Print All Similar Views ");
        
        Dimension buttonSize = printView.getPreferredSize();
        buttonSize.setSize(90.0, buttonSize.getHeight());
        printView.setSize(buttonSize);
        printAll.setSize(buttonSize);
        Border border = BorderFactory.createRaisedBevelBorder();
        printView.setBorder(border);
        printAll.setBorder(border);
    }
    
    private void doTheLayout() {
        JPanel rightPanel = new JPanel();
        leftPanel = new JPanel();
        
        GridBagLayout rightPanelGridBag = new GridBagLayout();
        rightPanel.setLayout(rightPanelGridBag);
        
        JPanel tableChartPanel = new JPanel(); // the table/chart radiobuttons
        makeTableChartPanel(tableChartPanel);
        
        JPanel selectivePanel = new JPanel(); // the selective view panel
        makeSelectivePanel(selectivePanel);
        
        JPanel printPanel = new JPanel();	// the print buttons
        makePrintPanel(printPanel);
        
        gbc = startingGBC();   // add 3 inner panels to the right panel
        gbc.ipady = 15;		// vertical spaces
        gbc.insets = new Insets(10, 10, 10, 10); 	// space around components
        gbc.anchor = GridBagConstraints.SOUTHWEST;
        
        rightPanel.add(tableChartPanel, gbc);
        
        gbc.gridheight = 3;
        gbc.anchor = GridBagConstraints.WEST;
        
        rightPanel.add(selectivePanel, gbc);
        
        gbc.gridwidth = GridBagConstraints.RELATIVE;
        gbc.gridheight = 1;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        
        rightPanel.add(printPanel, gbc);
        
        // the leftPanel will be a CardLayout with the table and the chart
        
        cardLayout = new CardLayout();
        leftPanel.setLayout(cardLayout);
        
        cardLayout.addLayoutComponent(tableScrollPane, "Table");  // give the views names
        cardLayout.addLayoutComponent(chartScrollPane, "Chart");
        cardLayout.show(leftPanel,"Table");
        
        leftPanel.add(tableScrollPane, "Table");
        leftPanel.add(chartScrollPane, "Chart");
        
        // now add the left and right panels to the top level panel
        GridBagLayout gridBag = new GridBagLayout();
        
        setLayout(gridBag);
        
        gbc = new GridBagConstraints();
        gbc.weightx = 6.0;  // allow components to resize?
        gbc.weighty = 1.0;
        gbc.insets = new Insets(7, 3, 7, 3);
        
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 5;
        gbc.gridheight = GridBagConstraints.REMAINDER;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.anchor = GridBagConstraints.WEST;	// move left against table
        
        add(leftPanel, gbc);
        cardLayout.show(leftPanel,"Table");
        
        gbc.weightx = 1.0;
        gbc.gridx = 5;
        gbc.gridwidth = 1;
        
        add(rightPanel, gbc);
    }
    
    private void makeSelectivePanel(JPanel selectivePanel) {
        titledBorder = BorderFactory.createTitledBorder(border, " Selective View ");
        selectivePanel.setBorder(titledBorder);
        
        selectivePanel.setLayout(new GridBagLayout());
        
        gbc.ipady = 0;		// vertical spaces
        selectivePanel.add(showAll, gbc);
        selectivePanel.add(showRoom, gbc);
        selectivePanel.add(showProfessor, gbc);
        
        // pick a room
        gbc.ipady = 10;		// vertical spaces
        
        JLabel label = new JLabel("Choose a classroom:");
        selectivePanel.add(label, gbc);
        gbc.ipady = 0;		// vertical spaces
        selectivePanel.add(roomCB, gbc);
        
        // pick a prof
        gbc.ipady = 2;		// vertical spaces
        label = new JLabel("Choose a professor:");
        selectivePanel.add(label, gbc);
        gbc.ipady = 0;		// vertical spaces
        selectivePanel.add(professorCB, gbc);
    }
    
    private void makeTableChartPanel(JPanel tableChartPanel) {
        titledBorder = BorderFactory.createTitledBorder(border, " View As ");
        tableChartPanel.setBorder(titledBorder);
        
        tableChartPanel.setLayout(new GridBagLayout());
        gbc = startingGBC();
        
        tableChartPanel.add(tableButton, gbc);
        tableChartPanel.add(chartButton, gbc);
    }
    
    private void makePrintPanel(JPanel printPanel) {
        titledBorder = BorderFactory.createTitledBorder(border, " Print ");
        printPanel.setBorder(titledBorder);
        
        printPanel.setLayout(new GridBagLayout());
        gbc = startingGBC();
        
        gbc.insets = new Insets(5,8,5,8);
        printPanel.add(printView, gbc);
        printPanel.add(printAll, gbc);
    }
    
    
    // These constraints are used for interior panels that are generally
    // using a vertical layout.  The constratins object needs to be modified
    // to allow for different sized components to be added, ie, change the weightx
    
    protected static GridBagConstraints startingGBC() {
        GridBagConstraints gbc = new GridBagConstraints();
        
        gbc.weightx = 0.0;
        gbc.gridheight = 1;
        gbc.ipady = 0;		// vertical spaces  was 2
        gbc.ipadx = 0;
        gbc.gridwidth = GridBagConstraints.REMAINDER;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        
        return gbc;
    }
    
    private void addToolTips() {
        
        
    }
    
    private void addListeners() {
        ListenToWhatToShow whatToShowListener = new ListenToWhatToShow();
        
        tableButton.addActionListener(whatToShowListener);
        chartButton.addActionListener(whatToShowListener);
        showAll.addActionListener(whatToShowListener);
        showRoom.addActionListener(whatToShowListener);
        showProfessor.addActionListener(whatToShowListener);
        roomCB.addActionListener(whatToShowListener);
        professorCB.addActionListener(whatToShowListener);
        printView.addActionListener(whatToShowListener);
        printAll.addActionListener(whatToShowListener);
    }
    
    // this is a helper for the actionListener
    
    private void printTheView(Pageable toPrint, boolean confirm) {
        
        PrinterJob printJob = PrinterJob.getPrinterJob();  // Get a printing object
        printJob.setPageable(toPrint);    // The schedule is printable
        
        if (confirm) {
            if (printJob.printDialog())   // Display print dialog
            {                                        // If true is returned...
                try {
                    printJob.print();                // then print
                }
                catch(PrinterException pe) {
                    System.out.println(pe);
                    JOptionPane.showMessageDialog(ClassroomScheduler.getFrame(),
                    "Error printing a schedule.",
                    "Unknown Printer Error",
                    JOptionPane.ERROR_MESSAGE);
                }
            }
        } else {
            try {
                printJob.print();                // then print
            }
            catch(PrinterException pe) {
                System.out.println(pe);
                JOptionPane.showMessageDialog(ClassroomScheduler.getFrame(),
                "Error printing a schedule.", "Unknown Printer Error",
                JOptionPane.ERROR_MESSAGE);
            }
        }
    }
    
    class ListenToWhatToShow implements ActionListener {
        
        public void actionPerformed(ActionEvent ae) {
            if (ae.getActionCommand() == "Table") {
                cardLayout.show(leftPanel,"Table");
                showAll.setEnabled(true);
                showAll.setSelected(true);
                roomCB.setEnabled(false);
                professorCB.setEnabled(false);
                printAll.setEnabled(false);
            }
            
            if (ae.getActionCommand() == "Time Chart") {
                cardLayout.show(leftPanel, "Chart");
                showAll.setEnabled(false);
                
                // select the room radio button in second panel
                showRoom.setSelected(true);
                roomCB.setEnabled(true);
                professorCB.setEnabled(false);
                viewGraphSched.setViewRoom((Classroom) (roomCB.getSelectedItem()));
                printAll.setEnabled(true);
            }
            
            if (ae.getActionCommand() == "All") {
                roomCB.setEnabled(false);
                professorCB.setEnabled(false);
            }
            
            if (ae.getActionCommand() == "Room") {
                roomCB.setEnabled(true);
                professorCB.setEnabled(false);
            }
            
            if (ae.getActionCommand() == "Professor") {
                roomCB.setEnabled(false);
                professorCB.setEnabled(true);
            }
            
            // if its a graphical view
            if (chartButton.isSelected()) {
                if (ae.getActionCommand() == "Professor")
                    viewGraphSched.setViewProf((Professor) (professorCB.getSelectedItem()));
                if (ae.getActionCommand() == "Room")
                    viewGraphSched.setViewRoom((Classroom) (roomCB.getSelectedItem()));
                if (ae.getSource() == roomCB)
                    viewGraphSched.setViewRoom((Classroom) (roomCB.getSelectedItem()));
                if (ae.getSource() == professorCB)
                    viewGraphSched.setViewProf((Professor) (professorCB.getSelectedItem()));
            }
            
            if (ae.getActionCommand() == " Print This View ") {
                if (chartButton.isSelected()) {
                    printTheView(viewGraphSched, true);
                } else {
                    printTheView(schedule, true);
                }
            }
            
            if (ae.getActionCommand() == " Print All Similar Views ") {
                if (!chartButton.isSelected())  // only for charts
                    return;
                
                if (showRoom.isSelected()) {
                    System.out.println("Print all the rooms");
                    
                    Iterator it = schedule.getClassrooms().iterator();
                    while (it.hasNext()) {
                        viewGraphSched.setViewRoom((Classroom)(it.next()));
                        printTheView(viewGraphSched, false);
                        
                    }
                    
                }
                
                if (showProfessor.isSelected()) {
                    System.out.println("Print all the full time profs");
                    
                    Iterator it = schedule.getProfessors().iterator();
                    while (it.hasNext()) {
                        Professor p = (Professor)(it.next());
                        if ( p.getStatus() == Professor.FULL_TIME) {
                            viewGraphSched.setViewProf(p);
                            printTheView(viewGraphSched, false);
                        }
                    }
                    
                }
                
            }
        }
    }
}
