/* REXX PROGRAM */
/* This REXX program expects CHECK+ to be the host environment */
/* It modifies an Ascii Import file to make it suitable for importing to Check+. */
/* If you are attempting to change this file, you will normally only need to change the CHANGE_FIELD_DATA subroutine. */
/* There are three external functions that Check+ provides:
		SetHeaderLine(headerLine) - tells Check+ to expect each data line to have this format.
		SetRexxVariables(line) - causes Check+ to parse the line and, for each field, set a rexx variable with the data from the line.
		AddTransaction() - causes Check+ to add a transaction using the data in the rexx variables.
		*/
/* The general outline is to load the header line and call SetHeaderLine to initialize Check+.
	Then, for each line in the input file:
		Read the line
		Call SetRexxVariables which loads rexx variables with the data from the line.
			(For each field you are importing Check+ will add a Rexx variable with the same
			name as the field and will assign it the data from the corresponding field in the line.)
		Modify the variables if desired.
		Call AddTransaction to tell Check+ to add a new transaction using the modified variables.
		*/
		


/* MAIN SUBROUTINE */
szEnvironment = address()
IF szEnvironment <> "Check+" THEN DO
	SAY "This rexx file can only be run by Check+, not directly."
	RETURN 1
	END

/* Load the Rexx variables based on the arguments the user passed in. */
FileIn = ARG(1)
HeaderFile = ARG(2)
Delimiter = ARG(3)
/* Make sure the required arguments where given. */
IF FileIn = "" THEN DO
	SAY "The following arguments are expected: FILEIN HEADERFILE DELIMITER"
	SAY "    FILEIN is required."
	SAY "    HEADERFILE will default to FILEIN if not given."
	SAY "    DELIMITER will default to ',' if not given."
	RETURN 1
	END
/* Set the default header file to the input file and the default delimiter to ','. */
IF HeaderFile = "" THEN HeaderFile = FileIn
IF Delimiter = "" THEN Delimiter = ','
/* Load the Header Line and pass it to Check+ in order to initialize the import. */
headerLine = LINEIN(HeaderFile)
SetHeaderLine(headerLine)
/* For each line in the input file, load the line, loads the Rexx variables, modify the Rexx variables, and tell Check+ to add the transaction. */
DO while LINES(FileIn)
	line = LINEIN(FileIn)
	SetRexxVariables(line)
	CALL CHANGE_FIELD_DATA
	AddTransaction()
	end
/* Close all the files. */
result = stream(FileOut, 'c', 'close')
result = stream(FileIn, 'c', 'close')
IF HeaderFile \= FileIn THEN stream(HeaderFile, 'c', 'close')
RETURN 0


/* SUBROUTINE */
CHANGE_FIELD_DATA:
/* Change fields to a format suitable for Check+ import. */
/* This subroutine is usually the only portion that the user needs to change. */

/* The following instructions are examples of some common changes: */

/* Change the type to a type compatible with Check+.  Setting type to "" causes it to default to a Computer Check or Hand Check. */
/*
if type = "Receive"
	THEN type = "Deposit"
	ELSE type = ""
*/

/* If the allocation amount is negative - make it positive. */
/*
IF allocationAmount < 0 THEN allocationAmount = 0 - allocationAmount
*/

/* Change dateCleared and datePrinted to a compatible format.  Check+ expects a date to indicate cleared or printed. */
/*
IF dateCleared = "R" THEN dateCleared = date
IF datePrinted = "P" THEN datePrinted = date
*/

RETURN

