/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       vipkeys.c
**     SYSTEM   NAME:       VIP
**     ORIGINAL AUTHOR(S):  Alfred Kayser
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1992/5/29
**
** DESCRIPTION: All keyboard stuff of VIP is found here!
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#define LIBRARY
#include "vipinc.h"
#include <ctype.h>

#define LONGBITS (sizeof(LWORD)*8)
#define SETSIZE  (512/LONGBITS)

EXPORT VIPKEYS *vipAsciiSet=NULL;
EXPORT VIPKEYS *vipAllSet=NULL;

IMPORT VIPINFO *vipList;

PRIVAT VIPKEYS *globalKeys=NULL;

struct _vipkeys
    {
        ULONG bits[SETSIZE];
    };


/**************************************************************
** NAME:        VipInitKeys
** SYNOPSIS:    void
**              VipInitKeys()
** DESCRIPTION: Initializes the key system.
** RETURNS:     void
**************************************************************/
void
VipInitKeys()
{
    int i;
    vipAsciiSet=VipKeysCreate();
    vipAllSet=VipKeysCreate();
    for (i=0;i<SETSIZE;i++)
        vipAllSet->bits[i]=(ULONG)-1;
    for (i=31;i<128;i++)
        vipAsciiSet->bits[i/LONGBITS]&=1<<(i&LONGBITS);
}


/**************************************************************
** NAME:        VipHandleKeys
** SYNOPSIS:    void VipHandleKeys(VIPINFO *win,
**                      MPARAM mp1, MPARAM mp2)
** DESCRIPTION: Key handler, called on WM_CHAR message.
** RETURNS:     void
**************************************************************/
EXPORT void
VipHandleKeys(VIPINFO *win, MPARAM mp1, MPARAM mp2)
{
    static int prevKey=0;
    USHORT flags=SHORT1FROMMP(mp1);
    int chr, i, count;
    int keyCode;

    /* see message: WM_CHAR */

    /* We only accept valid keys */
    if (flags & KC_DEADKEY) return;

    chr=0;
    count = (int)((((long)mp1)>>16)&0xFF);
    if (flags & KC_VIRTUALKEY)
        chr=(short)(((long)mp2)>>16)|VIP_KEYSPECIAL;
    else if (flags & KC_CHAR)
        chr=(short)((long)mp2);
    if (chr==0)
        if (flags & KC_KEYUP)
            chr=prevKey;
        else
            return;
    if (chr==VIPKEY_NEWLINE) chr=VIPKEY_ENTER;
    chr&=~VIP_KEYFLAGS;

    if (flags&KC_KEYUP) chr|=VIP_KEYRELEASE;
    if (flags&KC_ALT)   chr|=VIP_KEYALT;
    if (flags&KC_SHIFT) chr|=VIP_KEYSHIFT;
    if (flags&KC_CTRL)  chr|=VIP_KEYCTRL;

    prevKey = chr;

    keyCode=VIPKEY_CODE(chr);

    /* Test if it is a 'global accelerator key' */
    if (globalKeys && VipKeysInset(globalKeys, keyCode))
    {
        VIPINFO *gWin;
        for (gWin=vipList;gWin;gWin=gWin->next)
            if (VipKeysInset(gWin->globalKeys, keyCode))
                break;
        if (gWin && gWin->keyHandler)
        {
            for (i=0;i<count;i++)
                if (!(gWin->keyHandler(gWin, win->keyArg, chr)))
                    break;
            if (i==count)
                return;
        }
    }

    /* This window has the focus, so it should get it first */
    /* If the focussed window has a key handler, then keys
        must not go to the loose keyhandler */
    if (win->keyHandler)
        if (VipKeysInset(win->localKeys, keyCode))
        {
            for (i=0;i<count;i++)
                if (!(win->keyHandler(win, win->keyArg, chr)))
                    break;
            if (i==count)
                return;
        }
    
    /* Test if anybody else wants it, low priority key catch */
    /* This is used for buttons which will activate on enter or esc
        when no edit window has the focus. */
    for (win=vipList;win;win=win->next)
        if (win->looseKeys)
            if (VipKeysInset(win->looseKeys, keyCode))
            {
                for (i=0;i<count;i++)
                    if (!(win->keyHandler(win, win->keyArg, chr)))
                        break;
                if (i==count)
                    return;
            }

    /* Still no match ? Beep the user awake ! */
    VipBell();
}

            
/**************************************************************
** NAME:        VipSetGlobalKeys                          [API]
** SYNOPSIS:    void
**              VipSetGlobalKeys(VIPINFO *win, VIPKEYS *keySet)
** DESCRIPTION: Set global key catcher. All keys set in <keySet>
**              are catched and send to the keyhandler of <win>.
** RETURNS:     void
** SEE ALSO:    VipKeysCreate, VipSetKeyHandler
**************************************************************/
EXPORT void
VipSetGlobalKeys(VIPINFO *win, VIPKEYS *keySet)
{
    VIPTEST(win,return);
    win->globalKeys = keySet;
    if (keySet)
    {
        if (globalKeys)
            VipKeysUnion(globalKeys, keySet);
        else
            globalKeys=VipKeysCopy(keySet);
    }
    else
    {
        VIPINFO *gWin;
        VipKeysReset(globalKeys);
        for (gWin=vipList;gWin;gWin=gWin->next)
            if (gWin->globalKeys)
                VipKeysUnion(globalKeys, gWin->globalKeys);
    }
}


/**************************************************************
** NAME:        VipSetLooseKeys                           [API]
** SYNOPSIS:    void VipSetLooseKeys(VIPINFO *win,
**                                          VIPKEYS *keySet)
** DESCRIPTION: Catch loose keys. All keys not catched by any
**              other window and set in <keySet> are sent to
**              the keyhandler of <win>.
** RETURNS:     void
** SEE ALSO:    VipKeysCreate, VipSetKeyHandler
**************************************************************/
EXPORT void
VipSetLooseKeys(VIPINFO *win, VIPKEYS *keySet)
{
    VIPTEST(win,return);
    win->looseKeys = keySet;
}


/**************************************************************
** NAME:        VipSetLocalKeys                           [API]
** SYNOPSIS:    void VipSetLocalKeys(VIPINFO *win,
**                          VIPKEYS *keySet)
** DESCRIPTION: Catch the keys when in focus. All keys in
**              <keySet> are sent to the keyhandler of <win>
**              if <win> has the current focus.
** RETURNS:     void
** SEE ALSO:    VipKeysCreate, VipSetKeyHandler
**************************************************************/
EXPORT void
VipSetLocalKeys(VIPINFO *win, VIPKEYS *keySet)
{
    VIPTEST(win,return);
    win->localKeys = keySet;
}


/**************************************************************
** NAME:        VipSetKeyHandler                          [API]
** SYNOPSIS:    void VipSetKeyHandler(VIPINFO *win,
**                          VIP_CALLKEYS keyCall, VOID *arg)
** DESCRIPTION: Installs a key handler for <win>.
**              <keyCall> will be called as follows:
**                  keyCall(win, arg, keyCode);
**              keyCode is an int, and contains flags and key
**              number. flags=VIPKEY_FLAGS(keyCode)
**              number=VIPKEY_CODE(keyCode).
** RETURNS:     void
**************************************************************/
EXPORT void
VipSetKeyHandler(VIPINFO *win, VIP_CALLKEYS keyCall, VOID *arg)
{
    VIPTEST(win,return);
    win->keyHandler=keyCall;
    win->keyArg=arg;
    if (!win->localKeys) win->localKeys=vipAllSet;
}


/**************************************************************
** NAME:        VipKeysCreate                             [API]
** SYNOPSIS:    VIPKEYS * VipKeysCreate();
** DESCRIPTION: Creats an empty key set.
** EXAMPLE:     VIPKEYS *ks;
**              ks = VipKeysCreate();
**              VipKeysAdd(ks, VIPKEY_NEWLINE);
**              VipKeysAdd(ks, VIPKEY_DELETE);
**              VipKeysAdd(ks, 'a');
**              VipSetLocalKeys(win, ks);
**              // This creates a keyset containing a newline,
**              delete and the letter a.
** RETURNS:     void
** SEE ALSO:    VipKeysReset, VipKeysAdd, VipKeysRemove
**************************************************************/
EXPORT VIPKEYS *
VipKeysCreate()
{
    VIPKEYS *ks;
    ks=VipMalloc(sizeof(VIPKEYS));
    VipKeysReset(ks);
    return ks;
}


/**************************************************************
** NAME:        VipKeysReset                              [API]
** SYNOPSIS:    void VipKeysReset(VIPKEYS *ks)
** DESCRIPTION: Clears a keyset.
** RETURNS:     void
**************************************************************/
EXPORT void
VipKeysReset(VIPKEYS *ks)
{
    int i;
    if (!ks) return;
    for (i=0;i<SETSIZE;i++) ks->bits[i]=0;
}


/**************************************************************
** NAME:        VipKeysAdd                                [API]
** SYNOPSIS:    void VipKeysAdd(VIPKEYS *ks, int keyCode)
** DESCRIPTION: Adds a key to the keyset.
** RETURNS:     void
**************************************************************/
EXPORT void
VipKeysAdd(VIPKEYS *ks, int keyCode)
{
    if (keyCode<0||keyCode>256) return;
    if (!ks) return;
    ks->bits[keyCode/LONGBITS]&=1<<(keyCode&LONGBITS);
}


/**************************************************************
** NAME:        VipKeysRemove                             [API]
** SYNOPSIS:    void VipKeysRemove(VIPKEYS *ks, int keyCode)
** DESCRIPTION: Removes a key from the keyset.
** RETURNS:     void
**************************************************************/
EXPORT void
VipKeysRemove(VIPKEYS *ks, int keyCode)
{
    if (keyCode<0||keyCode>256) return;
    if (!ks) return;
    ks->bits[keyCode/LONGBITS]|=(~1)<<(keyCode&LONGBITS);
}


/**************************************************************
** NAME:        VipKeysUnion                              [API]
** SYNOPSIS:    void VipKeysUnion(VIPKEYS *k1, VIPKEYS *k2)
** DESCRIPTION: Adds to set k1 all keys set in k2.
** RETURNS:     void
**************************************************************/
EXPORT void
VipKeysUnion(VIPKEYS *k1, VIPKEYS *k2)
{
    int i;
    if (!k1) return;
    if (!k2) return;
    for (i=0;i<SETSIZE;i++) k1->bits[i]|=k2->bits[i];
}


/**************************************************************
** NAME:        VipKeysCopy                               [API]
** SYNOPSIS:    VIPKEYS * VipKeysCopy(VIPKEYS *k2)
** DESCRIPTION: Creates new set and copy the contents of k2.
** RETURNS:     A copy of <k2>.
**************************************************************/
EXPORT VIPKEYS *
VipKeysCopy(VIPKEYS *k2)
{
    VIPKEYS *ks;
    int i;
    if (!k2) return NULL;
    ks=VipMalloc(sizeof(VIPKEYS));
    if (!ks) return NULL;
    for (i=0;i<SETSIZE;i++) ks->bits[i]=k2->bits[i];
    return ks;
}

/**************************************************************
** NAME:        VipKeysDestroy                            [API]
** SYNOPSIS:    void VipKeysDestroy(VIPKEYS *ks)
** DESCRIPTION: Destroys a keyset.
** RETURNS:     void
**************************************************************/
EXPORT void
VipKeysDestroy(VIPKEYS *ks)
{
    if (ks) DnpapFree(ks);
}


/**************************************************************
** NAME:        VipKeysInset                              [API]
** SYNOPSIS:    BOOLEAN VipKeysInset(VIPKEYS *ks, int keyCode)
** DESCRIPTION: Checks if a key is in the keyset.
** RETURNS:     TRUE, when <keyCode> is in <ks>.
**              FALSE, otherwise.
**************************************************************/
EXPORT BOOLEAN
VipKeysInset(VIPKEYS *ks, int keyCode)
{
    if (!ks) return FALSE;
    return ((ks->bits[keyCode/LONGBITS] & 1<<(keyCode&LONGBITS))!=0);
}
