/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          info.c
** SYSTEM NAME:        beholder
** MODULE NAME:        info\view
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/7/13
** DESCRIPTION:        beholderInfo group: viewer
*************************************************************************/

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <dnpap.h>
#include <maxmem.h>
#include <config.h>
#include <message.h>
#include <snmp.h>
#include <mgmt.h>
#include <vip.h>                                        

#define MAXCOLLECTORS   1000

#define TOTALVAR        6
#define INDEXPOS        (BTNG_MIBLEN+4)

#define INFO_VAROBJ_OFFSET  10

static LONG             iface = 0;

static LONG             collIndex[MAXCOLLECTORS];

static LONG             currIndex = 0;
static LONG             infoIndex = 0;
static BYTE             infoOwner[SNMP_SIZE_BUFCHR];
static WORD             infoOwnerLen;
static LONG             infoSource[] = {1,3,6,1,2,1,2,2,1,1,1};
static VIPTIMER         *infoTimer = NULL;
static VIPINFO          *infoFrame, *mainFrame, *modeFrame, *menuFrame;
static VIPINFO          *graphFrame, *varSysFrame, *varInfoFrame, *varMemFrame;
static VIPINFO          *configList, *memDial;
static LWORD            infoInterval, infoClockOld, infoVarOld[TOTALVAR];
static BOOLEAN          showGraph = FALSE, configMode = TRUE;

static MGMT             *mgmt = NULL;

static SNMP_OBJECT  varObj[] =
{
    {SNMP_PDU_GET, {1,3,6,1,2,1,1,1,0}, 9},                  /* Description */
    {SNMP_PDU_GET, {1,3,6,1,2,1,1,2,0}, 9},                  /* ObjectID */
    {SNMP_PDU_GET, {1,3,6,1,2,1,1,3,0}, 9},                  /* UpTime */
    {SNMP_PDU_GET, {1,3,6,1,2,1,1,4,0}, 9},                  /* Contact */
    {SNMP_PDU_GET, {1,3,6,1,2,1,1,5,0}, 9},                  /* Name */
    {SNMP_PDU_GET, {1,3,6,1,2,1,1,6,0}, 9},                  /* Location */
    {SNMP_PDU_GET, {1,3,6,1,2,1,1,7,0}, 9},                  /* Services */
    {SNMP_PDU_GET, {BTNG_MIB,3,1,0}, BTNG_MIBLEN + 3, SNMP_INTEGER, 0},      /* maxmem */
    {SNMP_PDU_GET, {BTNG_MIB,3,2,0}, BTNG_MIBLEN + 3, SNMP_INTEGER, 0},      /* maxchunk */
    {SNMP_PDU_GET, {BTNG_MIB,3,3,0}, BTNG_MIBLEN + 3, SNMP_INTEGER, 0},      /* maxused */
    {SNMP_PDU_GET, {BTNG_MIB,1,1,1,3,1}, BTNG_MIBLEN + 5, SNMP_COUNTER, 0},  /* LostPkts */
    {SNMP_PDU_GET, {BTNG_MIB,1,1,1,4,1}, BTNG_MIBLEN + 5, SNMP_COUNTER, 0},  /* TotalPkts */
    {SNMP_PDU_GET, {BTNG_MIB,1,1,1,5,1}, BTNG_MIBLEN + 5, SNMP_COUNTER, 0},  /* Discarded */
    {SNMP_PDU_GET, {BTNG_MIB,1,1,1,6,1}, BTNG_MIBLEN + 5, SNMP_COUNTER, 0},  /* Truncated */
    {SNMP_PDU_GET, {BTNG_MIB,1,1,1,8,1}, BTNG_MIBLEN + 5, SNMP_COUNTER, 0},  /* Stacked */
    {SNMP_PDU_GET, {BTNG_MIB,1,1,1,7,1}, BTNG_MIBLEN + 5, SNMP_COUNTER, 0},  /* Buffered */
};


static VOID InfoConfig (VOID);                         /* reads config file */
static VOID InfoInit (VOID);                     /* init viewer & collector */
static VOID infoMenu(VIPINFO *wip, VOID *ptr, int elem);       /* top menu  */
static VOID TimerCall (VIPTIMER *timer, VOID *parm); /* requests data collector */
static VOID ConfigCall (VOID);                 /* frame with all collectors */
static VOID ViewCall (VOID);                   /* frame with data collector */
static VOID QuitCall (VOID);                                /* quits viewer */
static VOID ResetCall (VOID);                              /* resets viewer */
static VOID ModeCall (VOID);                      /* changes text <-> graph */
static VOID DrawConfig (SNMP_OBJECT *obj, WORD ind);
static VOID DrawGraph (VOID);
static VOID DrawText (VOID);
static VOID UpdateGraph (ULONG now);
static VOID UpdateText (ULONG now);
static VOID LineSelect(VIPINFO *wip, VOID *p, INT line);

static VOID SignalHandler(INT signum);

static VOID Usage (VOID);


VOID main (int argc, char *argv[])
{
    int argi;
    INT i;

    if (ConfigInit(argc, argv) == FALSE || ConfigLoad() == FALSE)
        DnpapMessage(DMC_ERROR, INFO_ERROR, "info: ConfigInit failed");

    MemoryInit(8000000L, 32000L);

    if (DnpapSignal(SIGINT, SignalHandler) == SIG_ERR)
        DnpapMessage(DMC_ERROR, INFO_ERROR, "info: signal handler initialization failed");
    
    MessageInit("stdio.out", MSGFILE, MSGVIP, NULL);
    MessageConfig(INFO_ERROR, "Info");

	for (argi = 1; argi < argc; argi++)
	{				 
		if (argv[argi][0] == '-')
        {
			for (i = 1; i < (INT)strlen(argv[argi]); i++)
				switch (argv[argi][i])
				{
				case 'h':
					Usage();
					DnpapExit(0);
				default:
					DnpapMessage(DMC_FATAL, INFO_ERROR, "info: unrecognized option");
					Usage();
					DnpapExit(1);
				}
        }
		else
		{
			if ((iface = atoi(argv[argi])) < 1)
            {
				DnpapMessage(DMC_FATAL, INFO_ERROR, "info: invalid iface number");
				DnpapExit(1);
            }
		}
	}

    if (VipInit (argc, argv) == FALSE)
    {
        DnpapMessage(DMC_ERROR, INFO_ERROR, "info: VipInit failed");
        DnpapExit(1);
    }
    if (MgmtInit() == FALSE)
    {
        DnpapMessage(DMC_ERROR, INFO_ERROR, "info: MgmtInit failed");
        DnpapExit(1);
    }
    InfoConfig();
    if ((mgmt = MgmtOpen (0, MGMT_TYPE_DEFAULT)) == NULL)
    {
        DnpapMessage(DMC_ERROR, INFO_ERROR, "info: MgmtOpen failed");
        DnpapExit(1);
    }
    InfoInit();
    VipCheck(TRUE);
    MgmtClose (mgmt);
    DnpapExit(0);
}


VOID InfoConfig (VOID)
{
    BYTE *p;
    LONG l;

    if (ConfigGetLong("info.interval", &l))
        infoInterval = l;
    else
        infoInterval = 2000L;
    if (iface > 0)
        infoSource[sizeof(infoSource)/sizeof(infoSource[0])-1]=iface;
    else
    if (ConfigGetLong("info.iface",&l))
        infoSource[sizeof(infoSource)/sizeof(infoSource[0])-1]=l;
    else
        infoSource[sizeof(infoSource)/sizeof(infoSource[0])-1]=1;
    if (!ConfigGetString("info.owner", &p) || strlen(p) > SNMP_SIZE_BUFCHR)
        p="monitorBeholderInfo";
    strcpy(infoOwner, p);
    infoOwnerLen = strlen(p);

    return;
}


VOID InfoInit (VOID)
{
    WORD status, index;

    SNMP_OBJECT obj[] =
    {
        {SNMP_PDU_GET, {BTNG_MIB,1,1,1,10,1}, BTNG_MIBLEN + 5},
        {SNMP_PDU_SET, {BTNG_MIB,1,1,1,10,1}, BTNG_MIBLEN + 5, SNMP_INTEGER, SNMP_CREATEREQUEST},
        {SNMP_PDU_SET, {BTNG_MIB,1,1,1,9,1}, BTNG_MIBLEN + 5, SNMP_DISPLAYSTR},
        {SNMP_PDU_SET, {BTNG_MIB,1,1,1,2,1}, BTNG_MIBLEN + 5, SNMP_OBJECTID}
    };                            
                                  
    strcpy (obj[2].Syntax.BufChr, infoOwner);
    obj[2].SyntaxLen = infoOwnerLen;
    memcpy (obj[3].Syntax.BufInt, infoSource, sizeof(infoSource));
    obj[3].SyntaxLen = sizeof(infoSource)/sizeof(infoSource[0]);

    infoFrame = VipOpenFrame (NULL, 450, 370, 545, 625);
    mainFrame = VipOpenFrame (infoFrame, 0, 0, 1000, 1000);
    VipSetSystemMenu(infoFrame, FALSE);

    status = SNMP_NOERROR;
    infoIndex = 0;

    obj[0].Id[INDEXPOS] = 0;

    while (status != SNMP_NOSUCHNAME)
    {
        if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_NEXT, &obj[0], 1, &status, &index, 0, 0) == FALSE)
        {
            DnpapMessage(DMC_ERROR, INFO_ERROR,"info: MgmtRequest failed");
            QuitCall();
            return;
        }
    }

    infoIndex = obj[0].Id[INDEXPOS]+1;

    obj[1].Id[INDEXPOS] = infoIndex;
    obj[2].Id[INDEXPOS] = infoIndex;
    obj[3].Id[INDEXPOS] = infoIndex;

    if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_SET, &obj[1], 3, &status, &index, 0, 0) == FALSE ||
        status != SNMP_NOERROR)
    {
        DnpapMessage(DMC_ERROR, INFO_ERROR,"info: can't initialize");
        QuitCall();
        return;
    }

    currIndex = infoIndex;

    VipShow (infoFrame);
    ConfigCall();

    return;
}


VOID InfoMenu(VIPINFO *wip, VOID *ptr, int elem)
{
    switch(elem)
    {
    case 0: QuitCall(); break;
    case 1:
        configMode = 1 - configMode;
        if (configMode == TRUE)
            ConfigCall();
        else
            ViewCall();
        break;
    case 2: ModeCall(); break;
    case 3: ResetCall();
    }
}


VOID ConfigCall (VOID)
{
    static int configTabs[]={150,600};

    LONG                indexId[]= {BTNG_MIB,1,1,1,1};
    WORD                status, index;
    WORD                ind = 0, IdLen = BTNG_MIBLEN + 4;
    VIPINFO             *configFrame, *configScrol, *configVar, *configText;

    SNMP_OBJECT          obj[] =
    {
        {SNMP_PDU_NEXT, {BTNG_MIB,1,1,1,1}, BTNG_MIBLEN + 4},
        {SNMP_PDU_NEXT, {BTNG_MIB,1,1,1,9}, BTNG_MIBLEN + 4},
        {SNMP_PDU_NEXT, {BTNG_MIB,1,1,1,10}, BTNG_MIBLEN + 4}
    };                            
                                  
    if (infoTimer != NULL)
        VipDestroyTimer (infoTimer);
    VipDestroy(mainFrame);

    /* Open all windows */
    mainFrame = VipOpenFrame (infoFrame, 0, 0, 1000, 1000);
    menuFrame = VipOpenMenu (mainFrame, 0, 0, 1000, 0);
    configFrame = VipOpenFrame (mainFrame, 0, 0, 1000, 1000);

    configScrol = VipOpenVScroll (configFrame, 0, 0, 0, 1000);
    configVar   = VipOpenFrame (configFrame, 0,0, 1000,1000);

    configText = VipOpenText (configVar, 0, 0, 1000, 0);
    configList = VipOpenList (configVar, 0, 0, 1000, 1000);
    VipSetListCallBack(configList, LineSelect, NULL);


    /* Configure all windows */
    VipSetTitle (infoFrame, "info: Configuration Window");

    VipSetMenuType (menuFrame, VIP_HMENU|VIP_FIXMENU);
    VipSetMenuCallBack (menuFrame, InfoMenu);
    VipSetMenuEntry(menuFrame, "~Quit!", 0, 0, NULL);
    VipSetMenuEntry(menuFrame, "~View!", 1, 0, NULL);
    VipSetMenuEntry(menuFrame, NULL,    2, VIP_MENUBAR, NULL);
    VipSetMenuEntry(menuFrame, NULL,    3, VIP_MENUBAR, NULL);

    VipSetTextTabs (configText, 2, configTabs, VIP_TABLINES);
    VipSetTextLine (configText, "Index\tOwner\tStatus", 0, VIP_LEFT);

    VipSetTextTabs (configList, 2, configTabs, VIP_TABLINES);
    VipSetTextScrollbars(configList, configScrol, NULL);

    /* Position all windows */
    VipSetFrameSplit (mainFrame, menuFrame, VIP_HSPLIT|VIP_FIXTOP);
    VipSetFrameSplit (configFrame, configScrol, VIP_VSPLIT);
    VipSetFrameSplit (configVar, configText, VIP_HSPLIT|VIP_FIXTOP);

    VipShow (mainFrame);

    while (MgmtRequest(mgmt, 0, 0, SNMP_PDU_NEXT, obj, 3, &status, &index, 0, 0) == TRUE &&
        status == SNMP_NOERROR &&
        !memcmp(obj[0].Id, indexId, IdLen * sizeof(LONG)))
        DrawConfig (obj, ind++);
    
    return;
}


VOID ViewCall (VOID)
{
    WORD    status, index;

    SNMP_OBJECT obj[] =
    {
        {SNMP_PDU_GET, {BTNG_MIB,1,1,1,10,1}, BTNG_MIBLEN + 5},
        {SNMP_PDU_SET, {BTNG_MIB,1,1,1,10,1}, BTNG_MIBLEN + 5, SNMP_INTEGER, SNMP_VALID}
    };

    VipDestroy(mainFrame);

    mainFrame = VipOpenFrame (infoFrame, 0, 0, 1000, 1000);
    menuFrame = VipOpenMenu (mainFrame, 0, 0, 1000, 0);
    modeFrame = VipOpenFrame (mainFrame, 0, 0, 1000, 1000);
    VipSetMenuType (menuFrame, VIP_HMENU|VIP_FIXMENU);

    VipSetMenuCallBack(menuFrame, InfoMenu);
    VipSetMenuEntry(menuFrame, "~Quit!", 0, 0, NULL);
    VipSetMenuEntry(menuFrame, "~Config!", 1, 0, NULL);
    VipSetMenuEntry(menuFrame, "~Reset!",  3, 0, NULL);

    VipShow (mainFrame);

    if (currIndex > 0)
    {
        obj[0].Id[INDEXPOS] = currIndex;
        obj[1].Id[INDEXPOS] = currIndex;
    }

    if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_GET, &obj[0], 1, &status, &index, 0, 0) == FALSE ||
        status != SNMP_NOERROR)
    {
        DnpapMessage(DMC_ERROR, INFO_ERROR,"info: can't get status");
        QuitCall();
        return;
    }
    if (status == SNMP_NOERROR && obj[0].Syntax.LngInt == SNMP_UNDERCREATION)
    {
        if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_SET, &obj[1], 1, &status, &index, 0, 0) == FALSE ||
            status != SNMP_NOERROR)
        {
            DnpapMessage(DMC_ERROR, INFO_ERROR,"info: can't set status to valid");
            QuitCall();
            return;
        }
    }
    if ((infoTimer = VipCreateTimer (infoInterval, TimerCall, NULL)) == NULL)
    {
        DnpapMessage(DMC_ERROR, INFO_ERROR,"info: can't register timer");
        QuitCall();
        return;
    }

    ResetCall ();
    return;
}


VOID QuitCall (VOID)
{
    WORD    status, index;

    SNMP_OBJECT obj[] =
    {
        {SNMP_PDU_SET, {BTNG_MIB,1,1,1,10,1}, BTNG_MIBLEN + 5, SNMP_INTEGER, SNMP_INVALID}
    };


    if (infoIndex > 0)
    {
        obj[0].Id[INDEXPOS] = infoIndex;

        if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_SET, &obj[0], 1, &status, &index, 0, 0) == FALSE ||
            status != SNMP_NOERROR)
            DnpapMessage(DMC_WARNING, INFO_ERROR,"info: can't set status to invalid");
    }

    VipClose (infoFrame);
}


VOID ResetCall (VOID)
{
    WORD status, index;
    WORD ind;

    VipDestroy (modeFrame);
    modeFrame = VipOpenFrame (mainFrame, 0, 0, 1000, 1000);
    VipSetFrameSplit (mainFrame, menuFrame, VIP_HSPLIT|VIP_FIXTOP);
    if (showGraph == TRUE)
        DrawGraph();
    else
        DrawText();
    for (ind = 0; ind < TOTALVAR; ind++)
        varObj[INFO_VAROBJ_OFFSET+ind].Id[INDEXPOS] = currIndex;
    if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_GET, &varObj[7], TOTALVAR + 3, &status, &index, 0, 0) == FALSE ||
        status != SNMP_NOERROR)
    {
        DnpapMessage(DMC_ERROR, INFO_ERROR,"info: can't get data");
        return;
    }
    for (ind = 0; ind < TOTALVAR; ind++)
        infoVarOld[ind] = varObj[INFO_VAROBJ_OFFSET+ind].Syntax.LngUns;
    infoClockOld = clock()/1000L;
    VipUpdate (menuFrame, TRUE);
    VipShow (modeFrame);
    return;
}


VOID ModeCall (VOID)
{
    showGraph = 1 - showGraph;
    ResetCall ();
    return;
}


VOID TimerCall (VIPTIMER *timer, VOID *parm)
{
    WORD status, index;
    WORD ind;
    LWORD now;

    for (ind = 0; ind < TOTALVAR; ind++)
        varObj[INFO_VAROBJ_OFFSET+ind].Id[INDEXPOS] = currIndex;
    if (MgmtRequest(mgmt, 0, 0, SNMP_PDU_GET, &varObj[0], INFO_VAROBJ_OFFSET, &status, &index, 0, 0) == FALSE ||
        status != SNMP_NOERROR ||
        MgmtRequest(mgmt, 0, 0, SNMP_PDU_GET, &varObj[INFO_VAROBJ_OFFSET], TOTALVAR, &status, &index, 0, 0) == FALSE ||
        status != SNMP_NOERROR)
    {
        DnpapMessage(DMC_ERROR, INFO_ERROR,"info: can't get data");
        return;
    }
    now = clock()/1000L;
    if (showGraph == TRUE)
        UpdateGraph(now);
    else
        UpdateText(now);
    return;
}


VOID UpdateText (ULONG now)
{
    WORD            ind;
    LWORD           current;
    char            str[SNMP_SIZE_BUFCHR], element[10];

    sprintf(str, "...............................................................................................%s", varObj[0].Syntax.BufChr);
    VipSetTextLine (varSysFrame, str, 0, VIP_RIGHT);
    strcpy (str,"..................................................................................................");
    for (ind = 0; ind < varObj[1].SyntaxLen; ind++)
    {
        sprintf(element, "%u", varObj[1].Syntax.BufInt[ind]);
        strcat(str, element);
        if (ind < varObj[1].SyntaxLen-1)
            strcat(str, ".");
    }
    VipSetTextLine (varSysFrame, str, 1, VIP_RIGHT);
    sprintf(str, "........................................................................................................................................................%lu", varObj[2].Syntax.LngUns);
    VipSetTextLine (varSysFrame, str, 2, VIP_RIGHT);
    sprintf(str, ".........................................................................................................................................................%s", varObj[3].Syntax.BufChr);
    VipSetTextLine (varSysFrame, str, 3, VIP_RIGHT);
    sprintf(str, ".........................................................................................................................................................%s", varObj[4].Syntax.BufChr);
    VipSetTextLine (varSysFrame, str, 4, VIP_RIGHT);
    sprintf(str, ".........................................................................................................................................................%s", varObj[5].Syntax.BufChr);
    VipSetTextLine (varSysFrame, str, 5, VIP_RIGHT);
    sprintf(str, "........................................................................................................................................................%ld", varObj[6].Syntax.LngInt);
    VipSetTextLine (varSysFrame, str, 6, VIP_RIGHT);
    sprintf(str, "........................................................................................................................................................%ld", varObj[7].Syntax.LngInt);
    VipSetTextLine (varMemFrame, str, 0, VIP_RIGHT);
    sprintf(str, "........................................................................................................................................................%ld", varObj[8].Syntax.LngInt);
    VipSetTextLine (varMemFrame, str, 1, VIP_RIGHT);
    sprintf(str, "........................................................................................................................................................%ld", varObj[9].Syntax.LngInt);
    VipSetTextLine (varMemFrame, str, 2, VIP_RIGHT);

    for (ind = 0; ind < TOTALVAR-1; ind++)
    {
        current = varObj[INFO_VAROBJ_OFFSET+ind].Syntax.LngUns - infoVarOld[ind];
        sprintf(str, ".............................................................................................................................................%lu", current);
        VipSetTextLine (varInfoFrame, str, ind, VIP_RIGHT);
    }
    current = varObj[INFO_VAROBJ_OFFSET+ind].Syntax.LngUns;
    sprintf(str, "..................................................................................................................................................%lu", current);
    VipSetTextLine (varInfoFrame, str, ind, VIP_RIGHT);
    VipUpdate(varSysFrame, TRUE);
    VipUpdate(varMemFrame, TRUE);
    VipUpdate(varInfoFrame, TRUE);
    return;
}


VOID UpdateGraph (ULONG now)
{
    SHORT ind;
    LWORD current, sec;

    sec = 1 + (now - infoClockOld) / 1000L;
    infoClockOld = now;
    for (ind = 0; ind < TOTALVAR-1; ind++)
    {
        current = varObj[INFO_VAROBJ_OFFSET+ind].Syntax.LngUns - infoVarOld[ind];
        infoVarOld[ind] = varObj[INFO_VAROBJ_OFFSET+ind].Syntax.LngUns;
        VipSetGraphLast (graphFrame, ind, (long) (current/sec));
    }
    current = varObj[INFO_VAROBJ_OFFSET+ind].Syntax.LngUns;
    VipSetGraphLast(graphFrame, ind, (long) current);
    VipSetDialCurrent(memDial, varObj[9].Syntax.LngUns);
    VipUpdate(modeFrame, FALSE);
    return;
}


VOID DrawGraph (VOID)
{
    VipSetTitle (infoFrame, "info: Graph Window");
    VipSetMenuEntry(menuFrame, "~Text!",   2, 0, NULL);
    graphFrame =  VipOpenGraph (modeFrame, 0, 0, 1000, 1000);
        /* Lost, Total, Discarded, Stacked, Buffered */
    memDial = VipOpenDial (modeFrame, 800, 800, 200, 200);
    VipSetTitle (memDial, "MemUsed");
    VipSetDialRange (memDial, 0, varObj[7].Syntax.LngUns, 10);
    VipSetBackground (graphFrame, VIP_WHITE);
    VipSetGraphDataSet (graphFrame, 5, 50, 0, 2000, "BufferedPkts", VIP_DARKRED);
    VipSetGraphDataSet (graphFrame, 4, 50, 0, 2000, "StackedPkts", VIP_BROWN);
    VipSetGraphDataSet (graphFrame, 3, 50, 0, 2000, "TruncatedPkts", VIP_CYAN);
    VipSetGraphDataSet (graphFrame, 2, 50, 0, 2000, "DiscardedPkts", VIP_RED);
    VipSetGraphDataSet (graphFrame, 1, 50, 0, 2000, "TotalPkts", VIP_BLUE);
    VipSetGraphDataSet (graphFrame, 0, 50, 0, 2000, "LostPkts", VIP_PINK);
    return;
}


VOID DrawText (VOID)
{
    WORD ind;
    VIPINFO     *sysFrame, *bInfoFrame, *memFrame;

    VipSetTitle (infoFrame, "info: Text Window");                                                 
    VipSetMenuEntry(menuFrame, "~Graph!",   2, 0, NULL);
    sysFrame    = VipOpenText (modeFrame, 0, 570, 300, 430);
    memFrame    = VipOpenText (modeFrame, 0, 380, 300, 190);
    bInfoFrame  = VipOpenText (modeFrame, 0, 0, 300, 380);
    varSysFrame = VipOpenText (modeFrame, 300, 570, 700, 430);
    varMemFrame = VipOpenText (modeFrame, 300, 380, 700, 190);
    varInfoFrame= VipOpenText (modeFrame, 300, 0, 700, 380);
    VipSetTextLine (sysFrame, "Description....................................................................................................................................................", 0, VIP_LEFT);
    VipSetTextLine (sysFrame, "ObjectID.......................................................................................................................................................", 1, VIP_LEFT);
    VipSetTextLine (sysFrame, "UpTime.........................................................................................................................................................", 2, VIP_LEFT);
    VipSetTextLine (sysFrame, "Contact........................................................................................................................................................", 3, VIP_LEFT);
    VipSetTextLine (sysFrame, "Name...........................................................................................................................................................", 4, VIP_LEFT);
    VipSetTextLine (sysFrame, "Location.......................................................................................................................................................", 5, VIP_LEFT);
    VipSetTextLine (sysFrame, "Services.......................................................................................................................................................", 6, VIP_LEFT);
    VipSetTextLine (memFrame, "Memory Avail...................................................................................................................................................", 0, VIP_LEFT);
    VipSetTextLine (memFrame, "Memory Chunk...................................................................................................................................................", 1, VIP_LEFT);
    VipSetTextLine (memFrame, "Memory Used....................................................................................................................................................", 2, VIP_LEFT);
    VipSetTextLine (bInfoFrame, "LostPkts.....................................................................................................................................................", 0, VIP_LEFT);
    VipSetTextLine (bInfoFrame, "TotalPkts....................................................................................................................................................", 1, VIP_LEFT);
    VipSetTextLine (bInfoFrame, "DiscardedPkts................................................................................................................................................", 2, VIP_LEFT);
    VipSetTextLine (bInfoFrame, "TruncatedPkts................................................................................................................................................", 3, VIP_LEFT);
    VipSetTextLine (bInfoFrame, "StackedPkts..................................................................................................................................................", 4, VIP_LEFT);
    VipSetTextLine (bInfoFrame, "BufferedPkts.................................................................................................................................................", 5, VIP_LEFT);
    for (ind = 0; ind < 7; ind++)
        VipSetTextLine (varSysFrame, "........................................................................................................................................................", ind, VIP_RIGHT);
    for (ind = 0; ind < 3; ind++)
        VipSetTextLine (varMemFrame, ".......................................................................................................................................................0", ind, VIP_RIGHT);
    for (ind = 0; ind < 6; ind++)
        VipSetTextLine (varInfoFrame,  ".....................................................................................................................................................0", ind, VIP_RIGHT);
    return;
}


VOID DrawConfig (SNMP_OBJECT *obj, WORD ind)
{
    BYTE str[SNMP_SIZE_BUFCHR];

    switch (obj[2].Syntax.LngInt)
    {
        case SNMP_INVALID:
            sprintf(str, "%ld\t%s\tINVALID", obj[0].Syntax.LngInt, obj[1].Syntax.BufChr);
            break;
        case SNMP_VALID:
            sprintf(str, "%ld\t%s\tVALID", obj[0].Syntax.LngInt, obj[1].Syntax.BufChr);
            break;
        case SNMP_UNDERCREATION:
            sprintf(str, "%ld\t%s\tUNDERCREATION", obj[0].Syntax.LngInt, obj[1].Syntax.BufChr);
            break;
        default:
            sprintf(str, "%ld\t%s\tUNKNOWN", obj[0].Syntax.LngInt, obj[1].Syntax.BufChr);
    }

    VipSetTextLine (configList, str, ind, VIP_LEFT);

    if (ind < MAXCOLLECTORS)
        collIndex[ind] = obj[0].Id[INDEXPOS];

    return;
}


VOID LineSelect(VIPINFO *wip, VOID *p, INT line)
{
    if (line < MAXCOLLECTORS)
        currIndex = collIndex[line];
    else
        DnpapMessage(DMC_WARNING, STATISTICS_ERROR, "stat: collector index array too small");
}


VOID SignalHandler(INT signum)
{
    QuitCall();
    DnpapExit(0);
}


VOID Usage (VOID)
{
	DnpapMessage(DMC_MESSAGE, INFO_ERROR, "usage: info [-h] iface");
}
