/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          statm.c
** SYSTEM NAME:        beholder
** MODULE NAME:        stat
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/7/14
** DESCRIPTION:        etherStat group: MIB support
**                     each function is preceded by etherStats
**                          Index, DataSource, DropEvents, Octets, Pkts,
**                          BroadcastPkts, MulticastPkts, CRCAlignErrors,
**                          UndersizePkts, OversizePkts, Fragments, Jabbers,
**                          Collisions, 64Octets, 65to127octets,
**                          128to255Octets, 256to511Octets, 512to1023Octets,
**                          1024to1518Octets, Owner, Status
*************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <snmp.h>
#include <mibsup.h>
#include <message.h>
#include <mac.h>

#include "state.h"
#include "statc.h"
#include "stat.h"


#define INDEXSIZE   1


static MIB_LOCAL    *etherStats = NULL;


static BOOLEAN RmonNext (SNMP_OBJECT *Obj, MIB_LOCAL **Local, WORD IdLen, WORD IdSze, void *Elm);



BOOLEAN EtherStatsMInit(VOID)
{
    MessageConfig(STATISTICS_ERROR, "EtherStat");
    return TRUE;
}


/*****************************************************************
** NAME:        RmonNext
** SYNOPSIS:    BOOLEAN RmonNext (SNMP_OBJECT *Obj,
**                                  MIB_LOCAL **Local, WORD IdLen,
**                                  WORD IdSze, VOID **Elm)
** PARAMETERS:  Obj: requested object
**              Local: local datastructure: a collector
**              IdLen: identifier length known to the MIB
**              IdSze: number of indices after the MIB
**              Elm: pointer to object if tables are used
** DESCRIPTION: application specific RMON NEXT function.
**              only called after MibRmon() for next requests 
**              searches the next object in the collector
** REMARKS:     ONLY FOR INTERNAL USE
** RETURNS:     TRUE: object found
**                    OPTIONAL: Elm -> pointer to object in table
*******************************************************************/

static BOOLEAN RmonNext (SNMP_OBJECT *Obj, MIB_LOCAL **Local, WORD IdLen, WORD IdSze, VOID **Elm)
{
    MIB_LOCAL    *local = *Local;

    if (local == NULL)
        return FALSE;
    if (Obj->IdLen == IdLen || local->Index > Obj->Id[IdLen])
    {
        /********************************************************************
        **  OPTIONAL: search function to find first obj in a table !! 
        **  if (not found)
        **      search next collector
        **  adjust object identifier to identifier first object
        **  Elm -> found table obj
        ********************************************************************/
        Obj->IdLen = IdLen + IdSze;
        Obj->Id[IdLen] = local->Index;
        return TRUE;
    }
    /********************************************************************
    **  OPTIONAL: search function to find next obj in a table !! 
    **  if (found)
    **      adjust object identifier to identifier next object
    **      Elm -> found table obj
    **      return TRUE
    ********************************************************************/
    *Local = local->Next;
    return (RmonNext(Obj, Local, IdLen, IdSze, Elm));
}

WORD etherStatsIndex (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;

    if ((local = MibRmon (Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);
    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            Obj->Syntax.LngInt = local->Index;
            return SNMP_NOERROR;
    }
}

WORD etherStatsDataSource (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;
    MAC_IFACE    *iface;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {           
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            memcpy (Obj->Syntax.BufInt, data->Source, data->SourceLen * sizeof (LONG));
            Obj->SyntaxLen = data->SourceLen;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (ETHER_STATS*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            if (data->SourceLen != Obj->SyntaxLen ||
                memcmp(data->Source, Obj->Syntax.BufInt, (data->SourceLen-1) * sizeof (LONG)))
                return SNMP_BADVALUE;
            if ((iface = MacIfaceGet((WORD) Obj->Syntax.BufInt[Obj->SyntaxLen-1])) == NULL)
                return SNMP_BADVALUE;
            memcpy (data->Source, Obj->Syntax.BufInt, Obj->SyntaxLen * sizeof (LONG));
            data->SourceLen = Obj->SyntaxLen;
            data->Iface = iface;
            return SNMP_NOERROR;
    }
}

WORD etherStatsDropEvents (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->DropEvents;
            return SNMP_NOERROR;
    }
}

WORD etherStatsOctets (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->Octets;
            return SNMP_NOERROR;
    }
}
WORD etherStatsPkts (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->Pkts;
            return SNMP_NOERROR;
    }
}
                                                
WORD etherStatsBroadcastPkts (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->BroadcastPkts;
            return SNMP_NOERROR;
    }
}

WORD etherStatsMulticastPkts (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->MulticastPkts;
            return SNMP_NOERROR;
    }
}

WORD etherStatsCRCAlignErrors (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->CRCAlignErrors;
            return SNMP_NOERROR;
    }
}

WORD etherStatsUndersizePkts (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->UndersizePkts;
            return SNMP_NOERROR;
    }
}

WORD etherStatsOversizePkts (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->OversizePkts;
            return SNMP_NOERROR;
    }
}

WORD etherStatsFragments (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->Fragments;
            return SNMP_NOERROR;
    }
}

WORD etherStatsJabbers (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->Jabbers;
            return SNMP_NOERROR;
    }
}

WORD etherStatsCollisions (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->Collisions;
            return SNMP_NOERROR;
    }
}

WORD etherStatsPkts64Octets (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->Pkts64Octets;
            return SNMP_NOERROR;
    }
}

WORD etherStatsPkts65to127Octets (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->Pkts65to127Octets;
            return SNMP_NOERROR;
    }
}

WORD etherStatsPkts128to255Octets (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->Pkts128to255Octets;
            return SNMP_NOERROR;
    }
}

WORD etherStatsPkts256to511Octets (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->Pkts256to511Octets;
            return SNMP_NOERROR;
    }
}

WORD etherStatsPkts512to1023Octets (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->Pkts512to1023Octets;
            return SNMP_NOERROR;
    }
}

WORD etherStatsPkts1024to1518Octets (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngUns = data->Pkts1024to1518Octets;
            return SNMP_NOERROR;
    }
}

WORD etherStatsOwner (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    if ((local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (ETHER_STATS*) local->Data;
            memcpy (Obj->Syntax.BufChr, data->Owner, data->OwnerLen);
            Obj->SyntaxLen = data->OwnerLen;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (ETHER_STATS*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            memcpy (data->Owner, Obj->Syntax.BufChr, Obj->SyntaxLen);
            data->OwnerLen = Obj->SyntaxLen;
            return SNMP_NOERROR;
    }
}

WORD etherStatsStatus (SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;
    ETHER_STATS  *data;

    local = MibRmon(Obj, etherStats, IdLen, INDEXSIZE);

    switch (Obj->Request)
    {
        case SNMP_PDU_GET:
            if (local == NULL)
            {
                Obj->Syntax.LngInt = SNMP_INVALID;
                return SNMP_NOSUCHNAME;
            }
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngInt = data->Status;
            return SNMP_NOERROR;
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            {
                Obj->Syntax.LngInt = SNMP_INVALID;
                return SNMP_NOSUCHNAME;
            }
            data = (ETHER_STATS*) local->Data;
            Obj->Syntax.LngInt = data->Status;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            if (local == NULL)
            {
                switch (Obj->Syntax.LngInt)
                {
                    case SNMP_CREATEREQUEST:
                        if ((local = MibInsert(Obj, &etherStats, IdLen, INDEXSIZE)) == NULL)
                            return SNMP_GENERROR;
                        if ((local->Data = DnpapMalloc(sizeof(ETHER_STATS))) == NULL)
                            return SNMP_GENERROR;
                        data = (ETHER_STATS*) local->Data;
                        memset(data, 0, sizeof(ETHER_STATS));
                        if (StatCInit(data) == TRUE)
                        {
                            data->Status = SNMP_UNDERCREATION;
                            DnpapMessage(DMC_MESSAGE, STAT_CREATE, "etherStat: collector %ld created", local->Index);
                            return SNMP_NOERROR;
                        }
                        DnpapFree(local->Data);
                        MibRemove (Obj, &etherStats, IdLen, INDEXSIZE);
                        return SNMP_GENERROR;
                    default:
                        return SNMP_NOSUCHNAME;
                }
            }
            data = (ETHER_STATS*) local->Data;
            switch (data->Status)
            {
                case SNMP_UNDERCREATION:
                    switch (Obj->Syntax.LngInt)
                    {
                        case SNMP_VALID:
                            if (StatCStart(data) == TRUE)
                            {
                                data->Status = SNMP_VALID;
                                DnpapMessage(DMC_MESSAGE, STAT_ACTIVE, "etherStat: collector %ld active", local->Index);
                                return SNMP_NOERROR;
                            }
                            else
                                return SNMP_GENERROR;
                        case SNMP_INVALID:
                            if (StatCStop(data) == TRUE)
                            {
                                DnpapFree(local->Data);
                                DnpapMessage(DMC_MESSAGE, STAT_DESTROY, "etherStat: collector %ld destroyed", local->Index);
                                if (MibRemove (Obj, &etherStats, IdLen, INDEXSIZE) == TRUE)
                                    return SNMP_NOERROR;
                            }
                            return SNMP_GENERROR;
                        default:
                            return SNMP_BADVALUE;
                    }
                case SNMP_VALID:
                    switch (Obj->Syntax.LngInt)
                    {
                        case SNMP_INVALID:
                            if (StatCStop(data) == TRUE)
                            {
                                DnpapFree(local->Data);
                                DnpapMessage(DMC_MESSAGE, STAT_DESTROY, "etherStat: collector %ld destroyed", local->Index);
                                if (MibRemove (Obj, &etherStats, IdLen, INDEXSIZE) == TRUE)
                                    return SNMP_NOERROR;
                            }
                            return SNMP_GENERROR;
                        case SNMP_VALID:
                            return SNMP_NOERROR;
                        default:
                            return SNMP_BADVALUE;
                    }
            }
    }
}
