#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dnpap.h>
#include <mac.h>
#include <snmp.h>
#include <mibsup.h>
#include <message.h>

#include "infoe.h"
#include "infoc.h"
#include "info.h"


#define INDEXSIZE       1


static MIB_LOCAL    *beholderInfo = NULL;


static BOOLEAN RmonNext (SNMP_OBJECT *Obj, MIB_LOCAL **Local, WORD IdLen, WORD IdSze, void *Elm);



BOOLEAN InfoMInit(VOID)
{
    MessageConfig(INFO_ERROR, "BeholderInfo");
    return TRUE;
}


/*****************************************************************
** NAME:        RmonNext
** SYNOPSIS:    BOOLEAN RmonNext (SNMP_OBJECT *Obj,
**                                  MIB_LOCAL **Local, WORD IdLen,
**                                  WORD IdSze, VOID **Elm)
** PARAMETERS:  Obj: requested object
**              Local: local datastructure: a collector
**              IdLen: identifier length known to the MIB
**              IdSze: number of indices after the MIB
**              Elm: pointer to object if tables are used
** DESCRIPTION: application specific RMON NEXT function.
**              only called after MibRmon() for next requests 
**              searches the next object in the collector
** REMARKS:     ONLY FOR INTERNAL USE
** RETURNS:     TRUE: object found
**                    OPTIONAL: Elm -> pointer to object in table
*******************************************************************/

BOOLEAN RmonNext (SNMP_OBJECT *Obj, MIB_LOCAL **Local, WORD IdLen, WORD IdSze, VOID **Elm)
{
MIB_LOCAL    *local = *Local;

    if (local == NULL)
        return FALSE;
    if (Obj->IdLen == IdLen || local->Index > Obj->Id[IdLen])
    {
        /********************************************************************
        **  OPTIONAL: search function to find first obj in a table !! 
        **  if (not found)
        **      search next collector
        **  adjust object identifier to identifier first object
        **  Elm -> found table obj
        ********************************************************************/
        Obj->IdLen = IdLen + IdSze;
        Obj->Id[IdLen] = local->Index;
        return TRUE;
    }
    /********************************************************************
    **  OPTIONAL: search function to find next obj in a table !! 
    **  if (found)
    **      adjust object identifier to identifier next object
    **      Elm -> found table obj
    **      return TRUE
    ********************************************************************/
    *Local = local->Next;
    return RmonNext(Obj, Local, IdLen, IdSze, Elm);
}

WORD infoIndex (SNMP_OBJECT *Obj, WORD IdLen)
{
MIB_LOCAL    *local = NULL;

    if ((local = MibRmon (Obj, beholderInfo, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;
    switch (Obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        Obj->Syntax.LngInt = local->Index;
        return SNMP_NOERROR;
    }
}

WORD infoDataSource (SNMP_OBJECT *Obj, WORD IdLen)
{
MIB_LOCAL       *local = NULL;
BEHOLDER_INFO   *data;
MAC_IFACE       *iface;

    if ((local = MibRmon(Obj, beholderInfo, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {           
    case SNMP_PDU_NEXT:
        if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (BEHOLDER_INFO*) local->Data;
        memcpy (Obj->Syntax.BufInt, data->Source, data->SourceLen * sizeof (LONG));
        Obj->SyntaxLen = data->SourceLen;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        data = (BEHOLDER_INFO*) local->Data;
        if (data->Status != SNMP_UNDERCREATION)
            return SNMP_READONLY;
        if (data->SourceLen != Obj->SyntaxLen ||
            memcmp(data->Source, Obj->Syntax.BufInt, (data->SourceLen-1) * sizeof (LONG)))
            return SNMP_BADVALUE;
        if ((iface = MacIfaceGet((WORD) Obj->Syntax.BufInt[Obj->SyntaxLen-1])) == NULL)
            return SNMP_BADVALUE;
        memcpy (data->Source, Obj->Syntax.BufInt, Obj->SyntaxLen * sizeof (LONG));
        data->SourceLen = Obj->SyntaxLen;
        data->Iface = iface;
        return SNMP_NOERROR;
    }
}

WORD infoLostPkts (SNMP_OBJECT *Obj, WORD IdLen)
{
MIB_LOCAL    *local = NULL;
BEHOLDER_INFO  *data;
MAC_STAT stat;

    if ((local = MibRmon(Obj, beholderInfo, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (BEHOLDER_INFO*) local->Data;
        if (data->Status != SNMP_VALID)
        {
            Obj->Syntax.LngUns = 0UL;
            return SNMP_NOERROR;
        }
        MacStatistics(data->Iface, &stat);
        Obj->Syntax.LngUns = stat.LostPkts - data->BaseStat.LostPkts;
        return SNMP_NOERROR;
    }
}

WORD infoPkts (SNMP_OBJECT *Obj, WORD IdLen)
{
MIB_LOCAL    *local = NULL;
BEHOLDER_INFO  *data;
MAC_STAT stat;

    if ((local = MibRmon(Obj, beholderInfo, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (BEHOLDER_INFO*) local->Data;
        if (data->Status != SNMP_VALID)
        {
            Obj->Syntax.LngUns = 0UL;
            return SNMP_NOERROR;
        }
        MacStatistics(data->Iface, &stat);
        Obj->Syntax.LngUns = stat.Pkts - data->BaseStat.Pkts;
        return SNMP_NOERROR;
    }
}
                                                
WORD infoDiscardedPkts (SNMP_OBJECT *Obj, WORD IdLen)
{
MIB_LOCAL    *local = NULL;
BEHOLDER_INFO  *data;
MAC_STAT stat;

    if ((local = MibRmon(Obj, beholderInfo, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (BEHOLDER_INFO*) local->Data;
        if (data->Status != SNMP_VALID)
        {
            Obj->Syntax.LngUns = 0UL;
            return SNMP_NOERROR;
        }
        MacStatistics(data->Iface, &stat);
        Obj->Syntax.LngUns = stat.DiscardedPkts - data->BaseStat.DiscardedPkts;
        return SNMP_NOERROR;
    }
}
                                                
WORD infoTruncatedPkts (SNMP_OBJECT *Obj, WORD IdLen)
{
MIB_LOCAL    *local = NULL;
BEHOLDER_INFO  *data;
MAC_STAT stat;

    if ((local = MibRmon(Obj, beholderInfo, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (BEHOLDER_INFO*) local->Data;
        if (data->Status != SNMP_VALID)
        {
            Obj->Syntax.LngUns = 0UL;
            return SNMP_NOERROR;
        }
        MacStatistics(data->Iface, &stat);
        Obj->Syntax.LngUns = stat.TruncatedPkts - data->BaseStat.TruncatedPkts;
        return SNMP_NOERROR;
    }
}
                                                
WORD infoBufferedPkts (SNMP_OBJECT *Obj, WORD IdLen)
{
MIB_LOCAL    *local = NULL;
BEHOLDER_INFO  *data;
MAC_STAT stat;

    if ((local = MibRmon(Obj, beholderInfo, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (BEHOLDER_INFO*) local->Data;
        if (data->Status != SNMP_VALID)
        {
            Obj->Syntax.LngUns = 0UL;
            return SNMP_NOERROR;
        }
        MacStatistics(data->Iface, &stat);
        Obj->Syntax.LngUns = stat.BufferedPkts /* - data->BaseStat.BufferedPkts */ ;   /* DMW */
        return SNMP_NOERROR;
    }
}

WORD infoStackedPkts (SNMP_OBJECT *Obj, WORD IdLen)
{
MIB_LOCAL    *local = NULL;
BEHOLDER_INFO  *data;
MAC_STAT stat;

    if ((local = MibRmon(Obj, beholderInfo, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (BEHOLDER_INFO*) local->Data;
        if (data->Status != SNMP_VALID)
        {
            Obj->Syntax.LngUns = 0UL;
            return SNMP_NOERROR;
        }
        MacStatistics(data->Iface, &stat);
        Obj->Syntax.LngUns = stat.StackedPkts - data->BaseStat.StackedPkts;
        return SNMP_NOERROR;
    }
}

WORD infoOwner (SNMP_OBJECT *Obj, WORD IdLen)
{
MIB_LOCAL       *local = NULL;
BEHOLDER_INFO   *data;

    if ((local = MibRmon(Obj, beholderInfo, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
    case SNMP_PDU_NEXT:
        if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            return SNMP_NOSUCHNAME;
    case SNMP_PDU_GET:
        data = (BEHOLDER_INFO*) local->Data;
        memcpy (Obj->Syntax.BufChr, data->Owner, data->OwnerLen);
        Obj->SyntaxLen = data->OwnerLen;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        data = (BEHOLDER_INFO*) local->Data;
        if (data->Status != SNMP_UNDERCREATION)
            return SNMP_READONLY;
        memcpy (data->Owner, Obj->Syntax.BufChr, Obj->SyntaxLen);
        data->OwnerLen = Obj->SyntaxLen;
        return SNMP_NOERROR;
    }
}

WORD infoStatus (SNMP_OBJECT *Obj, WORD IdLen)
{
MIB_LOCAL       *local = NULL;
BEHOLDER_INFO   *data;

    local = MibRmon(Obj, beholderInfo, IdLen, INDEXSIZE);

    switch (Obj->Request)
    {
    case SNMP_PDU_GET:
        if (local == NULL)
        {
            Obj->Syntax.LngInt = SNMP_INVALID;
            return SNMP_NOSUCHNAME;
        }
        data = (BEHOLDER_INFO*) local->Data;
        Obj->Syntax.LngInt = data->Status;
        return SNMP_NOERROR;
    case SNMP_PDU_NEXT:
        if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
        {
            Obj->Syntax.LngInt = SNMP_INVALID;
            return SNMP_NOSUCHNAME;
        }
        data = (BEHOLDER_INFO*) local->Data;
        Obj->Syntax.LngInt = data->Status;
        return SNMP_NOERROR;
    case SNMP_PDU_SET:
        if (local == NULL)
        {
            switch (Obj->Syntax.LngInt)
            {
            case SNMP_CREATEREQUEST:
                if ((local = MibInsert(Obj, &beholderInfo, IdLen, INDEXSIZE)) == NULL)
                    return SNMP_GENERROR;
                if ((local->Data = DnpapMalloc(sizeof(BEHOLDER_INFO))) == NULL)
                    return SNMP_GENERROR;
                data = (BEHOLDER_INFO*) local->Data;
                memset(data, 0, sizeof(BEHOLDER_INFO));
                if (InfoInit(data) == TRUE)
                {
                    ((BEHOLDER_INFO*) local->Data)->Status = SNMP_UNDERCREATION;
                    DnpapMessage(DMC_MESSAGE, INFO_CREATE, "beholderInfo: collector %ld created", local->Index);
                    return SNMP_NOERROR;
                }
                DnpapFree(local->Data);
                MibRemove (Obj, &beholderInfo, IdLen, INDEXSIZE);
                return SNMP_GENERROR;
            default:
                return SNMP_NOSUCHNAME;
            }
        }
        data = (BEHOLDER_INFO*) local->Data;
        switch (data->Status)
        {
        case SNMP_UNDERCREATION:
            switch (Obj->Syntax.LngInt)
            {
            case SNMP_VALID:
                if (InfoStart(data) == TRUE)
                {
                    data->Status = SNMP_VALID;
                    DnpapMessage(DMC_MESSAGE, INFO_ACTIVE, "beholderInfo: collector %ld active", local->Index);
                    return SNMP_NOERROR;
                }
                else
                    return SNMP_GENERROR;
            case SNMP_INVALID:
                if (InfoStop(data) == TRUE)
                {
                    DnpapFree(local->Data);
                    DnpapMessage(DMC_MESSAGE, INFO_DESTROY, "beholderInfo: collector %ld destroyed", local->Index);
                    if (MibRemove (Obj, &beholderInfo, IdLen, INDEXSIZE) == TRUE)
                        return SNMP_NOERROR;
                }
                return SNMP_GENERROR;
            default:
                return SNMP_BADVALUE;
            }
        case SNMP_VALID:
            switch (Obj->Syntax.LngInt)
            {
            case SNMP_INVALID:
                if (InfoStop(data) == TRUE)
                {
                    DnpapFree(local->Data);
                    DnpapMessage(DMC_MESSAGE, INFO_DESTROY, "beholderInfo: collector %ld destroyed", local->Index);
                    if (MibRemove (Obj, &beholderInfo, IdLen, INDEXSIZE) == TRUE)
                        return SNMP_NOERROR;
                }
                return SNMP_GENERROR;
            case SNMP_VALID:
                return SNMP_NOERROR;
            default:
                return SNMP_BADVALUE;
            }
        }
    }
}
