// copyright 2001-2002 by The Mind Electric

package electric.xml.xpath;

import electric.xml.*;

/**
 * <tt>TMEXPath</tt>
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public class TMEXPath implements IXPath
  {
  private String xpath;
  private NodeList nodes;
  private INamespaceContext namespaceContext;

  // ********** CONSTRUCTION ************************************************

  /**
   * Construct an XPath with the specified expression and no namespace context.
   * @param xpath The xpath expression.
   * @throws XPathException If the path is malformed.
   */
  public TMEXPath( String xpath )
    throws XPathException
    {
    this( xpath, (INamespaceContext) null );
    }

  /**
   * Construct an XPath with the specified expression and namespace context.
   * @param xpath The xpath expression.
   * @param namespaceContext The namespace context.
   * @throws XPathException If the path is malformed.
   */
  public TMEXPath( String xpath, INamespaceContext namespaceContext )
    throws XPathException
    {
    this.xpath = xpath.trim();
    this.nodes = XPathNodeFactory.newNodes( this.xpath );
    this.namespaceContext = namespaceContext;
    }

  /**
   * @param xpath
   * @param nodes
   * @param namespaceContext
   */
  TMEXPath( String xpath, NodeList nodes, INamespaceContext namespaceContext )
    {
    this.xpath = xpath;
    this.nodes = nodes;
    this.namespaceContext = namespaceContext;
    }

  // ********** STANDARD METHODS ********************************************

  /**
   * Return the xpath expression.
   */
  public String toString()
    {
    return xpath;
    }

  // ********** NAMESPACE CONTEXT *******************************************

  /**
   * Return the namespace context.
   */
  public INamespaceContext getNamespaceContext()
    {
    return namespaceContext;
    }

  /**
   * Set the namespace context.
   * @param namespaceContext The new value.
   */
  public void setNamespaceContext( INamespaceContext namespaceContext )
    {
    this.namespaceContext = namespaceContext;
    }

  // ********** ATTRIBUTES **************************************************

  /**
   * @param element
   * @throws XPathException
   */
  public Attribute getAttribute( Element element )
    throws XPathException
    {
    return getAttributes( element ).first();
    }

  /**
   * @param element
   * @throws XPathException
   */
  public Attributes getAttributes( Element element )
    throws XPathException
    {
    NodeList list = new NodeList();
    ((XPathNode) nodes.first).addNodes( this, element, list );
    return new Attributes( list );
    }

  // ********** ELEMENTS ****************************************************

  /**
   * @param parent
   * @throws XPathException
   */
  public Element getElement( Parent parent )
    throws XPathException
    {
    return getElements( parent ).first();
    }

  /**
   * @param parent
   * @throws XPathException
   */
  public Elements getElements( Parent parent )
    throws XPathException
    {
    NodeList list = new NodeList();
    ((XPathNode) nodes.first).addNodes( this, parent, list );
    return new Elements( list );
    }

  // ********** NAMESPACES **************************************************

  /**
   * Register a namespace with the specified prefix and namespace.
   * If I don't already have a namespace context, set it to a
   * HashtableNamespaceContext, which uses a Hashtable to record
   * the mappings.
   * @param prefix The prefix.
   * @param namespace The namespace.
   */
  public synchronized void setNamespace( String prefix, String namespace )
    {
    if( namespaceContext == null )
      namespaceContext = new HashtableNamespaceContext();

    namespaceContext.setNamespace( prefix, namespace );
    }

  /**
   * Return the namespace associated with the specified prefix, or null
   * if none exists.
   * @param prefix The prefix.
   */
  public synchronized String getNamespace( String prefix )
    {
    if( namespaceContext == null )
      return null;

    return namespaceContext.getNamespace( prefix );
    }
  }