// copyright 2001-2002 by The Mind Electric

package electric.xml;

import java.io.*;
import java.util.*;
import electric.util.*;

/**
 * <tt>Parent</tt> is the abstract root of all nodes that can have children,
 * and includes methods for adding and accessing child nodes.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public abstract class Parent extends Child
  {
  protected NodeList children = new NodeList(); // list of my Child nodes

  // ********** CONSTRUCTION ************************************************

  /**
   * Construct a Parent with no children.
   */
  public Parent()
    {
    }

  /**
   * Construct a copy of the specified Parent.
   * @param parent The parent to copy.
   */
  public Parent( Parent parent )
    {
    for( Node node = parent.children.first; node != null; node = node.next )
      addChild( (Child) ((Child) node).clone() );
    }

  // ********** CHILDREN ****************************************************

  /**
   * Return true if I have one or more children.
   */
  public boolean hasChildren()
    {
    return !children.isEmpty();
    }

  /**
   * Return an enumeration over my children.
   */
  public Children getChildren()
    {
    return new Children( children );
    }

  /**
   * Remove all of my children.
   */
  public void removeChildren()
    {
    children.clear();
    }

  /**
   * Add the specified child to my list of children and return the child.
   * @param child The child.
   */
  public Child addChild( Child child )
    {
    child.setParent( this );
    children.append( child );
    return child;
    }

  /**
   * Insert the specified child to my list of children and return the child.
   * @param child The child.
   */
  public Child insertChild( Child child )
    {
    child.setParent( this );
    children.insert( child );
    return child;
    }

  /**
   * Replace the old child with the new child.
   * @param oldChild The old child.
   * @param newChild The new child.
   */
  void replaceChild( Child oldChild, Child newChild )
    {
    children.replace( oldChild, newChild );
    newChild.setParent( this );
    }

  // ********** NAMESPACES **************************************************

  /**
   * Return the value of the namespace with the specified prefix, or null if
   * there is none. This operation searches up the element hierarchy starting
   * at this element.
   * @param prefix The prefix.
   */
  public String getNamespace( String prefix )
    {
    return null;
    }

  /**
   * Add all the prefixes that map to a particular namespace value in my context to
   * the specified vector.
   * @param namespace The namespace to match
   * @param prefixes All the prefixes so far.
   * @param matches All the prefixes that matched.
   */
  protected void addNamespacePrefixes( String namespace, Vector prefixes, Vector matches )
    {
    }

  /**
   * Return a prefix that maps to a particular
   * namespace value, searching from the current element up through its parents.
   * Return null if none is found.
   * @param namespace The namespace to match.
   */
  public String getNamespacePrefix( String namespace )
    {
    return null;
    }

  // ********** COMMENTS ****************************************************

  /**
   * Add a Comment node with the specified text.
   * @param text The text.
   */
  public void addComment( String text )
    {
    addChild( new Comment( text ) );
    }

  /**
   * Insert a Comment node with the specified text.
   * @param text The text.
   */
  public void insertComment( String text )
    {
    insertChild( new Comment( text ) );
    }

  // ********** ELEMENTS ****************************************************

  /**
   * Return true if I have one or more child Elements.
   */
  public boolean hasElements()
    {
    return getElements().current() != null;
    }

  /**
   * Return an enumeration of my child Elements.
   */
  public Elements getElements()
    {
    return new ElementsFilter( children );
    }

  /**
   * Return my first Element node with the specified name,
   * or null if I have none.
   * @param name The name.
   */
  public Element getElement( String name )
    {
    for( Node node = children.first; node != null; node = node.next )
      if( node instanceof Element && ((Element) node).name.equals( name ) )
        return (Element) node;

    return null;
    }

  /**
   * Return my first Element node with the specified namespace and name,
   * or null if I have none.
   * @param namespace The namespace.
   * @param name The name.
   */
  public Element getElement( String namespace, String name )
    {
    for( Node node = children.first; node != null; node = node.next )
      if( node instanceof Element && ((Element) node).hasName( namespace, name ) )
        return (Element) node;

    return null;
    }

  /**
   * Return my first Element node with the specified xpath,
   * or null if I have none.
   * @param xpath The XPath.
   */
  public Element getElement( IXPath xpath )
    {
    return xpath.getElement( this );
    }

  /**
   * Return true if I have an Element with the specified name.
   * @param name The name.
   */
  public boolean hasElement( String name )
    {
    return (getElement( name ) != null);
    }

  /**
   * Return the Element node at the specified index, using 1 as the base
   * index for Xpath compatibility.
   * @throws IndexOutOfBoundsException If the index was illegal.
   */
  public Element getElementAt( int index )
    throws IndexOutOfBoundsException
    {
    int n = 0;

    for( Node node = children.first; node != null; node = node.next )
      if( node instanceof Element && ++n == index )
        return (Element) node;

    throw new IndexOutOfBoundsException( index + " is an invalid index" );
    }

  /**
   * Return the first Element, or null if there is none.
   */
  public Element getFirstElement()
    {
    for( Node node = children.first; node != null; node = node.next )
      if( node instanceof Element )
        return (Element) node;

    return null;
    }

  /**
   * Set the element node at the specified index, using 1 as the base
   * index for Xpath compatibility, and return the previous value.
   * @param index The index.
   * @param element The element.
   * @throws IndexOutOfBoundsException If the index was illegal.
   */
  public Element setElementAt( int index, Element element )
    throws IndexOutOfBoundsException
    {
    Element previous = getElementAt( index );
    previous.replaceWith( element );
    return previous;
    }

  /**
   * Add a new Element with the specified name, replacing an existing one if
   * present.
   * @param name The name of the Element to add.
   * @return The new element.
   */
  public Element setElement( String name )
    {
    Element element = getElement( name );

    if( element == null )
      element = addElement( name );

    return element;
    }

  /**
   * Add a new Element with the specified namespace prefix name, replacing
   * an existing one if present.
   * @param prefix The namespace prefix.
   * @param name The name of the Element to add.
   * @return The new element.
   */
  public Element setElement( String prefix, String name )
    {
    Element element = getElement( prefix, name );

    if( element == null )
      element = addElement( prefix, name );

    return element;
    }

  /**
   * Remove and return the Element node at the specified index, using 1 as the base
   * index for Xpath compatibility.
   * @param index The index.
   * @throws IndexOutOfBoundsException If the index was illegal.
   */
  public Element removeElementAt( int index )
    {
    Element element = getElementAt( index );
    element.remove();
    return element;
    }

  /**
   * Return an enumeration over all my Elements with the specified name.
   * @param name The name.
   */
  public Elements getElements( String name )
    {
    NodeList selected = new NodeList();

    for( Node node = children.first; node != null; node = node.next )
      if( node instanceof Element && ((Element) node).name.equals( name ) )
        selected.append( new Selection( node ) );

    return new Elements( selected );
    }

  /**
   * Return an enumeration over all my Elements with the specified namespace and name.
   * @param namespace The namespace.
   * @param name The name.
   */
  public Elements getElements( String namespace, String name )
    {
    NodeList selected = new NodeList();

    for( Node node = children.first; node != null; node = node.next )
      if( node instanceof Element && ((Element) node).hasName( namespace, name ) )
        selected.append( new Selection( node ) );

    return new Elements( selected );
    }

  /**
   * Return an enumeration over all my Elements with the specified xpath.
   * @param xpath The XPath.
   */
  public Elements getElements( IXPath xpath )
    {
    return xpath.getElements( this );
    }

  /**
   * Add the specified Element.
   * @param element The Element.
   * @return The new element.
   */
  public Element addElement( Element element )
    {
    addChild( element );
    return element;
    }

  /**
   * Add a new Element with no name.
   * @return The new element.
   */
  public Element addElement()
    {
    return addElement( new Element() );
    }

  /**
   * Add a new Element with the specified name.
   * @param name The name of the Element to add.
   * @return The new element.
   */
  public Element addElement( String name )
    {
    return addElement().setName( name );
    }

  /**
   * Add a new Element with the specified namespace prefix name.
   * @param prefix The namespace prefix.
   * @param name The name of the Element to add.
   * @return The new element.
   */
  public Element addElement( String prefix, String name )
    {
    return addElement().setName( prefix, name );
    }

  /**
   * Insert the specified Element.
   * @param element The Element.
   * @return The new element.
   */
  public Element insertElement( Element element )
    {
    insertChild( element );
    return element;
    }

  /**
   * Insert a new Element with no name.
   * @return The new element.
   */
  public Element insertElement()
    {
    return insertElement( new Element() );
    }

  /**
   * Insert a new Element with the specified name.
   * @param name The name of the Element to add.
   * @return The new element.
   */
  public Element insertElement( String name )
    {
    return insertElement().setName( name );
    }

  /**
   * Insert a new Element with the specified namespace prefix name.
   * @param prefix The namespace prefix.
   * @param name The name of the Element to add.
   * @return The new element.
   */
  public Element insertElement( String prefix, String name )
    {
    return insertElement().setName( prefix, name );
    }

  /**
   * Remove all the elements.
   */
  public void removeElements()
    {
    getElements().remove();
    }

  /**
   * Remove and return the first element with the specified name, or return
   * null if none was found.
   * @param name The name.
   */
  public Element removeElement( String name )
    {
    Element element = getElement( name );

    if( element != null )
      element.remove();

    return element;
    }

  /**
   * Remove and return the first element with the specified namespace and name, or return
   * null if none was found.
   * @param namespace The namespace.
   * @param name The name.
   */
  public Element removeElement( String namespace, String name )
    {
    Element element = getElement( namespace, name );

    if( element != null )
      element.remove();

    return element;
    }

  /**
   * Remove and return the first element with the specified xpath, or return
   * null if none was found.
   * @param xpath The XPath.
   */
  public Element removeElement( IXPath xpath )
    {
    Element element = getElement( xpath );

    if( element != null )
      element.remove();

    return element;
    }

  /**
   * Remove and return an enumeration over the elements with the specified name.
   * @param name The name.
   */
  public Elements removeElements( String name )
    {
    Elements elements = getElements( name );
    elements.remove();
    elements.reset();
    return elements;
    }

  /**
   * Remove and return an enumeration over the elements with the specified namespace and name.
   * @param namespace The namespace.
   * @param name The name.
   */
  public Elements removeElements( String namespace, String name )
    {
    Elements elements = getElements( namespace, name );
    elements.remove();
    elements.reset();
    return elements;
    }

  /**
   * Remove and return an enumeration over the elements with the specified xpath.
   * @param xpath The XPath.
   */
  public Elements removeElements( IXPath xpath )
    {
    Elements elements = getElements( xpath );
    elements.remove();
    elements.reset();
    return elements;
    }

  /**
   * @param name The name.
   * @throws NumberFormatException
   */
  protected Element needElement( String name )
    throws NumberFormatException
    {
    Element child = getElement( name );

    if( child != null )
      return child;

    throw new NumberFormatException( "could not find element with name " + name );
    }

  /**
   * @param namespace The namespace.
   * @param name The name.
   * @throws NumberFormatException
   */
  protected Element needElement( String namespace, String name )
    throws NumberFormatException
    {
    Element child = getElement( namespace, name );

    if( child != null )
      return child;

    throw new NumberFormatException( "could not find element with name " + name );
    }

  // ********** IDS *********************************************************

  /**
   * Return the element whose "id" attribute is equal to the specified value,
   * or null if none exists.
   * @param value
   */
  public synchronized Element getElementWithId( String value )
    {
    for( Node node = children.first; node != null; node = node.next )
      if( node instanceof Element )
        {
        Element element = ((Element) node).getElementWithId( value );

        if( element != null )
          return element;
        }

    return null;
    }

  /**
   * @param idElement
   */
  synchronized void addIds( Hashtable idToElement )
    {
    for( Node node = children.first; node != null; node = node.next )
      if( node instanceof Element )
        ((Element) node).addIds( idToElement );
    }

  // ********** DOM *********************************************************


  /**
   * Returns whether this node has any children.
   */
  public boolean hasChildNodes()
    {
    return hasChildren();
    }

  /**
   * Return a NodeList that contains all children of this node.
   * If there are no children, this is a NodeList containing no nodes.
   */
  public org.w3c.dom.NodeList getChildNodes()
    {
    return children;
    }

  /**
   * Return the first child of this node.
   * If there is no such node, this returns null.
   */
  public org.w3c.dom.Node getFirstChild()
    {
    return children.first;
    }

  /**
   * Return the last child of this node.
   * If there is no such node, this returns null.
   */
  public org.w3c.dom.Node getLastChild()
    {
    return children.last;
    }

  /**
   * Adds the node newChild to the end of the list of children of this node.
   * If the newChild is already in the tree, it is first removed.
   * @param newChild The node to add.If it is a DocumentFragment object, the
   * entire contents of the document fragment are moved into the child list of this node
   */
  public org.w3c.dom.Node appendChild( org.w3c.dom.Node newChild )
    {
    if( newChild instanceof Fragment )
      {
      for( Children children = ((Fragment) newChild).getChildren(); children.hasMoreElements(); )
        appendChild( children.next() );
      }
    else
      {
      addChild( (Child) newChild );
      }

    return newChild;
    }

  /**
   * Removes the child node indicated by oldChild from the list of children, and returns it.
   * @param oldChild The node being removed.
   */
  public org.w3c.dom.Node removeChild( org.w3c.dom.Node oldChild )
    {
    if( oldChild instanceof Fragment )
      {
      for( Children children = ((Fragment) oldChild).getChildren(); children.hasMoreElements(); )
        removeChild( children.next() );
      }
    else
      {
      // should check that oldChild is in my list of children
      ((Node) oldChild).remove();
      }

    return oldChild;
    }

  /**
   * Inserts the node newChild before the existing child node refChild.
   * If refChild is null, insert newChild at the end of the list of children.
   * If newChild is a DocumentFragment object, all of its children are inserted,
   * in the same order, before refChild. If the newChild is already in the tree,
   * it is first removed.
   * @param newChild The node to insert.
   * @param refChild The reference node, i.e., the node before which the new node must be inserted.
   */
  public org.w3c.dom.Node insertBefore( org.w3c.dom.Node newChild, org.w3c.dom.Node refChild )
    {
    if( newChild instanceof Fragment )
      {
      for( Children children = ((Fragment) newChild).getChildren(); children.hasMoreElements(); )
        insertBefore( children.next(), refChild );
      }
    else
      {
      if( refChild == null )
        {
        appendChild( newChild );
        }
      else
        {
        // should check that refChild is in my list of children
        children.insertSiblingNode( (Child) refChild, (Child) newChild );
        ((Child) newChild).setParent( this );
        }
      }

    return newChild;
    }

  /**
   * Replaces the child node oldChild with newChild in the list of children,
   * and returns the oldChild node. If newChild is a DocumentFragment object,
   * oldChild is replaced by all of the DocumentFragment children, which are
   * inserted in the same order. If the newChild is already in the tree, it
   * is first removed.
   * @param newChild The new node to put in the child list.
   * @param oldChild The node being replaced in the list.
   */
  public org.w3c.dom.Node replaceChild( org.w3c.dom.Node newChild, org.w3c.dom.Node oldChild )
    {
    // should check that oldChild is in my list of children
    replaceChild( (Child) oldChild, (Child) newChild );
    return oldChild;
    }
  }