// copyright 2001-2002 by The Mind Electric

package electric.xml;

import java.io.*;
import java.util.*;
import electric.util.*;
import electric.util.io.*;

/**
 * <tt>Document</tt> represents an XML document, and includes methods for
 * parsing XML and accessing a document's root. When parsing XML, you can
 * supply an optional Hashtable of namespace values that acts as global context.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public class Document extends Parent implements org.w3c.dom.Document
  {
  /**
   * Keep whitespace during parsing.
   */
  public static final int KEEP_WHITESPACE = 1;

  /**
   * The major version.
   */
  public static final int MAJOR_VERSION = 6;

  /**
   * The minor version.
   */
  public static final int MINOR_VERSION = 0;

  private static final Implementation implementation = new Implementation();
  private static final String[] NO_STRINGS = new String[ 0 ];
  private static final String XMLDECL_START = "<?xml ";
  private static final String XMLDECL_STOP = "?>";

  private Hashtable context; // hashtable of global namespaces
  private boolean stripped = true;
  private String version = "1.0";
  private String encoding = "UTF-8";
  private boolean standalone = false;
  private boolean setStandalone = false;
  private boolean writeXMLDecl = true;

  // ********** CONSTRUCTION ************************************************

  /**
   * Construct an empty Document.
   */
  public Document()
    {
    }

  /**
   * Construct an empty Document.
   * @param context A Hashtable of global namespaces.
   */
  public Document( Hashtable context )
    {
    this.context = context;
    }

  /**
   * Construct a Document built by parsing the specified string.
   * @param string An XML document.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( String string )
    throws ParseException
    {
    this( string, 0 );
    }

  /**
   * Construct a Document built by parsing the specified string.
   * @param string An XML document.
   * @param flags Parse flags.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( String string, int flags )
    throws ParseException
    {
    parse( new FastReader( string ), null, flags );
    }

  /**
   * Construct a Document built by parsing the specified bytes.
   * @param bytes A byte array that is converted into a String.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( byte[] bytes )
    throws ParseException
    {
    this( bytes, null, 0 );
    }

  /**
   * Construct a Document built by parsing the specified bytes.
   * @param bytes A byte array that is converted into a String.
   * @param flags Parse flags.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( byte[] bytes, int flags )
    throws ParseException
    {
    this( bytes, null, flags );
    }

  /**
   * Construct a Document built by parsing the specified bytes.
   * @param bytes A byte array that is converted into a String.
   * @param context A Hashtable of global namespaces.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( byte[] bytes, Hashtable context )
    throws ParseException
    {
    this( bytes, context, 0 );
    }

  /**
   * Construct a Document built by parsing the specified bytes.
   * @param bytes A byte array that is converted into a String.
   * @param context A Hashtable of global namespaces.
   * @param flags Parse flags.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( byte[] bytes, Hashtable context, int flags )
    throws ParseException
    {
    try
      {
      parse( new FastReader( Strings.toString( bytes ) ), context, flags );
      }
    catch( UnsupportedEncodingException exception )
      {
      throw new ParseException( exception.toString() );
      }
    }

  /**
   * Construct a Document built by parsing the File.
   * @param file The File to parse.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( File file )
    throws ParseException
    {
    this( file, 0 );
    }

  /**
   * Construct a Document built by parsing the File.
   * @param file The File to parse.
   * @param flags Parse flags.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( File file, int flags )
    throws ParseException
    {
    try
      {
      parse( new FastBufferedReader( Streams.getReader( file ) ), null, flags );
      }
    catch( IOException exception )
      {
      throw new ParseException( exception.toString() );
      }
    }

  /**
   * Construct a Document built by parsing the InputStream.
   * @param stream The InputStream to parse.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( InputStream stream )
    throws ParseException
    {
    this( stream, 0 );
    }

  /**
   * Construct a Document built by parsing the InputStream.
   * @param stream The InputStream to parse.
   * @param flags Parse flags.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( InputStream stream, int flags )
    throws ParseException
    {
    try
      {
      parse( new FastBufferedReader( Streams.getReader( stream ) ), null, flags );
      }
    catch( IOException exception )
      {
      throw new ParseException( exception.toString() );
      }
    }

  /**
   * Construct a Document built by parsing the contents of the Reader.
   * @param reader The Reader.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( Reader reader )
    throws ParseException
    {
    parse( reader, null, 0 );
    }

  /**
   * Construct a Document built by parsing the contents of the Reader.
   * @param reader The Reader.
   * @param flags Parse flags.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( Reader reader, int flags )
    throws ParseException
    {
    this( reader, null, flags );
    }

  /**
   * Construct a Document built by parsing the contents of the Reader.
   * @param string The Reader.
   * @param context A Hashtable of global namespaces.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( Reader reader, Hashtable context )
    throws ParseException
    {
    this( reader, context, 0 );
    }

  /**
   * Construct a Document built by parsing the contents of the Reader.
   * @param string The Reader.
   * @param context A Hashtable of global namespaces.
   * @param flags Parse flags.
   * @throws ParseException If an error occurs during parsing.
   */
  public Document( Reader reader, Hashtable context, int flags )
    throws ParseException
    {
    parse( reader, context, flags );
    }

  /**
   * Construct a Document with the specified root element.
   * @param root The root Element.
   */
  public Document( Element root )
    {
    setRoot( root );
    }

  /**
   * Construct a copy of the specified document.
   * @param document The Document to copy.
   */
  public Document( Document document )
    {
    super( document );
    this.context = document.context;
    this.version = document.version;
    this.encoding = document.encoding;
    this.standalone = document.standalone;
    this.setStandalone = document.setStandalone;
    this.writeXMLDecl = document.writeXMLDecl;
    }

  // ********** STRIPPED ****************************************************

  /**
   * Return true if whitespace was stripped during parsing.
   */
  public boolean isStripped()
    {
    return stripped;
    }

  // ********** PARSING *****************************************************

  /**
   * @param reader
   * @param context
   * @param flags
   * @throws ParseException
   */
  private void parse( Reader reader, Hashtable context, int flags )
    throws ParseException
    {
    if( (flags & KEEP_WHITESPACE) != 0 )
      stripped = false;

    Lex lex = new Lex( reader, "<>=/:", Lex.SKIP_WS );

    try
      {
      parse( lex, context, stripped );
      }
    catch( Exception exception )
      {
      throw new ParseException( exception.getClass().getName() + ": " + exception.getMessage() + "\n" + lex.getLocation() );
      }
    finally
      {
      try
        {
        reader.close();
        }
      catch( IOException exception )
        {
        }
      }
    }

  /**
   * @param lex
   * @param context
   * @param strip
   * @throws IOException
   * @throws NamespaceException
   */
  private void parse( Lex lex, Hashtable context, boolean strip )
    throws IOException, NamespaceException
    {
    this.context = context;
    int elements = 0;
    writeXMLDecl = false;

    while( true )
      {
      StringBuffer whitespace = lex.readWhitespace();

      if( whitespace != null && !strip )
        new Text( this ).setString( whitespace.toString() );

      lex.mark( 2 );
      int ch1 = lex.peekRead();
      int ch2 = lex.peekRead();
      lex.reset();

      if( ch1 == -1 ) // eof
        break;
      else if( ch2 == '!' && lex.peekString( Comment.START ) )
        new Comment( lex, this );
      else if( ch2 == '!' && lex.peekString( DocType.START ) )
        new DocType( lex, this );
      else if( ch2 == '?' && lex.peekString( XMLDECL_START ) )
        {
        writeXMLDecl = true;
        parseXMLDecl( lex );
        }
      else if( ch2 == '?' )
        new Instruction( lex, this );
      else // start tag "<"
        {
        new Element( lex, this, strip );
        ++elements;
        }
      }

    if( elements != 1 )
      throw new IOException( "the document does not have exactly one root" );

    lex.skipWhitespace();

    if( lex.read() != -1 )
      throw new IOException( "extra stuff at the end" );
    }

  // ********** CLONING *****************************************************

  /**
   * Return a clone of this Document.
   */
  public Object clone()
    {
    return new Document( this );
    }

  // ********** DOCUMENT ****************************************************

  /**
   * Return myself.
   */
  public Document getDocument()
    {
    return this;
    }

  // ********** ROOT ********************************************************

  /**
   * Return the top-level element in this Document.
   */
  public Element getRoot()
    {
    for( Node node = children.first; node != null; node = node.next )
      if( node instanceof Element )
        return (Element) node;

    return null;
    }

  /**
   * Set the root to an empty element with no name.
   * @return The new root.
   */
  public Element newRoot()
    {
    return setRoot( new Element() );
    }

  /**
   * Set the root to the specified element.
   * @param element The new root.
   * @return The new root.
   */
  public Element setRoot( Element element )
    {
    Element root = getRoot();

    if( root != null )
      root.replaceWith( element );
    else
      addChild( element );

    return element;
    }

  /**
   * Set the root to an empty element with the specified name.
   * @param name The name of the new root.
   * @return The new root.
   */
  public Element setRoot( String name )
    {
    Element root = newRoot();
    root.setName( name );
    return root;
    }

  /**
   * Set the root to an empty element with the specified namespace prefix and name.
   * @param prefix The namespace prefix of the new root.
   * @param name The name of the new root.
   * @return The new root.
   */
  public Element setRoot( String prefix, String name )
    {
    Element root = newRoot();
    root.setName( prefix, name );
    return root;
    }

  // ********** DOCTYPE *****************************************************

  /**
   * Return the DocType, or null if there is none.
   */
  public DocType getDocType()
    {
    for( Node node = children.first; node != null; node = node.next )
      if( node instanceof DocType )
        return (DocType) node;

    return null;
    }

  // ********** WRITING *****************************************************

  /**
   * Write myself to the specified writer, starting at the specified indent level.
   * If the indent level is -1, no indentation will occur, otherwise the indent
   * level increases by two at each child node.
   * @param writer The nodeWriter.
   * @throws IOException If an I/O exception occurs.
   */
  public void write( NodeWriter writer )
    throws IOException
    {
    if( writeXMLDecl )
      writeXMLDecl( writer );

    for( Node node = children.first; node != null; node = node.next )
      {
      writer.write( node );

      if( node.next != null && stripped )
        writer.writeEOL();
      }
    }

  // ********** XML DECLARATION *********************************************

  /**
   * Return the version of this document, which is 1.0 by default.
   */
  public String getVersion()
    {
    return version;
    }

  /**
   * Set the version for this document.
   * @param version The new version.
   */
  public void setVersion( String version )
    {
    this.version = version;
    writeXMLDecl = true;
    }

  /**
   * Return the encoding for this document, which is UTF-8 by default.
   */
  public String getEncoding()
    {
    return encoding;
    }

  /**
   * Set the encoding for this document.
   * @param encoding The new encoding.
   */
  public void setEncoding( String encoding )
    {
    this.encoding = encoding;
    writeXMLDecl = true;
    }

  /**
   * Return true if this document is standalone, which is false by default.
   */
  public boolean getStandalone()
    {
    return standalone;
    }

  /**
   * Set the standalone value for this document.
   * @param standalone The new standalone value.
   */
  public void setStandalone( boolean standalone )
    {
    this.standalone = standalone;
    this.setStandalone = true;
    writeXMLDecl = true;
    }

  /**
   * If flag is true, write my XML declaration when printing.
   * By default, this flag is equal to true.
   * @param flag The new setting.
   */
  public void setWriteXMLDecl( boolean flag )
    {
    writeXMLDecl = flag;
    }

  /**
   * Return true if my XML declaration will be written.
   */
  public boolean getWriteXMLDecl()
    {
    return writeXMLDecl;
    }

  /**
   * @param lex
   * @throws IOException
   */
  private void parseXMLDecl( Lex lex )
    throws IOException
    {
    lex.skip( XMLDECL_START.length() );
    int peek = 0;

    do
      {
      String key = lex.readToken();
      lex.readChar( '=' );
      lex.skipWhitespace();
      int ch = lex.read();
      String value = null;

      if( ch == '\"' )
        value = lex.readToPattern( "\"", Lex.CONSUME | Lex.HTML );
      else if( ch == '\'' )
        value = lex.readToPattern( "'", Lex.CONSUME | Lex.HTML );
      else
        throw new IOException( "missing quote at start of XMLDecl attribute" );

      if( key.equals( "version" ) )
        setVersion( value );
      else if( key.equals( "encoding" ) )
        setEncoding( value );
      else if( key.equals( "standalone" ) )
        setStandalone( value.equals( "yes" ) );
      else
        throw new IOException( key + " is invalid attribute for XMLDecl" );

      lex.skipWhitespace();
      peek = lex.peek();
      }
    while( peek != '?' );

    lex.readChar( '?' );
    lex.readChar( '>' );
    }

  /**
   * @param writer
   * @throws IOException
   */
  private void writeXMLDecl( NodeWriter writer )
    throws IOException
    {
    writer.writeIndent();
    writer.write( XMLDECL_START );
    writer.write( "version='" );
    writer.write( version );
    writer.write( "' encoding='" );
    writer.write( encoding );
    writer.write( "'" );

    if( setStandalone )
      {
      writer.write( " standalone='" );
      writer.write( (standalone ? "yes" : "no") );
      writer.write( "'" );
      }

    writer.write ( XMLDECL_STOP );
    writer.writeEOL();
    }

  // ********** NAMESPACES **************************************************

  /**
   * Return my context.
   */
  public Hashtable getContext()
    {
    return context;
    }

  /**
   * Set my context.
   * @param context A Hashtable of global namespaces.
   */
  public void setContext( Hashtable context )
    {
    this.context = context;
    }

  /**
   * Return the value of the namespace with the specified prefix, or null if
   * there is none.
   * @param prefix The prefix.
   */
  public String getNamespace( String prefix )
    {
    return (context == null ? null : (String) context.get( prefix ));
    }

  /**
   * Add all the prefixes that map to a particular namespace value in my context to
   * the specified vector.
   * @param namespace The namespace to match
   * @param prefixes All the prefixes so far.
   * @param matches All the prefixes that matched.
   */
  protected void addNamespacePrefixes( String namespace, Vector prefixes, Vector matches )
    {
    if( context == null )
      return;

    for( Enumeration enum = context.keys(); enum.hasMoreElements(); )
      {
      String prefix = (String) enum.nextElement();

      if( (!prefixes.contains( prefix )) && context.get( prefix ).equals( namespace )  )
        matches.addElement( prefix );
      }
    }

  /**
   * Return a prefix that maps to a particular
   * namespace value, searching from the current element up through its parents.
   * Return null if none is found.
   * @param namespace The namespace to match.
   */
  public String getNamespacePrefix( String namespace )
    {
    if( context == null )
      return null;

    for( Enumeration enum = context.keys(); enum.hasMoreElements(); )
      {
      String prefix = (String) enum.nextElement();

      if( context.get( prefix ).equals( namespace )  )
        return prefix;
      }

    return null;
    }

  // ********** DOM *********************************************************

  /**
   * Adds the node newChild to the end of the list of children of this node.
   * @param newChild The node to add. If the newChild is a ProcessingInstruction,
   * check to see if it's the XML decl, and deal with it appropriately to avoid
   * duplicate decls.
   */
  public org.w3c.dom.Node appendChild( org.w3c.dom.Node newChild )
    throws org.w3c.dom.DOMException
    {
    if( newChild instanceof Instruction )
      {
      Instruction instruction = (Instruction) newChild;

      if( instruction.getTarget().startsWith( "xml" ) )
        {
        Lex lex = new Lex( "<?xml " + instruction.getContent() + "?>", "<>=/:", Lex.SKIP_WS );

        try
          {
          parseXMLDecl( lex );
          return instruction;
          }
        catch( IOException exception )
          {
          throw new org.w3c.dom.DOMException( org.w3c.dom.DOMException.INVALID_CHARACTER_ERR, exception.toString() );
          }
        }
      }


    return super.appendChild( newChild );
    }

  /**
   * Return DOCUMENT_NODE.
   */
  public short getNodeType()
    {
    return DOCUMENT_NODE;
    }

  /**
   * Return "#document".
   */
  public String getNodeName()
    {
    return "#document";
    }

  /**
   * @param name The name.
   */
  public org.w3c.dom.Attr createAttribute( String name )
    {
    return new Attribute( name, null );
    }

  /**
   * @param namespaceURI
   * @param qualifiedName
   */
  public org.w3c.dom.Attr createAttributeNS( String namespaceURI, String qualifiedName )
    {
    String[] parts = Element.getParts( qualifiedName );
    Attribute attribute = new Attribute( parts[ 0 ], parts[ 1 ], null );
    attribute.namespace = namespaceURI;
    return attribute;
    }

  /**
   * @param data
   */
  public org.w3c.dom.CDATASection createCDATASection( String data )
    {
    return new CData( data );
    }

  /**
   * @param data
   */
  public org.w3c.dom.Comment createComment( String data )
    {
    return new Comment( data );
    }

  /**
   *
   */
  public org.w3c.dom.DocumentFragment createDocumentFragment()
    {
    return new Fragment();
    }

  /**
   * @param tagName
   */
  public org.w3c.dom.Element createElement( String tagName )
    {
    return new Element( tagName );
    }

  /**
   * @param namespaceURI
   * @param qualifiedName
   */
  public org.w3c.dom.Element createElementNS( String namespaceURI, String qualifiedName )
    {
    String[] parts = Element.getParts( qualifiedName );
    Element element = new Element( parts[ 1 ] );
    element.prefix = parts[ 0 ];
    element.namespace = namespaceURI;
    return element;
    }

  /**
   * @param name
   */
  public org.w3c.dom.EntityReference createEntityReference( String name )
    {
    return null;
    }

  /**
   * @param target
   * @param data
   */
  public org.w3c.dom.ProcessingInstruction createProcessingInstruction( String target, String data )
    {
    return new Instruction( target, data );
    }

  /**
   * @param data
   */
  public org.w3c.dom.Text createTextNode( String data )
    {
    return new Text( data );
    }

  /**
   *
   */
  public org.w3c.dom.DocumentType getDoctype()
    {
    return getDocType();
    }

  /**
   *
   */
  public org.w3c.dom.Element getDocumentElement()
    {
    return getRoot();
    }

  /**
   * @param elementId
   */
  public org.w3c.dom.Element getElementById( String elementId )
    {
    return getElementWithId( elementId );
    }

  /**
   * @param tagName
   */
  public org.w3c.dom.NodeList getElementsByTagName( String tagName )
    {
    NodeList nodes = new NodeList();
    Element root = getRoot();

    if( root != null )
      root.addElementsByTagName( tagName, nodes );

    return nodes;
    }

  /**
   * @param namespaceURI
   * @param localName
   */
  public org.w3c.dom.NodeList getElementsByTagNameNS( String namespaceURI, String localName )
    {
    NodeList nodes = new NodeList();
    Element root = getRoot();

    if( root != null )
      root.addElementsByTagNameNS( namespaceURI, localName, nodes );

    return nodes;
    }

  /**
   *
   */
  public org.w3c.dom.DOMImplementation getImplementation()
    {
    return implementation;
    }

  /**
   * @param importedNode
   * @param deep
   */
  public org.w3c.dom.Node importNode( org.w3c.dom.Node importedNode, boolean deep )
    {
    return importedNode.cloneNode( deep );
    }
  }