/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide;

import ch.abacus.lib.ui.renderer.common.MetaCloneMode;
import ch.abacus.lib.ui.renderer.common.MetaObject;
import ch.abacus.lib.ui.renderer.common.MetaProperty;
import ch.abacus.lib.ui.renderer.common.MetaPropertyValueEx;

import java.awt.*;
import java.util.ArrayList;
import java.util.HashMap;

/**
 * <p>Title: uifactory</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2001</p>
 * <p>Company: Abacus Research</p>
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 */

public class IDEUndoRedoEvent {
    static final int UNDEFINED = 0;
    static final int CLEAR = 1;
    static final int IMPORT_OBJECT = 2;
    static final int DELETE_OBJECTS = 3;
    static final int CUT = 4;
    static final int PASTE = 5;
    static final int CHANGE_LOCATIONS = 6;
    static final int CHANGE_SIZE = 7;
    static final int CHANGE_PROPERTY = 8;
    static final int CHANGE_ALL_PROPERTY = 9;
    static final int CHANGE_DATA_LINKAGE = 10;
    int event = UNDEFINED;
    HashMap theValues = new HashMap(0);

    public IDEUndoRedoEvent(int event) {
        this.event = event;
    }

    public static IDEUndoRedoEvent createClearEvent() {
        IDEUndoRedoEvent theEvent = new IDEUndoRedoEvent(CLEAR);
        theEvent.theValues = null;  // no values for clear
        return theEvent;
    }

    // undo removes the object from the parent that it was placed in.
    // redo adds the object removed back into the parent
    public static IDEUndoRedoEvent createImportObjectEvent(MetaObject theMetaObject, MetaObject theParentObject) {
        IDEUndoRedoEvent theEvent = new IDEUndoRedoEvent(IMPORT_OBJECT);
        theEvent.theValues.put("Object", theMetaObject);
        theEvent.theValues.put("Parent", theParentObject);
        return theEvent;
    }

    // undo adds the objects removed back into their parents.
    // redo removes the objects from their parents.
    public static IDEUndoRedoEvent createDeleteObjectsEvent(ArrayList theParentObjects, ArrayList theMetaObjects) {
        IDEUndoRedoEvent theEvent = new IDEUndoRedoEvent(DELETE_OBJECTS);
        theEvent.theValues.put("Objects", theMetaObjects);
        theEvent.theValues.put("Parents", theParentObjects);
        return theEvent;
    }

    // undo adds the cut objects back to their parents
    // redo removes the cut objects from their parents
    public static IDEUndoRedoEvent createCutEvent(ArrayList theParentObjects,
                                                  ArrayList theMetaObjects) {
        IDEUndoRedoEvent theEvent = new IDEUndoRedoEvent(CUT);
        theEvent.theValues.put("Objects", theMetaObjects);
        theEvent.theValues.put("Parents", theParentObjects);
        return theEvent;
    }

    // undo removes the pasted objects from the target
    // redo adds the pasted objects back to the target
    public static IDEUndoRedoEvent createPasteEvent(MetaObject theTarget,
                                                    ArrayList theMetaObjects) {
        IDEUndoRedoEvent theEvent = new IDEUndoRedoEvent(PASTE);
        theEvent.theValues.put("Target", theTarget);
        theEvent.theValues.put("Objects", theMetaObjects);
        return theEvent;
    }

    // undo restores objects to their old location.
    // redo moves the objects back to new location.
    public static IDEUndoRedoEvent createChangeLocationEvent(ArrayList theMetaObjects) {
        IDEUndoRedoEvent theEvent = new IDEUndoRedoEvent(CHANGE_LOCATIONS);
        theEvent.theValues.put("Objects", theMetaObjects);
        ArrayList theOldLocations = new ArrayList(0);
        for (int i = 0; i < theMetaObjects.size(); i++) {
            Point ptOrigin = ((MetaObject) theMetaObjects.get(i)).getLocation();
            theOldLocations.add(ptOrigin);
        }
        theEvent.theValues.put("Locations", theOldLocations);
        return theEvent;
    }

    // undo returns objects to their old size.
    // redo changes objects back to the new size.
    public static IDEUndoRedoEvent createChangeSizeEvent(MetaObject theMetaObject) {
        IDEUndoRedoEvent theEvent = new IDEUndoRedoEvent(CHANGE_SIZE);
        theEvent.theValues.put("Object", theMetaObject);
        Dimension theSize = theMetaObject.getSize();
        theEvent.theValues.put("Size", theSize);
        return theEvent;
    }

    // undo returns object to its old property setting.
    // redo changes object back to the new setting.
    public static IDEUndoRedoEvent createChangePropertyEvent(MetaObject theMetaObject,
                                                             String sPropertyName,
                                                             int iItemIndex,
                                                             int iOrderIndex) {
        IDEUndoRedoEvent theEvent = new IDEUndoRedoEvent(CHANGE_PROPERTY);
        theEvent.theValues.put("Object", theMetaObject);
        theEvent.theValues.put("PropertyName", sPropertyName);
        theEvent.theValues.put("PropertyIndex", new Integer(iItemIndex));
        theEvent.theValues.put("OrderIndex", new Integer(iOrderIndex));
        MetaPropertyValueEx pv = theMetaObject.getPropertyValue(sPropertyName, iItemIndex);
        if (pv == null) {
            theMetaObject.setPropertyValue(sPropertyName, iItemIndex, iOrderIndex, new String(""), true);
            pv = theMetaObject.getPropertyValue(sPropertyName, iItemIndex);
        }
        pv = pv.Clone();
        theEvent.theValues.put("PropertyValue", pv);
        return theEvent;
    }

    public static IDEUndoRedoEvent createChangeAllPropertyEvent(MetaObject theMetaObject, String sPropertyName) {
        IDEUndoRedoEvent theEvent = new IDEUndoRedoEvent(CHANGE_ALL_PROPERTY);
        theEvent.theValues.put("Object", theMetaObject);
        theEvent.theValues.put("PropertyName", sPropertyName);
        MetaProperty prop = theMetaObject.getPropertyMetadata(sPropertyName);
        ArrayList thePropertyValues = new ArrayList(0);
        if (prop != null) {
            int iValueCount = prop.getValueCount();
            for (int i = 0; i < iValueCount; i++) {
                thePropertyValues.add(theMetaObject.getPropertyValue(sPropertyName, i).Clone());
            }
        }
        theEvent.theValues.put("PropertyValues", thePropertyValues);
        return theEvent;
    }

    public static IDEUndoRedoEvent createDataLinkageEvent(MetaObject theMetaObject) {
        IDEUndoRedoEvent theEvent = new IDEUndoRedoEvent(CHANGE_DATA_LINKAGE);
        theEvent.theValues.put("Object", theMetaObject);
        MetaProperty prop = theMetaObject.getPropertyMetadata("FieldNames");
        int iValueCount = prop.getValueCount();
        ArrayList theFields = new ArrayList(0);
        for (int i = 0; i < iValueCount; i++) {
            MetaPropertyValueEx pvTableName = theMetaObject.getPropertyValue("FieldNames", i);
            theFields.add(pvTableName.Clone());
        }
        theEvent.theValues.put("FieldNames", theFields);
        MetaPropertyValueEx pvTableName = theMetaObject.getPropertyValue("TableName", 0);
        theEvent.theValues.put("TableName", pvTableName.Clone());
        return theEvent;
    }


    void undo() {
        switch (event) {
            case IDEUndoRedoEvent.CLEAR:
                undoClear();
                break;
            case IDEUndoRedoEvent.IMPORT_OBJECT:
                undoImportObject();
                break;
            case IDEUndoRedoEvent.DELETE_OBJECTS:
                undoDeleteObjects();
                break;
            case IDEUndoRedoEvent.CUT:
                undoCut();
                break;
            case IDEUndoRedoEvent.PASTE:
                undoPaste();
                break;
            case IDEUndoRedoEvent.CHANGE_LOCATIONS:
                undoChangeLocations();
                break;
            case IDEUndoRedoEvent.CHANGE_SIZE:
                undoChangeSize();
                break;
            case IDEUndoRedoEvent.CHANGE_PROPERTY:
                undoChangeProperty();
                break;
            case IDEUndoRedoEvent.CHANGE_ALL_PROPERTY:
                undoChangeAllProperty();
                break;
            case IDEUndoRedoEvent.CHANGE_DATA_LINKAGE:
                undoChangeDataLinkage();
                break;
        }
    }

    void redo() {
        switch (event) {
            case IDEUndoRedoEvent.CLEAR:
                redoClear();
                break;
            case IDEUndoRedoEvent.IMPORT_OBJECT:
                redoImportObject();
                break;
            case IDEUndoRedoEvent.DELETE_OBJECTS:
                redoDeleteObjects();
                break;
            case IDEUndoRedoEvent.CUT:
                redoCut();
                break;
            case IDEUndoRedoEvent.PASTE:
                redoPaste();
                break;
            case IDEUndoRedoEvent.CHANGE_LOCATIONS:
                redoChangeLocations();
                break;
            case IDEUndoRedoEvent.CHANGE_SIZE:
                redoChangeSize();
                break;
            case IDEUndoRedoEvent.CHANGE_PROPERTY:
                redoChangeProperty();
                break;
            case IDEUndoRedoEvent.CHANGE_ALL_PROPERTY:
                redoChangeAllProperty();
                break;
            case IDEUndoRedoEvent.CHANGE_DATA_LINKAGE:
                redoChangeDataLinkage();
                break;
        }
    }


    private void undoClear() {
    }

    private void redoClear() {
    }

    // undo removes the object from the parent that it was placed in.
    // redo adds the object removed back into the parent

    private void undoImportObject() {
        MetaObject theMetaObject = (MetaObject) theValues.get("Object");
        SuperDesignCockpit theDesignCockpit = (SuperDesignCockpit) theMetaObject.theDesignProject.getMetaDataUser();
        theMetaObject.DeleteObject(theMetaObject.theDesignProject, theDesignCockpit);
    }

    private void redoImportObject() {
        MetaObject theMetaObject = (MetaObject) theValues.get("Object");
        MetaObject theParentObject = (MetaObject) theValues.get("Parent");
        SuperDesignCockpit theDesignCockpit = (SuperDesignCockpit) theMetaObject.theDesignProject.getMetaDataUser();
        theDesignCockpit.PasteObject(theParentObject, theMetaObject, theMetaObject.theDesignProject, false);
    }

    // undo adds the objects removed back into their parents.
    // redo removes the objects from their parents.

    private void undoDeleteObjects() {
        ArrayList theMetaObjects = (ArrayList) theValues.get("Objects");
        ArrayList theParentObjects = (ArrayList) theValues.get("Parents");
        for (int i = 0; i < theMetaObjects.size(); i++) {
            MetaObject theMetaObject = (MetaObject) theMetaObjects.get(i);
            MetaObject theParentObject = (MetaObject) theParentObjects.get(i);
            SuperDesignCockpit theDesignCockpit = (SuperDesignCockpit) theMetaObject.theDesignProject.getMetaDataUser();
            theDesignCockpit.PasteObject(theParentObject, theMetaObject, theMetaObject.theDesignProject, true);
        }
    }

    private void redoDeleteObjects() {
        ArrayList theMetaObjects = (ArrayList) theValues.get("Objects");
        for (int i = 0; i < theMetaObjects.size(); i++) {
            MetaObject theMetaObject = (MetaObject) theMetaObjects.get(i);
            SuperDesignCockpit theDesignCockpit = (SuperDesignCockpit) theMetaObject.theDesignProject.getMetaDataUser();
            theMetaObject.DeleteObject(theMetaObject.theDesignProject, theDesignCockpit);
        }
    }

    // undo adds the cut objects back to their parents
    // redo removes the cut objects from their parents

    private void undoCut() {
        ArrayList theMetaObjects = (ArrayList) theValues.get("Objects");
        ArrayList theParentObjects = (ArrayList) theValues.get("Parents");
        for (int i = 0; i < theMetaObjects.size(); i++) {
            MetaObject theMetaObject = (MetaObject) theMetaObjects.get(i);
            MetaObject theParentObject = (MetaObject) theParentObjects.get(i);
            SuperDesignCockpit theDesignCockpit = (SuperDesignCockpit) theMetaObject.theDesignProject.getMetaDataUser();
            MetaObject theNewObject = theDesignCockpit.PasteObject(theParentObject, theMetaObject, theMetaObject.theDesignProject, true);
            theMetaObjects.set(i, theNewObject);
        }
    }

    private void redoCut() {
        ArrayList theMetaObjects = (ArrayList) theValues.get("Objects");
        for (int i = 0; i < theMetaObjects.size(); i++) {
            MetaObject theMetaObject = (MetaObject) theMetaObjects.get(i);
            SuperDesignCockpit theDesignCockpit = (SuperDesignCockpit) theMetaObject.theDesignProject.getMetaDataUser();
            theMetaObjects.set(i, theMetaObject.Clone(new MetaCloneMode(MetaCloneMode.deepCopy), null, true));
            theMetaObject.DeleteObject(theMetaObject.theDesignProject, theDesignCockpit);
        }
    }

    // undo removes the pasted objects from the target
    // redo adds the pasted objects back to the target
    private void undoPaste() {
        ArrayList theMetaObjects = (ArrayList) theValues.get("Objects");
        for (int i = 0; i < theMetaObjects.size(); i++) {
            MetaObject theMetaObject = (MetaObject) theMetaObjects.get(i);
            SuperDesignCockpit theDesignCockpit = (SuperDesignCockpit) theMetaObject.theDesignProject.getMetaDataUser();
            theMetaObjects.set(i, theMetaObject.Clone(new MetaCloneMode(MetaCloneMode.deepCopy), null, true));
            theMetaObject.DeleteObject(theMetaObject.theDesignProject, theDesignCockpit);
        }
    }

    private void redoPaste() {
        ArrayList theMetaObjects = (ArrayList) theValues.get("Objects");
        MetaObject theTarget = (MetaObject) theValues.get("Target");
        for (int i = 0; i < theMetaObjects.size(); i++) {
            MetaObject theMetaObject = (MetaObject) theMetaObjects.get(i);
            SuperDesignCockpit theDesignCockpit = (SuperDesignCockpit) theMetaObject.theDesignProject.getMetaDataUser();
            MetaObject theNewObject = theDesignCockpit.PasteObject(theTarget, theMetaObject, theMetaObject.theDesignProject, true);
            theMetaObjects.set(i, theNewObject);
        }
    }

    // undo restores objects to their old location.
    // redo moves the objects back to new location.
    private void undoChangeLocations() {
        ArrayList theMetaObjects = (ArrayList) theValues.get("Objects");
        ArrayList thePoints = (ArrayList) theValues.get("Locations");
        ArrayList theOldPoints = new ArrayList(0);
        for (int i = 0; i < theMetaObjects.size(); i++) {
            MetaObject theMetaObject = (MetaObject) theMetaObjects.get(i);
            Point theLocation = (Point) thePoints.get(i);
            Point theOldLocation = theMetaObject.getLocation();
            theOldPoints.add(theOldLocation);
            theMetaObject.setLocation((int) theLocation.getX(), (int) theLocation.getY());
        }
        theValues.put("OldLocations", theOldPoints);
    }

    private void redoChangeLocations() {
        ArrayList theMetaObjects = (ArrayList) theValues.get("Objects");
        ArrayList thePoints = (ArrayList) theValues.get("OldLocations");
        for (int i = 0; i < theMetaObjects.size(); i++) {
            MetaObject theMetaObject = (MetaObject) theMetaObjects.get(i);
            Point theOldLocation = (Point) thePoints.get(i);
            theMetaObject.setLocation((int) theOldLocation.getX(), (int) theOldLocation.getY());
        }
    }

    // undo returns objects to their old size.
    // redo changes objects back to the new size.
    private void undoChangeSize() {
        MetaObject theMetaObject = (MetaObject) theValues.get("Object");
        Dimension theSize = (Dimension) theValues.get("Size");
        Dimension theOldSize = theMetaObject.getSize();
        theValues.put("OldSize", theOldSize);
        theMetaObject.setSize((int) theSize.getWidth(), (int) theSize.getHeight());
    }

    private void redoChangeSize() {
        MetaObject theMetaObject = (MetaObject) theValues.get("Object");
        Dimension theOldSize = (Dimension) theValues.get("OldSize");
        theMetaObject.setSize(theOldSize.width, theOldSize.height);
    }

    // undo returns object to its old property setting.
    // redo changes object back to the new setting.
    private void undoChangeProperty() {
        MetaObject theMetaObject = (MetaObject) theValues.get("Object");
        String thePropertyName = (String) theValues.get("PropertyName");
        MetaPropertyValueEx thePropertyValue = (MetaPropertyValueEx) theValues.get("PropertyValue");
        int iPropertyIndex = ((Integer) theValues.get("PropertyIndex")).intValue();
        int iOrderIndex = ((Integer) theValues.get("OrderIndex")).intValue();
        MetaPropertyValueEx theOldPropertyValue = theMetaObject.getPropertyValue(thePropertyName, iPropertyIndex).Clone();
        theValues.put("OldPropertyValue", theOldPropertyValue);
        theMetaObject.setPropertyValue(thePropertyName, iPropertyIndex, iOrderIndex, thePropertyValue.Clone());
    }

    private void redoChangeProperty() {
        MetaObject theMetaObject = (MetaObject) theValues.get("Object");
        String thePropertyName = (String) theValues.get("PropertyName");
        MetaPropertyValueEx theOldPropertyValue = (MetaPropertyValueEx) theValues.get("OldPropertyValue");
        int iPropertyIndex = ((Integer) theValues.get("PropertyIndex")).intValue();
        int iOrderIndex = ((Integer) theValues.get("OrderIndex")).intValue();
        theMetaObject.setPropertyValue(thePropertyName, iPropertyIndex, iOrderIndex, theOldPropertyValue.Clone());
    }


    private void undoChangeAllProperty() {
        MetaObject theMetaObject = (MetaObject) theValues.get("Object");
        String thePropertyName = (String) theValues.get("PropertyName");
        ArrayList propertyValues = (ArrayList) theValues.get("PropertyValues");
        ArrayList oldPropertyValues = new ArrayList(0);
        int iCount = propertyValues.size();
        for (int i = 0; i < iCount; i++) {
            MetaPropertyValueEx theOldPropertyValue = theMetaObject.getPropertyValue(thePropertyName, i).Clone();
            oldPropertyValues.add(theOldPropertyValue);
            theMetaObject.setPropertyValue(thePropertyName, i, 0, (MetaPropertyValueEx) propertyValues.get(i));
        }
        theValues.put("OldPropertyValues", oldPropertyValues);
    }

    private void redoChangeAllProperty() {
        MetaObject theMetaObject = (MetaObject) theValues.get("Object");
        String thePropertyName = (String) theValues.get("PropertyName");
        ArrayList propertyValues = (ArrayList) theValues.get("OldPropertyValues");
        int iCount = propertyValues.size();
        for (int i = 0; i < iCount; i++) {
            theMetaObject.setPropertyValue(thePropertyName, i, 0, (MetaPropertyValueEx) propertyValues.get(i));
        }
    }

    private void undoChangeDataLinkage() {
        MetaObject theMetaObject = (MetaObject) theValues.get("Object");
        String theTableName = (String) theValues.get("TableName");
        ArrayList fieldNames = (ArrayList) theValues.get("FieldNames");
        ArrayList oldFieldNames = new ArrayList(0);
        int iCount = fieldNames.size();
        String theOldTableName = theMetaObject.getPropertyValue("TableName", 0).getStringValue();
        for (int i = 0; i < iCount; i++) {
            String theOldField = theMetaObject.getPropertyValue("FieldNames", i).getStringValue();
            oldFieldNames.add(new String(theOldField));
            theMetaObject.setPropertyValue("FieldNames", i, 0, fieldNames.get(i), false);
        }
        theValues.put("OldFieldNames", oldFieldNames);
        theValues.put("OldTableName", theOldTableName);
        theMetaObject.setPropertyValue("TableName", 0, 0, theTableName, false);
    }

    private void redoChangeDataLinkage() {
        MetaObject theMetaObject = (MetaObject) theValues.get("Object");
        String theTableName = (String) theValues.get("OldTableName");
        ArrayList fieldNames = (ArrayList) theValues.get("OldFieldNames");
        int iCount = fieldNames.size();
        for (int i = 0; i < iCount; i++) {
            theMetaObject.setPropertyValue("FieldNames", i, 0, fieldNames.get(i), false);
        }
        theMetaObject.setPropertyValue("TableName", 0, 0, theTableName, false);
    }

    public String getDescription() {
        switch (event) {
            case IDEUndoRedoEvent.CLEAR:
                return "Clear";
            case IDEUndoRedoEvent.IMPORT_OBJECT:
                return "New Object";
            case IDEUndoRedoEvent.DELETE_OBJECTS:
                return "Delete";
            case IDEUndoRedoEvent.CUT:
                return "Cut";
            case IDEUndoRedoEvent.PASTE:
                return "Paste";
            case IDEUndoRedoEvent.CHANGE_LOCATIONS:
                return "Drag Object";
            case IDEUndoRedoEvent.CHANGE_SIZE:
                return "Resize Object";
            case IDEUndoRedoEvent.CHANGE_PROPERTY:
                return "Change Property";
            case IDEUndoRedoEvent.CHANGE_ALL_PROPERTY:
                return "Change Array Based Property";
            case IDEUndoRedoEvent.CHANGE_DATA_LINKAGE:
                return "Change Data Linkage";
        }
        return null;
    }

}
