/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide;

import ch.abacus.lib.ui.customizer.CustomizerDataConnectionInterface;
import ch.abacus.lib.ui.customizer.CustomizerDataElementInterface;
import ch.abacus.lib.ui.customizer.CustomizerDataTableInterface;
import ch.abacus.lib.ui.customizer.CustomizerAvailableClass;
import ch.abacus.lib.ui.renderer.common.*;

import javax.swing.*;
import javax.swing.tree.TreePath;
import java.awt.*;
import java.io.*;
import java.util.ArrayList;

/**
 * Created by IntelliJ IDEA.
 * User: Stuart
 * Date: Apr 28, 2004
 * Time: 3:18:27 PM
 * Acts as a conduit between the Customizer program and the IDE
 */
public class IDECustomizer implements EnumerateProjectMetaObjectInterface {
    private final SuperDesignCockpit theEditor;
    private final CustomizerDataConnectionInterface theDataConnection;
    private boolean isCreatingSubElements = false;
    private String groupElementName;
    private DesignProject theDesignProject;
    private boolean bCustomizerLogging;
    private static final int CUSTOMIZER_VERSIONED = 314159265; // if the int does not equal this value, then we know it wasn't saved with a version number
    private static final int CUSTOMIZER_DATA_VERSION = 2; // current version of data to be written to the preferences stream

    public IDECustomizer(SuperDesignCockpit editor, CustomizerDataConnectionInterface theDataConnection) {
        this.theEditor = editor;
        this.theDataConnection = theDataConnection;
    }


    /**
     * Loads the Customizer Elements and displays them in the Component tree
     */
    public void loadFormElements() {
        // load Customizer classes for the Form...
        if (this.theDataConnection != null) {
            final IDEComponentPalettePane theComponentPalettePane = theEditor.theComponentPalettePane;
            IDEComponentPaletteNode customizerNode = theComponentPalettePane.addCustomizerDataConnection(null, theDataConnection);
            createTables(customizerNode);
            theComponentPalettePane.theTree.expandPath(new TreePath(customizerNode.getPath()));
        }
    }

    private void createTables(IDEComponentPaletteNode customizerNode) {
        ArrayList tableList = theDataConnection.getDataTables();
        if (tableList != null) {
            for (int iTable = 0; iTable < tableList.size(); iTable++) {
                createTable(tableList, iTable, customizerNode);
            }
        }
    }

    private void createTable(ArrayList tableList, int iTable, IDEComponentPaletteNode customizerNode) {
        CustomizerDataTableInterface dataTable = (CustomizerDataTableInterface) tableList.get(iTable);
        //System.out.println(dataTable.getDisplayName());

        // if the table has sub-tables, then process them and ignore any elements. A Table can contain sub-tables or Elements, but not both...
        if (dataTable.getDataTableCount() > 0) {
            IDEComponentPaletteNode tableNode = theEditor.theComponentPalettePane.addCustomizerDataTable(customizerNode, dataTable, false, false);
            ArrayList subTableList = dataTable.getDataTables();

            for (int iSubTable = 0; iSubTable < subTableList.size(); iSubTable++) {
                createTable(subTableList, iSubTable, tableNode);
            }
        } else {
            boolean bDeferDatatableElements = (dataTable.getDataElementCount() == 0);
            IDEComponentPaletteNode tableNode = theEditor.theComponentPalettePane.addCustomizerDataTable(customizerNode, dataTable, false, bDeferDatatableElements);

            // Add the elements for each table...
            if (bDeferDatatableElements == false) {
                createElements(dataTable, tableNode);
            } else {
                // create a dummy leaf, just to force a branch.
                // later, we delete this node and insert the deferred DataElements
                theEditor.theComponentPalettePane.addClass(tableNode, "XXX");
            }
        }
    }

    private void createElements(CustomizerDataTableInterface dataTable, IDEComponentPaletteNode tableNode) {
        ArrayList elementList = dataTable.getDataElements();
        for (int iElement = 0; iElement < elementList.size(); iElement++) {
            createElement(elementList, iElement, tableNode);
        }
    }

    private void createElement(ArrayList elementList, int iElement, IDEComponentPaletteNode tableNode) {

        CustomizerDataElementInterface dataElement = (CustomizerDataElementInterface) elementList.get(iElement);
        MetaClass metaClassElement = theEditor.theMetadataDispenser.findClass(dataElement.getImplementingClass());

        IDEComponentPaletteNode elementNode = theEditor.theComponentPalettePane.addClass(tableNode, dataElement, metaClassElement);

        // Add any subElements...
        createSubElements(dataElement, elementNode);
    }

    private void createSubElements(CustomizerDataElementInterface dataElement, IDEComponentPaletteNode elementNode) {
        ArrayList subElementsList = dataElement.getListOfDataElements();
        if (subElementsList != null) {
            for (int iSubElement = 0; iSubElement < subElementsList.size(); iSubElement++) {
                createSubElement(subElementsList, iSubElement, elementNode);
            }
        }
    }

    private void createSubElement(ArrayList subElementsList, int iSubElement, IDEComponentPaletteNode elementNode) {
        CustomizerDataElementInterface subElement = (CustomizerDataElementInterface) subElementsList.get(iSubElement);
        MetaClass metaClassSubElement = theEditor.theMetadataDispenser.findClass(subElement.getImplementingClass());

        theEditor.theComponentPalettePane.addClass(elementNode, subElement, metaClassSubElement);
    }


    /**
     * setProperties - sets properties essential for the creation of Customizer objects.
     *
     * @param metaObject
     */
    public void setProperties(MetaObject metaObject, CustomizerDataElementInterface dataElement) {
        if (dataElement != null) {
            metaObject.setPropertyValue("DataTableName", 0, 0, dataElement.getDataTable().getDataTableName(), false);
            metaObject.setPropertyValue("DataColumnName", 0, 0, dataElement.getDataElementName(), false);
            metaObject.setPropertyValue("DataTableContext", 0, 0, dataElement.getDataTable().getDataTableContext(), false);
            metaObject.setPropertyValue("Name", 0, 0, getObjectName(dataElement), false);

            if (isCreatingSubElements) {
                String sGroupElement = dataElement.getParentDataElement().getDataElementName();
                metaObject.setPropertyValue("ElementGroup", 0, 0, sGroupElement, false);
            }

            //newInstanceCreated(dataElement, metaObject);
        }
    }

    /**
     * Inform the Data Element that a new instance has been created
     * @param dataElement the data element to update with news that an object has been created
     * @param metaObject the object that has just been created
     */
    public void newInstanceCreated(CustomizerDataElementInterface dataElement, MetaObject metaObject) {
        dataElement.newInstanceCreated(metaObject, dataElement); // send a callback to perform extra property settings
    }

    /**
     * getObjectName - returns an object name to use for Customizer elements
     *
     * @param dataElement - the element to make the object name for
     * @return - the object name
     */
    public String getObjectName(CustomizerDataElementInterface dataElement) {
        String sObjectName = dataElement.getObjectName();
        if (sObjectName == null || sObjectName.equals(""))
            sObjectName = "component_"; // fail-safe in case Interface doesn't implement name properly

        sObjectName = theEditor.getDesignProject().getNewObjectName(sObjectName);

        return sObjectName;
    }

    public void importElement(MetaObject theDesignObject, Point pt, boolean isShiftKeyDown) {
        IDEComponentPaletteNode selectedNode = theEditor.theComponentPalettePane.theTree.getSelectedNode();
        CustomizerDataElementInterface dataElement = selectedNode.getCustomizerDataElement();
        if (dataElement != null) {
            if (selectedNode.isCustomizerDataElementGroup() == false) {
                isCreatingSubElements = false;
                theEditor.theViewDisplayer.importObject(theDesignObject, new Point(pt.x, pt.y), isShiftKeyDown, dataElement);
            } else {
                isCreatingSubElements = true;
                ArrayList subElementList = dataElement.getListOfDataElements();
                for (int iSubElement = 0; iSubElement < subElementList.size(); iSubElement++) {
                    CustomizerDataElementInterface dataSubElement = ((CustomizerDataElementInterface) subElementList.get(iSubElement));
                    theEditor.theViewDisplayer.importObject(theDesignObject, new Point(pt.x, pt.y + (iSubElement * 32)), isShiftKeyDown, dataSubElement);
                }
            }
            dataElement.setImplementingClass(null);
        }
    }

    public void loadPreferences(DesignCockpitPreferences designCockpitPreferences) {
        ObjectInputStream in = theDataConnection.getPreferencesInputStream();
        writeToLog("Requested getPreferencesInputStream of the DataConnection. Return = ", in);
        if (in != null) {
            try {
                boolean bVersioned = in.readInt() == CUSTOMIZER_VERSIONED;
                if (bVersioned) {
                    int iVersion = in.readInt();
                    if (iVersion == 2) {
                        designCockpitPreferences.setFrameMaximized(in.readBoolean());
                        designCockpitPreferences.setFrameWidth(in.readInt());
                        designCockpitPreferences.setFrameHeight(in.readInt());
                        designCockpitPreferences.setFrameLeft(in.readInt());
                        designCockpitPreferences.setFrameTop(in.readInt());


                        designCockpitPreferences.setWidthColumnOne(in.readInt());
                        designCockpitPreferences.setWidthColumnTwo(in.readInt());
                        designCockpitPreferences.setSplitColumnOne(in.readInt());
                        designCockpitPreferences.setSplitColumnTwo(in.readInt());
                        designCockpitPreferences.setSplitColumnThree(in.readInt());

                        designCockpitPreferences.iSizeBoxHeight = in.readInt();
                        designCockpitPreferences.iSizeBoxWidth = in.readInt();

                        designCockpitPreferences.iViewDisplayerGridY = in.readInt();
                        designCockpitPreferences.iViewDisplayerGridX = in.readInt();

                        designCockpitPreferences.bDisplayGrid = in.readBoolean();
                        designCockpitPreferences.bSnapToGrid = in.readBoolean();

                        theDataConnection.preferencesLoaded(true);
                    } else
                        writeToLog("loadPreference failed as data version not supported");
            } else
                    writeToLog("loadPreference failed as data version not in data stream");

            } catch (IOException e) {
                theDataConnection.preferencesLoaded(false);
            }
        } else
            theDataConnection.preferencesLoaded(false);
    }

    public void savePreferences(DesignCockpitPreferences designCockpitPreferences) {
        ObjectOutputStream out = theDataConnection.getPreferencesOutputStream();
        writeToLog("Requested getPreferencesOutputStream of the DataConnection. Return = ", out);
        if (out != null) {
            try {
                out.writeInt(CUSTOMIZER_VERSIONED);
                out.writeInt(CUSTOMIZER_DATA_VERSION); // this is the version number of the data in the stream

                out.writeBoolean(theEditor.theFramePlacement.isMaximized());

                Dimension frameSize = theEditor.theFramePlacement.getNormalSize();
                Point frameLocation = theEditor.theFramePlacement.getNormalLocation();
                out.writeInt(frameSize.width);
                out.writeInt(frameSize.height);
                out.writeInt(frameLocation.x);
                out.writeInt(frameLocation.y);

                out.writeInt(designCockpitPreferences.getWidthColumnOne());
                out.writeInt(designCockpitPreferences.getWidthColumnTwo());

                out.writeInt(designCockpitPreferences.getSplitColumnOne());
                out.writeInt(designCockpitPreferences.getSplitColumnTwo());
                out.writeInt(designCockpitPreferences.getSplitColumnThree());

                out.writeInt(designCockpitPreferences.iSizeBoxHeight);
                out.writeInt(designCockpitPreferences.iSizeBoxWidth);

                out.writeInt(designCockpitPreferences.iViewDisplayerGridY);
                out.writeInt(designCockpitPreferences.iViewDisplayerGridX);

                out.writeBoolean(designCockpitPreferences.bDisplayGrid);
                out.writeBoolean(designCockpitPreferences.bSnapToGrid);

                theDataConnection.preferencesSaved(true);
            } catch (IOException e) {
                theDataConnection.preferencesSaved(false);
            }
        } else
            theDataConnection.preferencesSaved(false);
    }

    /**
     * This is the string to display in the Customizer's captionbar when editing a Form
     *
     * @return the string to display
     */
    public String getCustomizerTitle() {
        return theDataConnection.getCustomizerTitle();
    }

    /**
     * Selects objects in the Designer that have the same ElementGroup as the selected object(s)
     * This is used to delete objects that have the same ElementGroup
     */
    public void selectElementGroupElements() {
        theDesignProject = theEditor.getDesignProject();

        int iSelectedObjectCount = theDesignProject.getSelectedObjectCount();
        for (int i = 0; i < iSelectedObjectCount; i++) {
            MetaObject selectedObject = theDesignProject.getSelectedObject(i);
            if( selectedObject != null ){ //#ALEX: Sometimes this is null an then it crashes. No idea why
              MetaPropertyValueEx prop = selectedObject.getPropertyValue("ElementGroup", 0);
              if (prop != null)
                  selectObjectsWithGroupElement(prop.getStringValue());
            }
        }
    }

    /**
     * Selects objects in the Designer that have a specific ElementGroup
     *
     * @param sGroupElementName - the name of the ElementGroup
     */
    private void selectObjectsWithGroupElement(String sGroupElementName) {
        groupElementName = sGroupElementName;

        MetaProject metaProject = theDesignProject.theMetaProject;

        MetaObject theDesignObject = theDesignProject.getFirstObject().getFirstChild();

        // we now need to ask each object in the project if it has the same value for the ElementGroup property...
        metaProject.enumerateObjects(this, theDesignObject);
    }

    /**
     * A callback for each MetaObject in the current project.
     * We check the "ElementGroup" property of this object (if it exists), to see it it matches the value that was stored during selectObjectsWithGroupElement.
     * If the value matches, we select the object so that it will be deleted as part of the regular delete.
     *
     * @param metaObject - the MetaObject of the current enumeration
     */
    public void callbackProjectMetaObject(MetaObject metaObject) {
        MetaPropertyValueEx prop = metaObject.getPropertyValue("ElementGroup", 0);
        if (prop != null) {
            if (prop.getStringValue().equals(groupElementName)) {
                if (theDesignProject.isSelected(metaObject) == false)
                    theEditor.SelectObject(metaObject, true);
            }
        }
    }

    public OutputStream getSaveProjectOutputStream() {
        OutputStream outputStream = theDataConnection.getSaveProjectOutputStream();
        writeToLog("Requesting getSaveProjectOutputStream from the Data Connection. Return = ", outputStream);
        return outputStream;
    }

    public void projectSaved(boolean ok) {
        writeToLog("Sending projectSaved to the Data Connection. Boolean OK = " + ok);
        theDataConnection.projectSaved(ok);
    }

    public InputStream getLoadProjectIntputStream() {
        InputStream inputStream = theDataConnection.getLoadProjectInputStream();
        writeToLog("Requesting getLoadProjectInputStream from the Data Connection. Return = ", inputStream);
        return inputStream;
    }

    public void projectLoaded(MetaProject metaProject, boolean ok) {
        metaProject.setGenerateAccessor(false);
        metaProject.setNLSTranslation(false);
        writeToLog("Sending projectLoaded to the Data Connection. Boolean OK = " + ok);
        theDataConnection.projectLoaded(ok);
    }

    /**
     * This method is sent to enumerate each metaobject of a new project.
     * Note: This is not send immediately after a project is loaded, as the MetaObject's visual object will not have
     * been created at that time.
     * @param metaProject the project to enumerate
     */
    void doEnumProjectMetaObjects(MetaProject metaProject) {
        PostLoadMetaObjectsEnumerator projectMetaObjectsEnumerator = new PostLoadMetaObjectsEnumerator(this);
        MetaObject theDesignObject = metaProject.getFirstObject().getFirstChild();

        metaProject.enumerateObjects(projectMetaObjectsEnumerator, theDesignObject);
    }

    public void addToContextMenu(IDEObjectPopupMenu menu, MetaObject metaObject) {
        //System.out.println(metaObject.getPropertyMetadata("DataTableName"));
        if (metaObject.getPropertyMetadata("DataTableName") != null) {   // this is how it knows it is a Customizer-created object
            this.theDataConnection.addToContextMenu(menu, metaObject);
        }
    }

    /**
     * this is called when the DC is closing and the Form needs saving.
     *
     * @return A JOptionPane value (Yes, No, Cancel)
     */
    public int querySaveOnExit() {
        writeToLog("Requesting querySaveOnExit of the DataConnection");
        return theDataConnection.querySaveOnExit();
    }

    public boolean populateDeferredDataElements(CustomizerDataTableInterface dataTable, IDEComponentPaletteNode tableNode) {
        boolean bPopulated = dataTable.populateDeferredDataElements();

        if (bPopulated) {
            tableNode.remove(0); // remember, the first leaf is a dummy placeholder so it must be removed.
            createElements(dataTable, tableNode);
        }

        return bPopulated;
    }

    private void writeToLog(String sMessage) {
        if (bCustomizerLogging) {
            theEditor.theLogFile.doLogEntry("Customizer", sMessage);
        }
    }

    private void writeToLog(String sMessage, Object object) {
        if (bCustomizerLogging) {
            if (object != null)
                writeToLog(sMessage + object.getClass());
            else
                writeToLog(sMessage + "(NULL)");
        }
    }

    public void setCustomizerLogging(boolean bLogging) {
        this.bCustomizerLogging = bLogging;
    }

    public void languageChanged(int ilanguageId) {
        writeToLog("Language Changed To " + HammerLanguagePresentation.getLanguageName(ilanguageId));
        theDataConnection.languageChanged(ilanguageId);
    }

    public Icon getInstanceIcon(MetaObject metaObject) {
        Icon instanceIcon = theDataConnection.getInstanceIcon(metaObject);
        writeToLog("getInstanceIcon returns ", instanceIcon);
        return instanceIcon;
    }

    public String getProzName(){
        String sName = theDataConnection.getProzName();
        if ( sName == null || sName.equalsIgnoreCase ( "")){
            sName = "Customizer.proz";
        }
        writeToLog("getProzName returns ", sName);
        return  sName;

    }

    public void removeFiles(String sOutputDirectoryName, String sClassName) {
        if (theDataConnection.canDeleteCompilerFiles()) {
            if (!sOutputDirectoryName.endsWith(File.separator))
                sOutputDirectoryName += File.separator;

            try {
                File file = new File(sOutputDirectoryName + sClassName +".class");
                file.delete();
                file = new File(sOutputDirectoryName + sClassName +".java");
                file.delete();
            } catch (SecurityException e) {
                e.printStackTrace();
            }
        }
    }

    public String getCompilerOutputDirectory() {
        return System.getProperty("java.io.tmpdir");

    }

    /**
     * Returns the directory to "assume" the project would be in. This will be used to create temp proz files in (part
     * of the save process is to overwrite the proz file with a temp file)
     * @return the directory path
     */
    public String getProjectDirectory() {
        return System.getProperty("java.io.tmpdir");
    }

    public String getImplementingClass(final CustomizerDataElementInterface dataElement) {
        CustomizerAvailableClass[] classes = dataElement.getAvailableImplementingClasses();
        if (classes == null) {
            return dataElement.getImplementingClass();
        }

        // display list of alternative classes for this element...
        final String[] sClassToCreate = new String[1];
        String sCaption = theEditor.theLanguageManager.getMessage("SelectAlternativeClassDialog.label", "Select Class");
        final CustomizerAlternativeClassesDialog dialog = new CustomizerAlternativeClassesDialog(theEditor, sCaption, classes);

        dialog.setModal(true);
        dialog.show();
        //dialog.show(new Runnable() {
        //    public void run() {
                sClassToCreate[0] = dialog.getSelectedClassName();
                dataElement.setImplementingClass(sClassToCreate[0]);
        //    }
        //});
        //while (dialog.isVisible() || sClassToCreate[0] == null) {}
        return sClassToCreate[0];
    }

    public String getObjectDisplayName(MetaObject metaObject) {
        String sDisplayName = theDataConnection.getObjectDisplayName(metaObject);
        return sDisplayName;
    }

    class PostLoadMetaObjectsEnumerator implements EnumerateProjectMetaObjectInterface {
        private final IDECustomizer ideCustomizer;

        public PostLoadMetaObjectsEnumerator(IDECustomizer ideCustomizer) {
            this.ideCustomizer = ideCustomizer;
        }

        public void callbackProjectMetaObject(MetaObject metaObject) {
            this.ideCustomizer.theDataConnection.postLoadMetaObjectEnumerator(metaObject);
        }

    }

}
