/**
 * Title: Comedia Utils. Lexical scanner class.
 * Description: Abstract class for other specific scanners
 * Copyright: Copyright (c) Capella Development Group 2001
 * Company: Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.util.scanner;

/**
 * Abstract class for different specific lexical scanners.
 * Class provides general functionality and does not support
 * specific keywords or datatypes.
 * <p>
 * Example of scanner usage:
 * <p><pre>
 * System.out.println("*********** Scanner Test *************");
 *
 * CScanner scanner = new CScanner();
 * scanner.setBuffer("while(1.0e2*i := \t\r\n> \"string\'\'\")\n"
 *   + "// comment\n/.*second\ncomment*./{xxx}");
 * scanner.setShowEol(true);
 * scanner.setShowSpace(true);
 *
 * // Testing string convertions
 * String str = "The test \"string\"";
 * System.out.println("Start string: " + str);
 * str = scanner.wrapString(str);
 * System.out.println("Wrapped string: " + str);
 * str = scanner.unwrapString(str);
 * System.out.println("Unwrapped string: " + str);
 *
 * System.out.println();
 * System.out.println("Initial string: " + scanner.getBuffer());
 *
 * while (scanner.lex() != EOF) {
 *   switch (scanner.getTokenType()) {
 *     case UNKNOWN: System.out.print("Type: Unknown "); break;
 *     case COMMENT: System.out.print("Type: Comment "); break;
 *     case KEYWORD: System.out.print("Type: Keyword "); break;
 *     case TYPE: System.out.print("Type: Type "); break;
 *     case IDENT: System.out.print("Type: Ident "); break;
 *     case ALPHA: System.out.print("Type: Alpha "); break;
 *     case OPERATOR: System.out.print("Type: Operator "); break;
 *     case BRACE: System.out.print("Type: Brace "); break;
 *     case SEPARATOR: System.out.print("Type: Separator "); break;
 *     case EOL: System.out.print("Type: Eol "); break;
 *     case LF: System.out.print("Type: Lf "); break;
 *     case SPACE: System.out.print("Type: Space "); break;
 *     case INT: System.out.print("Type: Int "); break;
 *     case FLOAT: System.out.print("Type: Float "); break;
 *     case STRING: System.out.print("Type: String "); break;
 *     case BOOL: System.out.print("Type: Bool "); break;
 *     case EOF: System.out.print("Type: Eof "); break;
 *   }
 *   System.out.println("Value: '" + scanner.getToken()
 *     + "' Pos: " + scanner.getPosition() + " Line: " + scanner.getLineNo());
 * }
 * </pre>
 * The result:
 * <p><pre>
 * *********** Scanner Test *************
 * Start string: The test "string"
 * Wrapped string: "The test "string""
 * Unwrapped string: The test "string"
 *
 * Initial string: while(1.0e2*i :=
 * > "string''")
 * // comment
 * /.second
 * comment./{xxx}
 * Type: Ident Value: 'while' Pos: 0 Line: 0
 * Type: Brace Value: '(' Pos: 5 Line: 0
 * Type: Float Value: '1.0e2' Pos: 6 Line: 0
 * Type: Operator Value: '*' Pos: 11 Line: 0
 * Type: Ident Value: 'i' Pos: 12 Line: 0
 * Type: Space Value: ' ' Pos: 13 Line: 0
 * Type: Separator Value: ':' Pos: 14 Line: 0
 * Type: Operator Value: '=' Pos: 15 Line: 0
 * Type: Space Value: ' 	' Pos: 16 Line: 0
 * Type: Lf Value: '
 * ' Pos: 18 Line: 0
 * Type: Eol Value: '
 * ' Pos: 19 Line: 0
 * Type: Operator Value: '>' Pos: 20 Line: 1
 * Type: Space Value: ' ' Pos: 21 Line: 1
 * Type: String Value: '"string''"' Pos: 22 Line: 1
 * Type: Brace Value: ')' Pos: 32 Line: 1
 * Type: Eol Value: '
 * ' Pos: 33 Line: 1
 * Type: Operator Value: '/' Pos: 34 Line: 2
 * Type: Operator Value: '/' Pos: 35 Line: 2
 * Type: Space Value: ' ' Pos: 36 Line: 2
 * Type: Ident Value: 'comment' Pos: 37 Line: 2
 * Type: Eol Value: '
 * ' Pos: 44 Line: 2
 * Type: Operator Value: '/' Pos: 45 Line: 3
 * Type: Operator Value: '*' Pos: 46 Line: 3
 * Type: Ident Value: 'second' Pos: 47 Line: 3
 * Type: Eol Value: '
 * ' Pos: 53 Line: 3
 * Type: Ident Value: 'comment' Pos: 54 Line: 4
 * Type: Operator Value: '*' Pos: 61 Line: 4
 * Type: Operator Value: '/' Pos: 62 Line: 4
 * Type: Brace Value: '{' Pos: 63 Line: 4
 * Type: Ident Value: 'xxx' Pos: 64 Line: 4
 * Type: Brace Value: '}' Pos: 67 Line: 4
 * </pre>
 */
public class CScanner {
  /**
   * Unknown token constant.
   */
  public final static int UNKNOWN   = 0x0000;
  /**
   * Comment string token constant.
   */
  public final static int COMMENT   = 0x0001;
  /**
   * Keyword token constant. It depends on supported language syntax.
   */
  public final static int KEYWORD   = 0x0002;
  /**
   * Data type keyword token constant. It depends on supported language syntax.
   */
  public final static int TYPE      = 0x0004;
  /**
   * Identifier token constant. Rules depends on supported language syntax.
   */
  public final static int IDENT     = 0x0008;
  /**
   * Constant which covers <code>COMMENT, KEYWORD, TYPE</code> or
   * <code>IDENT</code> tokens.
   */
  public final static int ALPHA     = 0x000E;
  /**
   * Operator token constant. It depends on supported language syntax.
   * Can be simple ('-', '=' or '%'), complex ('>=', '<=') or presented by
   * keyword ('or', 'and').
   */
  public final static int OPERATOR  = 0x0010;
  /**
   * Different brace token constant. Braces can be square '[]', round '()' or
   * curve '{}'.
   */
  public final static int BRACE     = 0x0020;
  /**
   * Different lexem seperators token constant. Separators are presented by
   * ',', '.', ':' or ';' characters depends on supported language syntax.
   */
  public final static int SEPARATOR = 0x0040;
  /**
   * CHAR(13) token constant.
   */
  public final static int EOL       = 0x0080;
  /**
   * CHAR(10) token constant.
   */
  public final static int LF        = 0x00E0;
  /**
   * Space token constant. Token can consist with spaces and tab symbols.
   */
  public final static int SPACE     = 0x0100;
  /**
   * Constant which covers <code>OPERATOR, BRACE, SEPARATOR, EOL, LN</code>
   * and <code>SPACE</code> tokens.
   */
  public final static int DELIM     = 0x01F0;
  /**
   * Integer constant token.
   */
  public final static int INT       = 0x0200;
  /**
   * Float constant token.
   */
  public final static int FLOAT     = 0x0400;
  /**
   * String constant token. Escape symbols depend on supported language syntax.
   */
  public final static int STRING    = 0x0800;
  /**
   * Boolean constant token.
   */
  public final static int BOOL      = 0x1000;
  /**
   * Constant which covers all token constants: <code>INT, FLOAT, STRING</code>
   * and <code>BOOL</code>
   */
  public final static int CONST     = 0x1E00;
  /**
   * End-Of-File token constant. It means end of input stream and end
   * of parsing process.
   */
  public final static int EOF       = 0x8000;

  /**
   * Presents extracted token with information about token type and position
   * in input stream. This class is used as "Holder" to pass parameter
   * by reference into methods of class.
   */
  protected class Lexem {
    /**
     * Token string value.
     */
    public String token = "";
    /**
     * Token type value contained special constant.
     */
    public int tokenType = UNKNOWN;
    /**
     * Position in the input stream of the first token character.
     */
    public int position = 0;
    /**
     * Line number in teh input stream where token is started.
     */
    public int lineNo = 0;
  }

  /**
   * Buffer which contains input stream.
   */
  protected String buffer;
  /**
   * Pointer to current position in the input stream.
   */
  protected int bufferPos;
  /**
   * Current precessed line in the input stream.
   */
  protected int bufferLine;
  /**
   * The length of the input stream.
   */
  protected int bufferLen;
  /**
   * "Holder" class which contains current extracted token.
   */
  protected Lexem current;
  /**
   * "Holder" class which contains next available token.
   */
  protected Lexem next;
  /**
   * It means show or hide comment tokens. It is <code>FALSE</code> by default.
   */
  protected boolean showComment = false;
  /**
   * It shows how to present extracted string tokens: in ordinal
   * or escape format. <code>TRUE</code> means to present string in escape format.
   * It is <code>TRUE</code> by default.
   */
  protected boolean showString = true;
  /**
   * It means show or hide EOL/LF tokens. It is <code>FALSE</code> by default.
   */
  protected boolean showEol = false;
  /**
   * It shows do make a search for keywords or present them as identifiers.
   * It is <code>TRUE</code> by default.
   */
  protected boolean showKeyword = true;
  /**
   * It shows do make a search for data type keywords or present them
   * as identifiers. It is <code>TRUE</code> by default.
   */
  protected boolean showType = true;
  /**
   * It means show or hide space tokens. It is <code>FALSE</code> by default.
   */
  protected boolean showSpace = false;
  /**
   * List of language specified operators.
   */
  protected String operators[];
  /**
   * List of language specified data type keywords.
   */
  protected String types[];
  /**
   * List of language specified reserved keywords.
   */
  protected String keywords[];

  /**
   * Default class constructor.
   */
  public CScanner() {
    setBuffer("");
  }

  /**
   * Gets a lowlevel token. Presents the main parsing process and should
   * be overrided by specific language scanners.
   * @param curr a "Holder" which containes extracted token.
   * @result extracted token type represented by special constant.
   */
  protected int lowRunLex(Lexem curr) {
    char temp, quote;
    int result = innerStartLex(curr);
    if (result != UNKNOWN) return result;

    temp = curr.token.charAt(0);
    // Check for brace
    if ((new String("(){}[]").indexOf(temp)) >= 0)
      return (curr.tokenType = BRACE);
    // Check for separator
    if ((temp == ',') || (temp == '.') || (temp == ';') || (temp == ':'))
      return (curr.tokenType = SEPARATOR);
    // Check for delimiters
    if (isDelim(temp))
      return (curr.tokenType = OPERATOR);
    // Check for string
    if (isQuote(temp)) {
      quote = temp;
      while (bufferPos < bufferLen) {
        temp = buffer.charAt(bufferPos);
        if (temp == '\n' || temp == '\r') break;
        curr.token += temp;
        bufferPos++;
        if (temp == quote) break;
      }
      return (curr.tokenType = STRING);
    }
    // Check for digits and identifiers
    {
      result = isDigit(temp)? INT: IDENT;
      while (bufferPos < bufferLen) {
        temp = buffer.charAt(bufferPos);
        if (isDelim(temp) || isQuote(temp))
          break;
        if ((result == INT) && (temp == '.'))
          result = FLOAT;
        curr.token += temp;
        bufferPos++;
      }
      return (curr.tokenType = result);
    }
  }

  /**
   * Extracts next available token from the input stream.
   * It can extract "current" or "next" token depends iplementation logic.
   * @param curr a "Holder" class which contains extracted token.
   * @result extracted token type represented by special constant.
   */
  protected int runLex(Lexem curr) {
    // Get current token
    int result;
    do {
      result = lowRunLex(curr);
      if (((result == EOL) || (result == LF)) && showEol)
        break;
      if ((result == COMMENT) && showComment)
        break;
    } while ((result == EOL) || (result == LF) || (result == COMMENT));
    // Convert string if needed
    if ((result == STRING) && !showString)
      curr.token = unwrapString(curr.token);
    return (curr.tokenType = result);
  }

  /**
   * Extract "current" token or copies it from "next" token if it is available.
   */
  protected void extractToken() {
    if ((next.token.length() != 0) && (next.tokenType != UNKNOWN)) {
      // Move next token to current token
      current.tokenType = next.tokenType;
      current.lineNo = next.lineNo;
      current.position = next.position;
      current.token = next.token;
      // Clear next token
      next = new Lexem();
    } else
      runLex(current);
  }

  /**
   * Extracts "next" token from the input stream.
   */
  protected void extractNextToken() {
    if ((next.token.length() == 0) && (next.tokenType == UNKNOWN))
      runLex(next);
  }

  /**
   * Starts the first stage of lexical parsing.
   * It initializes a token and skips white spaces from the current
   * position of the input stream. Each parsing process should begins
   * with this method.
   * @param curr a "Holder" class which contains an extracting token.
   * @result a special token type constant if token was extracted on
   * the first stage of <code>UNKNOWN</code> constant otherwise.
   */
  protected int innerStartLex(Lexem curr) {
    // Initialize values
    curr.lineNo = bufferLine;
    curr.position = bufferPos;
    curr.token = "";

    // Check position
    if (bufferPos >= bufferLen)
      return (curr.tokenType = EOF);

    // Process whitespaces
    if (showSpace && isWhite(buffer.charAt(bufferPos))) {
      do {
        curr.token += buffer.charAt(bufferPos);
        bufferPos++;
      } while ((bufferPos < bufferLen) && isWhite(buffer.charAt(bufferPos)));
      return (curr.tokenType = SPACE);
    }

    // Skip whitespaces
    while (isWhite(buffer.charAt(bufferPos))) {
      bufferPos++;
      if (bufferPos >= bufferLen) {
        curr.position = bufferPos;
        return (curr.tokenType = EOF);
      }
    }
    curr.position = bufferPos;
    curr.token = "" + buffer.charAt(bufferPos);
    bufferPos++;

    // Check for LF
    if (curr.token.charAt(0) == '\r')
      return (curr.tokenType = LF);

    // Check for EOL
    if (curr.token.charAt(0) == '\n') {
      bufferLine++;
      return (curr.tokenType = EOL);
    }

    return (curr.tokenType = UNKNOWN);
  }

  /**
   * Processes the rest single-line comment. The method can be used in
   * defferent lexical procedures to skip the rest of the line.
   * @param curr a "Holder" class whci contains an extracting token.
   * @result extracted token type represented by special constant.
   */
  protected int innerProcLineComment(Lexem curr) {
    while (bufferPos < bufferLen) {
      char temp = buffer.charAt(bufferPos);
      curr.token += temp;
      bufferPos++;
      if (temp == '\n') {
        bufferLine++;
        break;
      }
    }
    return (curr.tokenType = COMMENT);
  }

  /**
   * Parses C-like multi-line comment. Comment starts with '/.*'
   * and ends with '*./'.
   * @param curr a "Holder" class whci contains an extracting token.
   * @result extracted token type represented by special constant.
   */
  protected int innerProcCComment(Lexem curr) {
    curr.tokenType = UNKNOWN;
    if ((curr.token.charAt(0) == '/') && (bufferPos < bufferLen)
      && (buffer.charAt(bufferPos) == '*')) {
      curr.tokenType = COMMENT;
      char temp1 = (char)0;
      while (bufferPos < bufferLen) {
        char temp = buffer.charAt(bufferPos);
        curr.token += temp;
        bufferPos++;
        if ((temp == '/') && (temp1 == '*') && (curr.token.length() > 3))
          break;
        if (temp == '\n') bufferLine++;
        temp1 = temp;
      }
    }
    return curr.tokenType;
  }

  /**
   * Parses an identificator or numeric constant tokens.
   * Indetifier starts with alpha and can contain alphas, digits or
   * special characters like '_$'. Numeric contants stars with digit and
   * can contain except digit also alphas. Float contants can also contain
   * colon symbol ('.') inside.
   * @param curr a "Holder" class which contains an extracting token.
   * @result extracted token type represented by special constant.
   */
  protected int innerProcIdent(Lexem curr) {
    char temp = curr.token.charAt(0);
    curr.tokenType = UNKNOWN;
    if (isDigit(temp) || isLetter(temp)
      || ((new String("._,$").indexOf(temp)) >= 0)) {
      if (temp == '.') {
        if ((bufferPos < bufferLen) && isDigit(buffer.charAt(bufferPos)))
          curr.tokenType = FLOAT;
        else
          return (curr.tokenType = SEPARATOR);
      }
      else if (isDigit(temp)) curr.tokenType = INT;
      else curr.tokenType = IDENT;

      while (bufferPos < bufferLen) {
        temp = buffer.charAt(bufferPos);
        if (!(isDigit(temp) || isLetter(temp)
          || ((new String("._,$").indexOf(temp)) >= 0)))
          break;
        if ((curr.tokenType == INT) && (temp == '.'))
          curr.tokenType = FLOAT;
        else if (temp == '.')
          break;
        curr.token += temp;
        bufferPos++;
      }
    }
    return curr.tokenType;
  }

  /**
   * Parses a string. String should be limited with single or double quotes.
   * @param curr a "Holder" class which contains an extracting token.
   * @result extracted token type represented by special constant.
   */
  protected int innerProcString(Lexem curr) {
    curr.tokenType = UNKNOWN;
    char temp = curr.token.charAt(0);
    if (isQuote(temp)) {
      curr.tokenType = STRING;
      char quote = temp;
      while (bufferPos < bufferLen) {
        temp = buffer.charAt(bufferPos);
        if (temp == '\n' || temp == '\r') break;
        curr.token += temp;
        bufferPos++;
        if (temp == quote) break;
      }
    }
    return curr.tokenType;
  }

  /**
   * Parses C-like escape string. String should be limited with double quotes
   * ('"') and contain special characters in c-like escape format
   * (CHAR(13) -> '\n', '"' -> '\"', etc).
   * @param curr a "Holder" class which contains an extracting token.
   * @result extracted token type represented by special constant.
   */
  protected int innerProcCString(Lexem curr) {
    curr.tokenType = UNKNOWN;
    char temp = curr.token.charAt(0);
    if (isQuote(temp)) {
      curr.tokenType = STRING;
      char quote = temp;
      while (bufferPos < bufferLen) {
        temp = buffer.charAt(bufferPos);
        if (temp == '\n' || temp == '\r') break;
        curr.token += temp;
        bufferPos++;
        if ((temp == '\\') && ((bufferPos < bufferLen)
          && (buffer.charAt(bufferPos) == quote))) {
          curr.token += quote;
          bufferPos++;
        } else if (temp == quote)
          break;
      }
    }
    return curr.tokenType;
  }

  /**
   * Parses Pascal-like escape string. String should be limited with single quotes
   * (''') and contain double all single quotes inside.
   * @param curr a "Holder" class which contains an extracting token.
   * @result extracted token type represented by special constant.
   */
  protected int innerProcPasString(Lexem curr) {
    curr.tokenType = UNKNOWN;
    char temp = curr.token.charAt(0);
    if (isQuote(temp)) {
      curr.tokenType = STRING;
      char quote = temp;
      while (bufferPos < bufferLen) {
        temp = buffer.charAt(bufferPos);
        if (temp == '\n' || temp == '\r') break;
        curr.token += temp;
        bufferPos++;
        if ((temp == quote) && ((bufferPos < bufferLen)
          && (buffer.charAt(bufferPos) == quote))) {
          curr.token += quote;
          bufferPos++;
        } else if (temp == quote)
          break;
      }
    }
    return curr.tokenType;
  }

  /**
   * Searches a string value inside a string array.
   * This method is used for searching registered keywords, operators or
   * data types.
   * @param s a searching string value.
   * @param a a string array.
   * @result the result of the searching.
   */
  protected boolean searchForString(String s, String a[]) {
    if (a != null) {
      for (int i = 0; i < a.length; i++)
        if (s.equals(a[i]))
          return true;
    }
    return false;
  }

  /**
   * Restarts the parsing process by reassinging the same input buffer.
   */
  public void restart() {
    setBuffer(buffer);
  }

  /**
   * Converts a string from ordinary into escape format limited with quotes.
   * Escape format depends on supported language syntax.
   * @param s a string in ordinary (local) presentation.
   * @result a result string in special escape format.
   */
  public static String wrapString(String s) {
    return "\"" + s + "\"";
  }

  /**
   * Converts a string from special escape format limited with quotes
   * into oridinary (local) presentation.
   * Escape format depends on supported language syntax.
   * @param s a string in special escape format.
   * @result a result string in ordinary (local) presentation.
   */
  public static String unwrapString(String s) {
    if (s.length() == 0) return s;
    char quote = s.charAt(0);
    int start = 0, end = s.length()-1;
    if (quote == '\"' || quote == '\'') {
      start++;
      if (end > 0 && s.charAt(end) == quote)
        end--;
    }
    return s.substring(start, end+1);
  }

  /**
   * Checks is character an alpha.
   * Alpha means some letter which is not a white space, delimiter or digit.
   * @param c a checking character.
   * @result the result of the checking.
   */
  public static boolean isAlpha(char c) {
    return !((c < ' ') || isDelim(c) || isDigit(c));
  }

  /**
   * Checks is character a letter. Letters can be only from latin alphabet and
   * do not include letters from other alphabets.
   * @param c a checking character.
   * @result the result of the checking.
   */
  public static boolean isLetter(char c) {
    return ((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z'));
  }

  /**
   * Checks is character a digit.
   */
  public static boolean isDigit(char c) {
    return ((c >= '0') && (c <= '9'));
  }

  /**
   * Checks is character a delimiter.
   * @param c a checking character.
   * @result the result of the checking.
   */
  public static boolean isDelim(char c){
    return (new String(":;,+-<>/*%^=()[]|&~@#$\\`{}!? \t\n\r").indexOf(c)) >= 0;
  }

  /**
   * Checks is character a white space.
   * @param c a checking character.
   * @result the result of the checking.
   */
  public static boolean isWhite(char c) {
    return ((c == ' ') || (c == '\t'));
  }

  /**
   * Checks is character EOL (CHAR(13) symbol.
   * @param c a checking character.
   * @result the result of the checking.
   */
  public static boolean isEol(char c) {
    return (c == '\n');
  }

  /**
   * Checks is character a quote. Quote are represented by ''' or '"' characters.
   * @param c a checking character.
   * @result the result of the checking.
   */
  public static boolean isQuote(char c) {
    return (c == '\"') || (c == '\'');
  }

  /**
   * Starts the parsing process and extract a current token.
   * @result a current token type reresented by special constant.
   */
  public int lex() {
    extractToken();
    return current.tokenType;
  }

  /**
   * Continues the parsing process and extracts a current token.
   * It means the same as <code>lex</code> method.
   * @result a current token type reresented by special constant.
   */
  public int gotoNextToken() {
    return lex();
  }

  /**
   * Gets a <code>ShowComment</code> property value.
   * <code>ShowComment</code> means show or hide comment tokens.
   * It is <code>FALSE</code> by default.
   * @result a <code>ShowComment</code> property value.
   */
  public boolean isShowComment() {
    return showComment;
  }

  /**
   * Sets a new <code>ShowComment</code> property value.
   * <code>ShowComment</code> means show or hide comment tokens.
   * It is <code>FALSE</code> by default.
   * @param value a new <code>ShowComment</code> property value.
   */
  public void setShowComment(boolean value) {
    showComment = value;
  }

  /**
   * Gets a <code>ShowEol</code> property value.
   * <code>ShowEol</code> means show or hide EOL/LF tokens.
   * It is <code>FALSE</code> by default.
   * @result a <code>ShowEol</code> property value.
   */
  public boolean isShowEol() {
    return showEol;
  }

  /**
   * Sets a new <code>ShowEol</code> property value.
   * <code>ShowEol</code> means show or hide EOL/LF tokens.
   * It is <code>FALSE</code> by default.
   * @param value a new <code>ShowEol</code> property value.
   */
  public void setShowEol(boolean value) {
    showEol = value;
  }

  /**
   * Gets a <code>ShowString</code> property value.
   * <code>ShowString</code> shows how to present extracted string tokens:
   * in ordinal or escape format. <code>TRUE<code> means to present string
   * in escape format. It is <code>TRUE</code> by default.
   * @result a <code>ShowString</code> property value.
   */
  public boolean isShowString() {
    return showString;
  }

  /**
   * Sets a new <code>ShowString</code> property value.
   * <code>ShowString</code> shows how to present extracted string tokens:
   * in ordinal or escape format. <code>TRUE<code> means to present string
   * in escape format. It is <code>TRUE</code> by default.
   * @param value a new <code>ShowString</code> property value.
   */
  public void setShowString(boolean value) {
    showString = value;
  }

  /**
   * Gets a <code>ShowKeyword</code> property value.
   * <code>ShowKeyword</code> shows do make a search for keywords or present
   * them as identifiers. It is <code>TRUE</code> by default.
   * @result a <code>ShowKeyword</code> property value.
   */
  public boolean isShowKeyword() {
    return showKeyword;
  }

  /**
   * Sets a new <code>ShowKeyword</code> property value.
   * <code>ShowKeyword</code> shows do make a search for keywords or present
   * them as identifiers. It is <code>TRUE</code> by default.
   * @param value a new <code>ShowKeyword</code> property value.
   */
  public void setShowKeyword(boolean value) {
    showKeyword = value;
  }

  /**
   * Gets a <code>ShowType</code> property value.
   * <code>ShowType</code> shows do make a search for data type keywords
   * or present them as identifiers. <code>ShowType</code> is <code>TRUE</code>
   * by default.
   * @result a <code>ShowType</code> property value.
   */
  public boolean isShowType() {
    return showType;
  }

  /**
   * Sets a new <code>ShowType</code> property value.
   * <code>ShowType</code> shows do make a search for data type keywords
   * or present them as identifiers. <code>ShowType</code> is <code>TRUE</code>
   * by default.
   * @param value a new <code>ShowType</code> property value.
   */
  public void setShowType(boolean value) {
    showType = value;
  }

  /**
   * Gets a <code>ShowSpace</code> property value.
   * <code>ShowSpace</code> means show or hide space tokens.
   * It is <code>FALSE</code> by default.
   * @result a <code>ShowSpace</code> property value.
   */
  public boolean isShowSpace() {
    return showSpace;
  }

  /**
   * Sets a new <code>ShowSpace</code> property value.
   * <code>ShowSpace</code> means show or hide space tokens.
   * It is <code>FALSE</code> by default.
   * @param value a new <code>ShowSpace</code> property value.
   */
  public void setShowSpace(boolean value) {
    showSpace = value;
  }

  /**
   * Gets an input buffer string.
   * @result an input buffer string value.
   */
  public String getBuffer() {
    return buffer.toString();
  }

  /**
   * Sets a new input buffer and resets buffer pointers.
   * @param s a new input stream.
   */
  public void setBuffer(String s) {
    buffer = s;
    bufferLen = s.length();
    bufferPos = 0;
    bufferLine = 0;
    current = new Lexem();
    current.tokenType = EOF;
    next = new Lexem();
  }

  /**
   * Gets a current position in the input stream.
   * @result a current position in the input stream.
   */
  public int getBufferPos() {
    return bufferPos;
  }

  /**
   * Gets position ot the first character of a current token in the input stream.
   * @result position of a current token in the input stream.
   */
  public int getPosition() {
    return current.position;
  }

  /**
   * Gets a line number of the first character of a current token.
   * @result line number of a current token.
   */
  public int getLineNo() {
    return current.lineNo;
  }

  /**
   * Gets a current token value.
   * @result a current token string value.
   */
  public String getToken() {
    return current.token;
  }

  /**
   * Gets a current token type represented by special constant.
   * @result a current token type value.
   */
  public int getTokenType() {
    return current.tokenType;
  }

  /**
   * Gets position ot the first character of a next token in the input stream.
   * @result position of a next token in the input stream.
   */
  public int getNextPosition() {
    extractNextToken();
    return next.position;
  }

  /**
   * Gets a line number of the first character of a next token.
   * @result line number of a next token.
   */
  public int getNextLineNo() {
    extractNextToken();
    return next.lineNo;
  }

  /**
   * Gets a next token value.
   * @result a next token string value.
   */
  public String getNextToken() {
    extractNextToken();
    return next.token;
  }

  /**
   * Gets a next token type represented by special constant.
   * @result a next token type value.
   */
  public int getNextTokenType() {
    extractNextToken();
    return next.tokenType;
  }

  /**
   * The main function for test purposes.
   */
  public static void main(String[] args) {
    System.out.println("*********** Scanner Test *************");

    CScanner scanner = new CScanner();
    scanner.setBuffer("while(1.0e2*i := \t\r\n> \"string\'\'\")\n"
      + "// comment\n/*second\ncomment*/{xxx}");
    scanner.setShowEol(true);
    scanner.setShowSpace(true);

    // Testing string convertions
    String str = "The test \"string\"";
    System.out.println("Start string: " + str);
    str = scanner.wrapString(str);
    System.out.println("Wrapped string: " + str);
    str = scanner.unwrapString(str);
    System.out.println("Unwrapped string: " + str);

    System.out.println();
    System.out.println("Initial string: " + scanner.getBuffer());

    while (scanner.lex() != EOF) {
      switch (scanner.getTokenType()) {
        case UNKNOWN: System.out.print("Type: Unknown "); break;
        case COMMENT: System.out.print("Type: Comment "); break;
        case KEYWORD: System.out.print("Type: Keyword "); break;
        case TYPE: System.out.print("Type: Type "); break;
        case IDENT: System.out.print("Type: Ident "); break;
        case ALPHA: System.out.print("Type: Alpha "); break;
        case OPERATOR: System.out.print("Type: Operator "); break;
        case BRACE: System.out.print("Type: Brace "); break;
        case SEPARATOR: System.out.print("Type: Separator "); break;
        case EOL: System.out.print("Type: Eol "); break;
        case LF: System.out.print("Type: Lf "); break;
        case SPACE: System.out.print("Type: Space "); break;
        case INT: System.out.print("Type: Int "); break;
        case FLOAT: System.out.print("Type: Float "); break;
        case STRING: System.out.print("Type: String "); break;
        case BOOL: System.out.print("Type: Bool "); break;
        case EOF: System.out.print("Type: Eof "); break;
      }
      System.out.println("Value: '" + scanner.getToken()
        + "' Pos: " + scanner.getPosition() + " Line: " + scanner.getLineNo());
    }
  }
}
