/**
 * Title: Comedia Utils. Java scanner class.
 * Description: Lexical scanner for Java language
 * Copyright: Copyright (c) Capella Development Group 2001
 * Company: Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.util.scanner;

/**
 * Performs lexical scanning for Java-like languages.
 * Scanner supports all standard operators, keywords or datatypes of Java.
 * <p>
 * Example of scanner usage:
 * <p><pre>
 * System.out.println("*********** Java Scanner Test *************");
 *
 * CJavaScanner scanner = new CJavaScanner();
 * scanner.setBuffer("final(1.0e2=i.a >>= \t\r\n-> \"string\\\"\")\n"
 *   + "// comment\n/.*second\ncomment*./{xxx}");
 * scanner.setShowEol(true);
 * scanner.setShowSpace(true);
 * scanner.setShowComment(true);
 * scanner.setShowKeyword(true);
 * scanner.setShowType(true);
 *
 * // Testing string convertions
 * String str = "The test \"string\"";
 * System.out.println("Start string: " + str);
 * str = scanner.wrapString(str);
 * System.out.println("Wrapped string: " + str);
 * str = scanner.unwrapString(str);
 * System.out.println("Unwrapped string: " + str);
 *
 * System.out.println();
 * System.out.println("Initial string: " + scanner.getBuffer());
 *
 * while (scanner.lex() != EOF) {
 *   switch (scanner.getTokenType()) {
 *     case UNKNOWN: System.out.print("Type: Unknown "); break;
 *     case COMMENT: System.out.print("Type: Comment "); break;
 *     case KEYWORD: System.out.print("Type: Keyword "); break;
 *     case TYPE: System.out.print("Type: Type "); break;
 *     case IDENT: System.out.print("Type: Ident "); break;
 *     case ALPHA: System.out.print("Type: Alpha "); break;
 *     case OPERATOR: System.out.print("Type: Operator "); break;
 *     case BRACE: System.out.print("Type: Brace "); break;
 *     case SEPARATOR: System.out.print("Type: Separator "); break;
 *     case EOL: System.out.print("Type: Eol "); break;
 *     case LF: System.out.print("Type: Lf "); break;
 *     case SPACE: System.out.print("Type: Space "); break;
 *     case INT: System.out.print("Type: Int "); break;
 *     case FLOAT: System.out.print("Type: Float "); break;
 *     case STRING: System.out.print("Type: String "); break;
 *     case BOOL: System.out.print("Type: Bool "); break;
 *     case EOF: System.out.print("Type: Eof "); break;
 *   }
 *   System.out.println("Value: '" + scanner.getToken()
 *     + "' Pos: " + scanner.getPosition() + " Line: " + scanner.getLineNo());
 * }
 * </pre>
 * The result:
 * <p><pre>
 * *********** Java Scanner Test *************
 * Start string: The test "string"
 * Wrapped string: "The test \"string\""
 * Unwrapped string: The test "string"
 *
 * Initial string: final(1.0e2=i.a >>=
 * -> "string\"")
 * // comment
 * /.*second
 * comment*./{xxx}
 * Type: Keyword Value: 'final' Pos: 0 Line: 0
 * Type: Brace Value: '(' Pos: 5 Line: 0
 * Type: Float Value: '1.0e2' Pos: 6 Line: 0
 * Type: Operator Value: '=' Pos: 11 Line: 0
 * Type: Ident Value: 'i' Pos: 12 Line: 0
 * Type: Separator Value: '.' Pos: 13 Line: 0
 * Type: Ident Value: 'a' Pos: 14 Line: 0
 * Type: Space Value: ' ' Pos: 15 Line: 0
 * Type: Operator Value: '>>=' Pos: 16 Line: 0
 * Type: Space Value: ' 	' Pos: 19 Line: 0
 * Type: Lf Value: '
 * ' Pos: 21 Line: 0
 * Type: Eol Value: '
 * ' Pos: 22 Line: 0
 * Type: Operator Value: '-' Pos: 23 Line: 1
 * Type: Operator Value: '>' Pos: 24 Line: 1
 * Type: Space Value: ' ' Pos: 25 Line: 1
 * Type: String Value: '"string\""' Pos: 26 Line: 1
 * Type: Brace Value: ')' Pos: 36 Line: 1
 * Type: Eol Value: '
 * ' Pos: 37 Line: 1
 * Type: Comment Value: '// comment
 * ' Pos: 38 Line: 2
 * Type: Comment Value: '/*second
 * comment*./' Pos: 49 Line: 3
 * Type: Brace Value: '{' Pos: 67 Line: 4
 * Type: Ident Value: 'xxx' Pos: 68 Line: 4
 * Type: Brace Value: '}' Pos: 71 Line: 4
 * </pre>
 */
public class CJavaScanner extends CCppScanner {
  /**
   * List of Java specified data type keywords.
   */
  String javaTypes[] = {
    "int","long","short","char","boolean","float","double","byte","void"
  };
  /**
   * List of Java specified reserved keywords.
   */
  String javaKeywords[] = {
    "return","if","else","while","do","switch","case","default","typedef",
    "class","true","false","break","continue","for","static","public",
    "private","protected","extends","implements","interface","final","null",
    "package","import","instanceof","new","try","catch","finally","this"
  };

  /**
   * Default class constructor.
   */
  public CJavaScanner() {
    super();
    keywords = javaKeywords;
    types = javaTypes;
  }

  /**
   * Gets a lowlevel token. Presents the main parsing process.
   * @param curr a "Holder" which containes extracted token.
   * @result extracted token type represented by special constant.
   */
  protected int lowRunLex(Lexem curr) {
    super.lowRunLex(curr);
    if (curr.token.equals("->")) {
      curr.token = "-";
      bufferPos--;
    }
    return curr.tokenType;
  }

  /**
   * The main function for test purposes.
   */
  public static void main(String[] args) {
    System.out.println("*********** Java Scanner Test *************");

    CJavaScanner scanner = new CJavaScanner();
    scanner.setBuffer("final(1.0e2=i.a >>= \t\r\n-> \"string\\\"\")\n"
      + "// comment\n/*second\ncomment*/{xxx}");
    scanner.setShowEol(true);
    scanner.setShowSpace(true);
    scanner.setShowComment(true);
    scanner.setShowKeyword(true);
    scanner.setShowType(true);

    // Testing string convertions
    String str = "The test \"string\"";
    System.out.println("Start string: " + str);
    str = scanner.wrapString(str);
    System.out.println("Wrapped string: " + str);
    str = scanner.unwrapString(str);
    System.out.println("Unwrapped string: " + str);

    System.out.println();
    System.out.println("Initial string: " + scanner.getBuffer());

    while (scanner.lex() != EOF) {
      switch (scanner.getTokenType()) {
        case UNKNOWN: System.out.print("Type: Unknown "); break;
        case COMMENT: System.out.print("Type: Comment "); break;
        case KEYWORD: System.out.print("Type: Keyword "); break;
        case TYPE: System.out.print("Type: Type "); break;
        case IDENT: System.out.print("Type: Ident "); break;
        case ALPHA: System.out.print("Type: Alpha "); break;
        case OPERATOR: System.out.print("Type: Operator "); break;
        case BRACE: System.out.print("Type: Brace "); break;
        case SEPARATOR: System.out.print("Type: Separator "); break;
        case EOL: System.out.print("Type: Eol "); break;
        case LF: System.out.print("Type: Lf "); break;
        case SPACE: System.out.print("Type: Space "); break;
        case INT: System.out.print("Type: Int "); break;
        case FLOAT: System.out.print("Type: Float "); break;
        case STRING: System.out.print("Type: String "); break;
        case BOOL: System.out.print("Type: Bool "); break;
        case EOF: System.out.print("Type: Eof "); break;
      }
      System.out.println("Value: '" + scanner.getToken()
        + "' Pos: " + scanner.getPosition() + " Line: " + scanner.getLineNo());
    }
  }
}
