/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.ui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import org.comedia.text.*;

/**
 * Implements an MS Access like navigator for tables.
 * <p><img src="CTableNavigator.gif">
 * <p>Usage example:
 * <pre>
 * JFrame frame = new JFrame("Comedia Data Grid Test");
 *
 * JTable table = new JTable(
 *   new String[][] {{"1x1", "1x2","1x3"}, {"2x1", "2x2", "2x3"},
 *   {"3x1", "3x2", "3x3"}, {"4x1", "4x2", "4x3"}},
 *   new String[] {"Column 1", "Column 2", "Column 3"});
 * CTableScroller scroll = new CTableScroller(table);
 * frame.getContentPane().add(scroll, BorderLayout.CENTER);
 *
 * CTableNavigator navigator = new CTableNavigator(table);
 * frame.getContentPane().add(navigator, BorderLayout.SOUTH);
 * </pre>
 */
public class CTableNavigator extends JPanel implements ListSelectionListener,
  TableModelListener, ActionListener, FocusListener {

  JLabel captionLabel = new JLabel();
  JButton firstButton = new JButton();
  JButton priorButton = new JButton();
  JTextField recordText = new JTextField();
  JButton nextButton = new JButton();
  JButton lastButton = new JButton();
  JButton lastNewButton = new JButton();
  JLabel recordLabel = new JLabel();
  FlowLayout flowLayout1 = new FlowLayout();
  Border border1;
  JPanel splitPanel1 = new JPanel();
  JPanel splitPanel2 = new JPanel();

  /**
   * The related table control.
   */
  JTable table = null;

  /**
   * The last selection row.
   */
  private int lastSelection = 0;

  /**
   * The integer model for text editor.
   */
  private CNumericDocument document = new CNumericDocument();


  /**
   * Constructs this panel with default properties.
   */
  public CTableNavigator() {
    try {
      jbInit();
    }
    catch(Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Constructs this panel and assignes the main parameters.
   * @param table a related table control.
   */
  public CTableNavigator(JTable table) {
    this();
    setTable(table);
  }

  /**
   * Initializes visual controls of this console. This method is managed by
   * JBuilder visial designers.
   */
  private void jbInit() throws Exception {
    captionLabel.setText("  Record:  ");
    priorButton.setPreferredSize(new Dimension(21, 17));
    priorButton.setActionCommand("Prior");
    firstButton.setPreferredSize(new Dimension(21, 17));
    firstButton.setActionCommand("First");
    recordText.setPreferredSize(new Dimension(80, 19));
    nextButton.setPreferredSize(new Dimension(21, 17));
    nextButton.setActionCommand("Next");
    lastButton.setPreferredSize(new Dimension(21, 17));
    lastButton.setActionCommand("Last");
    lastNewButton.setPreferredSize(new Dimension(21, 17));
    lastNewButton.setActionCommand("LastNew");
    recordLabel.setText("  of 0");
    this.setLayout(flowLayout1);
    flowLayout1.setAlignment(FlowLayout.LEFT);
    flowLayout1.setHgap(0);
    flowLayout1.setVgap(0);
    splitPanel1.setPreferredSize(new Dimension(5, 10));
    splitPanel2.setPreferredSize(new Dimension(5, 10));
    this.setMaximumSize(new Dimension(32767, 19));
    this.setMinimumSize(new Dimension(270, 19));
    this.setPreferredSize(new Dimension(270, 19));
    this.add(captionLabel, null);
    this.add(firstButton, null);
    this.add(priorButton, null);
    this.add(splitPanel1, null);
    this.add(recordText, null);
    this.add(splitPanel2, null);
    this.add(nextButton, null);
    this.add(lastButton, null);
    this.add(lastNewButton, null);
    this.add(recordLabel, null);

    recordText.setEnabled(false);
    recordText.setHorizontalAlignment(JLabel.RIGHT);
    recordText.addActionListener(this);
    recordText.setActionCommand("Goto");
    recordText.addFocusListener(this);
    recordText.setDocument(document);
    recordText.setText("0");

    initButton(firstButton, CEncodedIcon.FIRST_SIGN);
    initButton(priorButton, CEncodedIcon.PRIOR_SIGN);
    initButton(nextButton, CEncodedIcon.NEXT_SIGN);
    initButton(lastButton, CEncodedIcon.LAST_SIGN);
    initButton(lastNewButton, CEncodedIcon.LAST_NEW_SIGN);
  }

  /**
   * Initializes a button with icon and listeners.
   * @param button a button to initialize.
   * @param icon a icon for the button.
   */
  private void initButton(JButton button, CEncodedIcon icon) {
    button.setIcon(icon);
    icon = new CEncodedIcon(icon.getImageArray(), icon.getIconWidth(),
      icon.getIconHeight());
    icon.setEnabled(false);
    button.setDisabledIcon(icon);
    button.setEnabled(false);
    button.addActionListener(this);
    button.setFocusPainted(false);
    button.setRequestFocusEnabled(false);
  }

  /**
   * Performs an event when selection of the related table is changed.
   * @param e an object which describes occured event.
   */
  public void valueChanged(ListSelectionEvent e) {
    updateContent();
  }

  /**
   * Performs an event when model of the related table is changed.
   * @param e an object which describes occured event.
   */
  public void tableChanged(TableModelEvent e) {
    updateContent();
  }

  /**
   * Gets the related table control.
   * @result the related table control.
   */
  public JTable getTable() {
    return table;
  }

  /**
   * Sets a new related table control.
   * @param table a related table control.
   */
  public void setTable(JTable table) {
    if (this.table != null) {
      this.table.getSelectionModel().removeListSelectionListener(this);
      this.table.getModel().removeTableModelListener(this);
    }
    this.table = table;
    this.lastSelection = (table.getSelectedRow() >= 0) ? table.getSelectedRow() : 0;
    if (this.table != null) {
      this.table.getSelectionModel().addListSelectionListener(this);
      this.table.getModel().addTableModelListener(this);
      updateContent();
    }
  }

  /**
   * Updates the contents of this scroller.
   */
  private void updateContent() {
    if (table == null) {
      firstButton.setEnabled(false);
      priorButton.setEnabled(false);
      nextButton.setEnabled(false);
      lastButton.setEnabled(false);
      lastNewButton.setEnabled(false);
      recordText.setEnabled(false);
      recordLabel.setText("  of 0");
      validate();
      return;
    }

    // Count the numbers
    int currentRow = (table.getSelectedRow() >= 0) ? table.getSelectedRow() : 0;
    if (table.getSelectedRowCount() > 1)
      currentRow = lastSelection;
    else lastSelection = currentRow;
    int insertRow = ((table instanceof CDataTable) && ((CDataTable) table).isInsert()) ?
      table.getRowCount() - 1: -1;
    int rowCount = (insertRow >= 0)? table.getRowCount()-1: table.getRowCount();

    firstButton.setEnabled(currentRow > 0);
    priorButton.setEnabled(firstButton.isEnabled());
    lastButton.setEnabled(currentRow < rowCount-1);
    lastNewButton.setEnabled(insertRow >= 0 && currentRow != insertRow);
    nextButton.setEnabled(lastButton.isEnabled() || lastNewButton.isEnabled());
    recordText.setEnabled(rowCount > 0);
    recordText.setText(new Integer(currentRow + 1).toString());
    recordLabel.setText("  of " + new Integer(rowCount).toString());

    validate();
  }

  /**
   * Performs actions from user interface.
   * @param e description of the action.
   */
  public void actionPerformed(ActionEvent e) {
    String cmd = e.getActionCommand();

    // Count the numbers
    int currentRow = (table.getSelectedRow() >= 0) ? table.getSelectedRow() : 0;
    if (table.getSelectedRowCount() > 1)
      currentRow = lastSelection;
    else lastSelection = currentRow;
    int insertRow = ((table instanceof CDataTable) && ((CDataTable) table).isInsert()) ?
      table.getRowCount() - 1: -1;
    int rowCount = (insertRow >= 0)? table.getRowCount()-1: table.getRowCount();

    if (cmd.equals("First")) gotoRow(0);
    else if (cmd.equals("Prior")) gotoRow(currentRow - 1);
    else if (cmd.equals("Next")) gotoRow(currentRow + 1);
    else if (cmd.equals("Last")) gotoRow(rowCount-1);
    else if (cmd.equals("LastNew")) gotoRow(insertRow);
    else if (cmd.equals("Goto")) {
      int row = new Integer(recordText.getText()).intValue();
      gotoRow(row - 1);
    }
  }

  /**
   * Performs event when this table box gets a focus.
   * @param e an object which described occured event.
   */
  public void focusGained(FocusEvent e) {
  }

  /**
   * Performs event when this table box lost a focus.
   * @param e an object which described occured event.
   */
  public void focusLost(FocusEvent e) {
    if (document.isModified())
      recordText.postActionEvent();
  }

  /**
   * Moves the pointer in the table.
   * @param row the selected row.
   */
  private void gotoRow(int row) {
    if (table != null && row >= 0 && row < table.getRowCount()) {
      table.removeEditor();
      table.setRowSelectionInterval(row, row);

      if (table.getAutoscrolls()) {
        Rectangle cellRect = table.getCellRect(row, table.getSelectedColumn(), false);
        if (cellRect != null) {
          table.scrollRectToVisible(cellRect);
        }
      }
    }
  }

 /**
   * The main routine to run this module as standalone application.
   */
  public static void main(String[] args) {
    JFrame frame = new JFrame("Comedia Data Grid Test");

    JTable table = new JTable(
      new String[][] {{"1x1", "1x2","1x3"}, {"2x1", "2x2", "2x3"},
        {"3x1", "3x2", "3x3"}, {"4x1", "4x2", "4x3"}},
      new String[] {"Column 1", "Column 2", "Column 3"});
//    CTableScroller scroll = new CTableScroller(table);
    frame.getContentPane().add(table, BorderLayout.CENTER);

    CTableNavigator navigator = new CTableNavigator(table);
    frame.getContentPane().add(navigator, BorderLayout.SOUTH);

    frame.setLocation(100, 100);
    frame.setSize(500, 300);
    frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
    frame.show();

    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      SwingUtilities.updateComponentTreeUI(navigator);
    }
    catch(Exception e) {}
  }
}