/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.ui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import java.util.*;

/**
 * Presents a table with extended capabilities to add or remove data rows.
 * Moreover this table close interact with CTableScroller to show the table
 * state in row headers and CTableNavigator to navigate in rowset.
 * <p>
 * The table supports keyboard insertion and deletion operations through
 * keyboard. <code>INSERT</code> key adds a new row in the current position.
 * <code>CTRL-DELETE</code> key removes all selected rows from the table.
 * <p>
 * Note: you should use models descanded from DefaultTableModel to use the
 * features of this class.
 * <p><img src="CDataTable.gif">
 * <p>Usage examples:
 * <pre>
 * JFrame frame = new JFrame("Comedia Data Grid Test");
 * CDataTable table = new CDataTable();
 * CTableScroller scroll = new CTableScroller();
 * frame.getContentPane().add(scroll, BorderLayout.CENTER);
 *
 * CTableNavigator navigator = new CTableNavigator(table);
 * frame.getContentPane().add(navigator, BorderLayout.SOUTH);
 * </pre>
 */
public class CDataTable extends JTable {
  /**
   * The list of listeners.
   **/
  protected EventListenerList listenerList = new EventListenerList();

  /**
   * The delete flag.
   */
  private boolean delete = true;

  /**
   * The insert flag.
   */
  private boolean insert = true;

  /**
   * Constructs this class with default parameters.
   */
  public CDataTable() {
    this.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
  }

  /**
  * Adds a listener to the list that's notified each time a change
  * to the data model occurs.
  * @param l the TableModelListener.
  */
  public void addItemListener(ItemListener l) {
    listenerList.add(ItemListener.class, l);
  }

  /**
  * Removes a listener from the list that's notified each time a
  * change to the data model occurs.
  * @param l the TableModelListener.
  */
  public void removeItemListener(ItemListener l) {
    listenerList.remove(ItemListener.class, l);
  }

  /**
  * Forwards the given notification event to all
  * <code>TableModelListeners</code> that registered
  * themselves as listeners for this table model.
  * @param e the event to be forwarded
  */
  public void fireItemStateChanged(ItemEvent e) {
    // Guaranteed to return a non-null array
    Object[] listeners = listenerList.getListenerList();
    // Process the listeners last to first, notifying
    // those that are interested in this event
    for (int i = listeners.length-2; i>=0; i-=2) {
      if (listeners[i] == ItemListener.class) {
        ((ItemListener)listeners[i+1]).itemStateChanged(e);
      }
    }
  }

  /**
   * Checks the current deletion row status.
   * @result <code>TRUE</code> if deletion is allowed, <code>FALSE</code>
   *   otherwise.
   */
  public boolean isDelete() {
    return delete;
  }

  /**
   * Sets a new deletion status.
   * @param  <code>TRUE</code> to allow deletion, <code>FALSE</code>
   *   otherwise.
   */
  public void setDelete(boolean delete) {
    this.delete = delete;
  }

  /**
   * Checks the current insertion row status.
   * @result <code>TRUE</code> if insertion is allowed, <code>FALSE</code>
   *   otherwise.
   */
  public boolean isInsert() {
    return insert;
  }

  /**
   * Sets a new insertion status.
   * @param  <code>TRUE</code> to allow insertion, <code>FALSE</code>
   *   otherwise.
   */
  public void setInsert(boolean insert) {
    this.insert = insert;
  }

  /**
   * Gets a current row count.
   * @result a number of current rows.
   */
  public int getRowCount() {
    int count = super.getRowCount();
    if (insert) count++;
    return count;
  }

  /**
   * Returns the cell value at <code>row</code> and <code>column</code>.
   * @param row the row whose value is to be queried.
   * @param column the column whose value is to be queried.
   * @return the Object at the specified cell
   */
  public Object getValueAt(int row, int column) {
    if (row < getModel().getRowCount())
      return getModel().getValueAt(row, convertColumnIndexToModel(column));
    return null;
  }

  /**
   * Sets the value for the cell in the table model at <code>row</code>
   * and <code>column</code>.
   * @param aValue the new value.
   * @param row the row of the cell to be changed.
   * @param column the column of the cell to be changed.
   */
  public void setValueAt(Object aValue, int row, int column) {
    if (row >= getModel().getRowCount()) {
      if (getModel() instanceof DefaultTableModel)
        ((DefaultTableModel) getModel()).addRow(new Object[getModel().getColumnCount()]);
      else return;
    }
    getModel().setValueAt(aValue, row, convertColumnIndexToModel(column));
  }

  /**
   * Programmatically starts editing the cell at <code>row</code> and
   * <code>column</code>, if the cell is editable.
   * @param row the row to be edited.
   * @param column the column to be edited.
   * @return <code>FALSE</code> if for any reason the cell cannot be edited.
   */
  public boolean editCellAt(int row, int column, EventObject e){
    boolean result = super.editCellAt(row, column, e);
    if (result) fireItemStateChanged(null);
    return result;
  }

  /**
   * Invoked when editing is finished. The changes are saved and the
   * editor is discarded.
   * @param e the event received.
   */
  public void editingStopped(ChangeEvent e) {
    super.editingStopped(e);
    fireItemStateChanged(null);
  }

  /**
   * Invoked when editing is canceled. The editor object is discarded
   * and the cell is rendered once again.
   * @param e the event received.
   */
  public void editingCanceled(ChangeEvent e) {
    super.editingCanceled(e);
    fireItemStateChanged(null);
  }

  /**
   * Invoked when key event is accepted from user.
   * @param e the event received.
   */
  protected void processKeyEvent(KeyEvent e) {
    if (e.isAltDown() || e.isMetaDown() || e.isAltGraphDown() ||
      e.getID() != e.KEY_PRESSED) {
      super.processKeyEvent(e);
      return;
    }

    if (e.getKeyCode() == e.VK_INSERT && !e.isShiftDown() && !e.isControlDown()) {
      int row = (getSelectedRow() >= 0)? getSelectedRow(): 0;
      if (getModel() instanceof DefaultTableModel) {
        ((DefaultTableModel) getModel()).insertRow(row, new Object[getColumnCount()]);
        this.removeEditor();
        this.setRowSelectionInterval(row, row);
        if (getAutoscrolls()) {
          Rectangle cellRect = getCellRect(row, getSelectedColumn(), false);
          if (cellRect != null) {
            scrollRectToVisible(cellRect);
          }
        }
      }
      e.consume();
    } else if (e.getKeyCode() == e.VK_DELETE && !e.isShiftDown() && e.isControlDown()
      && !e.isActionKey()) {
      if (getSelectedRow() >= 0 && getModel() instanceof DefaultTableModel) {
        if (getAutoscrolls()) {
          Rectangle cellRect = getCellRect(getSelectedRow(), getSelectedColumn(), false);
          if (cellRect != null) {
            scrollRectToVisible(cellRect);
          }
        }
        if (JOptionPane.showConfirmDialog(null, "You are about to delete "
          + this.getSelectedRowCount()
          + " row(s)?", "Confirmation", JOptionPane.YES_NO_OPTION,
          JOptionPane.QUESTION_MESSAGE) == JOptionPane.YES_OPTION) {

          this.removeEditor();
          int[] columns = this.getSelectedRows();
          for (int i = columns.length-1; i >= 0; i--) {
            if (!insert || columns[i] != getRowCount()-1)
              ((DefaultTableModel) getModel()).removeRow(columns[i]);
          }
        }
      }
      e.consume();
    } else
      super.processKeyEvent(e);
  }

  /**
   * The main routine to run this module as standalone application.
   */
  public static void main(String[] args) {
    try {
//      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      UIManager.setLookAndFeel("com.sun.java.swing.plaf.motif.MotifLookAndFeel");
    }
    catch(Exception e) {}

    JFrame frame = new JFrame("Comedia Data Grid Test");
    CDataTable table = new CDataTable();
    CTableScroller scroll = new CTableScroller();
    frame.getContentPane().add(scroll, BorderLayout.CENTER);

    CTableNavigator navigator = new CTableNavigator(table);
    frame.getContentPane().add(navigator, BorderLayout.SOUTH);

    frame.setLocation(100, 100);
    frame.setSize(305, 310);
    frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
    frame.show();

    DefaultTableModel model = (DefaultTableModel) table.getModel();

    String[] row = {"0", "value 1", "value 2", "value 3"};
    model.addColumn("Id");
    model.addColumn("Column 1");
    model.addColumn("Column 2");
    model.addColumn("Column 3");
    for (int i = 1; i < 100; i++) {
      row[0] = new Integer(i).toString();
      model.addRow(row);
    }

/*
    table.setShowHorizontalLines(false);
    table.setShowVerticalLines(false);
*/
    scroll.setTable(table);
//    table.setModel(table.getModel());
  }

}