/**
 * Title:        Comedia Beans
 * Module:       Console Window
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.ui;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import javax.swing.text.*;
import javax.swing.*;

/**
 * Implements a text console for output different information.
 * <p>
 * Example usage of this <code>CConsole</code> class:
 * <pre>
 * Style regular = StyleContext.getDefaultStyleContext().
 *   getStyle(StyleContext.DEFAULT_STYLE);
 *
 * for (int i = 0; i < 10; i++) {
 *   Style newStyle = console.getTextPane().addStyle("new"+i, regular);
 *   StyleConstants.setFontSize(newStyle, 10+i);
 *   console.println("Hello World!", newStyle);
 * }
 * </pre>
 */
public class CConsole extends JScrollPane {
  /**
   * The text control of this console.
   */
  private JTextPane text = new JTextPane();

  /**
   * Constructs this console class with default parameters.
   */
  public CConsole() {
    try {
      jbInit();
    }
    catch(Exception e) {}
  }

  /**
   * Initializes visual controls of this console. This method is managed by
   * JBuilder visial designers.
   */
  private void jbInit() throws Exception {
    text.setFont(new java.awt.Font("Monospaced", 0, 12));
    text.setEditable(false);
    this.setViewportView(text);
  }

  /**
   * Gets the text control of this console class.
   * @result a text control to this console class.
   */
  public JTextPane getTextPane() {
    return text;
  }

  /**
   * Prints a string on this console.
   * @param s a string to print on this console.
   */
  public synchronized void print(String s) {
    Document doc = text.getDocument();
    try {
      doc.insertString(doc.getLength(), s, null);
    }
    catch (Exception e) {};
  }

  /**
   * Prints a line on this console.
   * @param s a string to print on this console.
   */
  public synchronized void println(String s) {
    Document doc = text.getDocument();
    try {
      doc.insertString(doc.getLength(), s + '\n', null);
    }
    catch (Exception e) {};
  }

  /**
   * Prints a string on this console with Attribute.
   * @param s a string to print on this console.
   * @param a an attribute of printed text. it should be registered in
   *   text control.
   */
  public synchronized void print(String s, AttributeSet a) {
    Document doc = text.getDocument();
    try {
      doc.insertString(doc.getLength(), s, a);
    }
    catch (Exception e) {};
  }

  /**
   * Prints a line on the console with Attribute.
   * @param s a string to print on this console.
   * @param a an attribute of printed text. it should be registered in
   *   text control.
   */
  public synchronized void println(String s, AttributeSet a) {
    Document doc = text.getDocument();
    try {
      doc.insertString(doc.getLength(), s + '\n', a);
    }
    catch (Exception e) {};
  }

  /**
   * Implements a thread to print an InputStream on this console.
   */
  private class PrintThread extends Thread {
    /**
     * The printing input stream.
     */
    private InputStream in;

    /**
     * Constructs this input stream and assign main parameters.
     * @param in an input stream to print on console.
     */
    public PrintThread(InputStream in) {
      this.in = in;
    }

    /**
     * Performs the thread functionality.
     */
    public void run() {
      byte buf[] = new byte[512];
      int r;
      try {
        while ((r = in.read(buf, 0, 512)) > -1)
          print(new String(buf, 0, r));
      }
      catch (Exception e) {}
    }
  }

  /**
   * Print a stream content on this console.
   * @param in an input stream to print on this console.
   */
  public synchronized void copy(InputStream in) {
    PrintThread thread = new PrintThread(in);
    thread.start();
  }

  /**
   * Clears the content of this console.
   */
  public void clear() {
    text.setText("");
  }

  /**
   * Runs this console as standalone application for test purposes.
   * @param args arguments of command line.
   */
  public static void main(String args[]) {
    JFrame window = new JFrame("Console");
    CConsole console = new CConsole();

    window.setContentPane(console);
    window.setSize(400, 300);
    window.setLocation(300, 300);
    window.setDefaultCloseOperation(3);
    window.show();

    Style regular = StyleContext.getDefaultStyleContext().
      getStyle(StyleContext.DEFAULT_STYLE);

    for (int i = 0; i < 10; i++) {
      Style newStyle = console.getTextPane().addStyle("new"+i, regular);
      StyleConstants.setFontSize(newStyle, 10+i);
      console.println("Hello World!", newStyle);
    }
  }
}