/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.db.view;

import java.awt.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.border.*;

/**
 * Implements a table entity box on database schemas.
 * <p>
 * <img src="CTableBox.gif">
 * <p>
 * Usage examples:
 * <pre>
 * String[] fields = {"Field1", "Field2", "Field3", "Field4", "Field5", "Field6"};
 * CTableBox table = new CTableBox();
 * table.setLocation(50, 50);
 * table.setTitle("The Table");
 * table.setFields(fields);
 * </pre>
 */
public class CTableBox extends JPanel implements MouseMotionListener,
  MouseListener, FocusListener, AdjustmentListener {
  /**
   * The caption label of this table box.
   */
  private JLabel label = new JLabel();

  /**
   * The field list of this table box.
   */
  private JList listBox = new JList();

  /**
   * The scroller of the list box.
   */
  private JScrollPane scroller = new JScrollPane();

  /**
   * The point of resizing this table box.
   */
  private Point resizePoint;

  /**
   * The point of reposition this table box.
   */
  private Point reposPoint;

  /**
   * The quadrant where move clicked.
   */
  private int quadrant;

  /**
   * The list of table fields.
   */
  private Object[] fields;

  /**
   * The list of related table links.
   */
  private ArrayList links = new ArrayList();

  /**
   * Creates this table box with default properties.
   */
  public CTableBox() {
    this.setMinimumSize(new Dimension(95, 26));
    this.setSize(new Dimension(112, 112));
    this.setPreferredSize(new Dimension(112, 112));
    this.setRequestFocusEnabled(true);
    BorderLayout layout = new BorderLayout();
    layout.setVgap(3);
    this.setLayout(layout);
    Border border = BorderFactory.createCompoundBorder(
      BorderFactory.createBevelBorder(BevelBorder.RAISED,Color.white,
      Color.white,new Color(148, 145, 140),new Color(103, 101, 98)),
      BorderFactory.createEmptyBorder(3,3,3,3));
    this.setBorder(border);
    this.setBackground(Color.lightGray);
    this.addMouseListener(this);
    this.addMouseMotionListener(this);
    this.addFocusListener(this);

    this.add(label, BorderLayout.NORTH);
    label.setFont(new Font("Dialog", 0, 11));
    label.setOpaque(true);
    label.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
    label.addMouseListener(this);
    label.addMouseMotionListener(this);
    this.setSelectedColor(false);

    listBox.addFocusListener(this);
    listBox.setFont(label.getFont());
    listBox.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
    scroller.setViewportView(listBox);
    scroller.getVerticalScrollBar().addAdjustmentListener(this);
    this.add(scroller, BorderLayout.CENTER);
  }

  /**
   * Gets the table box title.
   * @result the table box title.
   */
  public String getTitle() {
    return label.getText();
  }

  /**
   * Sets new table box title.
   * @param title a new table box title.
   */
  public void setTitle(String title) {
    label.setText(title);
  }

  /**
   * Sets selected or unselected colors for this table box.
   * @param selected <code>TRUE</code> to set selected color
   *   and <code>FALSE</code> otherwise.
   */
  private void setSelectedColor(boolean selected) {
    if (selected) {
      label.setBackground(new Color(0,0,100));
      label.setForeground(Color.white);
    } else {
      label.setBackground(Color.gray);
      label.setForeground(Color.lightGray);
    }
  }

  /**
   * Gets the current table fields list.
   * @result the current table fields list.
   */
  public Object[] getFields() {
    return fields;
  }

  /**
   * Sets table fields list.
   * @param fields an array of table fields
   */
  public void setFields(Object[] fields) {
    listBox.setListData(fields);
    this.fields = fields;
  }

  /**
   * Performs event when this table box gets a focus.
   * @param e an object which described occured event.
   */
  public void focusGained(FocusEvent e) {
    listBox.requestFocus();
    setSelectedColor(true);
  }

  /**
   * Performs event when this table box lost a focus.
   * @param e an object which described occured event.
   */
  public void focusLost(FocusEvent e) {
    setSelectedColor(false);
  }

  /**
   * Performs event when mouse is dragged over the component.
   * @param e an object which described occured event.
   */
  public void mouseDragged(MouseEvent e) {
    if (e.getSource() == this)
      borderMouseDragged(e);
    else if (e.getSource() == label)
      labelMouseDragged(e);
//    System.out.println("Mouse dragged: " + e.getX() + "-" + e.getY());
  }

  /**
   * Performs event when mouse is moved over the component.
   * @param e an object which described occured event.
   */
  public void mouseMoved(MouseEvent e) {
    if (e.getSource() == this)
      borderMouseMoved(e);
//    System.out.println("Mouse moved: " + e.getX() + "-" + e.getY());
  }

  /**
   * Performs event when mouse is clicked on the component.
   * @param e an object which described occured event.
   */
  public void mouseClicked(MouseEvent e) {
//    System.out.println("Mouse clicked: " + e.getX() + "-" + e.getY());
  }

  /**
   * Performs event when mouse is pressed on the component.
   * @param e an object which described occured event.
   */
  public void mousePressed(MouseEvent e) {
    listBox.requestFocus();
    if (e.getSource() == this) {
      quadrant = countQuadrant(e.getPoint());
      resizePoint = e.getPoint();
    } else if (e.getSource() == label) {
      label.setCursor(new Cursor(Cursor.MOVE_CURSOR));
      reposPoint = e.getPoint();
    }
//    System.out.println("Mouse pressed: " + e.getX() + "-" + e.getY());
  }

  /**
   * Performs event when mouse is released on the component.
   * @param e an object which described occured event.
   */
  public void mouseReleased(MouseEvent e) {
    if (e.getSource() == label)
      label.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
//    System.out.println("Mouse released: " + e.getX() + "-" + e.getY());
  }

  /**
   * Performs event when mouse is entered to the component.
   * @param e an object which described occured event.
   */
  public void mouseEntered(MouseEvent e) {
//    System.out.println("Mouse entered: " + e.getX() + "-" + e.getY());
  }

  /**
   * Performs event when mouse is exited from the component.
   * @param e an object which described occured event.
   */
  public void mouseExited(MouseEvent e) {
//    System.out.println("Mouse exited: " + e.getX() + "-" + e.getY());
  }

  /**
   * Performs event when fields list scroller is changed.
   * @param e an object which described occured event.
   */
  public void adjustmentValueChanged(AdjustmentEvent e) {
    updateLinks();
  }

  /**
   * The width of the table conner.
   */
  private static final int CONNER_SIZE = 5;

  /**
   * Counts a quadrant side of this table box.
   * @param x X coordinate of the point.
   * @param y Y coordinate of the point.
   * @result a quadrant number.
   */
  private int countQuadrant(Point point) {
    int result = 2;
    // Detect X position
    if (point.x <= CONNER_SIZE)
      result = 1;
    else if (point.x >= (this.getWidth() - CONNER_SIZE))
      result = 3;
    // Detect Y position
    if (point.y > CONNER_SIZE) {
      if (point.y < (this.getHeight() - CONNER_SIZE))
        result += 3;
      else
        result += 6;
    }
    return result;
  }

  /**
   * Performs event when mouse is moved over the border of this table box.
   * @param e an object which described occured event.
   */
  public void borderMouseMoved(MouseEvent e) {
    // Change cursor depending mouse position
    switch (countQuadrant(e.getPoint())) {
      case 1:
        this.setCursor(new Cursor(Cursor.NW_RESIZE_CURSOR));
        break;
      case 9:
        this.setCursor(new Cursor(Cursor.SE_RESIZE_CURSOR));
        break;
      case 2:
        this.setCursor(new Cursor(Cursor.N_RESIZE_CURSOR));
        break;
      case 8:
        this.setCursor(new Cursor(Cursor.S_RESIZE_CURSOR));
        break;
      case 3:
        this.setCursor(new Cursor(Cursor.NE_RESIZE_CURSOR));
        break;
      case 7:
        this.setCursor(new Cursor(Cursor.SW_RESIZE_CURSOR));
        break;
      case 4:
        this.setCursor(new Cursor(Cursor.W_RESIZE_CURSOR));
        break;
      case 6:
        this.setCursor(new Cursor(Cursor.E_RESIZE_CURSOR));
        break;
      default:
        this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
    }
  }

  /**
   * Performs event when mouse is dragged over the border of this table box.
   * @param e an object which described occured event.
   */
  public void borderMouseDragged(MouseEvent e) {
    Point pos = this.getLocation();
    Dimension dim = this.getSize();
    boolean resized = false;
    int dx = e.getPoint().x - resizePoint.x;
    int dy = e.getPoint().y - resizePoint.y;

    switch (quadrant) {
      case 1:
      case 4:
      case 7:
        if (dim.width - dx >= this.getMinimumSize().width) {
          pos.x += dx;
          dim.width -= dx;
          resized = true;
        }
        break;
      case 3:
      case 6:
      case 9:
        if (dim.width + dx >= this.getMinimumSize().width)
          dim.width += dx;
        resized = true;
        break;
    }
    switch (quadrant) {
      case 1:
      case 2:
      case 3:
        if (dim.height - dy >= this.getMinimumSize().height) {
          pos.y += dy;
          dim.height -= dy;
          resized = true;
        }
        break;
      case 7:
      case 8:
      case 9:
        if (dim.height + dy >= this.getMinimumSize().height)
          dim.height += dy;
        resized = true;
        break;
    }
    if (resized) {
      this.setLocation(pos);
      this.setSize(dim);
      updateLinks();
      this.validate();
    }

    if (quadrant == 3 || quadrant == 6 || quadrant == 9)
      resizePoint.x = e.getPoint().x;
    if (quadrant == 7 || quadrant == 8 || quadrant == 9)
      resizePoint.y = e.getPoint().y;
  }

  /**
   * Performs event when mouse is dragged over the label of this table box.
   * @param e an object which described occured event.
   */
  public void labelMouseDragged(MouseEvent e) {
    Point pos = this.getLocation();
    pos.x += e.getPoint().x - reposPoint.x;
    pos.y += e.getPoint().y - reposPoint.y;
    this.setLocation(pos);
    updateLinks();
  }

  /**
   * Adds new link to this table.
   * @param link a link to add.
   */
  public void addLink(CTableLink link) {
    links.add(link);
    link.updateShape();
  }

  /**
   * Clears all related links.
   */
  public void clearLinks() {
    links.clear();
  }

  /**
   * Removes a link from this table.
   * @param link a link to remove.
   */
  public void removeLink(CTableLink link) {
    links.remove(link);
  }

  /**
   * Updates all related links.
   */
  public void updateLinks() {
    for (int i = 0; i < links.size(); i++)
      try {
        ((CTableLink) links.get(i)).updateShape();
      }
      catch (Exception e) {}
  }

  /**
   * Updates new related links.
   */
  public void updateNewLinks() {
    for (int i = 0; i < links.size(); i++)
      try {
        CTableLink link = ((CTableLink) links.get(i));
        if (!link.isUpdated())
          link.updateShape();
      }
      catch (Exception e) {}
  }

  /**
   * Paints this table box.
   * @param g the current graphics context.
   */
  public void paint(Graphics g) {
    super.paint(g);
    updateNewLinks();
  }

  /**
   * Gets Y position of the specified field in screen coordinates.
   * @param index an index of a field.
   * @result an Y coordinate of the field.
   */
  public int getItemPos(int index) {
    int y;

    if (index < 0)
      y = this.getHeight() / 2;
    else if (index < listBox.getFirstVisibleIndex()
      || listBox.getVisibleRowCount() <= 0 || scroller.getHeight() <= 3)
      y = label.getLocation().y + label.getHeight() / 2;
    else if (index > listBox.getLastVisibleIndex()
      && listBox.getLastVisibleIndex() >= 0)
      y = this.getHeight() - 4;
    else {
      Rectangle rect = listBox.getCellBounds(index, index);
      y = scroller.getLocation().y + listBox.getLocation().y + rect.y + rect.height / 2;
    }

    return this.getLocationOnScreen().y + y;
  }

  /**
   * Runs a standalone test application.
   * @param args a command line arguments.
   */
  public static void main(String args[]) {
    try {
//      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      UIManager.setLookAndFeel("com.sun.java.swing.plaf.motif.MotifLookAndFeel");
    }
    catch(Exception e) {}

    JFrame frame = new JFrame("Test Table View Application");
    frame.getContentPane().setLayout(null);
    frame.getContentPane().setBackground(Color.white);
    String[] fields = {"Field1", "Field2", "Field3", "Field4", "Field5", "Field6"};

    CTableBox table1 = new CTableBox();
    table1.setLocation(50, 50);
    table1.setTitle("The First Table");
    table1.setFields(fields);

    CTableBox table2 = new CTableBox();
    table2.setLocation(250, 250);
    table2.setTitle("The Second Table");
    table2.setFields(fields);

    CTableLink link = new CTableLink();
    link.setLeftTable(table1);
    link.setLeftLinkType(CTableLink.JOIN_LINK);
    link.setLeftRelation(CTableLink.ONE_RELATION);
    link.setLeftFieldIndex(0);
    link.setRightTable(table2);
    link.setRightLinkType(CTableLink.JOIN_LINK);
    link.setRightRelation(CTableLink.MANY_RELATION);
    link.setRightFieldIndex(5);
    link.setLocation(150, 150);
    link.setSize(100, 100);

    frame.getContentPane().add(table1);
    frame.getContentPane().add(table2);
    frame.getContentPane().add(link);

    JButton button = new JButton("OK");
    button.setSize(50, 50);
    button.setLocation(100, 100);
    frame.getContentPane().add(button);

    frame.setSize(500, 500);
    frame.setLocation(100, 100);
    frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
    frame.show();
  }

}
