/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;


/**
 * Title:        UIFactoryCodeGenerator
 * Description:  This is the code generator for the ui factory.  It generates
 *              java code based on the state of the objects it receives in
 *              a broadcast object and the metadata provided in an xml document.
 * Copyright:    Copyright (c) 2001
 * Company:      Abacus Software
 * @author       Michael Gouker (Cagey Logic)
 * @version 0.1
 */


import java.io.StringReader;
import java.io.StringWriter;
import java.lang.reflect.Constructor;
import java.util.ArrayList;

/**
 * UIFactoryCodeGenerator - This class generates source code that describe objects that
 * are generated by the UIFactory.  There are two levels of interface:
 *
 *    1.  A low-level interface that works like an output stream that collects
 *    all the source output into a buffer that can be returned at any time.
 *
 *    2.  A high-level interface that implements code generation with full
 *    knowledge of the kind of code being generated (declaration, property
 *    settings, preamble, epilogue, custom, frame attachment, container
 *    intialization.)
 *
 */

/**
 * UIFactoryLowLevelCodeGenerator - This interface defines a simple I/O interface
 *  to a buffer that stores source code and returns the buffer upon request.
 */

interface UIFactoryLowLevelCodeGenerator {

    /**
     * getSource - returns the current contents of the code generator, a string buffer.
     */
    public String getSource();

    /**
     * getFormattedSource - returns the current contents of the code generator, a string buffer.
     * Paases it through the formatting bean.
     */
    public String getFormattedSource(int iMaxLineLength, int iIndentation, String sDelims, String sStickyDelims);


    /**
     * print - Output code to the code generator buffer with no cr/lf.
     *
     * @param  s - a String - The code to be appended to the buffer.
     */
    public void print(String s);

    /**
     * println - Output code to the code generator buffer with a cr/lf.
     *
     * @param  s - a String - The code to be appended to the buffer.
     */
    public void println(String s);

    /**
     * endPrintln - Output code to the end of the code generator buffer with a cr/lf.
     *
     * @param  s - a String - The code to be appended to the buffer.
     */
    public void endPrintln(String s);

}

/**
 * UIFactoryHighLevelCodeGenerator - This interface defines a more implementation-
 * specific code generator for the UIFactory.  The methods defined here
 * are more tailored to solve specific problems of code generation for
 * UI objects.
 */

interface UIFactoryHighLevelCodeGenerator {

    /**
     * generateDeclaration - Writes a declaration for the specified object with
     * the given identifier to the source buffer.
     *
     * @param  obj - The object for which to write source.
     * @param  sObjectIdentifier - This is the identifier to use for the
     * object when writing the source.
     * @param  sPreCode - code before the declaration
     * @param  bIsMember - make it a class member
     *
     */
    public void generateDeclaration(Object obj, String sObjectIdentifier, String sPreCode, boolean bIsMember) throws UIFactoryException;

    /**
     * generateBasicContainerInitialization - Writes the special code for a
     * container that sets up the content pane to which controls attach.
     *
     * @param  sFrameIdentifier - This is the identifier to use for the
     * object when writing the frame initialization source.
     * @param  sLayoutType - This is the requested layout type
     *
     */
    public void generateBasicContainerInitialization(String sFrameIdentifier, String sLayoutType) throws UIFactoryException;

    /**
     * generateBasicControlToContentPaneAttachment - Writes the special code
     * that attaches a control to a container's content pane.
     *
     * @param  sControlIdentifier - This is the identifier to use for the
     * object when writing the control attachment source.
     * @param  sPaneName - Used to assign layout to content pane.
     * @param  sContainerName - Used to assign layout.
     * @param  bAnchorLeft - anchor to left
     * @param  bAnchorRight - anchor to right
     * @param  bAnchorTop - anchor to top
     * @param  bAnchorBottom - anchor to bottom
     *
     */
    public void generateBasicControlToContentPaneAttachment(String sControlIdentifier,
                                                            String sPaneName,
                                                            String sContainerName,
                                                            boolean bAnchorLeft, boolean bAnchorRight,
                                                            boolean bAnchorTop, boolean bAnchorBottom);


    /**
     * generateAdditionalCustomCode - This method writes the object's stored
     * custom code into the source file.
     *
     * @param  sCustomCode - This is the custom code to write.
     *
     */
    public void generateAdditionalCustomCode(String sCustomCode);

    /**
     * generatePreamble - This method writes the code up to the first object
     * declaration, including package declaration, import declarations, and the
     * main class declaration.
     *
     * @param  sPackageName - The name of the package the class will reside in.
     * @param  sClassName - The name of the main class for the UI interface.
     *
     */
    public void generatePreamble(String sPackageName, String sClassName) throws UIFactoryException;

    /**
     * generateEpilogue - This method writes the code after the last object is
     * processed.  This is the code that contains the event handler to close the
     * program along with associated specialized subclasses.
     *
     * @param  sFrameObjectName - The name of the main frame.
     * @param  sLookAndFeel - the name of the platform lnf.
     *
     */
    public void generateEpilogue(String sFrameObjectName, String sLookAndFeel) throws UIFactoryException;


    /**
     *
     * generateClasses - This method creates all the classes defined in the program aside
     * from the main class.  Class definitions are stored in the metadata.
     *
     */
    public void generateClasses() throws UIFactoryException;


}


/**
 * Code Generator
 *
 *  This class implements both the high-level and low-level code generator.
 *  It is used to generate java source code in the UIFactory family of classes.
 *
 */

public class UIFactoryCodeGenerator implements UIFactoryLowLevelCodeGenerator, UIFactoryHighLevelCodeGenerator {

    /**
     * elementMetaDataRoot - Element.
     *
     *  This object is set to the element at the root of the metadata xml document.
     */
    MetaProject theProject;
    /**
     * metadataProvider - a MetadataProvider object
     *
     * This object defines the interface between the code generator and the metadata
     * provider which researches class info both from java introspection and the
     * metadata XML document.
     */
    protected MetadataProvider metadataProvider;

    /**
     * sSource - The code generator's buffer that contains all of the source
     * generated.  Although this UIFactoryCodeGenerator uses a buffer to implement the
     * access to the written source, other CodeGenerators may use other means
     * to store the source.
     */
    protected String sSource = "";

    /**
     * sBegSource - Another code generator buffer that accumulates source to be
     * written before any object declaration besides main class.
     */
    protected String sBegSource = "";

    /**
     * sBeg2Source - Another code generator buffer that accumulates source to be
     * written before any object declaration besides main class, but after the
     * member declarations.
     */
    protected String sBeg2Source = "";


    /**
     * sEndSource - Another code generator buffer that accumulates source to be
     * written after the class definitions.
     */
    protected String sEndSource = "";

    /**
     * sMemberDeclarations - A code generation buffer that allows declarations to
     * be made within the main class declaration.
     */

    protected String sMemberDeclSource = "";

    /**
     * sDeclSource - Another code generator buffer that accumulates source to be
     * written for declarations.
     */

    protected String sDeclSource = "";

    /**
     * sListenerSource - Classes for listeners.  These are added before the epilogue.
     */

    protected String sListenerSource = "";

    /**
     * bGenerateAbalet - Property set when an abalet is generated instead of an application.
     */

    protected boolean bAbalet = false;

    /**
     * bGeneratedTabbedPaneListener - Property set to stop the listener for tab pages from
     * being created more than once. Only one per program.
     */

    protected boolean bGeneratedTabbedPaneListener = false;

    /**
     * sMainObject - the Main Object of the program.
     */

    protected String sMainObject = null;
    private String sMainClass;

    /**
     * UIFactoryCodeGenerator - This is the constructor for the UIFactoryCodeGenerator class.
     *
     * @param  theMetadataProvider - a MetadataProvider object that links the
     * code generator to the metadata of the system.
     */
    public UIFactoryCodeGenerator(MetadataProvider theMetadataProvider,
                                  MetaProject theProject,
                                  MetaObject theMainObject)
            throws java.io.FileNotFoundException, java.io.IOException {
        // Load up XML data for use in code generation.
        metadataProvider = theMetadataProvider;
        this.theProject = theProject;
        sMainObject = theMainObject.getName();
        // generate main class name.
        sMainClass = new String(theProject.theProgramData.sProjectName.trim());
        int iKillSpaces = sMainClass.indexOf(" ");
        while (iKillSpaces != -1) {
            String sLeftSide = sMainClass.substring(0, iKillSpaces);
            String sRightSide = sMainClass.substring(iKillSpaces + 1);
            sMainClass = sLeftSide + "_" + sRightSide;
            iKillSpaces = sMainClass.indexOf(" ");
        }
    }

    /**
     * getSource - This method returns the source code to the caller.
     */
    public String getSource() {
        return sBegSource + sMemberDeclSource + sBeg2Source + sDeclSource + sSource + sListenerSource + sEndSource;
    }

    /**
     * getFormattedSource - This method returns the source code to the caller.
     * Passes it through the CodeFormatter bean.
     */


    public String getFormattedSource(int iMaxLineLength, int iIndentation, String sDelims, String sStickyDelims) {
        String sSourceCode = sBegSource + sMemberDeclSource + sBeg2Source +
                sDeclSource + sSource + sListenerSource + sEndSource;
        com.ibm.cf.CodeFormatter theFormatter = new com.ibm.cf.CodeFormatter();
        StringReader theReader = new StringReader(sSourceCode);
        StringWriter theWriter = new StringWriter(sSourceCode.length());
        theFormatter.setDelimiters(sDelims);
        theFormatter.setStickyDelimiters(sStickyDelims);
        theFormatter.setIndentationStep(iIndentation);
        theFormatter.setMaxLineLength(iMaxLineLength);
        theFormatter.formatCode(theReader, theWriter);
        return theWriter.getBuffer().toString();
    }

    public String getFormattedSource() {                       // use default delims
        String sSourceCode = sBegSource + sMemberDeclSource + sBeg2Source +
                sDeclSource + sSource + sListenerSource + sEndSource;
        com.ibm.cf.CodeFormatter theFormatter = new com.ibm.cf.CodeFormatter();
        StringReader theReader = new StringReader(sSourceCode);
        StringWriter theWriter = new StringWriter(sSourceCode.length());
        theFormatter.formatCode(theReader, theWriter);
        return theWriter.getBuffer().toString();
    }

    /**
     * generateBasicContainerInitialization - Writes the special code for a
     * container that sets up the content pane to which controls attach.
     *
     * @param  sFrameIdentifier - This is the identifier to use for the
     * object when writing the frame initialization source.
     * @param  sLayoutType - This is the requested layout type
     *
     */
    public void generateBasicContainerInitialization(
            String sFrameIdentifier,
            String sLayoutType) throws UIFactoryException {
        println("//  Get the content pane of the frame.");
        println("JPanel contentPane" + sFrameIdentifier + " = (JPanel) " + sFrameIdentifier +
                ".getContentPane();");
        println("// Set the layout.");
//      println("contentPane"+sFrameIdentifier+".setLayout(" + sLayoutType + ");");
        println("AnchoringLayoutManager layoutManager" + sFrameIdentifier + " = new AnchoringLayoutManager();");
        println("contentPane" + sFrameIdentifier + ".setLayout(layoutManager" + sFrameIdentifier + ");");
    }

    /**
     * generateBasicPanelInitialization - Writes the special code for a
     * panel.
     *
     * @param  sPanelIdentifier - This is the identifier to use for the
     * object when writing the panel initialization source.
     * @param  sLayoutType - This is the requested layout type
     *
     */
    public void generateBasicPanelInitialization
            (String sPanelIdentifier,
             String sLayoutType) throws UIFactoryException {
//      println("contentPane"+sPanelIdentifier+".setLayout(" + sLayoutType + ");");
        println("AnchoringLayoutManager layoutManager" + sPanelIdentifier + " = new AnchoringLayoutManager();");
        println(sPanelIdentifier + ".setLayout(layoutManager" + sPanelIdentifier + ");");
    }

    /**
     * generateBasicGroupInitialization - Writes the special code for a
     * group.
     *
     * @param  sGroupIdentifier - This is the identifier to use for the
     * object when writing the panel initialization source.
     * @param  sLayoutType - This is the requested layout type
     *
     *
     */
    public void generateBasicGroupInitialization(String sGroupIdentifier,
                                                 String sLayoutType) throws UIFactoryException {
    }

    /**
     * generateAbaletLinkage - If executed, an abalet is generated in the epilog.
     */

    public void generateAbaletLinkage() {
        bAbalet = true;
    }

    /**
     * generateBasicControlToContentPaneAttachment - Writes the special code
     * that attaches a control to a container's content pane.
     *
     * @param  sControlIdentifier - This is the identifier to use for the
     * object when writing the control attachment source.
     * @param  sContentPaneName - The name of the content pane.
     *
     */
    public void generateBasicControlToContentPaneAttachment(String sControlIdentifier,
                                                            String sContentPaneName,
                                                            String sContainerName,
                                                            boolean bAnchorLeft, boolean bAnchorRight,
                                                            boolean bAnchorTop, boolean bAnchorBottom) {
        println("//  Add the control " + sControlIdentifier + " to the content pane.");
        if (sContentPaneName.equals("") == false)
            print(sContentPaneName + ".");
        println("add(" + sControlIdentifier + ", \"" + sControlIdentifier + "\");");
        println("layoutManager" + sContainerName + ".setAnchoring(" + sControlIdentifier + ", " +
                (bAnchorLeft ? "true, " : "false, ") +
                (bAnchorRight ? "true, " : "false, ") +
                (bAnchorTop ? "true, " : "false, ") +
                (bAnchorBottom ? "true);" : "false);"));

    }

    public void generateBasicControlToTabbedPaneAttachment(String sControlIdentifier,
                                                           String sContentPaneName, String sCaption) {
        println("//  Add the control " + sControlIdentifier + " to the Tabbed pane.");
        if (sContentPaneName.equals("") == false)
            print(sContentPaneName + ".");
        println("addTab(\"" + sCaption + "\", " + sControlIdentifier + ");");
    }

    public void generateBasicControlToGroupAttachment(String sControlIdentifier,
                                                      String sContentPaneName) {
        println("//  Add the control " + sControlIdentifier + " to the Group.");
        if (sContentPaneName.equals("") == false)
            print(sContentPaneName + ".");
        println("add(" + sControlIdentifier + ");");
    }

    public void generateBasicControlToMenuAttachment(String sControlIdentifier,
                                                           String sContentPaneName) {
        println("//  Add menu item" + sControlIdentifier + " to Menu.");
        if (sContentPaneName.equals("") == false)
            print(sContentPaneName + ".");
        println("add(" + sControlIdentifier + ", \"" + sControlIdentifier + "\");");
    }

    public void generateBasicControlToScrollPaneAttachment(String sControlIdentifier,
                                                           String sContentPaneName) {
        println("//  Add the control " + sControlIdentifier + " to the content pane.");
        if (sContentPaneName.equals("") == false)
            print(sContentPaneName + ".");
        println("setViewportView(" + sControlIdentifier + ");");
    }

    public void generateBasicTabbedPaneInitialization(String sIdentifier) throws UIFactoryException {
        println(sIdentifier + ".addChangeListener(new DefaultUIFactoryTabbedPaneListener());");
    }

    public void generateTabbedPaneListener() {
        // Just once.
        if (bGeneratedTabbedPaneListener == true)
            return;
        bGeneratedTabbedPaneListener = true;
        endPrintln("class DefaultUIFactoryTabbedPaneListener implements ChangeListener {");
        endPrintln("int selectedIndex = -1; // -1 = nothing has ever been selected.");
        endPrintln("JTabbedPane tp;         // current pane.");
        endPrintln("public void stateChanged(ChangeEvent evt) {");
        endPrintln("tp = (JTabbedPane) evt.getSource();");
        endPrintln("if ((selectedIndex == -1) || (selectedIndex != tp.getSelectedIndex())) {");
        endPrintln("tp.setEnabledAt(tp.getSelectedIndex(), false);");
        endPrintln("if (selectedIndex != -1)");
        endPrintln("tp.setEnabledAt(selectedIndex, true);");
        endPrintln("}");
        endPrintln("selectedIndex = tp.getSelectedIndex();");
        endPrintln("}");
        endPrintln("}");
    }

    /**
     * generateBasicControlToPanelAttachment - Writes the special code
     * that attaches a control to a Panel.
     *
     * @param  sControlIdentifier - This is the identifier to use for the
     * object when writing the control attachment source.
     * @param  sPanelName - The name of the panel to attach to.
     *
     */
    public void generateBasicControlToPanelAttachment(String sControlIdentifier,
                                                      String sPanelName,
                                                      boolean bAnchorLeft, boolean bAnchorRight,
                                                      boolean bAnchorTop, boolean bAnchorBottom) {
        println("//  Add the control " + sControlIdentifier + " to the panel " + sPanelName + ".");
        println(sPanelName + ".add(" + sControlIdentifier + ", \"" + sControlIdentifier + "\");");
    }

    /**
     * print - Output code to the code generator buffer with no cr/lf.
     *
     * @param  s - a String - The code to be appended to the buffer.
     */
    public void print(String s) {
        sSource = sSource + s;
    }

    /**
     * declPrint - Output code to the declaration buffer with no cr/lf.
     *
     * @param  s - a String - The code to be appended to the buffer.
     */
    public void declPrint(String s) {
        sDeclSource = sDeclSource + s;
    }

    /**
     * begPrint - Output code to the declaration buffer with no cr/lf.
     *
     * @param  s - a String - The code to be appended to the beginning buffer.
     */
    public void begPrint(String s) {
        sBegSource = sBegSource + s;
    }

    /**
     * begPrint - Output code to the declaration buffer with  cr/lf.
     *
     * @param  s - a String - The code to be appended to the beginning buffer.
     */
    public void begPrintln(String s) {
        sBegSource = sBegSource + s + '\n';
    }

    /**
     * beg2Print - Output code to the declaration buffer with no cr/lf.
     *
     * @param  s - a String - The code to be appended to the beginning buffer.
     */
    public void beg2Print(String s) {
        sBeg2Source = sBeg2Source + s;
    }

    /**
     * beg2Print - Output code to the declaration buffer with  cr/lf.
     *
     * @param  s - a String - The code to be appended to the beginning buffer.
     */
    public void beg2Println(String s) {
        sBeg2Source = sBeg2Source + s + '\n';
    }

    /**
     * begPrint - Output code to the declaration buffer with no cr/lf.
     *
     * @param  s - a String - The code to be appended to the beginning buffer.
     */
    public void memberDeclPrint(String s) {
        sMemberDeclSource = sMemberDeclSource + s;
    }

    /**
     * begPrint - Output code to the declaration buffer with  cr/lf.
     *
     * @param  s - a String - The code to be appended to the beginning buffer.
     */
    public void memberDeclPrintln(String s) {
        sMemberDeclSource = sMemberDeclSource + s + '\n';
    }

    /**
     * listenerPrint - Output code to the declaration buffer with no cr/lf.
     *
     * @param  s - a String - The code to be appended to the listener buffer.
     */
    public void listenerPrint(String s) {
        sListenerSource = sListenerSource + s;
    }

    /**
     * listenerPrintln - Output code to the declaration buffer with  cr/lf.
     *
     * @param  s - a String - The code to be appended to the listener buffer.
     */
    public void listenerPrintln(String s) {
        sListenerSource = sListenerSource + s + '\n';
    }


    /**
     * println - Output code to the code generator buffer with a cr/lf.
     *
     * @param  s - a String - The code to be appended to the buffer.
     */
    public void println(String s) {
        sSource = sSource + s + '\n';
    }

    /**
     * endPrintln - Output code to another buffer that is added at the end.
     *
     * @param  s - a String - The code to be appended to the buffer at the end.  It gets cr/lf.
     */
    public void endPrintln(String s) {
        sEndSource = sEndSource + s + '\n';
    }

    /**
     * endPrint - Output code to another buffer that is added at the end.
     *
     * @param  s - a String - The code to be appended to the buffer.
     */
    public void endPrint(String s) {
        sEndSource = sEndSource + s;
    }

    /**
     * declPrintln - Output code for declarations.  Done at beginning of main constructor.
     *
     * @param  s - a String - The code to be appended to the decl buffer.
     */
    public void declPrintln(String s) {
        sDeclSource = sDeclSource + s + '\n';
    }


    /**
     * generateDeclaration - Writes a declaration for the specified object with
     * the given identifier to the source buffer. The form of the declaration is:
     *
     *  Class ObjName = new Class();
     *
     * @param  theObject - The object for which to write source. This is always
     * a blueprint object.
     * @param  sObjectIdentifier - This is the identifier to use for the
     * object when writing the source.
     * @param  sPreCode - code before the declaration
     * @param  bMember - make the decl as a class member.
     *
     */

    public void generateDeclaration(Object theObject, String sObjectIdentifier, String sPreCode, boolean bMember) throws UIFactoryException {
        UIFactoryBlueprint obj = (UIFactoryBlueprint) theObject;
        String sFullClassName = obj.getClassName();
        String sClassName = sFullClassName.substring(sFullClassName.lastIndexOf('.') + 1);

        if (bMember == true)
            memberDeclPrint(sPreCode);
        else
            declPrint(sPreCode);

        try {
            String sConstructorParameterList = obj.theObject.getConstructorParameters();
            if (sConstructorParameterList == null)
                sConstructorParameterList = "";
            // Invoke the constructor with parameters.
            if (bMember == true) // All members are 'protected' for now.
                memberDeclPrint("protected " + sClassName + " " + sObjectIdentifier + " = new " + sClassName);
            else
                declPrint(sClassName + " " + sObjectIdentifier + " = new " + sClassName);
            if (bMember == true)
                memberDeclPrintln("(" + sConstructorParameterList + ");");
            else
                declPrintln("(" + sConstructorParameterList + ");");
        } catch (Exception e) {
            throw new UIFactoryException("Error in declaration of " + sObjectIdentifier + " during code generation.");
        }
    }


    /**
     * generateAdditionalCustomCode - This method writes the object's stored
     * custom code into the source file.
     *
     * @param  sCustomCode - This is the custom code to write.
     *
     */
    public void generateAdditionalCustomCode(String sCustomCode) {
        println(sCustomCode);
    }

    /**
     * generatePreamble - This method writes the code up to the first object
     * declaration, including package declaration, import declarations, and the
     * main class declaration.
     *
     * @param  sPackageName - The name of the package the class will reside in.
     * @param  sClassName - The name of the main class for the UI interface.
     *
     */
    public void generatePreamble(String sPackageName, String sClassName) throws UIFactoryException {
        begPrintln("// PREAMBLE CODE");
        begPrintln("package " + sClassName + ";");
        begPrintln("");
        MetaImport theImport = theProject.theProgramData.theFirstImport;
        if (theImport != null)
            begPrintln("// These are the import packages.");
        while (theImport != null) {
            String sImportName = theImport.sImportName;
            begPrintln("import " + sImportName + ";");
            theImport = theImport.theNextImport;
        }
        begPrintln("");
    }


    /**
     *
     * generateClasses - This method creates all the classes defined in the program aside
     * from the main class.  Class definitions are stored in the metadata.
     *
     */


    public String getCustomCodeGenerator(String sClassName) {
        // Now get access to the class metadata
        MetaClass theMetaClass = theProject.findClass(sClassName);
        if (theMetaClass != null) {
            if (theMetaClass.theMetadata.sCustomCodeGenerationMethod != null)
                return theMetaClass.theMetadata.sCustomCodeGenerationMethod.trim();
        }
        return "";
    }


    ListenerDiscriminator findDiscriminator(ArrayList theDiscriminators, MetaClassDetail theClass) {
        int iDiscriminators = theDiscriminators.size();
        for (int i = 0; i < iDiscriminators; i++) {
            ListenerDiscriminator theDiscriminator = (ListenerDiscriminator) theDiscriminators.get(i);
            if (theDiscriminator.theClass.equals(theClass))
                return theDiscriminator;
        }
        return null;
    }

    public void generateListenerClasses(String sOutputIdentifier, UIFactoryBlueprint theObject) {
        // Make a class for each listener for the object.
        // The class name is identifier of object plus name of listener concatenated
        // Implements is the name of the listener
        // One method for each event
        // Class is written to buffer that will be added before epilog code.
        MetaObject theDesignObject = theObject.theObject;
        MetaMethodLinkage theMethod = theDesignObject.theFirstMethod;
        ArrayList theListeners = new ArrayList(0);
        ArrayList theDiscriminators = new ArrayList(0);
        while (theMethod != null) {
            MetaClassDetail theClass = theMethod.theListener;
            if (theListeners.contains(theClass) == false) {
                theListeners.add(theMethod.theListener);
                ListenerDiscriminator theDiscriminator = new ListenerDiscriminator(theClass);
                theDiscriminators.add(theDiscriminator);
                theDiscriminator.addMethod(theMethod);
            } else {
                ListenerDiscriminator theDiscriminator = findDiscriminator(theDiscriminators, theClass);
                theDiscriminator.addMethod(theMethod);
            }
            theMethod = theMethod.theNextMethod;
        }
        // remove listeners that have no special implementation.
        for (int i = 0; i < theDiscriminators.size(); i++) {
            ListenerDiscriminator theDiscriminator = (ListenerDiscriminator) theDiscriminators.get(i);
            if (theDiscriminator.isSpecial() == false)
                theDiscriminators.remove(theDiscriminator);
        }
        // generate classes for the others.
        for (int i = 0; i < theDiscriminators.size(); i++) {
            ListenerDiscriminator theDiscriminator = (ListenerDiscriminator) theDiscriminators.get(i);
            MetaClassDetail theListener = theDiscriminator.theClass;
            String sListenerName = theListener.sClassName;
            listenerPrintln("/***__@Listener: [" + sOutputIdentifier + "$$$" + sListenerName + "] ***/");
            listenerPrint("class " + sOutputIdentifier + "$$$" + sListenerName);
            // Check if it's a listener or an adapter.  Adapter extends.  Listener implements interface.
            boolean bIsAdapter = (sListenerName.lastIndexOf("Adapter") == -1) ? false : true;
            if (bIsAdapter)
                listenerPrintln(" extends " + sListenerName + "  {");
            else
                listenerPrintln(" implements " + sListenerName + "  {");
            // Now write the methods.
            MetaMethod theListenerMethod = theListener.theFirstMethod;
            while (theListenerMethod != null) {
                String sCode = theListenerMethod.getCode();
                if (sCode.trim().length() != 0) {
                    listenerPrintln("/***__@Method: [" + theListenerMethod.sMethodName + "] ***/");
                    listenerPrintln(sCode);
                    listenerPrintln("/***__@@Method: [" + theListenerMethod.sMethodName + "] ***/");
                }
                theListenerMethod = theListenerMethod.getNextMethod();
            }
            // Close the class.
            listenerPrintln("  }");
            listenerPrintln("/***__@@Listener: [" + sListenerName + "] ***/");
        }
        theObject.theDiscriminators = theDiscriminators;
    }

    public void generateMainClass(String sClassName) {
        // Write smart-tag that allows us to find the class later.
        begPrintln("/***__@Class: [" + sClassName + "] ***/");
        begPrintln("public class " + sClassName);
        begPrintln("{");
        if (sMainObject != null)
            begPrintln("public " + sClassName + " theProjectWrapper = this;");
        if (!bAbalet) {
            beg2Println("public " + sClassName + " () {");
        } else {
            beg2Println("Object theServerLink;");
            beg2Println("boolean bIsShutdown = false");
            beg2Println("public " + sClassName + " (Object theServerLink) {");
            beg2Println("bIsShutdown = false;");
            beg2Println("this.theServerLink = theServerLink;");
            beg2Println("registerWithServer();");
        }
        beg2Println("try {");
        beg2Println("initializeInterface();");
        beg2Println("} catch (Exception e) {");
        beg2Println("e.printStackTrace();");
        beg2Println("}");
        beg2Println("}");
        beg2Println("");
        beg2Println("public void initializeInterface() {");
    }

    public void generateClasses() throws UIFactoryException {

        // Write the subclass declarations.
        MetaClass theClass = theProject.theFirstClass;
        boolean bSpecializedClassesExist = false;
        generateMainClass(sMainClass);
        while (theClass != null) {
            String sClassName = theClass.theMetadata.sClassName;
            // Now get access to the class metadata (which is off of the root)
            if (theClass.theFirstObject != null) {  // if there are instances
                String sFullSuperClassName = null;
                String sSuperClass = null;
                // Get the subclass method declarations from the metadata xml doc.
                int iMethods = theClass.theMetadata.iNumberOfMethods;
                int iMemberData = theClass.theMetadata.iNumberOfDataElements;

                // If there are no new methods, the class was just made to implement
                // code generation and can be skipped.

                if ((iMethods != 0) || (iMemberData != 0)) {
                    if (bSpecializedClassesExist == false) {
                        endPrintln("// Declare specialized classes.");
                        bSpecializedClassesExist = true;
                    }

                    // In program subclsses never instantiate so we need to copy
                    // constructors from the superclasses.  Start by getting the
                    // first instantiable class.

                    Class clsSuper = null;
                    if (sFullSuperClassName != null) {
                        String sInstantiableClass = sFullSuperClassName;
                        while ((clsSuper == null) && (sInstantiableClass != null)) {
                            try {
                                clsSuper = Class.forName(sInstantiableClass);  // Need better solution for this!
                            } catch (java.lang.ClassNotFoundException e) {
                                int iLastIndex = sInstantiableClass.lastIndexOf(".") + 1;
                                String sTestClass = sInstantiableClass.substring(iLastIndex);
                                MetaClass theTestClass = theProject.findClass(sTestClass);
                                sInstantiableClass = null;
                                if (theTestClass != null) {  // should be valid since it's a subclass of ours.
                                    sSuperClass = theTestClass.theMetadata.sClassName;
                                    if (sSuperClass != null)
                                        sInstantiableClass = sSuperClass;
                                    // For code generation purposes, the class is defined in the file.
                                    // Use the short cut name for the superclass
                                    sFullSuperClassName = sSuperClass;
                                }
                            }
                        }

                        //  clsSuper is what we will use for constructors and searches outside of the metadoc.
                        // Prepare the class declaration.
                        // Get the interface metadata for the class declaration.

                        // Create a class declaration for the subclass.
                        // Write smart-tag that allows us to find the class later.
                        endPrintln("/***__@Class: [" + sClassName + "] ***/");
                        endPrintln("class " + sClassName +
                                " extends " + sFullSuperClassName);

                        endPrintln("{");

                        MetaDataMember theMemberData = theClass.theMetadata.theFirstData;
                        while (theMemberData != null) {
                            String s = "/**\n" +
                                    "  *\n" +
                                    "  *  ";
                            if (theMemberData.getDoc() != null)
                                s = s + theMemberData.getDataName() + " - " + theMemberData.getDoc();
                            else
                                s = s + theMemberData.getDataName() + " - Needs documentation";
                            s = s + "\n";
                            s = s + "*\n";
                            s = s + "*/\n\n";
                            s = s + "protected " + theMemberData.getTypeName() + "    " +
                                    theMemberData.getDataName();
                            if (theMemberData.getValue() != null)
                                s = s + " = " + theMemberData.getValue();
                            s = s + ";\n\n";
                            endPrintln("/***__@Member Data: [" + theMemberData.getDataName() + "] ***/");
                            endPrintln(s);
                            endPrintln("/***__@@Member Data: [" + theMemberData.getDataName() + "] ***/");
                        }

                        // Write the constructors. Constructors for main class execute the init code.
                        if (clsSuper != null) {
                            Constructor[] theConstructors = clsSuper.getConstructors();
                            int iConstructorCount = theConstructors.length;

                            int iConstructor = 0;
                            for (iConstructor = 0; iConstructor < iConstructorCount; iConstructor++) {
                                String sConstructorMethod = "  public " + sClassName + "(";
                                String sConstructorDoc = "  /**\n";
                                sConstructorDoc = sConstructorDoc + "    *  Constructor for class " + sClassName + " derived from constructor of " + sFullSuperClassName + "\n";
                                Constructor constructorTest = theConstructors[iConstructor];
                                Class[] classesConstructorParam = constructorTest.getParameterTypes();
                                String sSuperParamList = " ";
                                for (int j = 0; j < classesConstructorParam.length; j++) {
                                    Class clsParam = classesConstructorParam[j];
                                    String sLongParamClassName = clsParam.getName();
                                    StringBuffer sShortParamClassName = new StringBuffer(sLongParamClassName.substring(sLongParamClassName.lastIndexOf('.') + 1));
                                    int iNameLength = sShortParamClassName.length();
                                    // Handle arrays of objects and strings!
                                    if (sLongParamClassName.endsWith(";")) {
                                        sShortParamClassName.setLength(iNameLength - 1);
                                        sShortParamClassName = sShortParamClassName.append("[]");
                                    }
                                    sConstructorMethod = sConstructorMethod + sShortParamClassName + " param" + j;
                                    sSuperParamList = sSuperParamList + "param" + j;

                                    if (j != (classesConstructorParam.length - 1)) {
                                        sSuperParamList = sSuperParamList + ", ";
                                        sConstructorMethod = sConstructorMethod + ", ";
                                    }

                                }
                                sConstructorMethod = sConstructorMethod + " ) {\n";

                                sConstructorMethod = sConstructorMethod + "    super(" + sSuperParamList + " );\n";
                                sConstructorMethod = sConstructorMethod + "  }\n";
                                endPrintln(sConstructorDoc + "    */\n\n" + sConstructorMethod + "\n");
                            }
                        }
                        // Write methods for the specialized classes.  Methods are stored
                        // in the metadata xml document and are just copied into the
                        // final source.
                        MetaMethod theMethod = theClass.theMetadata.theFirstMethod;
                        while (theMethod != null) {
                            endPrintln("/***__@Method: [" + theMethod.getMethodName() + "] ***/");
                            if (theMethod.getDoc() != null) {
                                String s = theMethod.getDoc();
                                endPrintln(s);
                            }
                            if (theMethod.getCode() != null) {
                                String s = theMethod.getCode();
                                endPrintln(s);
                            }
                            endPrintln("/***__@@Method: [" + theMethod.getMethodName() + "] ***/");
                        }

                        endPrintln("}");
                        endPrintln("/***__@@Class: [" + sClassName + "] ***/");
                        endPrintln("");
                    }  // End the block for real class with methods

                }
            }
            theClass = theClass.theNextSibling;
        }
    }

    public void generateAbalet() {

        // Properties
        println("private Object theServerLink;");
        println("private String theTitle;");
        println("private boolean bIsShutdown;");
        println("HashMap theParameters = null;");
        // Methods.
        println("public HashMap getAbaletParameters() {");
        println("HashMap objRetVal = null;");
        println("Class cls = theServerLink.getClass();  // actually an applet link");
        println("Class clsParams[] = new Class[1];");
        println("try {");
        println("clsParams[0] = Class.forName(\"java.lang.Object\");");
        println("Method m = cls.getMethod(\"getParameters\", clsParams);");
        println("objRetVal = (HashMap)m.invoke(theServerLink, null);");
        println("}");
        println("catch (java.lang.NoSuchMethodException e1) {");
        println("return null;");
        println("}");
        println("catch (java.lang.reflect.InvocationTargetException e2) {");
        println("return null;");
        println("}");
        println("catch (java.lang.IllegalAccessException e3) {");
        println("return null;");
        println("}");
        println("catch (java.lang.ClassNotFoundException e4) {");
        println("return null;");
        println("}");
        println("return objRetVal;");
        println("}");


        println("public String getParameterValueByName(String sName) {");
        println("Class cls = theServerLink.getClass();  // actually an applet link");
        println("Class clsParams[] = new Class[1];");
        println("String sRetVal = null;");
        println("try {");
        println("clsParams[0] = Class.forName(\"java.lang.String\");");
        println("Method m = cls.getMethod(\"getParameterValueByName\", clsParams);");
        println("Object objParams[] = new Object[1];");
        println("objParams[0] = sName;");
        println("sRetVal = (String)m.invoke(theServerLink, objParams);");
        println("}");
        println("catch (java.lang.NoSuchMethodException e1) {");
        println("return null;");
        println("}");
        println("catch (java.lang.reflect.InvocationTargetException e2) {");
        println("return null;");
        println("}");
        println("catch (java.lang.IllegalAccessException e3) {");
        println("return null;");
        println("}");
        println("catch (java.lang.ClassNotFoundException e4) {");
        println("return null;");
        println("}");
        println("return sRetVal;");
        println("}");


        println("public Boolean canShutdown() {");
        println("Object confirmText = \"Do you really want to shut down the application?\";");
        println("String confirmTitle = \"Shutdown Confirmation for \"+theTitle;");
        println("int optionType = JOptionPane.YES_NO_OPTION;");
        println("int messageType = JOptionPane.QUESTION_MESSAGE;");
        println("// if selection is 'yes', selected value is 0");
        println("// if selection is 'no', selected value is 1");
        println("int selectedValue = JOptionPane.showConfirmDialog(this, confirmText, confirmTitle, optionType, messageType);");
        println("if (selectedValue==0) {");
        println("return new Boolean(true);");
        println("}");
        println("else return new Boolean(false);");
        println("}");
        println("public void doShutdown() {");
        println("hide();");
        println("bIsShutdown = true;");
        println("}");
        println("public Boolean isShutdown() {");
        println("return new Boolean(bIsShutdown);");
        println("}");

        println("public boolean registerWithServer() {");
        println("Class cls = theServerLink.getClass();  // actually an applet link");
        println("Class clsParams[] = new Class[1];");
        println("try {");
        println("clsParams[0] = Class.forName(\"java.lang.Object\");");
        println("Method m = cls.getMethod(\"addObject\", clsParams);");
        println("Object objParams[] = new Object[1];");
        println("objParams[0] = this;");
        println("m.invoke(theServerLink, objParams);");
        println("}");
        println("catch (java.lang.NoSuchMethodException e1) {");
        println("return false;");
        println("}");
        println("catch (java.lang.reflect.InvocationTargetException e2) {");
        println("return false;");
        println("}");
        println("catch (java.lang.IllegalAccessException e3) {");
        println("return false;");
        println("}");
        println("catch (java.lang.ClassNotFoundException e4) {");
        println("return false;");
        println("}");
        println("return true;");
        println("}");

        println("public boolean unregisterWithServer() {");
        println("Class cls = theServerLink.getClass();  // actually an applet link");
        println("Class clsParams[] = new Class[1];");
        println("try {");
        println("clsParams[0] = Class.forName(\"java.lang.Object\");");
        println("Method m = cls.getMethod(\"removeObject\", clsParams);");
        println("Object objParams[] = new Object[1];");
        println("objParams[0] = this;");
        println("m.invoke(theServerLink, objParams);");
        println("}");
        println("catch (java.lang.NoSuchMethodException e1) {");
        println("return false;");
        println("}");
        println("catch (java.lang.reflect.InvocationTargetException e2) {");
        println("return false;");
        println("}");
        println("catch (java.lang.IllegalAccessException e3) {");
        println("return false;");
        println("}");
        println("catch (java.lang.ClassNotFoundException e4) {");
        println("return false;");
        println("}");
        println("return true;");
        println("}");

        println("public void exitingCode() { ");
        println("bIsShutdown = true;");
        println("unregisterWithServer();");
        println("}");


    }

    /**
     * generateEpilogue - This method writes the code after the last object is
     * processed.  This is the code that contains the event handler to close the
     * program along with associated specialized subclasses.
     *
     * @param  sFrameObjectName - The name of the main frame.
     * @param  sLookAndFeel - The name of the platform lnf.
     *
     */
    public void generateEpilogue(String sFrameObjectName, String sLookAndFeel) throws UIFactoryException {
        println("// Show the main window");
        println(sFrameObjectName + ".show();");
        println("  }  // End of the initializeInterface method. ");
        if (bAbalet) {  // put in all the abalet stuff.
            generateAbalet();
        }
        if (bAbalet) {  // cannot change look and feel in an abalet (multiple abalets are running!)
            println("public void jammMain(String[] args, Object objServerLink, Object userObject) {");
            println(sMainClass + " objMain = new " + sMainClass + "(objServerLink);");
            println("}");
        } else {
            println("public static void main(String[] args) {");
            if (sLookAndFeel != null) {
                println("try {");
                println("UIManager.setLookAndFeel(\"" + sLookAndFeel + "\");");
                println("} catch(Exception evt) {");
                println("JOptionPane.showMessageDialog(null, \"Cannot set look and feel to " + sLookAndFeel +
                        "\", \"Swing Error\", JOptionPane.INFORMATION_MESSAGE);");
                println("}");
            }
            println(sMainClass + " objMain = new " + sMainClass + "();");
            println("}");
        }


        endPrintln("}");
        // Main class ends here since jbinit shouldn't give errors.
        endPrintln("/***__@@Class: [" + sMainClass + "] ***/");


        endPrintln("");
        endPrintln("");
        generateClasses();

        endPrintln("");

    }
}
