/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;


/**
 * Title:        uifactory
 * Description:  Takes XML file of object descriptions and builds a java program.
 * Copyright:    Copyright (c) 2001
 * Company:      Abacus Software
 * @author       Michael Gouker (Cagey Logic)
 * @version 0.1
 */

import ch.abacus.lib.ui.renderer.programmableCompiler.ProgrammableCompiler;

import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;

/**
 * UIFactoryCodeGeneration is a simple interface describing the methods required
 * to invoke the code generator.  Each object stores its state and generates
 * the appropriate code that describes its state.
 */

interface UIFactoryCodeGeneration {
    /**
     * Command the object to generate its code.
     * @param theOutput  Code generator that writes source to an embedded print writer.
     * @param sObjectIdentifier  The identifier that names the object in the source.  This parameter
     * is necessary because the object name is hashed by Java.
     */

    public void generateCode(UIFactoryCodeGenerator theOutput,
                             String sObjectIdentifier) throws Throwable;
}


/**
 * Assembly line is a class that controls all aspects of UI construction.
 * It invokes the XML parser to load both metadata and object descriptions.
 * Using the object descriptions, it creates objects and sets the properties
 * of the objects to the values that are provided in the object descriptions.
 * Once it finishes, it generates code (by invoking code generation on each
 * object) and then calls a java compiler to build the class files.
 */

public class UIFactoryAssemblyLine {

    /**
     * sObjectDescriptionDocumentName - the XML document that contains the object
     * descriptions.  Object descriptions are names of objects, names of their
     * classes, messages to invoke (can be more than just properties), and custom
     * code to insert for each object.
     */
    protected String sObjectDescriptionDocumentName;
    /**
     * sMetaDataDocumentName - the XML document that contains the class metadata.
     * This includes the names of classes, the properties that will be included
     * (both the set and get methods) and code that is automatically plugged in
     * by the code generator.
     */
    protected String sMetaDataDocumentName;
    /**
     * sPackageName - the name of the package that will contain all classes
     * generated by the system.
     */
    protected String sPackageName;
    /**
     * BroadcasterHead - The objects created are kept in a heirarchical tree for
     * later invocation of the code generator.  Each level of the tree is a
     * UI container level.
     */
    protected UIFactoryBroadcaster BroadcasterHead;
    /**
     * bDebugCall - Just for debugging.  Prints a nice message before and after
     * sending a message to the object.
     */
    protected boolean bDebugCall = false;
    /**
     * factoryMetadata - This object contains all the logic that describes/interprets
     * the metadata that is used to do code generation.
     */
    protected MetadataProvider factoryMetadata;
    /**
     * dc - This is a compiler that is invoked to generate the class files once
     * we have generated the java code to describe the objects.
     */
    protected ProgrammableCompiler dc;

    /**
     * invHelper - An object that facilitates invocations for object generation.
     */
    protected UIFactoryInvokeHelper invHelper;

    /**
     * XMLComponentConstructor - Creates components of objects or classes from an
     * xml input.  This is used in object construction and class construction.
     */

    protected UIFactoryComponentFactory componentFactory = null;
    /**
     * sConstructorParamList - This is the parameter list for the constructor as text.
     * It will be added to the source at code generation time.  It is created from the
     * constructor nodes of the object.
     */
    protected String sConstructorParamList = null;  // If there is a param list for constructor, make it too.

    /**
     * sPreConstrutionCode - This is code that is executed before the constructor is called.
     * It is added to the source before the constructor.
     */
    protected String sPreConstructionCode = null;

    /**
     * sOutputDirectoryName = This is the place the output of the compiler will be copied upon completion.
     */
    public String sOutputDirectoryName = "";

    /**
     * theCodeJumper - Object stores the positions of different parts of the source code for error lookup.
     */
    public UIFactoryCodeJumper theCodeJumper = null;

    public MetaProject theDesignProject;

    /**
     * The Project - we use the project to get access to the classes.
     */

    /**
     * UIFactoryAssemblyLine - Constructor for UIFactoryAssemblyLine class.
     *
     *
     * The constructor sets member variables and creates a metadata provider,
     * a class pool, and a java compiler interface.
     */

    public UIFactoryAssemblyLine(MetaProject theProject, String sWorkingDirectory) throws electric.xml.ParseException {
        sOutputDirectoryName = sWorkingDirectory;
        theDesignProject = theProject;
        dc = new ProgrammableCompiler();
    }


    /**
     * ProcessObjects - This function processes an XML document of object descriptions
     * creating objects and sending messages to them to set properties and so forth.
     * It stores the objects in a broadcast tree structure.
     *
     *   @param theParent - UIFactoryBroadcaster - the parent node in the broadcaster where the
     *   object (described by the element) will be placed as a child.
     *
     *
     *  This is the order of events for each object:
     *
     *  First, custom code is processed.
     *  Second, all the methods are called to set the object's properties and state.
     *  Third, subobjects are processed.  It is a recursive process.
     *
     */

    public boolean ProcessObjects(UIFactoryBroadcaster theParent, MetaObject theObject) throws Throwable {

        // Get information from the xml document that describes the object.
        String sObjectIdentifier = theObject.getName();
        String sPostConstructionCode = theObject.getPostConstructionCode();
        String sPreConstructionCode = theObject.getPreConstructionCode();
        String sConstructorParameters = theObject.getConstructorParameters();

        UIFactoryBlueprint objMetaObject = new UIFactoryBlueprint(theObject);

        if (objMetaObject == null)
            throw new UIFactoryException("Cannot create object " + sObjectIdentifier);

        // sParentName is used to name the container for assignments of layout.
        String sParentName = null;

        // Put the new object in the broadcaster so it can generate code later.
        UIFactoryBroadcaster oThisObjectAccessor;
        if (BroadcasterHead == null) {
            BroadcasterHead = new UIFactoryBroadcaster(objMetaObject, sObjectIdentifier);
            oThisObjectAccessor = BroadcasterHead;
        } else {
            oThisObjectAccessor = new UIFactoryBroadcaster(objMetaObject, sObjectIdentifier);
            theParent.addChildObject(oThisObjectAccessor);
            sParentName = theParent.sObjectIdentifier;
            objMetaObject.setParentMeta(theParent.oTheObject);
        }

        // Store the custom code first.  Custom code is stored in the object.
        // Each class is subclassed for code generation purposes adding the
        // setCustomCode method and the codeGeneration method.
        if (sPostConstructionCode != null) {
            // Set the custom data by invoking setCustomCode.
            objMetaObject.setCustomCode(sPostConstructionCode);
        }

        if (sConstructorParameters != null) {
            // Constructor parameter list from metadata.
            objMetaObject.setConstructorParameterList(sConstructorParameters);
        }

        if (sPreConstructionCode != null) {
            // Constructor parameter list from metadata.
            objMetaObject.setPreConstructionCode(sPreConstructionCode);
        }

        MetaProperty theProperty = theObject.theFirstProperty;
        while (theProperty != null) {
            int iCount = theProperty.getValueCount();
            for (int iIndex = 0; iIndex < iCount; iIndex++) {
                String oValue = theProperty.getLiteralValue(iIndex);
                // Set the properties.  These can be queried at code generation time.
                if (oValue != null) {
                    objMetaObject.setPropertyValue(theProperty.theName, iIndex, oValue);
                }
            }
            theProperty = theProperty.theNextProperty;
        }

        // Set the container name using the parent child structure here.
        if (sParentName != null)
            objMetaObject.setContainerName(sParentName);

        // 4/2/2002 mg changed default to member
        if (theObject.theParentObject != null)
            objMetaObject.setDeclareAsMember(Boolean.TRUE);

        MetaObject theChild = theObject.theFirstChild;
        while (theChild != null) {
            ProcessObjects(oThisObjectAccessor, theChild);
            theChild = theChild.theNextObject;
        }

        return true;
    }

    /**
     * CreateUI - This method (which takes no parameters) is just a driver for the
     * processing of the XML document that describes the objects.  This is what it
     * does:
     *
     *    1.  Gets the metadata from the metadata xml document.
     *    2.  Loads an array of simple objects (one for each class) that are a
     *    template for creating the objects that will actually be processed.
     *    3.  Create a code generator, passing the factory and the xml document
     *    that descibes the metadata.
     *    4.  Get access to the xml document that contains the descriptions.
     *    5.  Process all the objects setting their properties as defined in the
     *    input xml object descriptor document.
     *    6.  Generate all the code using the broadcaster built while processing
     *    the objects in step 5.
     *    7.  Call the java compiler to build the class files.
     *    8.  Copy the files to the output directory.
     *
     *  That's all, folks!
     *
     */

    public boolean CreateUI() throws Throwable {

        MetaObject theRootObject = theDesignProject.getFirstObject();
        String sTransientDirectory = theDesignProject.theProgramData.sTransientDirectory;
        String sProjectName = theDesignProject.theProgramData.sProjectName;
        String sMainClass = theDesignProject.getFirstObject().theClass.getMetadata().sClassName;
        boolean bAbalet = theDesignProject.theProgramData.bAbalet;
        boolean bApplication = theDesignProject.theProgramData.bApplication;
        String sPackage = theDesignProject.theProgramData.sPackageName;
        String sPlatformLookAndFeel = theDesignProject.theProgramData.sPlatformLookAndFeel;

        MetadataProvider factoryMetadata = new MetadataProvider(theDesignProject);
        // Set up component factory - used by class pool and by object construction.
        componentFactory = new UIFactoryComponentFactory(null, factoryMetadata);

        //  Create code generator that holds source that is generated by parsing the xml file of object decls.
        UIFactoryCodeGenerator codeOutput = new UIFactoryCodeGenerator(factoryMetadata, theDesignProject, theRootObject);

        //  Parse input file generating objects.
        ProcessObjects(BroadcasterHead, theRootObject);

        //  Do code generation.
        //   Write the beginning of the class file.
        codeOutput.generatePreamble(sPackageName, sMainClass);
        //   Tell the objects to generate their code.
        BroadcasterHead.generateCode(codeOutput);
        //   Write the end of the class file.
        codeOutput.generateEpilogue(BroadcasterHead.sObjectIdentifier, sPlatformLookAndFeel);

        //  Invoke the compiler.
        //   Now get everything back into memory for the compilation.

        String sSourceCode = codeOutput.getFormattedSource();

//  Write file before compilation to test for error.
        File outstuff = new File("TheStuff");
        PrintWriter pw = new PrintWriter(new FileWriter(outstuff));
        pw.print(sSourceCode);
        pw.close();

        String[] defns = {sSourceCode};     // Only one at a time, may create multiple classes.

        theCodeJumper = new UIFactoryCodeJumper(sSourceCode);

        //   Set up class path & compile the new application.
        String sClassPath = System.getProperty("java.class.path");
        int iOtherClassPaths = theDesignProject.theProgramData.getAdditionalClassPathCount();
        String sPathSeparator = System.getProperty("path.separator");
        for (int iClassPath = 0; iClassPath < iOtherClassPaths; iClassPath++) {
            sClassPath = sClassPath + sPathSeparator + theDesignProject.theProgramData.getAdditionalClassPath(iClassPath);
        }
        dc.setProperty("javaClassPath", sClassPath);
        if (sTransientDirectory != null)
            dc.setProperty("tempDirNameRoot", sTransientDirectory);
        File[] files = dc.compileClassesToFiles(defns);
        // Now move the compiled classes and the source to the output directory.
        int iFilesToWrite = files.length;
        String sTempDir = dc.getProperty("tempDirNameRoot");
        for (int i = 0; i < iFilesToWrite; i++) {
            int iMatch = files[i].getPath().indexOf(sTempDir);
            if (iMatch > 0) {
                // Get the intransient part of the file name (the package and the class) by itself.
                String sSeparator = files[i].getPath().substring(iMatch - 1, iMatch);
                String sFileSubName = files[i].getPath().substring(iMatch);
                iMatch = sFileSubName.indexOf(sSeparator);
                // sFileSubName is the useful part of the file (still has .class extension)
                sFileSubName = sFileSubName.substring(iMatch);
                // Prepare old names and new names for copy of source and class files to output directory.
                String sOldClassFileName = files[i].getPath();
                if (sOutputDirectoryName==null || sOutputDirectoryName.trim().length() == 0)
                    sOutputDirectoryName = "." + sSeparator;
                String sNewClassFileName = sOutputDirectoryName + sFileSubName;
                int iSuffix = sNewClassFileName.lastIndexOf(".class");
                String sNewJavaFileName = sNewClassFileName.substring(0, iSuffix) + ".java";
                iSuffix = sOldClassFileName.lastIndexOf(".class");
                String sOldJavaFileName = sOldClassFileName.substring(0, iSuffix) + ".java";
                // Make sure the output directory exists.
                iSuffix = sNewClassFileName.lastIndexOf(sSeparator);
                String sOutputDirectoryFullPath = sNewClassFileName.substring(0, iSuffix);
                new File(sOutputDirectoryFullPath).mkdirs();
                // Copy the files.
                FileCopy.copyFile(sOldClassFileName, sNewClassFileName);
                FileCopy.copyFile(sOldJavaFileName, sNewJavaFileName);
//              try {
                File oldFile = new File(sOldClassFileName);
                oldFile.delete();
                oldFile = new File(sOldJavaFileName);
                oldFile.delete();
//              }
            }
        }
        dc.cleanup();
        return true;
    }

}

