/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

import ch.abacus.designcockpit.ide.SuperDesignCockpit;
import ch.abacus.lib.ui.renderer.common.generator.AbacusCodeGenerator;
import ch.abacus.lib.ui.renderer.common.generator.DropletCodeGenerator;
import ch.abacus.lib.ui.renderer.common.generator.ProjectCodeGenerator;
import ch.abacus.lib.ui.renderer.programmableCompiler.CompilationException;
import ch.abacus.lib.ui.renderer.programmableCompiler.ProgrammableCompiler;

import java.io.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;

/**
 * <p>Title: uifactory</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2001</p>
 * <p>Company: Abacus Research</p>
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 *
 *  Reads and writes saved project documents.
 *
 */

public class ProjectDocument {

    private SuperDesignCockpit theCockpit;
    public String sOutputDocumentName = "";
    MetaProject theDesignProject = null;

    private boolean bOpenSource = false;

    private FileOutputStream fostream = null;
    private JarOutputStream jarOutputStream = null;
    private ProjectCodeGenerator theCodeGenerator;

    public ProjectDocument(SuperDesignCockpit theCockpit, String theDocumentName, MetaProject theDesignProject) {
        this.theCockpit = theCockpit;

        boolean bDroplets = theDesignProject.getMetaDataUser().getDropletsState();
        if (!bDroplets)
            theDesignProject.setCodeGenerator(new AbacusCodeGenerator(this));
        else
            theDesignProject.setCodeGenerator(new DropletCodeGenerator(this));

        bOpenSource = theDesignProject.getMetaDataUser().getOpenSourceState();

        sOutputDocumentName = theDocumentName;
        this.theDesignProject = theDesignProject;

        // Due to jar paths the jar they will always point to root
        if (bOpenSource) {
            String sFileSeparator = System.getProperty("file.separator");
            int iMatchSlash = sOutputDocumentName.lastIndexOf(sFileSeparator);
            if (iMatchSlash != -1) {
                sOutputDocumentName = sOutputDocumentName.substring(iMatchSlash + sFileSeparator.length());
            }
            sOutputDocumentName = theDesignProject.theProgramData.sWorkspace + sFileSeparator + sOutputDocumentName;
        }
    }

    public boolean isOpenSource() {
        return this.bOpenSource;
    }

    public boolean save(MetaProjectHeader theProgInfo, boolean bCompileClass) throws HammerException,
            java.io.IOException {  // Outputs xml from progInfo of parser.

        if (theCockpit.isCustomizing() == false) {
            theDesignProject.getCodeGenerator().reset();
            writeDecl();
        }
        if (theDesignProject.getInputType() == MetaProject.FROM_RESOURCE)
            return saveToFile(theProgInfo, true ? bOpenSource : false, bCompileClass);
        else
            return saveToFile(theProgInfo, false, bCompileClass);
    }

    public boolean writeDecl() {
        PrintWriter pwOutput = null;
        // write the decl file.
        File fileOutput = new File(sOutputDocumentName + ".decl");
        try {
            pwOutput = new PrintWriter(new FileWriter(fileOutput));
        } catch (IOException e) {
            return false;
        }
        theDesignProject.writeDeclarations(this);
        pwOutput.println("// This file contains declarations of the user interface objects in project " + this.sOutputDocumentName);
        pwOutput.println();
        pwOutput.println("// Declarations of variables for this user interface.");
        pwOutput.println();
        pwOutput.println(getDeclarations());
        pwOutput.println();
        pwOutput.println("// Assignments for this user interface");
        pwOutput.println();
        pwOutput.println("public void getReferences(){");
        pwOutput.print(getReferences());
        pwOutput.println("}");
        pwOutput.println();
        pwOutput.println("// Using the renderer");
        pwOutput.println();

        pwOutput.println("// Step 1: Declare a variable 'm_abaRenderer' that is a reference to the AbaRenderer");
        pwOutput.println("// For example:  AbaRenderer m_AbaRenderer = new AbaRenderer(sDocumentName, false, theGlobalInterfaceObject);");
        pwOutput.println("// ");
        pwOutput.println("//               First parameter is name of the project document. ");
        pwOutput.println("//               Second parameter is false (unless rendering inside design cockpit. ");
        pwOutput.println("//               Third parameter is the global interface (if NULL, second parameter must be true");
        pwOutput.println("// ");
        pwOutput.println("// Step 2: Load the project.");
        pwOutput.println("// ");
        pwOutput.println("//               boolean bTestLoad = m_AbaRenderer.load();");
        pwOutput.println("// ");
        pwOutput.println("// Step 3: You can set the language of the renderer like this:");
        pwOutput.println("// ");
        pwOutput.println("//               m_AbaRenderer.setLanguage(HammerLanguagePresentation.DEUTSCH);");
        pwOutput.println("// ");
        pwOutput.println("// Step 4: Render the interface.");
        pwOutput.println("// ");
        pwOutput.println("//               m_AbaRenderer.renderInterface();");
        pwOutput.println();
        pwOutput.close();
        return true;
    }

    private String getDeclarations() {
        return getCodeGenerator().getDeclarationBuffer();
    }

    private String getReferences() {
        return getCodeGenerator().getAssignmentBuffer();
    }

    void createNewJar(String sNewJarName) throws HammerException {
        File newJarFile = null;
        JarFile newJar = null;

        try {
            newJarFile = new File(sNewJarName + ".jar");

            if (newJarFile.exists())
                newJarFile.delete();

            newJarFile.createNewFile();
        } catch (IOException e0) {
            System.err.println("Cannot create jar file " + sNewJarName + ".tmp" + ".jar");
        }
        try {
            fostream = new FileOutputStream(newJarFile);

            try {
                jarOutputStream = new JarOutputStream(fostream);
            } catch (IOException e1) {
                System.err.println("Cannot create output stream for jar file " + sNewJarName + " " + e1);
            }

        } catch (FileNotFoundException e1) {
            e1.printStackTrace();
        }

    }

    private void closeNewJarStream() {
        try {
            jarOutputStream.finish();
            jarOutputStream.close();
            fostream.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * store in jar
     */

    boolean storeResultInJar() throws HammerException {
        HammerJarAccess theJarAccess = theDesignProject.theJarAccess;

        if (theJarAccess != null) {
            JarEntry entry = theJarAccess.theJarEntry;
            JarFile jarf = theJarAccess.theJarFile;
            HammerJarFileEx jarEx = new HammerJarFileEx(jarf);
            jarEx.load();
            jarEx.createJarEntry(entry.getName());
            File fileInput = new File(sOutputDocumentName + ".temp");
            FileInputStream in = null;
            try {
                in = new FileInputStream(fileInput);
            } catch (FileNotFoundException e) {
                return false;
            }
            BufferedInputStream inStr = new BufferedInputStream(in);
            int b;
            try {
                while ((b = inStr.read()) != -1) {
                    jarEx.write(b);
                }
                in.close();
            } catch (IOException e) {
                return false;
            }
            boolean bTest = jarEx.save();
            if (bTest == false) {
                throw new HammerException(HammerException.CANNOT_SAVE_TO_OPEN_JAR, "Cannot Save. Access to jar file " + sOutputDocumentName + " is not exclusive.");
            }
            jarEx.finishJarSave();
        }
        return true;
    }

    public String convertUTF8(String sConvertMe) throws HammerException {
        ByteArrayOutputStream bytes = null;
        try {
            bytes = new ByteArrayOutputStream();
            BufferedOutputStream bos = new BufferedOutputStream(bytes);
            Writer writer = new OutputStreamWriter(bos, "UTF8");
            writer.write(sConvertMe);
        } catch (IOException e) {
            throw new HammerException(HammerException.BAD_UTF8_CONVERSION, "Cannot convert output to UTF8 Bytes");
        }
        return bytes.toString();
    }


    public boolean saveToMainFile(MetaProjectHeader theProgInfo, boolean bJar, boolean bCompileClass) throws HammerException {

        boolean bRetVal = true;
        ByteArrayOutputStream outputProject = new ByteArrayOutputStream();
        electric.xml.Document theDocument = new electric.xml.Document();
        electric.xml.Element theRootElement = new electric.xml.Element("Metadata");
        theDocument.setRoot(theRootElement);

        boolean bNova = theDesignProject.getMetaDataUser().getNovaState();

        String jarName = sOutputDocumentName;

        theDesignProject.save(theProgInfo, theRootElement);

        byte[] bytes = null;
        String sOutput = null;

        if (bNova) {
            // Create the output document - proz file as a string of text.
            try {
                theDocument.write(outputProject);
                bytes = outputProject.toByteArray();

            } catch (IOException e) {
            }
            if (bytes == null)
                return false;
        } else {
            try {
                OutputStreamWriter os = new OutputStreamWriter(outputProject);
                // electric will write in utf8 - we need to recreate a bug here.
                theDocument.write(outputProject);
                sOutput = outputProject.toString();
                sOutput = ProjectDocument.stripUTFEncoding(sOutput);
            } catch (IOException e) {
            }
            OldProjectDocument oldDoc = new OldProjectDocument(sOutputDocumentName);
            if (oldDoc == null)
                throw new HammerException(HammerException.CANNOT_CREATE_PROZ_FILE, "Unable to create old style proz file");
            bRetVal = oldDoc.println(sOutput);
            if (bRetVal == false)
                return bRetVal;
        }

        // Proj always writes as just a text file.
        if (sOutputDocumentName.endsWith("proj") == true || bOpenSource == true) {
            try {
                File fileOutput = new File(sOutputDocumentName + ((bJar) ? ".temp" : ""));
                FileOutputStream os = null;
                os = new FileOutputStream(fileOutput);
                BufferedOutputStream bos = new BufferedOutputStream(os);
                if (bNova) {
                    bos.write(bytes);  // write properly encoded data.
                    bos.close();
                } else {
                    PrintWriter writer = new PrintWriter(bos);
                    writer.println(sOutput);
                    writer.close();
                }

                try {

                    os.close();
                    // if (bJar is set than we have to store back into the jar)
                    if (bJar)
                        return storeResultInJar();
                    else if (!bOpenSource)
                        return true;
                } catch (FileNotFoundException e) {
                    return false;
                }
            } catch (IOException e) {
                return false;
            }
        }

        if (!bNova && !bOpenSource)
            return true;

        // ********************** NEW FILE FORMAT ***********************
        // ********************** ADDITIONAL OPEN SOURCE FORMAT*********
        // output document is a proz. Unless we have the OpenSource Flag on
        // if the OpenSource flag is true then create a jar and store the .proj
        // and the .class files in this jar. In any case, the default should be an Abacus
        // PROZ file.
        String sClassName = new String();
        CompressedProjectDocument storage = null;
        String sOutputFileName = new String();
        RandomAccessFile fileOutput = null;

        // If OpenSource we will let the DC create a PROZ and we shall delete it.
        if (bOpenSource)
            sOutputFileName = sOutputDocumentName + ".proz";
        else if (bJar)
        // Create PROZ Open Source creates JAR files
            sOutputFileName = sOutputDocumentName + ".temp";
        else
            sOutputFileName = sOutputDocumentName;

        try {
            fileOutput = new RandomAccessFile(sOutputFileName, "rw");
        } catch (FileNotFoundException e) {
            return false;
        }

        if (fileOutput == null)
            return false;

        storage = new CompressedProjectDocument(theDesignProject);
        storage.open(fileOutput);

        // get the classname of the generated class for the project.
        sClassName = sOutputDocumentName;
        int iMatchClass = sOutputDocumentName.lastIndexOf('.');
        if (iMatchClass != -1) {
            sClassName = sClassName.substring(0, iMatchClass);
        }
        String sFileSeparator = System.getProperty("file.separator");
        int iMatchSlash = sClassName.lastIndexOf(sFileSeparator);
        if (iMatchSlash != -1) {
            sClassName = sClassName.substring(iMatchSlash + sFileSeparator.length());
        }
        // make class name ugly and unique and legal
        sClassName = "AbaRenderer_" + sClassName;
        // guru flag is for writing compiled class too.
        //boolean bGuru = theDesignProject.getMetaDataUser().getGuruState();

        String sOutputDirectoryName = null;
        if (bCompileClass) {
            // Do code generation.
            // compile stuff
            sOutputDirectoryName = theDesignProject.theProgramData.sOutputDirectory;
            if (sOutputDirectoryName == null)
                sOutputDirectoryName = "output";

            // Added this for clarity for OpenSource MHC
            boolean bJaninoCompile = false;
            if(bOpenSource==false)
                bJaninoCompile = this.theCockpit.isJaninoCompilerWanted();

            if (bJaninoCompile) // For the Janino compiler, we use the user's home dir to store the output files
                sOutputDirectoryName = System.getProperty("user.home") + File.separator + "abacus" +File.separator +sOutputDirectoryName;

            if (this.theCockpit.isCustomizing()) {
                sOutputDirectoryName = theCockpit.theCustomizer.getCompilerOutputDirectory();
            }

            ProgrammableCompiler pc = new ProgrammableCompiler(bJaninoCompile);
            try {
                File[] files = compileProject(pc, sClassName, sOutputDirectoryName);
                // Now move the compiled classes and the source to the output directory.
                moveCompiledClasses(files, pc, sOutputDirectoryName, storage, bytes);
                pc.cleanup();
            } catch (Throwable throwable) { // MHC Fixed bug leaving proz file in a REALLY bad state
                throwable.printStackTrace();  //To change body of catch statement use Options | File Templates.
                theDesignProject.getMetaDataUser().tellProgress(throwable.getMessage());
                try {
                    pc.cleanup();
                    // make sure we write the project.
                    storage.writeProjSource(bytes);
                } catch (Throwable e) {
                    e.printStackTrace();
                    return false;
                }
                bRetVal = false;
                //return false;
            }
        } else {
            // just write the project.
            try {
                storage.writeProjSource(bytes);
            } catch (IOException e) {
                return false;
            }
        }
        // finally close the proz file
        if (this.theCockpit.isCustomizing()) {
            OutputStream os = theCockpit.theCustomizer.getSaveProjectOutputStream();
            boolean bClosedOk = storage.close(sOutputFileName, os);
            theCockpit.theCustomizer.projectSaved(bClosedOk);
            if (sOutputDirectoryName != null)
                theCockpit.theCustomizer.removeFiles(sOutputDirectoryName, sClassName);
            if (!bClosedOk)
                return false;
        }
        else if(bOpenSource==true){ // Added in order to skip all the proz manipulation.
            if (storage.close()== false)
                return false;
        }
        else {
            if (storage.close(sOutputFileName) == false)
                return false;
        }
        try {
            fileOutput.close();
        } catch (IOException e) {
            return false;
        }
        // now if we loaded from a resource we have to store back in a jar.
        if (bJar)
            return storeResultInJar();
        bRetVal = true;

        if (bOpenSource == true && sOutputFileName.endsWith("proz")) {
            File fDelete = new File(sOutputFileName);
            if (fDelete.exists())
                fDelete.delete();
        }

        return bRetVal;
    }

    /**
     * Save project back to proj file or proz file.
     *
     * @param theProgInfo
     * @return boolean - true/false - depending on if the operation succeeds.
     */


    public boolean saveToFile(MetaProjectHeader theProgInfo, boolean bJar, boolean bCompileClass) throws HammerException {

        boolean bRetVal = true;
        bRetVal = saveToMainFile(theProgInfo, bJar, bCompileClass);
        if (bRetVal == true) {
            boolean bDroplets = theDesignProject.getMetaDataUser().getDropletsState();
            if (bDroplets) {
                DropletCodeGenerator theGenerator = (DropletCodeGenerator) theDesignProject.getCodeGenerator();
                theGenerator.doTemplateFiles(sOutputDocumentName);
            }
        }
        return bRetVal;
    }

    File[] compileProject(ProgrammableCompiler pc, String sClassName, String sOutputDirectoryName) throws HammerException {

        theDesignProject.getCodeGenerator().reset();
        theDesignProject.getMetaDataUser().tellProgress("Generating source for " + sClassName);
        theDesignProject.getCodeGenerator().generateProjectCode(sClassName, theDesignProject);
        // Get the source and format it.
        String sSource = theDesignProject.getCodeGenerator().getFormattedSource();
        String[] defns = {sSource};     // Only one at a time, may create multiple classes.
//        PrintWriter pwOutput = null;
//        try {
//            pwOutput = new PrintWriter(new FileOutputStream("temp.java"));
//        } catch (FileNotFoundException e) {
//            e.printStackTrace();  //To change body of catch statement use Options | File Templates.
//        }
//        pwOutput.print(sSource);
//        pwOutput.close();
        // Set up the class path.
        String sClassPath = System.getProperty("java.class.path");
        int iOtherClassPaths = theDesignProject.theProgramData.getAdditionalClassPathCount();
        String sPathSeparator = System.getProperty("path.separator");
        for (int iClassPath = 0; iClassPath < iOtherClassPaths; iClassPath++) {
            sClassPath = sClassPath + sPathSeparator + theDesignProject.theProgramData.getAdditionalClassPath(iClassPath);
        }
        // Make the compiler interface.
        String sTransientDirectory = theDesignProject.theProgramData.sTransientDirectory;
        pc.setProperty("javaClassPath", sClassPath);
        if (sTransientDirectory != null)
            pc.setProperty("tempDirNameRoot", sTransientDirectory);
        // And invoke the compiler.
        File[] files = new File[0];
        theDesignProject.getMetaDataUser().tellProgress("Compiling Class " + sClassName);
        try {
            files = pc.compileClassesToFiles(defns);
            theDesignProject.getMetaDataUser().tellProgress("Finish Compiling" + sClassName);
        } catch (CompilationException e) {
            throw new HammerException(HammerException.COMPILATION_ERRORS_OCCURRED, e.getMessage(), e.getLocalizedMessage());
        }
        theDesignProject.getMetaDataUser().tellProgress("Saving Compiled Class " + sClassName);
        return files;
    }

    boolean moveCompiledClasses(File[] files, ProgrammableCompiler pc, String sOutputDirectoryName, CompressedProjectDocument storage, byte[] bytes) {
        boolean bRetVal = true;
        int iFilesToWrite = files.length;
        String sTempDir = pc.getProperty("tempDirNameRoot");
        JarOutputStream jarOutPut = null;

        if (bOpenSource) {
            try {
                String sNewJarFileName = sOutputDocumentName;
                String sFileSeparator = ".";
                JarFile newJarFile = null;
                int iMatchSlash = sNewJarFileName.lastIndexOf(sFileSeparator);
                if (iMatchSlash != -1) {
                    sNewJarFileName = sOutputDocumentName.substring(0, iMatchSlash);
                }
                try {
                    createNewJar(sNewJarFileName);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        for (int i = 0; i < iFilesToWrite; i++) {
            int iMatch = files[i].getPath().indexOf(sTempDir);
            if (iMatch > 0) {
                // Get the intransient part of the file name (the package and the class) by itself.
                String sSeparator = files[i].getPath().substring(iMatch - 1, iMatch);
                String sFileSubName = files[i].getPath().substring(iMatch);
                iMatch = sFileSubName.indexOf(sSeparator);
                // sFileSubName is the useful part of the file (still has .class extension)
                sFileSubName = sFileSubName.substring(iMatch);
                // Prepare old names and new names for copy of source and class files to output directory.
                String sOldClassFileName = files[i].getPath();
                if (sOutputDirectoryName.trim().length() == 0)
                    sOutputDirectoryName = "." + sSeparator;
                String sNewClassFileName = sOutputDirectoryName + sFileSubName;
                int iSuffix = sNewClassFileName.lastIndexOf(".class");
                String sNewJavaFileName = sNewClassFileName.substring(0, iSuffix) + ".java";
                iSuffix = sOldClassFileName.lastIndexOf(".class");
                String sOldJavaFileName = sOldClassFileName.substring(0, iSuffix) + ".java";
                // Make sure the output directory exists.
                iSuffix = sNewClassFileName.lastIndexOf(sSeparator);
                String sOutputDirectoryFullPath = sNewClassFileName.substring(0, iSuffix);
                new File(sOutputDirectoryFullPath).mkdirs();
                // Copy the files to output directory.
                try {
                    FileCopy.copyFile(sOldClassFileName, sNewClassFileName);
                } catch (IOException e) {
                    return false;
                }
                try {
                    FileCopy.copyFile(sOldJavaFileName, sNewJavaFileName);
                } catch (IOException e) {
                    return false;
                }

                File oldFile = new File(sOldClassFileName);
                oldFile.delete();
                oldFile = new File(sOldJavaFileName);
                //boolean bSourceExists = oldFile.canRead();   // MHC
                oldFile.delete();
// NOTE:  Comment out delete call and set delete flag to false
// if you want to keep the source.  For Alex.
//                oldFile.delete();
//                dc.deleteFilesOnFinalize = false;

                // Now the new files need to be stored in the proz file.
                try {
                    // ok to limit to int - class size is small.
                    File newFile = new File(sNewClassFileName);
                    FileInputStream in = new FileInputStream(newFile);
                    BufferedInputStream inStr = new BufferedInputStream(in);
                    int iSize = (int) newFile.length();
                    byte[] classBytes = new byte[iSize];
                    inStr.read(classBytes, 0, iSize);
                    String sEntryClassFileName = new String();

                    if (bOpenSource) {
                        String sFileSeparator = System.getProperty("file.separator");
                        int iMatchSlash = sNewJavaFileName.lastIndexOf(sFileSeparator);
                        if (iMatchSlash != -1) {
                            sEntryClassFileName = sNewClassFileName.substring(iMatchSlash + sFileSeparator.length());
                        }

                        ZipEntry ze = new ZipEntry(sEntryClassFileName);
                        jarOutputStream.putNextEntry(ze);

                        File fileInput = new File(sNewClassFileName);
                        FileInputStream xin = null;
                        try {
                            xin = new FileInputStream(fileInput);
                        } catch (FileNotFoundException e) {
                            return false;
                        }
                        BufferedInputStream xinStr = new BufferedInputStream(xin);
                        int b;
                        try {
                            while ((b = xinStr.read()) != -1) {
                                jarOutputStream.write(b);
                            }
                            xin.close();
                        } catch (IOException e) {
                            return false;
                        }

                    }

                    inStr.close();
                    in.close();

                    // write the class bytes to the proz.
                    storage.writeClass(classBytes);
                    // write the project.
                    storage.writeProjSource(bytes);
                    String repoPath=this.theCockpit.theConfigurationManager.getConfigurationVariable("RepositoryPath");
                    if(repoPath!=null)
                    {
                        String defaultConfPath = this.theCockpit.thePreferences.getRepositPath();
                        if(defaultConfPath!=null)
                            repoPath = repoPath + ";" + defaultConfPath;
                    }
                    if(repoPath!=null)
                        storage.writeCustomMetadata(repoPath);

                } catch (FileNotFoundException e) {
                    return false;
                } catch (IOException e) {
                    e.printStackTrace();
                    return false;
                }
            }
        }

        if (jarOutputStream != null && bOpenSource == true) {
            this.closeNewJarStream();
        }

        return bRetVal;
    }


    public boolean load
            () {
        return true;
    }

    class OldProjectDocument {

        HammerJarFileEx jarEx;
        String sOutputEntryName = "";
        ByteArrayOutputStream pwByteOutput;
        GZIPOutputStream fzip;
        FileOutputStream fon = null;
        File fileXMLOutput;
        HammerJarAccess theJarAccess;
        BufferedOutputStream jar_os;


        public OldProjectDocument(String sOutputDocumentName) throws HammerException {
            if (theDesignProject.iInputType == MetaProject.FROM_XML_DOCUMENT) {
                fileXMLOutput = new File(sOutputDocumentName);
                if (sOutputDocumentName.endsWith("proz")) {
                    try {
                        fon = new FileOutputStream(fileXMLOutput);
                    } catch (FileNotFoundException e) {
                        throw new HammerException(HammerException.CANNOT_CREATE_PROZ_FILE, "Cannot open file: " + sOutputDocumentName);
                    }
                    try {
                        fzip = new GZIPOutputStream(fon);
                    } catch (IOException e) {
                        throw new HammerException(HammerException.CANNOT_CREATE_PROZ_FILE, "Cannot open gzip file: " + sOutputDocumentName);
                    }
                } else {
                    try {
                        fon = new FileOutputStream(fileXMLOutput);
                    } catch (IOException e) {
                        throw new HammerException(HammerException.CANNOT_CREATE_PROJ_FILE, "Cannot create proj file: " + sOutputDocumentName);
                    }
                }
            } else {
                theJarAccess = theDesignProject.theJarAccess;
                if (theJarAccess != null) {
                    JarEntry entry = theJarAccess.theJarEntry;
                    JarFile jarf = theJarAccess.theJarFile;
                    jarEx = new HammerJarFileEx(jarf);
                    jarEx.load();
                    jarEx.removeJarEntry(entry);
                    sOutputEntryName = entry.getName();
                    pwByteOutput = new ByteArrayOutputStream(4096);
                    jar_os = new BufferedOutputStream(pwByteOutput);
                    if (sOutputDocumentName.endsWith("proz")) {
                        try {
                            fzip = new GZIPOutputStream(jar_os);
                        } catch (IOException e) {
                            throw new HammerException(HammerException.CANNOT_CREATE_PROZ_FILE, "Cannot open gzip file: " + sOutputDocumentName);
                        }
                    }
                }
            }

        }

        public boolean println(String s) throws HammerException {
            try {
                OutputStreamWriter writer = null;
                if (fzip != null) {
                    writer = new OutputStreamWriter(fzip);
                } else if (fon != null) {
                    writer = new OutputStreamWriter(fon);
                }
                String sEncoding = writer.getEncoding();
                writer.write(s);
                writer.close();
                if (fzip != null)
                    fzip.close();
                if (fon != null)
                    fon.close();
                if (jar_os != null) {
                    jar_os.close();
                }
            } catch (IOException e1) {
                throw new HammerException(HammerException.CANNOT_SAVE_PROZ_FILE, "Error writing " + sOutputDocumentName);
            }
            return true;
        }


    }

    /**
     * stripUTFEncoding - this method strips utf encoding - it helps us recreate a bug
     * that makes our files compatible.
     *
     * @param s
     * @return
     */
    public static String stripUTFEncoding(String s) {
        ByteArrayInputStream bytesIn = new ByteArrayInputStream(s.getBytes());
        InputStreamReader reader;
        try {
            reader = new InputStreamReader(bytesIn, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return null;
        }
        try {
            StringBuffer buffer = new StringBuffer();
            int ch;
            while ((ch = reader.read()) > -1) {
                buffer.append((char) ch);
            }
            reader.close();
            s = buffer.toString();
        } catch (IOException e) {
            return null;
        }
        return s;
    }

    public ProjectCodeGenerator getCodeGenerator() {
        return this.theDesignProject.getCodeGenerator();
    }

}

