/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

import java.io.*;
import java.util.*;
import java.util.zip.ZipFile;
import java.util.zip.ZipEntry;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * <p>Title: Abajamm</p>
 * <p>Description: Abacus Java Module Manager</p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: Abacus </p>
 * @author Michael Gouker - Cagey Logic
 * @version 1.0
 */

public class HammerClassByteFileLoader {

    private ArrayList classPathArrayList;
    private ArrayList shortClassPathArrayList;
    private ArrayList customMetadataFiles;
    AbaMetaDataUser theUser;
    private HashMap properties;

    /**
     * @param classPathArrayList	ArrayList of directories forming class path with index 0 being searched first
     * @param properties		hash table containing entries for various constants including "packageSeparator" and "compiledSuffix"
     */
    public HammerClassByteFileLoader(ArrayList classPathArrayList, ArrayList shortClassPathArrayList, HashMap properties, AbaMetaDataUser theUser) {
        this.classPathArrayList = classPathArrayList;
        this.shortClassPathArrayList = shortClassPathArrayList;
        this.properties = properties;
        this.theUser = theUser;
    }

    /**
     * Finds the byte array corresponding to the given class name.
     *
     * @param name	a fully qualified class name
     * @return		an array containing the byte code for the given class, or null if it is not found
     */
    public byte[] getClassBytes(String name) {
        ArrayList theClassBytes = getClassesNotInJars(name, true, false);
        if (theClassBytes != null) {
            Object classBytes = theClassBytes.get(0);
            if (classBytes instanceof byte[])
                return (byte[]) classBytes;
        }
        ArrayList theJarAccessArray = getJarAccess(name, true, false);
        // Get the file containing the byte code.
        if ((theJarAccessArray != null) && (theJarAccessArray.size() > 0)) {
            HammerJarAccess theJarAccess = (HammerJarAccess) theJarAccessArray.get(0);
            byte[] classBytes = readClassFile(theJarAccess);
            return classBytes;
        } else
            return null;
    }


    public byte[] readClassFile(HammerJarAccess theJarAccess) {
        JarFile jarf = theJarAccess.theJarFile;
        JarEntry entry = theJarAccess.theJarEntry;
        // Get class bytes from zip file.
        try {
            InputStream is = jarf.getInputStream(entry);
            int fileLength = (int) entry.getSize();
            int totalbytes = 0;

            // Create an array the same length as the file to hold the compiled class.
            byte[] classBytes = new byte[fileLength];

            while (totalbytes != fileLength) {
                // Read class definition into the array.
                int iBytesToRead = 512;
                if ((fileLength - totalbytes) < 512)
                    iBytesToRead = fileLength - totalbytes;
                int bytesread = is.read(classBytes, totalbytes, iBytesToRead);
                totalbytes += bytesread;
            }
            // Close the file.
            is.close();
            return classBytes;
        } catch (java.io.IOException e2) {
        }
        return null;
    }

    /**
     * Finds the .class file corresponding to the given class name.
     *
     * @param name	a full class name
     * @return the file containing the byte code for the given class
     */
    public String findJarFileWithResource(String name) {
        ArrayList theJarAccessArray = getJarAccess(name, false, false);
        if ((theJarAccessArray != null) && (theJarAccessArray.size() > 0)) {
            HammerJarAccess theJarAccess = (HammerJarAccess) theJarAccessArray.get(0);
            return theJarAccess.theJarFile.getName();
        }
        return null;
    }


    // IDE jar access looks in named jars and scans all jars in named
    // directories.
    public ArrayList getJarAccess(String fullClassName, boolean bAddCompiledSuffix, boolean bMultiple) {
        ArrayList retVal = new ArrayList(0);
        if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
            theUser.getLogFile().doClassLoaderLogEntry("GetJarAccess:", "Searching for " + fullClassName);
        String packageSeparator = (String) properties.get("packageSeparator");
        String compiledSuffix = (String) properties.get("compiledSuffix");
        String fileSeparator = (String) properties.get("fileSeparator");
        String jarClassName = "";
        String theSuffix = "";
        // For resources we need to preserve the last "." - it's part of the filename.
        if ((bAddCompiledSuffix == false) && (packageSeparator.equals("."))) {
            int iLastDot = fullClassName.lastIndexOf(packageSeparator);
            if (iLastDot != -1) {
                String left = fullClassName.substring(0, iLastDot);
                theSuffix = fullClassName.substring(iLastDot);
                fullClassName = left;
            }
        }
        StringTokenizer st1 = new StringTokenizer(fullClassName, packageSeparator);
        // Get the first token.
        String s1 = new String();

        try
        {
            s1 = st1.nextToken();
        }
        catch(NoSuchElementException e)
        {
            return null;
        }

        // Add the sub-directories for the package structure in turn, if any.
        // This converts the "."s to "/"s
        while (st1.hasMoreTokens()) {
            jarClassName = jarClassName + s1 + "/";
            s1 = st1.nextToken();
        }
        jarClassName = jarClassName + s1;
        if (bAddCompiledSuffix == true)  // classes need to add compiled suffix ".class"
            jarClassName = jarClassName + compiledSuffix;
        else
            jarClassName = jarClassName + theSuffix;

        // This class loader searches all of the jars that are installed
        // in a directory of the permanent cache and then all of the jars
        // that are specific to this abalet.
        // Each element of the class directory is actually a jar file
        // Use the short class path ArrayList for optimization.
        for (int iListEntry = 0; iListEntry < shortClassPathArrayList.size(); iListEntry++) {
            // Start at the root directory.
            String sClassPathEntry = (String) shortClassPathArrayList.get(iListEntry);
            if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
                theUser.getLogFile().doClassLoaderLogEntry("GetJarAccess:", "Searching for " + fullClassName + " using " + sClassPathEntry);
            // Class file not found?  Look in the jars then.
            if (sClassPathEntry.endsWith(".jar")) {
                // Look for named class in the named jar.
                if (bMultiple) {
                    ArrayList theAccess = getMatchingJarAccess(sClassPathEntry, jarClassName);
                    if (theAccess != null) {
                        for (int iAccess = 0; iAccess < theAccess.size(); iAccess++) {
                            retVal.add(theAccess.get(iAccess));
                        }
                    }
                } else {
                    HammerJarAccess theAccess = getJarAccessByName(sClassPathEntry, jarClassName);
                    if (theAccess != null) {
                        retVal.add(theAccess);
                        if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
                            theUser.getLogFile().doClassLoaderLogEntry("GetJarAccess:", "Found " + jarClassName + " using " + sClassPathEntry);
                        return retVal;
                    }
                    // else just keep searching.
                }
            } else {  // Directory. Search for names in the abalet jar list.
                File theDirectoryOrJar = new File(sClassPathEntry);
                if (theDirectoryOrJar.isDirectory()) {
                    FilenameFilter theJarFilter = new HammerJarFilter();
                    String[] files = theDirectoryOrJar.list(theJarFilter);
                    if (files != null) {
                        for (int iFile = 0; iFile < files.length; iFile++) {
                            // For each jar...
                            String sJarName = sClassPathEntry + fileSeparator + files[iFile];
                            if (bMultiple) {
                                ArrayList theAccess = getMatchingJarAccess(sJarName, jarClassName);
                                if (theAccess != null) {
                                    for (int iAccess = 0; iAccess < theAccess.size(); iAccess++) {
                                        retVal.add(theAccess.get(iAccess));
                                    }
                                }
                            } else {
                                HammerJarAccess theAccess = getJarAccessByName(sJarName, jarClassName);
                                if (theAccess != null) {
                                    retVal.add(theAccess);
                                    if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
                                        theUser.getLogFile().doClassLoaderLogEntry("GetJarAccess:", "Found " + jarClassName + " using " + sClassPathEntry);
                                    return retVal;
                                }
                            }
                        }
                    }
                }
            }
        }
        if (retVal.size() > 0) {
            if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
                theUser.getLogFile().doClassLoaderLogEntry("GetJarAccess:", "Found " + fullClassName);
            return retVal;
        }
        return null;
    }

    private String getConvertedSearchString(String fullClassName, boolean bAddCompiledSuffix) {
        String packageSeparator = (String) properties.get("packageSeparator");
        String fileSeparator = (String) properties.get("fileSeparator");
        String ClassFileName = "";
        String theSuffix = "";
        // For resources we need to preserve the last "." - it's part of the filename.
        if ((bAddCompiledSuffix == false) && (packageSeparator.equals("."))) {
            int iLastDot = fullClassName.lastIndexOf(packageSeparator);
            if (iLastDot != -1) {
                String left = fullClassName.substring(0, iLastDot);
                theSuffix = fullClassName.substring(iLastDot);
                fullClassName = left;
            }
        }
        StringTokenizer st1 = new StringTokenizer(fullClassName, packageSeparator);
        // Get the first token.
        String s1 = st1.nextToken();
        // Add the sub-directories for the package structure in turn, if any.
        // This converts the "."s to "/"s
        while (st1.hasMoreTokens()) {
            ClassFileName = ClassFileName + s1 + fileSeparator;
            s1 = st1.nextToken();
        }
        ClassFileName = ClassFileName + s1;
        if (bAddCompiledSuffix == true)  // classes need to add compiled suffix ".class"
            ClassFileName = ClassFileName + ".class";
        else
            ClassFileName = ClassFileName + theSuffix;
        return ClassFileName;
    }

    private String getConvertedJarSearchString(String fullClassName, boolean bAddCompiledSuffix) {
        String packageSeparator = (String) properties.get("packageSeparator");
        String fileSeparator = (String) properties.get("fileSeparator");
        String ClassFileName = "";
        String theSuffix = "";
        // For resources we need to preserve the last "." - it's part of the filename.
        if ((bAddCompiledSuffix == false) && (packageSeparator.equals("."))) {
            int iLastDot = fullClassName.lastIndexOf(packageSeparator);
            if (iLastDot != -1) {
                String left = fullClassName.substring(0, iLastDot);
                theSuffix = fullClassName.substring(iLastDot);
                fullClassName = left;
            }
        }
        StringTokenizer st1 = new StringTokenizer(fullClassName, packageSeparator);
        // Get the first token.
        String s1 = st1.nextToken();
        // Add the sub-directories for the package structure in turn, if any.
        // This converts the "."s to "/"s
        while (st1.hasMoreTokens()) {
            ClassFileName = ClassFileName + s1 + "/";
            s1 = st1.nextToken();
        }
        ClassFileName = ClassFileName + s1;
        if (bAddCompiledSuffix == true)  // classes need to add compiled suffix ".class"
            ClassFileName = ClassFileName + ".class";
        else
            ClassFileName = ClassFileName + theSuffix;
        return ClassFileName;
    }

    public ArrayList getClassesNotInJars(String fullClassName, boolean bAddCompiledSuffix, boolean bMultiple) {
        ArrayList retVal = new ArrayList(0);
        HashMap discrim = new HashMap(311);
        String fileSeparator = (String) properties.get("fileSeparator");
        String ClassFileName = getConvertedSearchString(fullClassName, bAddCompiledSuffix);
        // This class loader searches all of the jars that are installed
        // in a directory of the permanent cache and then all of the jars
        // that are specific to this abalet.
        // Each element of the class directory is actually a jar file
        if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
            theUser.getLogFile().doClassLoaderLogEntry("GetClassesNotInJars:", "Searching for " + ClassFileName);

        for (int iClassPathEntry = 0; iClassPathEntry < shortClassPathArrayList.size(); iClassPathEntry++) {

            // Start at the root directory.
            String sClassPathEntry = (String) shortClassPathArrayList.get(iClassPathEntry);
            // Class file not found?  Look in the jars then.
            if (sClassPathEntry.endsWith(".jar")) {
            } else {  // Directory. Search for names in the abalet class list.
                if (bMultiple == false) {
                    File theClassFile = new File(sClassPathEntry + fileSeparator + ClassFileName);
                    if (theClassFile != null) {
                        byte[] classBytes = getClassBytesFromClassFile(theClassFile);
                        retVal.add(classBytes);
                        if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
                            theUser.getLogFile().doClassLoaderLogEntry("GetClassesNotInJars:", "Found " + sClassPathEntry + fileSeparator + ClassFileName);
                        return retVal;
                    }
                } else {
                    File theDirectoryOrJar = new File(sClassPathEntry + fileSeparator + ClassFileName);
                    if (theDirectoryOrJar.isDirectory()) {
                        retVal = exploreDirectoryForClasses(retVal, theDirectoryOrJar, discrim);
                    }
                }
            }
        }
        if (retVal.size() > 1) {
            if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
                theUser.getLogFile().doClassLoaderLogEntry("GetClassesNotInJars:", "Found " + ClassFileName);
            return retVal;
        }
        return null;
    }

    ArrayList exploreDirectoryForClasses(ArrayList retVal, File thisDirectoryNode, HashMap discrim) {
        String fileSeparator = (String) properties.get("fileSeparator");
        String[] files = thisDirectoryNode.list();
        if (files != null) {
            for (int iFile = 0; iFile < files.length; iFile++) {
                // For each jar...
                String sFileName = thisDirectoryNode.getAbsolutePath() + fileSeparator + files[iFile];
                if (discrim.get(sFileName) == null) {
                    File theDirectoryOrFile = new File(sFileName);
                    if (theDirectoryOrFile != null) {
                        if (theDirectoryOrFile.isDirectory())
                            retVal = exploreDirectoryForClasses(retVal, theDirectoryOrFile, discrim);
                        else {
                            if (sFileName.toLowerCase().endsWith(".class")) {
                                byte[] classBytes = getClassBytesFromClassFile(theDirectoryOrFile);
                                HammerClassDescriptor theEntry = new HammerClassDescriptor(classBytes, sFileName);
                                retVal.add(theEntry);
                                if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
                                    theUser.getLogFile().doClassLoaderLogEntry("ExploreDirectoryForClasses:", "Found " + sFileName);
                            }
                        }
                    }
                    discrim.put(sFileName, new Boolean(true));
                }
            }
        }
        return retVal;
    }

    byte[] getClassBytesFromClassFile(File theClassFile) {
        try {
            FileInputStream is = new FileInputStream(theClassFile);
            int fileLength = (int) theClassFile.length();
            int totalbytes = 0;

            // Create an array the same length as the file to hold the compiled class.
            byte[] classBytes = new byte[fileLength];
            while (totalbytes != fileLength) {
                // Read class definition into the array.
                int iBytesToRead = 512;
                if ((fileLength - totalbytes) < 512)
                    iBytesToRead = fileLength - totalbytes;
                int bytesread = is.read(classBytes, totalbytes, iBytesToRead);
                totalbytes += bytesread;
            }
            // Close the file.
            is.close();
            return classBytes;
        } catch (java.io.IOException e1) {
            return null;
        }
    }

    private HammerJarAccess getJarAccessByName(String sJarName, String sKeyName) {
        if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
            theUser.getLogFile().doClassLoaderLogEntry("Invoking GetJarAccessByName:", "Jar Name: " + sJarName + " Key Name: " + sKeyName);
        try {
            JarFile jarf = new JarFile(sJarName);
            JarEntry entry = jarf.getJarEntry(sKeyName);
            if (entry != null)
                return new HammerJarAccess(jarf, entry);
        } catch (IOException e1) {
        }
        return null;
    }

    private ArrayList getMatchingJarAccess(String sJarName, String sKeyName) {
        ArrayList theMatches = new ArrayList(0);
        try {
            JarFile jarf = new JarFile(sJarName);
            Enumeration allentries = jarf.entries();
            while (allentries.hasMoreElements()) {
                JarEntry entry = (JarEntry) allentries.nextElement();
                String sTestName = entry.getName();
                if ((sTestName.startsWith(sKeyName)) && (sTestName.endsWith("/") == false)) {
                    theMatches.add(new HammerJarAccess(jarf, entry));
                }
            }
            return theMatches;
        } catch (IOException e1) {
        }
        return null;
    }

    // What returns is an array list of byte[].
    public ArrayList getClassesFromPath(String sSearchPath) {
        // Look everywhere for classes that start with sSearchPath.
        // theClassBytes is an array list of byte[]
        ArrayList theClassBytes = getClassesNotInJars(sSearchPath, false, true);
        // theJarAccess is an array list of HammerJarAccess.
        ArrayList theJarAccessArray = getJarAccess(sSearchPath, false, true);
        // Get the file containing the byte code.
        if (theClassBytes == null) {
            if (theJarAccessArray == null)
                return null;
            theClassBytes = new ArrayList(0);
        }
        if (theJarAccessArray != null) {
            for (int i = 0; i < theJarAccessArray.size(); i++) {
                HammerJarAccess theJarAccess = (HammerJarAccess) theJarAccessArray.get(i);
                if (theJarAccess != null) {
                    byte[] classBytes = readClassFile(theJarAccess);
                    HammerClassDescriptor theEntry = new HammerClassDescriptor(classBytes, theJarAccess.theJarEntry.getName());
                    theClassBytes.add(theEntry);
                }
            }
        }
        return theClassBytes;
    }

    public ArrayList getFilesInPath(String sPath) {
           // This uses the long class path ArrayList including all the defaults.
            if(customMetadataFiles!=null)
                return customMetadataFiles;
            File theRetVal = null;
            String fileSeparator = (String) properties.get("fileSeparator");
            customMetadataFiles = new ArrayList();

            for (int iClassPathEntry = 0; iClassPathEntry < classPathArrayList.size(); iClassPathEntry++)
            {
                String sClassPathEntry = (String) classPathArrayList.get(iClassPathEntry);
                File isDir = new File(sClassPathEntry);

                try
                {
                    // We check for the name as the indetifier that that this is custom.jar
                    // containg an XML file definition
                    if (!isDir.isDirectory() && sClassPathEntry.endsWith("AbacusCustom.jar"))
                    {
                        try
                        {
                            ZipFile zf=new ZipFile(sClassPathEntry);
                            Enumeration e=zf.entries();

                            while (e.hasMoreElements())
                            {
                                ZipEntry ze=(ZipEntry)e.nextElement();
                                String fullPath = ze.toString();
                                boolean begins=fullPath.startsWith(sPath);
                                boolean ends=fullPath.endsWith(".xml");
                                if(begins && ends)
                                    customMetadataFiles.add(fullPath);
                            }
                            zf.close();

                        }
                        catch(Exception eJar)
                        {
                            System.out.println("Warning: " + sClassPathEntry +" in classpath not found.");
                        }

                    }
                }
                catch(Exception e)
                {
                    System.out.println(sClassPathEntry);
                    e.printStackTrace();
                }
            }

        return customMetadataFiles;
    }

    public HammerInputStream getFileInPathAsStream(String sFileName , boolean bDevFlag) {
        if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
            theUser.getLogFile().doClassLoaderLogEntry("FindFileInPath:", "Searching for " + sFileName);
        // This uses the long class path ArrayList including all the defaults.
        File theRetVal = null;
        String ClassFileName = getConvertedSearchString(sFileName, false);
        String JarClassFileName = getConvertedJarSearchString(sFileName, false);
        String fileSeparator = (String) properties.get("fileSeparator");
        for (int iClassPathEntry = 0; iClassPathEntry < classPathArrayList.size(); iClassPathEntry++) {
            String sClassPathEntry = (String) classPathArrayList.get(iClassPathEntry);
            // Just look in directory entries.
            // Loading files from jars uses getResourceAsStream.
            if (sClassPathEntry.toLowerCase().endsWith("jar") == false) {
                if (sClassPathEntry.endsWith(fileSeparator) == false)
                    sClassPathEntry = sClassPathEntry + fileSeparator;
                if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
                    theUser.getLogFile().doClassLoaderLogEntry("FindFileInPath:", "Searching in " + sClassPathEntry);
                theRetVal = new File(sClassPathEntry + ClassFileName);
                if (theRetVal.exists()) {
                    if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
                        theUser.getLogFile().doClassLoaderLogEntry("FindFileInPath:", "Found " + sFileName + " in " + sClassPathEntry);
                    try {
                        FileInputStream in = new FileInputStream(theRetVal);
                        String sCanonicalPath = sClassPathEntry + ClassFileName;
                        try {
                            sCanonicalPath = theRetVal.getCanonicalPath();
                        } catch (java.io.IOException e1) {

                        }
                        return new HammerInputStream(sClassPathEntry, sCanonicalPath, in);
                    } catch (FileNotFoundException e) {
                        return null;
                    }
                }
                File theDirectoryOrJar = new File(sClassPathEntry);
                if (theDirectoryOrJar.isDirectory()) {
                    FilenameFilter theJarFilter = new HammerJarFilter();
                    String[] files = theDirectoryOrJar.list(theJarFilter);
                    if (files != null) {
                        for (int iFile = 0; iFile < files.length; iFile++) {
                            // For each jar...
                            String sJarName = sClassPathEntry + fileSeparator + files[iFile];
                            HammerJarAccess theAccess = getJarAccessByName(sJarName, JarClassFileName);
                            if (theAccess != null) {
                                if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
                                    theUser.getLogFile().doClassLoaderLogEntry("GetJarAccess:", "Found " + ClassFileName + " using " + sClassPathEntry);
                                JarFile jarf = theAccess.theJarFile;
                                JarEntry entry = theAccess.theJarEntry;
                                // Get class bytes from zip file.
                                try {
                                    return new HammerInputStream(sClassPathEntry, sJarName, JarClassFileName, jarf.getInputStream(entry));
                                } catch (IOException e) {
                                    return null;
                                }
                            }
                        }
                    }
                }
            }
            else if(bDevFlag)
            {
                if (sClassPathEntry.toLowerCase().endsWith("jar") == true)
                {
                // Look for named class in the named jar.
                    String packageSeparator = (String) properties.get("packageSeparator");
                    String compiledSuffix = (String) properties.get("compiledSuffix");
                    String sTmpFileName ="";

                    String jarClassName = "";
                    String theSuffix = "";
                    // For resources we need to preserve the last "." - it's part of the filename.
                    if ((packageSeparator.equals("."))) {
                        int iLastDot = sFileName.lastIndexOf(packageSeparator);
                        if (iLastDot != -1) {
                            String left = sFileName.substring(0, iLastDot);
                            theSuffix = sFileName.substring(iLastDot);
                            sTmpFileName= left;
                        }
                    }
                    StringTokenizer st1 = new StringTokenizer(sTmpFileName, packageSeparator);
                    // Get the first token.
                    String s1 = st1.nextToken();
                    // Add the sub-directories for the package structure in turn, if any.
                    // This converts the "."s to "/"s
                    while (st1.hasMoreTokens()) {
                        jarClassName = jarClassName + s1 + "/";
                        s1 = st1.nextToken();
                    }
                    jarClassName = jarClassName + s1;
                    jarClassName = jarClassName + theSuffix;

                    HammerJarAccess theAccess = getJarAccessByName(sClassPathEntry, jarClassName);

                    if (theAccess != null)
                    {
                        JarFile jarf = theAccess.theJarFile;
                        JarEntry entry = theAccess.theJarEntry;

                        if ((theUser.getLogFile() != null) && (theUser.getLogFile().bClassLoaderLogging))
                            theUser.getLogFile().doClassLoaderLogEntry("GetJarAccess:", "Found " + sClassPathEntry  + " using " + sClassPathEntry);
                        try
                        {
                            return new HammerInputStream(sClassPathEntry, sClassPathEntry, JarClassFileName, jarf.getInputStream(entry));
                        }
                        catch (IOException e)
                        {
                            return null;
                        }
                    }
                }
            }
        }
        return null;
    }


    public void setPaths(ArrayList vClassPath, ArrayList vShortPath) {
        this.classPathArrayList = vClassPath;
        this.shortClassPathArrayList = vShortPath;
    }

}

