/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

import java.io.*;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.JarFile;

/**
 * class  compressedProjectDocument - used to store project components in one file.
 *
 * Multiple components may be stored in the file.  The header contains offsets and
 * sizes of each component.
 *
 */
public class CompressedProjectDocument {
    private boolean bOpen = false;

    /** Header is uncompressed **/
    class CompressedProjectDocumentHeader {
        public static final int WATERMARK_VALUE = 0xabac;
        public static final int HEADER_SIZE = 256;
        public static final int COMPRESSION_BUFFER_SIZE=16384;
        int iWatermark;
        int iVersion;
        int iRevision;
        int iSubrevision;
        int iBuild;
        int iHeaderSize;
        int iProjOffset;
        int iClassOffset;
        int iJavaOffset;
        int iProjSize;
        int iClassSize;
        int iJavaSize;
    }

    /** all the rest of the data of the file is compressed **/

    private MetaProject theMetaProject;
    private GZIPInputStream zinputStream;
    private InputStream inputStream;
    private int iBytesLastRead = 0;
    private int iBytesLastWritten = 0;
    private RandomAccessFile outputFile;
    private int inputPosition = 0;

    CompressedProjectDocumentHeader theHeader = new CompressedProjectDocumentHeader();

    CompressedProjectDocument(MetaProject theMetaProject) {
        this.theMetaProject = theMetaProject;
    }

    void setInputStream(GZIPInputStream zinputStream) {
        this.zinputStream = zinputStream;
    }

    InputStream getInputStream() {
        return zinputStream;
    }

    void setOutputFile(RandomAccessFile outputFile) {
        this.outputFile = outputFile;
    }

    RandomAccessFile getOutputFile() {
        return outputFile;
    }

    int getBytesLastWritten() {
        return iBytesLastWritten;
    }

    int getBytesLastRead() {
        return iBytesLastRead;
    }

    boolean open(InputStream inputStream) throws HammerException {
        this.inputStream = inputStream;
        this.outputFile = null;
        // read header into memory.
        try {
            if (bOpen == false) {
                boolean bRetVal = readHeader();
                this.bOpen = true;
                return bRetVal;
            } else
                return true;
        } catch (IOException e1) {
            return false;
        }
    }

    void open(RandomAccessFile outputFile) {
        this.outputFile = outputFile;
        this.zinputStream = null;
        this.inputStream = null;
        // header starts off as zero - need to initialize.
        this.theHeader.iWatermark = 0xabac;
        this.theHeader.iHeaderSize = CompressedProjectDocumentHeader.HEADER_SIZE;
        this.theHeader.iVersion = theMetaProject.getMetaDataUser().getVersionNumber();
        this.theHeader.iRevision = theMetaProject.getMetaDataUser().getRevisionNumber();
        this.theHeader.iSubrevision = theMetaProject.getMetaDataUser().getSubRevisionNumber();
        this.theHeader.iBuild = theMetaProject.getMetaDataUser().getBuildNumber();
        inputPosition = 0;
    }

    boolean close() {
        if (zinputStream != null) {
            try {
                zinputStream.close();
                if (inputStream != null)
                    inputStream.close();
            } catch (IOException e) {
                return false;
            }
        }
        return true;
    }

    boolean close(String sFileName, OutputStream os) {
        //todo this is a copy of close(String) - until I have verified that it works!
        File fileOutput = null;
        if (outputFile != null) {
            boolean bStreamProvided = (os != null);
            try {
                writeHeader();
                outputFile.close();

                // File is written.  OK now compress.
                File fileInput = new File(sFileName);
                if (bStreamProvided == false) {
                    fileOutput = new File(sFileName + ".zzz");
                    os = new BufferedOutputStream(new FileOutputStream(fileOutput));
                }
                BufferedInputStream is = new BufferedInputStream(new FileInputStream(sFileName));
                // write header uncompressed.
                byte[] headerBytes = new byte[theHeader.iHeaderSize];
                is.read(headerBytes);
                os.write(headerBytes);
                // write zip file.
                GZIPOutputStream fgzipOutput = new GZIPOutputStream(os, 8192);

                //16K but we will make configurable on next release.
                // We can also make configurable so the whole binary is buffered but it depends
                // int iSize = new Long(fileInput.length()).intValue();

                int iSize = CompressedProjectDocumentHeader.COMPRESSION_BUFFER_SIZE;
                byte tmpbuf[] = new byte[iSize];
                while (is.available() != 0) {
                    int b = is.read(tmpbuf);
                    fgzipOutput.write(tmpbuf);
                }
                fgzipOutput.close();
                if (bStreamProvided == false)
                    os.close();
                is.close();
                fileInput.delete();
                if (bStreamProvided == false)
                    fileOutput.renameTo(fileInput);
            } catch (IOException e) {
                return false;
            }
        }
        return true;
    }

    boolean close(String sFileName) {
        if (outputFile != null) {
            try {
                writeHeader();
                outputFile.close();

                // File is written.  OK now compress.
                File fileInput = new File(sFileName);
                File fileOutput = new File(sFileName + ".zzz");
                FileOutputStream os = new FileOutputStream(sFileName + ".zzz");
                FileInputStream fin = new FileInputStream(sFileName);
                // write header uncompressed.

                byte tmpHeadBuf[] = new byte[theHeader.iHeaderSize];
                fin.read(tmpHeadBuf);
                os.write(tmpHeadBuf);
                // write zip file.
                GZIPOutputStream fgzipOutput = new GZIPOutputStream(os);

                int iSize = CompressedProjectDocumentHeader.COMPRESSION_BUFFER_SIZE;
                byte tmpbuf[] = new byte[iSize];

                while (fin.available() != 0) {
                    int b = fin.read(tmpbuf);
                    fgzipOutput.write(tmpbuf);
                }
                fgzipOutput.close();
                os.close();
                fin.close();
                fileInput.delete();
                fileOutput.renameTo(fileInput);
            } catch (IOException e) {
                return false;
            }
        }
        return true;
    }

    byte[] read(int iOffset, int iBytesToRead) throws IOException {

        while (inputPosition < iOffset) {
            zinputStream.read();
            inputPosition++;
        }
        if (iBytesToRead == 0)
            iBytesToRead = 0xffff;  // java compiled class is limited to 64K.
        int totalbytes = 0;

        // Create an array the same length as the file to hold the compiled class.
        byte[] prozBytes = new byte[iBytesToRead];

        while ((totalbytes != iBytesToRead) && (zinputStream.available() != 0)) {
            // Read class definition into the array.
            int iBytesToReadThisTime = 512;
            if ((iBytesToRead - totalbytes) < 512)
                iBytesToReadThisTime = iBytesToRead - totalbytes;
            int bytesread = zinputStream.read(prozBytes, totalbytes, iBytesToReadThisTime);
            totalbytes += bytesread;
        }

        iBytesLastRead = totalbytes;
        return prozBytes;
    }

    void write(int iOffset, byte[] bytes) throws IOException {
        outputFile.write(bytes);
        iBytesLastWritten = bytes.length;
    }

    boolean readHeader() throws IOException, HammerException {
        DataInputStream dataIn = new DataInputStream(inputStream);
        this.theHeader.iWatermark = dataIn.readInt();
        if (theHeader.iWatermark != CompressedProjectDocumentHeader.WATERMARK_VALUE) {
            throw new HammerException(HammerException.BAD_COMPRESSED_PROJECT_FILE_FORMAT, "Watermark bad - does not equal 0xabac");
        }
        inputPosition += 4;

        this.theHeader.iVersion = dataIn.readInt();
        inputPosition += 4;
        this.theHeader.iRevision = dataIn.readInt();
        inputPosition += 4;
        this.theHeader.iSubrevision = dataIn.readInt();
        inputPosition += 4;
        this.theHeader.iBuild = dataIn.readInt();
        inputPosition += 4;
        this.theHeader.iHeaderSize = dataIn.readInt();
        inputPosition += 4;
        this.theHeader.iProjOffset = dataIn.readInt();
        inputPosition += 4;
        this.theHeader.iClassOffset = dataIn.readInt();
        inputPosition += 4;
        this.theHeader.iJavaOffset = dataIn.readInt();
        inputPosition += 4;
        this.theHeader.iProjSize = dataIn.readInt();
        inputPosition += 4;
        this.theHeader.iClassSize = dataIn.readInt();
        inputPosition += 4;
        this.theHeader.iJavaSize = dataIn.readInt();
        inputPosition += 4;
        while (inputPosition < theHeader.iHeaderSize) {
            dataIn.readInt();
            inputPosition += 4;
        }
        zinputStream = new GZIPInputStream(inputStream);

        // do some validation.
        if (theHeader.iWatermark == CompressedProjectDocumentHeader.WATERMARK_VALUE) {
            if (theHeader.iHeaderSize == CompressedProjectDocumentHeader.HEADER_SIZE)
                return true;
            else
                return false;
        } else
            return false;
    }

    void writeHeader() throws IOException {
        // initialize header area.
        byte[] filler = new byte[theHeader.iHeaderSize];
        outputFile.write(filler);
        outputFile.seek(0);
        // make sure header is padded with header size bytes first.
        outputFile.writeInt(theHeader.iWatermark);
        outputFile.writeInt(theHeader.iVersion);
        outputFile.writeInt(theHeader.iRevision);
        outputFile.writeInt(theHeader.iSubrevision);
        outputFile.writeInt(theHeader.iBuild);
        outputFile.writeInt(theHeader.iHeaderSize);
        outputFile.writeInt(theHeader.iProjOffset);
        outputFile.writeInt(theHeader.iClassOffset);
        outputFile.writeInt(theHeader.iJavaOffset);
        outputFile.writeInt(theHeader.iProjSize);
        outputFile.writeInt(theHeader.iClassSize);
        outputFile.writeInt(theHeader.iJavaSize);
    }

    String readJavaSource() throws IOException {
//        if (theHeader.iJavaSize==0)
//            return null;
//        byte[] javaSrc = read(theHeader.iJavaOffset, theHeader.iJavaSize);
//        if (javaSrc != null)
//            return new String(javaSrc);
//        else
        return null;
    }

    byte[] readProjSource() throws IOException {
        if (theHeader.iProjSize == 0)
            return null;
        byte[] projSrc = read(theHeader.iProjOffset, theHeader.iProjSize);
        return projSrc;
    }

    Class readRenderingClass(InputStream inputStream, String sClassName) throws IOException, HammerException {
        // only load the class once from the proz file per execution.
        try {
             // When using Open Source the JAR is loaded not the class so the Class needs to be loaded first. MHC

              if(this.theMetaProject.getMetaDataUser().getOpenSourceState()==false)
               {
            theMetaProject.renderingClass =
                    theMetaProject.getMetaDataUser().getClassLoader().theLoader.loadLoadedClass(sClassName);
               }
                else
               {
                    theMetaProject.renderingClass =
                     theMetaProject.getMetaDataUser().getClassLoader().getLoader().loadClass(sClassName);
               }

        } catch (ClassNotFoundException e1) {
            open(inputStream);  // throws hammer exception
            if (theHeader.iClassSize == 0)
                return null;
            byte[] classBytes = read(theHeader.iClassOffset, theHeader.iClassSize);
            close();
            if (classBytes == null)
                return null;
            else {
                theMetaProject.renderingClass = theMetaProject.getMetaDataUser().getClassLoader().getLoader().defineClass(sClassName, classBytes);
            }
        }
        return theMetaProject.renderingClass;
    }

    void writeJavaSource(String sSource) throws IOException {
//        theHeader.iJavaSize = 0;
//        if (sSource==null)
//            return;
//        byte[] bytes = sSource.getBytes();
//        theHeader.iJavaOffset = theHeader.iHeaderSize+theHeader.iProjSize + theHeader.iClassSize;
//        outputFile.seek(theHeader.iJavaOffset);
//        outputFile.write(bytes);
//        theHeader.iJavaSize = bytes.length;
    }

    void writeProjSource(byte[] bytes) throws IOException {
        theHeader.iProjSize = 0;
        if (bytes == null)
            return;
        theHeader.iProjOffset = theHeader.iHeaderSize + theHeader.iClassSize;
        outputFile.seek(theHeader.iProjOffset);
        outputFile.write(bytes);
        theHeader.iProjSize = bytes.length;
    }

    void writeClass(byte[] bytes) throws IOException {
        theHeader.iClassSize = 0;
        if (bytes == null)
            return;
        theHeader.iClassOffset = theHeader.iHeaderSize + inputPosition; // MHC
        outputFile.seek(theHeader.iClassOffset);
        outputFile.write(bytes);
        theHeader.iClassSize = bytes.length;
        inputPosition += bytes.length;
    }

    protected  boolean writeCustomMetadata(String customMetadataPath) throws IOException
    {
        FilenameFilter filter = new FilenameFilter() {
            public boolean accept(File dir, String name) {
                return name.endsWith(".xml");
            }
        };

        String dirRepositories[] = customMetadataPath.split(";");
        for(int c=0;c < dirRepositories.length;c++)
        {
            File dir = new File(dirRepositories[c]);
            String[] children = dir.list(filter);
            if (children == null) {
                return false;
            }

        }

        for(int c=0;c < dirRepositories.length;c++)
        {

            String dirSeparator = System.getProperty("file.separator");
            File dir = new File(dirRepositories[c]);
            String[] children = dir.list(filter);
            if (children == null) {
                return false;
            }

            FileOutputStream fostream = null;
            JarOutputStream jarOutputStream = null;

            boolean singleFileSepa = dirRepositories[c].contains("/");
            if(singleFileSepa) 
                dirSeparator = "/";

            String customJarName = dirRepositories[c].substring(3,dirRepositories[c].length()).replace(dirSeparator,".");
            File newJarFile = null;
            String sNewJarName = new String(dirRepositories[c]+ dirSeparator + customJarName + "." + "AbacusCustom" + ".jar");

            // This checks if any XML file is older than the jar file
            // if so, then it will be skip on the next block. No need to
            // recreate the jar when nothing has changed.
            if(isXMLTouched(dirRepositories[c],children,sNewJarName)==false)
                continue;

            try {
                newJarFile = new File( sNewJarName);

                if (newJarFile.exists())
                    newJarFile.delete();

                newJarFile.createNewFile();
            } catch (IOException e0) {
                System.err.println("Cannot create jar file " + sNewJarName + ".tmp" + ".jar");
            }
            try {
                fostream = new FileOutputStream(newJarFile);

                try {
                    jarOutputStream = new JarOutputStream(fostream);
                } catch (IOException e1) {
                    System.err.println("Cannot create output stream for jar file " + sNewJarName + " " + e1);
                }

            } catch (FileNotFoundException e1) {
                e1.printStackTrace();
            }

            for (int i=0; i<children.length; i++)
            {
                // Get filename of file or directory
                String filename = dirRepositories[c] + dirSeparator + children[i];
                InputStream  repoHammerMetadataStream = null;
                try
                {
                    ZipEntry ze = new ZipEntry("ch\\abacus\\components\\imported\\" + children[i]);
                    jarOutputStream.putNextEntry(ze);

                    File fileInput = new File(filename);
                    FileInputStream xin = null;
                    try {
                        xin = new FileInputStream(fileInput);
                    } catch (FileNotFoundException e) {
                        return false;
                    }
                    BufferedInputStream xinStr = new BufferedInputStream(xin);
                     int iSize = new Long(fileInput.length()).intValue();
                    byte tmpbuf[] = new byte[iSize];

                    int b;
                    try {
                        while ((b = xinStr.read(tmpbuf)) != -1) {
                            jarOutputStream.write(tmpbuf);
                        }
                        xin.close();
                    } catch (IOException e) {
                        return false;
                    }

                }
                catch (java.io.FileNotFoundException e) {
                    e.printStackTrace();
                }
            }
            jarOutputStream.close();
            fostream.close();
        }

        return false;
    }

    private boolean isXMLTouched(String path, String files[] , String jarName)
    {
        boolean bFlag = false;

        File jarFile = new File(jarName);
        if(jarFile.exists()==false)
            return true;
        long jarModDate=jarFile.lastModified();

        for(int c=0;c < files.length;c++)
        {
            if(files[c]==null)
            continue;

            String sXmlFile = path + "\\" + files[c];
            File xmlFile = new File(sXmlFile);
            long modDate=xmlFile.lastModified();
            if(modDate>jarModDate)
            {
                bFlag=true;
                break;
            }
        }

        return bFlag;
    }
}
