/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * JAEntry.java
 *
 * Creator:
 * 15.05.2003 10:08:18 lauchenauer
 *
 * Maintainer:
 * 15.05.2003 10:08:18 lauchenauer
 *
 * Last Modification:
 * $Id: JAEntry.java,v 1.1 2005/10/14 06:31:13 agraham Exp $
 *
 * Copyright (c) 2003 ABACUS Research AG, All Rights Reserved
 */
package ch.abacus.lib.ui.entry;

import ch.abacus.lib.ui.vorgaben.ApplicationSettingsInterface;
import ch.abacus.lib.ui.*;
import ch.abacus.lib.util.GlobalInterface;

import javax.swing.*;
import java.awt.event.*;
import java.awt.*;
import java.util.ArrayList;
import java.util.Iterator;


/**
 * The Abacus Entry Component
 */
public class JAEntry extends JComponent implements ApplicationSettingsInterface, KeyListener {

  /**
   * @see #getUIClassID
   * @see #readObject
   */
  private static final String mUIClassID = "EntryUI";

  /**
   * Constant for the Entry state. NORMAL state allows editing, browsing, etc. The edit field has a white background and
   * the writing is black. This is the default state.
   */
  public static final int ENTRYSTATE_NORMAL = 0;

  /**
   * Constant for the Entry state. BROWSEONLY state browsing, etc. No editing can be done in the edit field, even though
   * the field can receive the focus. The edit field has a white background and the writing is black.
   */
  public static final int ENTRYSTATE_BROWSEONLY = 1;

  /**
   * Constant for the Entry state. READONLY state browsing, etc. No editing and no browsing. The field is essentially
   * disabled with the difference that the value can be selected. The edit field has a grey background and the writing
   * is black. Caption is black, no prompt
   */
  public static final int ENTRYSTATE_READONLY = 2;

  /**
   * This constant is used for the focus lost behavior of the Entry. If this behavior is set, and a Object is set,
   * the entry value is compared with the current DateElement. If they don't match, the value is cleared. This is the
   * standard behabior
   */
  public static final int SIMPLE_VALUE_CHECK = 0;

  /**
   * This constant is used for the focus lost behavior of the Entry. If this behavior is set, and a Object is set, a
   * check is made on the Object to see if this entryvalue is present. If that is the case, the value is adjusted,
   * else, the value is cleared.
   */
  public static final int EXACT_VALUE_CHECK = 1;

  /**
   * This constant is used for the focus lost behavior of the Entry. If this behavior is set, and a Object is set, a
   * findnearest is performed on focuslost to make sure that no invalid entryvalues are set.
   */
  public static final int NO_INVALIDS_VALUE_CHECK = 2;

  /**
   * Behavior on exit: no update is called
   */
  public static final int NO_UPDATE_ON_EXIT = 1;

  /**
   * Behavior on exit: update is called if the entry was modified
   */
  public static final int UPDATE_ONLY_IF_MODIFIED = 2;


  /**
   * Creates an entry setting the caption width to fit the caption text, The Editor width is set to fit the length of
   * the string provided.
   */
  public static JAEntry createEntry(GlobalInterface global, String caption, String editorWidth, boolean showPrompt, boolean showSpin, boolean showValue) {
    JAEntry entry = new JAEntry(global, caption, 100, 100, showPrompt, showSpin, showValue);
    entry.setWidthCaption(caption);
    entry.setWidthEdit(editorWidth);
    return entry;
  }


  /**
   * This flag is used to determine the focus lost behavior of the entry.
   */
  protected int mFocusLostBehavior = SIMPLE_VALUE_CHECK;

  /**
   * This flag is used to determine the exit behavior of the entry.
   */
  protected int mExitUpdateBehavior = UPDATE_ONLY_IF_MODIFIED;

  /**
   * The setting name of this component used for saving the Application settings
   */
  private String mSettingName = "";

  /**
   * all Navigationlisteners of this component are tracked in this vector
   */
  private EventListenerContainer mNavigationListenerContainer;

  /**
   * all KeyListeners of this component are tracked in this vector
   */
  private EventListenerContainer mKeyListenerContainer;

  /**
   * all ValueChangedListeners of this component are tracked in this vector
   */
  private EventListenerContainer mValueChangedContainer;

  /**
   * all AllowNavigationListeners of this component are tracked in this vector
   */
  private EventListenerContainer mAllowNavigationContainer;

  /**
   * all EntryModificationListeners of this component are tracked in this vector
   */
  private EventListenerContainer mEntryModificationContainer;

  /**
   * The Focus Listener attached to the Editor Component of this JAEntry
    */
  private EditorFocusListener mEditorFocusListener;


  /**
   * Every entry requires an instance of  GlobalInterface to access the users FormatPatterns
   */
  protected GlobalInterface mGlobalInterface = null;

  /**
   * variable to manage the EntryState
   */
  protected int mEntryState = ENTRYSTATE_NORMAL;

  // flags that store which parts of the entry are visible
  protected boolean mShowCaption = true;
  protected boolean mShowSpinner = true;
  protected boolean mShowDateButton = false;
  protected boolean mShowPromptButton = false;
  protected boolean mShowValue = true;

  /**
   * The caption text is stored in this variable. This is necessary to allow for an underlined caption if the prompt has
   * been initialized. This text has to be set back to the label component whenever the Entry is disabled, as HTML
   * labels are not greyed out when they are disabled
   */
  protected String mCaptionText;

  protected Icon mCaptionIcon;

  // the different components of the Entry
  protected Object mEditor;
  protected JAEntryLabel mCaption;
  protected Object mValueRenderer;

  /**
   * To be able to fire the valueChanged events at the right time we have to keep track of the previous value
   */
  protected Object mOldEntryValue = null;

  // from to variables
  protected JAEntry mEntryTo;
  protected JAEntry mEntryFrom;

  // the EntryModel
  protected Object mModel = null;
  protected Object mData = null;

  // the lookup model
  protected Object mLookupData = null;

  /**
   * flag to track how sorting is adjusted. This affects the way that from to entries are adjusted against each other,
   * not the sorting from the Object. The entry can never change the sorting of the Object
   */
  protected boolean mUseDataFlexSort = true;

  /**
   * flag to track how sorting is adjusted. This affects the way that from to entries are adjusted against each other,
   * not the sorting from the Object. The entry can never change the sorting of the Object
   */
  protected boolean mUppercaseSort = false;

  /**
   * this is the flag that shows if a late action (browsNext, etc, has to be called 0= no action, 1 = browseNext, 2 =
   * browseprior, 3 = prompt, 4 = date
   */
  protected int mLateAction = 0;

  protected Object mUserValue;
  protected boolean mIsUserEntry;
  protected Runnable mPromptCallback;

  /**
   * flag that keeps track if the data in the entry has been modified
   */
  protected boolean mModified = false;

  /**
   * Flag that keeps track if the browse operation is to be exectued. The Navigation events are fired, no matter what
   * the setting of this flag is, but the events are not passed on to the model if the flag is set to false.
   */
  protected boolean mPerformBrowseOperation = true;

  private String m_DataColumnName;

  private boolean m_AutoAbaDisplaySameColumns = true;

  private EventListenerContainer m_CFWEventListeners = new EventListenerContainer();

  private ArrayList m_CFWDataSources = new ArrayList();

  private Color m_BackgroundEditor = null;
  private Color m_ForegroundEditor = null;

  protected boolean mFireEntryModificationEvents = true; // flag used to stop entry modification events from being fired
  protected boolean mAbaDisplayTriggersModifications = false;  // flag that is used to define whether abaDisplay events also cause entryModification events to be fired

  /**
   * Flag to force a conversion of the Editor value to a string
   */
  protected boolean mForceStringValue = false;

  /**
   * For saving the current cursor between mouse events on the caption
    */
  private Cursor mAppCursor;//=Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR);

  /**
   * Extended Label class for the caption, required for that Validation
   */
  public class JAEntryLabel extends JALabel {
    public JAEntryLabel() {
      super();
    }

    public JAEntryLabel(String text) {
      super(text);
    }

    public void abaDisplay() {
    }

    public void abaClear() {
    }

    public void abaUpdate() {
    }

    public void abaValidate() {
    }

    public void requestEnter(Component context) {
    }

    public void requestExit(Component context) {
    }

    /**
     * remove an actionlistener
     *
     * @param l the actionlistener to be removed
     */
    public void removeActionListener(ActionListener l) {
    }

    /**
     * add a new actionlistener
     *
     * @param l the new listener
     */
    public void addActionListener(ActionListener l) {
    }

    public Object getDispatcher() {
      return null;
    }
  }

  protected JAEntry() {
  }

  /**
   * Constructor
   *
   * @param global instance of the GlobalInterface used for the FormatPatterns
   */
  public JAEntry(GlobalInterface global) {
    mGlobalInterface = global;
    updateUI();
    init();
  }

  /**
   * Constructor that hooks up a Object as well. Make sure the EntryData is completely initialized as the Object
   * is retrieved from this class
   *
   * @param global instance of the GlobalInterface used for the FormatPatterns
   * @param data   the EntryData for this entry defining which fields are shown
   */
  public JAEntry(GlobalInterface global, Object data) {
    this(global);
  }

  public JAEntry(GlobalInterface global, String caption, int capWidth, int editorWidth, boolean showPrompt,
                 boolean showSpin, boolean showValue) {
    this(global);
    setCaption(caption);
    setWidthCaption(capWidth);
    setWidthEdit(editorWidth);
    setShowPromptButton(showPrompt);
    setShowSpinButtons(showSpin);
    setShowValue(showValue);
  }

  public JAEntry(GlobalInterface global, Object data, String caption, int capWidth, int editorWidth, boolean showPrompt,
                 boolean showSpin, boolean showValue) {
    this(global, data);
    setCaption(caption);
    setWidthCaption(capWidth);
    setWidthEdit(editorWidth);
    setShowPromptButton(showPrompt);
    setShowSpinButtons(showSpin);
    setShowValue(showValue);
  }


  protected void init() {
    // make sure that a caption and value is initialized
    getCaptionComponent();
  }

  /**
   * Returns the name used to identify the instance in the ApplicationSettings class.
   *
   * @return the Setting name of the instance of this object
   */
  public String getSettingName() {
    return mSettingName;
  }

  /**
   * Sets the name used to identify the instance in the ApplicationSettings class.
   *
   * @param newName the nwe Setting name for the instance of this object
   */
  public void setSettingName(String newName) {
    mSettingName = newName;

    // if history popup is enabled remove it from the entry and attach it again
    // just to make sure the history values are saving under the correct setting
    // name
    if (m_HistoryEnabled){
      setHistroyEnabled(false);
      setHistroyEnabled(true);
    }
  }

  //********************************************************************************************************************
  //                          Methods for the visual appearance of the Entry
  //********************************************************************************************************************

  public boolean getShowCaption() {
    return mShowCaption;
  }

  public void setShowCaption(boolean showCaption) {
    boolean oldValue = mShowCaption;
    mShowCaption = showCaption;
    firePropertyChange("Caption.visible", oldValue, mShowCaption);
  }

  public boolean getShowSpinButtons() {
    return mShowSpinner;
  }

  public void setShowSpinButtons(boolean showSpinner) {
    boolean oldValue = mShowSpinner;
    mShowSpinner = showSpinner;
    firePropertyChange("Spinner.visible", oldValue, mShowSpinner);
  }

  public boolean getShowDateButton() {
    return mShowDateButton;
  }

  public void setShowDateButton(boolean showDateButton) {
    boolean oldValue = mShowDateButton;
    mShowDateButton = showDateButton;
    firePropertyChange("Date.visible", oldValue, mShowDateButton);
  }

  public boolean getShowPromptButton() {
    return mShowPromptButton;
  }

  public void setShowPromptButton(boolean showPromptButton) {
    boolean oldValue = mShowPromptButton;
    mShowPromptButton = showPromptButton;
    firePropertyChange("Prompt.visible", oldValue, mShowPromptButton);
  }

  public boolean getShowValue() {
    return mShowValue;
  }

  public void setShowValue(boolean showValue) {
    boolean oldValue = mShowValue;
    mShowValue = showValue;
    firePropertyChange("Value.visible", oldValue, mShowValue);
  }

  public void setWidthCaption(int newWidth) {
    if (newWidth <= 0) setShowCaption(false);
    firePropertyChange("Caption.width", 0, newWidth);
  }

  public void setWidthCaption(String maxString) {
    JALabel l = new JALabel(maxString);
    setWidthCaption((int) l.getPreferredSize().getWidth());
  }

  public void setWidthEdit(int newWidth) {
    firePropertyChange("Editor.width", 0, newWidth);
  }

  public void setWidthEdit(String maxString) {
    JALabel l = new JALabel(maxString);
    setWidthEdit((int) l.getPreferredSize().getWidth() + 6);
  }


  public void setWidthValue(int newWidth) {
    if (newWidth <= 0) setShowValue(false);
    firePropertyChange("Value.width", 0, newWidth);
  }

  public void setEnabled(boolean setting) {
    super.setEnabled(setting);

    firePropertyChange("enabled", null, null);
  }

  protected void underlineCaption(boolean underline) {
    mCaption.setText(mCaptionText);
  }

  //********************************************************************************************************************
  //                          The Model and the EntryData
  //********************************************************************************************************************

  /**
   * Returns the entryData instance
   *
   * @return the internal EntryData instance
   */
  public Object getEntryData() {
    return mData;
  }

  /**
   * Returns the EntryModel that was set, null if no model is set
   *
   * @return the set EntryModel
   */
  public Object getModel() {
    return mModel;
  }

  /**
   * Sets a lookup model that is used for browsing and displaying, only writing values into the main model on updates
   *
   * @param data the EntryData for the lookup model
   */
  public void setLookupModel(Object data) {
    mLookupData = data;
  }

  /**
   * Returns the LookupModel that was set, null if no model is set
   *
   * @return the set LookupModel
   */
  public Object getLookupModel() {
    return mLookupData;
  }

  //********************************************************************************************************************
  //                          The EntryValue and the Editor
  //********************************************************************************************************************

  /**
   * Returns the current value stored in the Editor, make sure the object that is returned supports the Comparable
   * interface, else From-To Entris will not work
   *
   * @return returns the value from the editor
   *
   * @see Comparable
   */
  public Object getEntryValue() {
      return null;
  }

  /**
   * Set the value for the entry, make sure that the type of the value is supported by the editor
   *
   * @param value the new value for the entry
   */
  public void setEntryValue(Object value) {
  }

  public void setEntryValue(String value) {
    setEntryValue((Object) value);
  }

  public void setEntryValue(Integer value) {
    setEntryValue((Object) value);
  }

  public void setEntryValue(Long value) {
    setEntryValue((Object) value);
  }

  public void setEntryValue(Boolean value) {
    setEntryValue((Object) value);
  }

  /**
   * set to true if you want th eentry value to be converted to a string
   */
  public void setForceStringValue(boolean forceStringValue) {
    mForceStringValue = forceStringValue;
  }

  /**
   * You can replace the Editor that the Entry uses. When replacing the editor take into account that each Editor has
   * its own datatype. Thus the Editor has to match the data that is passed to it, else Exceptions can occur.
   *
   * @param editor the new editor
   *
   * @exception IllegalArgumentException if editor is <code>null</code>
   * @see #getEditor
   */
  public void setEditor(Object editor) {
  }


   protected class EditorFocusListener extends FocusAdapter
   {
   }



  /**
   * Returns the current EntryEditor
   *
   * @return the current EntryEditor
   */
  public Object getEditor() {
    return mEditor;
  }

  /**
   * Used to pass settings to the Editor. Every editor is free as to which properties are available. These are
   * documented in each Editors documentation. All this method does is pass the property on to the currently active
   * editor.
   *
   * @param propertyName the name of the properts that is to be adjusted
   * @param setting      the new setting for the property, check the datatype
   */
  public void setEditorProperty(String propertyName, Object setting) {
  }

  /**
   * Sets the behavior for the entry when a DatamOdel is present and the entry loses it's focus. Use:
   * SIMPLE_VALUE_CHECK, EXACT_VALUE_CHECK, NO_INVALIDS_VALUE_CHECK, default is SIMPLE_VALUE_CHECK.
   *
   * @param behavior the new behavior
   */
  public void setFocusLostBehavior(int behavior) {
    mFocusLostBehavior = behavior;
  }

  /**
   * Sets the behavior for the entry when a DatamOdel is present and the entry loses it's focus. Use:
   * UPDATE_ONLY_IF_MODIFED, NO_UPDATE_ON_EXIT default is UPDATE_ONLY_IF_MODIFED.
   * <p/>
   * Certain combinations with the FocusLostBehavior will cause Exceptions as they are not possible. The following
   * combinations work:
   * <p/>
   * SIMPLE_VALUE_CHECK with both exit behaviors EXACT_VALUE_CHECK only with UPDATE_ONLY_IF_MODIFED  (EXACT_VALUE_CHECK
   * calls findNEarest in some cases which modifies the model) NO_INVALIDS_VALUE_CHECK only with UPDATE_ONLY_IF_MODIFED
   * (NO_INVALIDS_VALUE_CHECK always calls findNearest modifying the model)
   *
   * @param behavior the new behavior
   */
  public void setExitBehavior(int behavior) {
    mExitUpdateBehavior = behavior;
  }

  public int getExitBehavior() {
    return mExitUpdateBehavior;
  }

  public boolean requestFocusInWindow() {
      return false;
  }


  //********************************************************************************************************************
  //                          EntryState Code
  //********************************************************************************************************************

  /**
   * Returns the currently set EntryState, being one of the following constants: ENTRYSTATE_NORMAL,
   * ENTRYSTATE_BROWSEONLY, ENTRYSTATE_READONLY
   *
   * @return the currently set EntryState
   */
  public int getEntryState() {
    return mEntryState;
  }

  /**
   * Sets the EntryState, use the following constants: ENTRYSTATE_NORMAL, ENTRYSTATE_BROWSEONLY, ENTRYSTATE_READONLY
   *
   * @param entryState sets the EntryState
   */
  public void setEntryState(int entryState) {
  }

  //********************************************************************************************************************
  //                          The Caption and the Caption component
  //********************************************************************************************************************

  /**
   * Gets the caption of the entry.
   *
   * @return the current caption
   */
  public String getCaption() {
    return mCaptionText;
  }

  /**
   * Sets the caption of the entry.
   *
   * @param newCaption the new caption for the entry
   */
  public void setCaption(String newCaption) {
    String oldValue = getCaption();
    mCaptionText = newCaption;
    firePropertyChange("Caption.text", oldValue, mCaptionText);
  }

  protected void setCaptionIcon(Icon _Icon){
      Icon oldValue = mCaptionIcon;
      mCaptionIcon = _Icon;
      if ( mCaption != null){
        mCaption.setHorizontalTextPosition ( SwingConstants.LEADING );
        mCaption.setIcon ( mCaptionIcon );
      }

      firePropertyChange("Caption.icon",  oldValue, mCaptionIcon);

  }

  /**
   * Do not call this method externally. It is only public so it can be called from the UI class. Override this method
   * if you want to replace the standard caption component with one of your own.
   *
   * @return the Caption component
   */
  public JComponent getCaptionComponent() {
    return mCaption;
  }

  /**
   * Sets the orientation of the catption in relation to the rest of the components in the entry. Use the following
   * constants as parameteres: SwingConstants.TOP, .LEFT, .RIGHT, .BOTTOM, default is .LEFT
   *
   * @param orientation the orientation of the caption in relation to the rest of the entry parts
   */
  public void setCaptionOrientation(int orientation) {
    firePropertyChange("Caption.orientation", 0, orientation);
  }

  /**
   * Sets the alignment of the label's contents along the X axis. This will only only work if the caption is an instance
   * of JLabel, which is the case unless the method getCaptionComponent() has been overwritten to return a different
   * component.
   *
   * @param alignment One of the following constants defined in <code>SwingConstants</code>: <code>LEFT</code>,
   *                  <code>CENTER</code>, <code>RIGHT</code>, <code>LEADING</code> (the default) or
   *                  <code>TRAILING</code>.
   */
  public void setCaptionHorizontalAlignment(int alignment) {
    firePropertyChange("Caption.alignment", 0, alignment);
  }

  /**
   * Sets the caption to be underlined. Is only necessary in cases where a prompt event has to be triggered through the
   * caption without having an F4List conencted to this entry (via initPrompt)
   *
   * @param underline true if the caption should be underlined
   */
  public void setUnderlineCaption(boolean underline) {
    setCaption(getCaption());
  }

//********************************************************************************************************************
//                          The Value and the ValueRenderer
//********************************************************************************************************************

  /**
   * Sets the new value
   *
   * @param value the new value
   */
  public void setValue(String value) {
  }

  /**
   * Returns the current String from the ValueRenderer
   *
   * @return the current value
   */
  public String getValue() {
      return null;
  }

  /**
   * Do not call this method externally. It is only public so it can be called from the UI class. Override this method
   * if you want to replace the standard value component with one of your own.
   *
   * @return the value component
   */
  public Object getValueComponent() {
    return null;
  }

  /**
   * If you want to change the way that the value string is rendered, you can set a new Renderer here
   *
   * @param renderer the new Renderer used to render the value string
   */
  public void setValueRenderer(Object renderer) {
  }

//********************************************************************************************************************
//                          Methods for From-To Entries
//********************************************************************************************************************

  /**
   * Sets the From Entry if there is to be one, this results in the From entry alway containing a value that is less
   * than or equal to the To Entry. Remember to set the current entry as the To Entry in the From Entry
   *
   * @param newEntryFrom sets the From entry
   */
  public void setEntryFrom(JAEntry newEntryFrom) {
    mEntryFrom = newEntryFrom;
  }

  /**
   * Returns the From Entry if one is set
   *
   * @return the From Entry if one is set
   */
  public JAEntry getEntryFrom() {
    return mEntryFrom;
  }

  /**
   * Sets the To Entry if there is to be one, this results in the To entry alway containing a value that is greater than
   * or equal to the From Entry. Remember to set the current entry as the From Entry in the To Entry
   *
   * @param newEntryTo sets the To entry
   */
  public void setEntryTo(JAEntry newEntryTo) {
    mEntryTo = newEntryTo;
  }

  /**
   * Returns the To Entry if one is set
   *
   * @return the To Entry if one is set
   */
  public JAEntry getEntryTo() {
    return mEntryTo;
  }

  /**
   * Can be called to force the entry to perform the FromTo check
   */
  public void updateFromTo() {
    updateEntryFrom();
    updateEntryTo();
  }

  /**
   * Internal method that adjusts the value of the From Entry if necessary
   */
  protected void updateEntryFrom() {
  }

  /**
   * Internal method that adjusts the value of the To Entry if necessary
   */
  protected void updateEntryTo() {
  }

//********************************************************************************************************************
//                          Cursor changes
//********************************************************************************************************************

  /**
   * Show the Hourglass cursor
   */
  protected void setWaitCursor() {
  }

  /**
   * Show the hand cursor. This is called on mouseEntered event on the captionComponent
   */
  protected void setHandCursor() {
  }

  /**
   * Reset the cursor to default cursor
   */
  protected void setDefaultCursor() {
  }

//********************************************************************************************************************
//                          Navigation and button reactions
//********************************************************************************************************************

  /**
   * Sets whether the browse operation is to be exectued. The Navigation events are fired, no matter what the setting of
   * this flag is, but the events are not passed on to the model if the flag is set to false.
   */
  public void setPerformBrowseOperation(boolean setting) {
    mPerformBrowseOperation = setting;
  }

  /**
   * Go to the first value
   */
  public void first() {
  }

  protected void firstMain() {
  }

  protected void firstLookup() {
  }

  /**
   * Go to the last value
   */
  public void last() {
  }

  protected void lastMain() {
  }

  protected void lastLookup() {
  }

  /**
   * Browse to the previous value
   */
  public void browsePrior() {
  }

  protected void browsePriorMain () {
  }


    protected void browsePriorLookup () {
    }

  /**
   * Browse to the next value
   */
  public void browseNext() {
  }

  protected void browseNextMain () {
  }

    protected void browseNextLookup () {
    }

  /**
   * Search the nearest Value, used internally for cases where the from or to entries do not have to be updated
   */
  protected void findNearest(boolean updateFromTo) {
  }

  protected void findNearestMain() {
  }

  protected void findNearestLookup() {
  }

  public boolean executeFindNearestLookup() {
      return false;
  }

  /**
   * Search for the nearest value and update from to fields
   */
  public void findNearest() {
  }

  public void findBySysKey(String syskey) {
  }

  protected void findBySysKeyMain(String syskey) {
  }

  protected void findBySysKeyLookup(String syskey) {
  }

//********************************************************************************************************************
//          All methods concerning the calendar
//********************************************************************************************************************

  protected Object getCalendarPopupInstance() {
      return null;
  }

  protected Object getMonthPopupInstance() {
      return null;
  }

  public void calendarAction() {
  }

  public Object getMonthPopup() {
      return null;
  }

  public void setMonthPopup(Object monthPopup) {
  }

  public Object getCalendarPopup() {
    return null;
  }

  public void setCalendarPopup(Object calendarPopup) {
  }


//********************************************************************************************************************
//          Initializing and showing the F4-List (prompt)
//********************************************************************************************************************

  /**
   * Check if an extended prompt is set, if so call that one, else call the normal prompt
   *
   * @param callback the method that is run when this method terminates (null is valid)
   */
  public void promptCaption(Runnable callback) {
  }

  /**
   * Due to a change in the Modal window policy this method now requires a parameter Runnable that is called when the
   * complete method is done.
   *
   * @param callback the method that is run when this method terminates (null is valid)
   */
  public void prompt(Runnable callback) {
  }

  /**
   * Due to a change in the Modal window policy this method now requires a parameter Runnable that is called when the
   * complete method is done.
   *
   * @param callback the method that is run when this method terminates (null is valid)
   */
  protected void promptExecute(Runnable callback) {
  }

  /**
   * This method is called when the F4-List Dialog was shown and closed
   */
  protected void promptExecute2() {
  }

  private Window findContainingWindow() {
    // if the entry is used as a celleditor we have to discover the parent a bit differently
    Component component;
    if (mIsCellEditor)
      component = mCellEditorParent;
    else
      component = this;
    do {
      component = component.getParent();
    } while (!(component instanceof Window));
    return (Window) component;
  }

  public boolean isPromptInitialized() {
      return false;
  }

  /**
   * If you want to use an implementation of F4ListBaseDialog other than the standard implementation, set it with this
   * method.
   *
   * @param dialog the F4Dialog class that is to be sued instead of the standard one
   */
  public void setF4BaseDialog(Object dialog) {
  }

  /**
   * Returns the current F4List Dialog, returns null if no dialog has been instantiated/set yet
   *
   * @return Returns the current F4List Dialog, returns null if no dialog has been instantiated/set yet
   */
  public Object getF4ListBaseDialog() {
    return null;
  }

  /**
   * If you want to use an implementation of F4ListBaseDialog other than the standard implementation, set it with this
   * method. This is for the extended F4List
   *
   * @param dialog the F4Dialog class that is to be sued instead of the standard one
   */
  public void setExtendedF4BaseDialog(Object dialog) {
  }

  public void initPrompt(Object data) {
  }

  public void initExtendedPrompt(Object data) {
  }

  /**
   * @deprecated use initPrompt(EntryData, F4ListDefinition) instead, since the EntryData already contains the
   *             Object
   */
  public void initPrompt(Object model, Object data, Object def) {
  }

  public void initExtendedPrompt(Object data, Object def) {
  }

  /**
   * Initializes the prompt / F4-List with the current model used in the entry
   */
  public void initPromptWithModel() {
  }

  /**
   * Initializes the prompt / F4-List with the current model used in the entry
   */
  public void initExtendedPromptWithModel() {
  }

  /**
   * Reset all Prompt settings
   */
  public void resetPrompt() {
  }

  /**
   * Reset all Extended Prompt settings
   */
  public void resetExtendedPrompt() {
  }

  public void initPromptWithModel(Object def) {
  }

  public void initExtendedPromptWithModel(Object def) {
  }

  /**
   * Set the ActionListener that is triggered if Shift-F4 is pressed on the Editor of the entry
   */
  public void setShiftF4ListListener(ActionListener listener) {
  }

  public void setEntryToF4ListSynch(Object synch) {
  }


//********************************************************************************************************************
//                          NavigationListener Stuff
//********************************************************************************************************************


//********************************************************************************************************************
//                          StandardEvents Stuff
//********************************************************************************************************************

  /**
   * Removes the registered StandardListener.
   *
   * @param l the listener that is to be removed
   *
   * @see ch.abacus.lib.ui.StandardListener
   */
  public void removeStandardListener(Object l) {
  }

  /**
   * Registers a new StandardListener Because the Editor is the only focusable component we hook the listener up to that
   * component.
   *
   * @param l the new StandarListener
   *
   * @see ch.abacus.lib.ui.StandardListener
   */
  public void addStandardListener(Object l) {
  }

  /**
   * Implementation of the StandardEvents interface
   *
   * @see ch.abacus.lib.ui.StandardEvents#abaDisplay
   */
  public void abaDisplay() {
  }


  /**
   * There are inherited classes which need to update their entrydata
   */

  protected void refreshEntryData() {
  }

  protected void abaDisplayMain() {
  }

  private boolean entryValueEqualsValue(Object entryValue, Object lookupEntryValue) {
      return false;
  }

  protected void abaDisplayLookupValue () {
  }

  /**
   * Implementation of the StandardEvents interface
   *
   * @see ch.abacus.lib.ui.StandardEvents#abaUpdate
   */
  public void abaUpdate() {
  }

  protected void abaUpdateLookup() {
  }

  /**
   * Implementation of the StandardEvents interface
   *
   * @see ch.abacus.lib.ui.StandardEvents#abaClear
   */
  public void abaClear() {
  }

  /**
   * Implementation of the StandardEvents interface
   *
   * @see ch.abacus.lib.ui.StandardEvents#abaValidate
   */
  public void abaValidate() {
  }

  /**
   * Implementation of the StandardEvents interface
   *
   * @see ch.abacus.lib.ui.StandardEvents#requestEnter
   */
  public void requestEnter(Component context) {
  }

  /**
   * Implementation of the StandardEvents interface
   *
   * @see ch.abacus.lib.ui.StandardEvents#requestExit
   */
  public void requestExit(Component context) {
  }






//********************************************************************************************************************
//    KeyListeners added to the Entry are fired from the editors
//********************************************************************************************************************

  /**
   * called by the Editor if a key is typed
   */
  public void fireKeyTyped(KeyEvent e) {
    if (mKeyListenerContainer == null) return;
    Iterator i = mKeyListenerContainer.iterator();
    while (i.hasNext()) {
      ((KeyListener) i.next()).keyTyped(e);
    }
  }

  /**
   * called by the Editor if a key is pressed
   */
  public void fireKeyPressed(KeyEvent e) {
  }

  /**
   * called by the Editor if a key is released
   */
  public void fireKeyReleased(KeyEvent e) {
  }

  /**
   * removes a keylistener from the list
   */
  public void removeKeyListener(KeyListener l) {
  }

  /**
   * adds a keylistener that reacts to keyevents of the editor
   */
  public void addKeyListener(KeyListener l) {
  }


//********************************************************************************************************************
//                          Validation Stuff
//********************************************************************************************************************

  /**
   * removes the registered FocusListener
   *
   * @param l the listener that is to be removed
   */
  public void removeFocusListener(FocusListener l) {
  }

  /**
   * register a new FocusListener
   *
   * @param l the new FocusListener
   */
  public void addFocusListener(FocusListener l) {
  }

  /**
   * Internal variable for the validation stuff
   */
  protected Object mEventDispatcher;

  /**
   * construct the internal EventDispatcher, only used by the validation stuff
   *
   * @return the instance of a dispatcher
   */
  public Object getDispatcher() {
    return null;
  }

//********************************************************************************************************************
//                          CellEditor Stuff
//********************************************************************************************************************

  /**
   * Flag to store whether this entry is used as a cell editor
   */
  protected boolean mIsCellEditor = false;

  /**
   * If this entry is a celleditor, we need to be able to get a parent for the F4-List. This table is the parent used
   */
  protected JTable mCellEditorParent;

  /**
   * Check if the entry is used as a cell editor
   *
   * @return true if this entry is used as a celleditor
   */
  public boolean isCellEditor() {
    return mIsCellEditor;
  }

  /**
   * Change this entry to a celleditor entry. DO NOT CALL THIS METHOD UNLESS YOU KNOW WHAT YOU ARE DOING. It is normally
   * not necessary to call this method by hand as the library classes take care of this
   *
   * @param isCellEditor set to true if this entry is to be used as a cell editor
   * @param table        the component to be used as parent
   */
  public void setIsCellEditor(boolean isCellEditor, JTable table) {
  }

//********************************************************************************************************************
//                          ValueChanged Stuff
//********************************************************************************************************************

  /**
   * set this to true if you want abaDisplay events to trigger entryModification events as well. default is false
   */
  public void setTiggerModificationEventsOnDisplay(boolean setting) {
    mAbaDisplayTriggersModifications = setting;
  }


//********************************************************************************************************************
//                          DataFlex Collating sequence in Strings Stuff
//********************************************************************************************************************
  /**
   * Returns whether strings are sorted using Uppercase sort. This only affects the sorting if DataFlex Sort is turned
   * on. Default is false.
   *
   * @return true if upper case sort is turned on
   */
  public boolean isUppercaseSort() {
    return mUppercaseSort;
  }

  /**
   * Sets whether strings are sorted using Uppercase sort. This only affects the sorting if DataFlex Sort is turned on.
   * Default is false.
   *
   * @param uppercaseSort true if uppercase sort is to be used
   */
  public void setUppercaseSort(boolean uppercaseSort) {
    mUppercaseSort = uppercaseSort;
  }

  /**
   * Returns true if DataFlex sorting is turned on. Default is true.
   *
   * @return true if DataFlex sorting is turned on
   */
  public boolean isUseDataFlexSort() {
    return mUseDataFlexSort;
  }

  /**
   * Sets whether the DataFlex sort is turned on.
   *
   * @param useDataFlexSort set to true to use the DataFlex sorting of strings
   */
  public void setUseDataFlexSort(boolean useDataFlexSort) {
    mUseDataFlexSort = useDataFlexSort;
  }

//********************************************************************************************************************
//                          AllowNavigationListener Stuff
//********************************************************************************************************************

//********************************************************************************************************************
//                          CompoundInterface implementation
//********************************************************************************************************************
  public Component getFirstFocusableComponent() {
    return null;
  }

/* *******************************************************************************************************************
 ValidationContainerInterface implementation
******************************************************************************************************************* */

  protected Object mContValHandler;

  public void setWeight(long weight) {
  }

  public long getWeight() {
      return 0;
  }

  public int getComponentPos(Component comp) {
      return 0;
  }


/* *******************************************************************************************************************
 KeyListener implementation to set the modified flag
******************************************************************************************************************* */


  public void keyPressed(KeyEvent e) {
  }

  public void keyReleased(KeyEvent e) {
  }

  /**
   * Check if this was an entry into the editor, if so set the modified flag
   */
  public void keyTyped(KeyEvent e) {
    mModified = true;
  }

  /**
   * returns whether the entry has been modified. This is the case if any keypresses were done, any browse operations,
   * etc.
   */
  public boolean isEntryModified() {
    return mModified;
  }

  public void setEntryModified(boolean modified) {
    mModified = modified;
  }

  public boolean isEntryValueModified() {
    return !mOldEntryValue.equals(getEntryValue());
  }

/* *******************************************************************************************************************
 misc. methods
******************************************************************************************************************* */

//************************************************************************************
  //Support for CFW Stuff
  //************************************************************************************

  public void abaDisplay(Object _Sender, Object _Parameters) {
    abaDisplay();
    fireCFWAbaDisplay(_Sender, _Parameters);
  }

  public String getDataColumnName() {
    return m_DataColumnName;
  }

  public void setDataColumnName(String _Name) {
    m_DataColumnName = _Name;
  }



  public void valueChanged(Object _Sender) {
    fireCFWComponentValueChanged(_Sender);

  }

  public void handleMessage(Object _Sender, Object _Message, Object _Parameters) {
    fireCFWHandleMessage(_Sender, _Message, _Parameters);
  }

  public boolean isModified() {
    return isEntryModified();
  }

  public void setModified(boolean _Value) {
    setEntryModified(_Value);
  }

  public void setAutoSynchronizeSameColumns(boolean _value) {
    m_AutoAbaDisplaySameColumns = _value;
  }

  public boolean getAutoSynchronizeSameColumns() {
    return m_AutoAbaDisplaySameColumns;
  }


  public void synchronizeSameColumns(Object _Parameter) {
  }



  public void fireCFWComponentValueChanged(Object _Sender) {
  }

  public void fireCFWAbaDisplay(Object _Sender, Object _Parameters) {
  }

  public void fireCFWHandleMessage(Object _Sender, Object _Message, Object _Parameters) {
  }

  public Object getMainDataSource() {
    return null;
  }

  public int getDataSourceCount() {
    return m_CFWDataSources.size();
  }

  public void setBackgroundEditor(Color bg) {
    m_BackgroundEditor = bg;
    setEditorColors();
  }

  public void setForegroundEditor(Color fg) {
    m_ForegroundEditor = fg;
    setEditorColors();
  }

  //If no background color is set then do nothing because the editor paints different if disabled, readonly etc etc
  public void setEditorColors() {
  }

  public Color getBackgroundEditor() {
    return m_BackgroundEditor;
  }

  public Color getForegroundEditor() {
    return m_ForegroundEditor;
  }

  //************************************************************************************
  // History Popup Stuff
  //************************************************************************************

  /**
   * <code>boolean</code> whether the history feature is enabled
   */
  private boolean m_HistoryEnabled = false;

  /**
   * Reference on the history textfield
   */
  private Object m_HistoryPopup;

  public void setHistroyEnabled(boolean enabled) {
  }

  /**
   * Returns whether the history feature is enabled.
   *
   * @return <code>boolean</code> whether history feature is enabled.
   */
  public boolean isHistoryEnabled(){
    return m_HistoryEnabled;
  }

  /************************************************************************************
  // Quicksearch Popup Stuff
  //************************************************************************************

  /**
   * <code>boolean</code> whether the quicksearch feature is enabled
   */
  protected boolean mQuickSearchEnabled = false;

  /**
   * the actual quicksearch object
   */
  protected Object mQuickSearch;

  /**
   * the identifier object that maintains the settings
   */
  protected Object mQuickSearchIdentifier;

  /**
   * Returns whether the quicksearch feature is enabled.
   *
   * @return <code>boolean</code> whether quicksearch feature is enabled.
   */
  public boolean isQuickSearchEnabled(){
    return mQuickSearchEnabled;
  }

  public void removeQuickSearch() {
  }

  /**
   * initialize the quicksearch on this entry
   */
  public void initQuickSearch(String alias, String file, String appChar, String fieldName) {
  }

  /**
   * adjust the quicksearch to use a different field (leave empty or null to search over all fields)
   */
  public void reInitQuickSearch(String fieldName) {
  }

  /**
   * adjust the quicksearch to use a different table and field
   */
  public void reInitQuickSearch(String alias, String file, String appChar, String fieldName) {
  }
}