/* 7zdate - Sync 7z file timestamp with content

   FIXME to work for files in /tmp - PathRewriter interferes

   Copyright (c) 2013-2023 Steven Levine and Associates, Inc.
   All rights reserved.

   2013-06-29 SHL Baseline - clone from elsewhere
   2023-02-05 SHL Convert to REXX
   2023-02-05 SHL Support seconds
   2023-02-07 SHL Use relative path names when possible
   2023-02-23 SHL Pass batch flags to SyncFileTimes
*/

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

gVersion = '0.2 2023-02-23'

Globals = 'gArgList. gBatch gCmdName gDbgLvl gEnv gVerbose gVersion'

call Initialize

Main:

  if 0 & trace() <> '?A' then trace '?A' ; nop	/* FIXME to be gone debug */

  parse arg cmdLine
  call ScanArgs cmdLine
  drop cmdLine

  do argNum = 1 to gArgList.0
    curArg = gArgList.argNum
    call DoWildArgs curArg
  end

  exit

/* end Main */

/*=== DoOneArg(fileInfo) Adjust one 7z file ===*/

DoOneArg: procedure expose (Globals)

  parse arg fileInfo

  parse var fileInfo oldFileDate oldFileTime fileBytes fileAttrib fileName
  fileName = strip(fileName)

  /* Try to use relative path name to shorten prompts and to
     avoid path rewriter conflicts for files in /tmp
  */
  cwd = translate(directory() || '\')
  if pos(cwd, translate(fileName)) = 1 then do
    shortName = substr(fileName, length(cwd) + 1)
    if \ IsFile(shortName) then
      call Die 'Cannot access' fileName 'by relative name' shortName
    fileName = shortName
  end

  oldDateTime = oldFileDate oldFileTime
  newDateTime = Get7zDateTime(fileName)

  if newDateTime == oldDateTime then do
    '7z l' Map7zFileName(fileName)
    say
    'dir /kmt' fileName
    say
    say 'No change needed for' fileName
  end
  else do
    newFileInfo = newDateTime fileBytes fileAttrib fileName
    say
    say fileName 'current date time is' oldDateTime
    if gBatch then
      mode = 'info batch'
    else
      mode = 'info'
    call SyncFileTimes newFileInfo, fileName, mode
    fileInfo = GetFileInfo(fileName)
    parse var fileInfo fileDate fileTime fileBytes fileAttrib fileName
    fileName = strip(fileName)
    fileDateTime = fileDate fileTime
    if fileDateTime \== oldDateTime then do
      say
      'dir /kmt' fileName
      say
      say 'Date time to the second is now' fileDateTime
    end

  end

  return

/* end DoOneArg */

/*=== DoWildArgs() Process wildcarded argument list ===*/

DoWildArgs: procedure expose (Globals)

  do argNum = 1 to gArgList.0
    wildCard = gArgList.argNum

    /* Format options F: files D: directories S: recurse */
    /* MM-DD-YY HH:MM Size ADHRS Name (default) */
    /* L: MM-DD-YYYY HH:MM Size ADHRS Name */
    /* O: Name only */
    /* T: YY/MM/DD/HH/MM Size ADHRS Name */
    /* TL: YYYY-MM-DD HH:MM:SS Size ADHRS Name */
    call SysFileTree wildCard, 'fileList', 'FL'
    if RESULT \= 0 then
      call Die 'SysFileTree' wildCard 'failed'
    if fileList.0 = 0 then
      call ScanArgsUsage 'No matches for' wildCard
    do fileNum = 1 to fileList.0
      call DoOneArg fileList.fileNum
    end /* fileNum */
  end /* argNum */

  return

/* end DoWildArgs */

/*=== Get7zDateTime(fileName) Get 7z timestamp from content ===*/

Get7zDateTime: procedure expose (Globals)

  parse arg fileName

  /* Extract timestamped lines */
  cmd = '7z l' Map7zFileName(fileName) '| grep "^....-..-.. ..:..:.." | rxqueue'
  call RunExtCmdNoDie cmd
  if RESULT \= 0 then
    call Die cmd 'return error' RESULT

  /* Try to select newest */
  selected = ''
  do while queued() > 0
    parse pull line
    if line > selected then
      selected = line
  end

  if selected == '' then
    call Die fileName 'does not appear to be a valid 7z file'

  /* 7z l outputs:
	 Date      Time    Attr         Size   Compressed  Name
      2021-10-10 13:53:58 ....A       770432       311241  anpm.exe

	0123456789012345679
     Have yyyy-mm-dd hh:mm:ss
     Convert to dd-mm-yy hh:mm:ss
  */
  newDateTime = word(selected, 1) word(selected, 2)

  return newDateTime

/* end Get7zDateTime */

/*=== Map7zFileName(arg) Map file name to what 7z expects ===*/

Map7zFileName: procedure expose (Globals)

  parse arg fileName

  /* Try to use relative path name to
     avoid path rewriter conflicts for files in /tmp
     Map slashes for 7z
  */

  cwd = translate(directory() || '\')
  if pos(cwd, translate(fileName)) = 1 then do
    shortName = substr(fileName, length(cwd) + 1)
    if \ IsFile(shortName) then
      call Die 'Cannot access' fileName 'by relative name' shortName
    fileName = shortName
  end

  if substr(fileName, 2, 1) == ':' then
    fileName = substr(fileName, 3)

  fileName = translate(fileName, '/', '\')

  return fileName

/* end Map7zFileName */

/*=== Initialize() Initialize globals ===*/

Initialize: procedure expose (Globals)
  call SetCmdName
  call LoadRexxUtil
  gEnv = 'OS2ENVIRONMENT'
  return

/* end Initialize */

/*=== ScanArgsInit() ScanArgs initialization exit routine ===*/

ScanArgsInit: procedure expose (Globals) cmdTail swCtl keepQuoted

  if cmdTail == '' then
    call ScanArgsHelp

  /* Preset defaults */
  gBatch = 0				/* Batch mode flag */
  gDbgLvl = 0				/* Debug mode level */
  gVerbose = 0				/* Verbose mode level */
  gArgList.0 = 0			/* Argument count */

  return

/* end ScanArgsInit */

/*=== ScanArgsSwitch() ScanArgs switch option exit routine ===*/

ScanArgsSwitch: procedure expose (Globals) curSw curSwArg

  select
  when curSw == 'b' then
    gBatch = 1
  when curSw == 'd' then
    gDbgLvl = gDbgLvl + 1
  when curSw == 'h' | curSw == '?' then
    call ScanArgsHelp
  when curSw == 'v' then
    gVerbose = gVerbose + 1
  when curSw == 'V' then do
    say
    say gCmdName gVersion
    exit
  end
  otherwise
    call ScanArgsUsage 'Switch "-' || curSw || '" unexpected'
  end /* select */

  return

/* end ScanArgsSwitch */

/*=== ScanArgsArg() ScanArgs argument option exit routine ===*/

ScanArgsArg: procedure expose (Globals) curArg

  if \ IsFile(curArg) then
    call ScanArgsUsage 'Cannot access' curArg

  i = gArgList.0 + 1
  gArgList.i = curArg
  gArgList.0 = i

  return

/* end ScanArgsArg */

/*=== ScanArgsTerm() ScanArgs scan end exit routine ===*/

ScanArgsTerm: procedure expose (Globals)

  /* FIXME
    say Checking for .7z files dated yesterday or today
    for /[d-1] fileName in ( *.7z ) ( gosub fixdate fileName %+ if %? ge 2 cancel )
    cmd = '@echo off %+ dir /b /[d-1] *.7z | rxqueue'
    call RunIntCmdNoDie cmd
    do while queued() > 0
      parse pull s
    end
  */

  if gArgList.0 = 0 then
    call ScanArgsUsage 'At least one filespec required'

  return

/* end ScanArgsTerm */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'Sync 7z file timestamp with content.'
  say
  say 'Usage:' gCmdName '[-b] [-d] [-h] [-v] [-V] [-?] filespec...'
  say
  say '  -b           Run in batch mode'
  say '  -d           Run in debug mode, repeat for more verbosity'
  say '  -h -?        Display this message'
  say '  -v           Run in verbose mode, repeat for more verbosity'
  say '  -V           Display version number and quit'
  say
  say '  filespec  7z file to adjust, wildcards supported'
  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report ScanArgs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-b] [-d] [-h] [-v] [-V] [-?] filespec...'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== AskYNQ([prompt][, noskip[, nofocus]])) returns 0=Yes, 1=No, 2=Quit, skips line unless noskip ===*/

AskYNQ: procedure
  parse arg msg, noskip, nofocus

  /* Take focus with 4OS2 or fail if cannot match window title */
  /* If script designed for CMD too, use nofocus arg to avoid error noise */
  signal off Error
  /* Map 1st left bracket to wild card - [pid] seems to confuse activate */
  if nofocus = '' | nofocus \= 1 then
    '@if defined _WINTITLE activate "%@replace[[,*,%_WINTITLE]"'
  signal on Error

  /* Skip line unless suppressed by noskip arg - any non-zero value requests noskip */
  if noskip = '' | noskip = 0 then
    call lineout 'STDERR', ''

  if msg == '' then
    msg = 'Continue'
  call charout 'STDERR', msg '(y/n/q) ? '
  do forever
    key = translate(SysGetKey('NOECHO'))
    if key == 'Y' | key == 'N' then do
      call lineout 'STDERR', key
      if key == 'Y' then
	ynq = 0
      else
	ynq = 1
      leave
    end
    if key == 'Q' | c2x(key) == '1B' then do
      call lineout 'STDERR', ''
      ynq = 2
      leave
    end
  end /* forever */
  return ynq

/* end AskYNQ */

/*=== GetEnv(var) Return value for environment variable or empty string ===*/

GetEnv: procedure expose (Globals)
  parse arg var
  if var = '' then
    call Die 'GetEnv requires an argument'
  return value(var,, gEnv)

/* end GetEnv */

/*=== GetFileInfo(fileName[, option]) Get info for one file in SysFileTree format ===*/

/**
 * options is SysFileTree format option, defaults to FTL
 */

GetFileInfo: procedure expose (Globals)

  parse arg fileName, option

  /* Z: MM-DD-YY HH:MM Size ADHRS Name (default) */
  /* L: MM-DD-YYYY HH:MM Size ADHRS Name */
  /* O: Name only */
  /* T: YY/MM/DD/HH/MM Size ADHRS Name */
  /* TL: YYYY-MM-DD HH:MM:SS Size ADHRS Name */

  if option == 'Z' then
    option = ''				/* Force SysFileTree default */
  if option == '' then
    option = 'FTL'			/* Force GetFileInfo default */

  call SysFileTree fileName, 'fileList', option
  if RESULT \= 0 then
    call Die 'SysFileTree' fileName 'failed'
  if fileList.0 \= 1 then
    call File 'Expected SysFileTree' fileName 'to match exactly one file - matched' fileList.0

  return fileList.1

/* end GetFileInfo */

/*=== GetFileName(pathName) Return file name stripping drive and directory ===*/

GetFileName: procedure
  parse arg s
  /* returns file name with with drive and directory stripped */
  s = filespec('N', s)
  return s
/* end GetFileName */

/*=== IsFile(file) return true if arg is file and file exists ===*/

IsFile: procedure expose (Globals)
  parse arg file
  if file == '' then
    yes = 0
  else do
    /* '.' and '..' returns files in '.' or '..' - so avoid false positives */
    call SysFileTree file, 'files', 'F'
    if RESULT \= 0 then
      call Die 'SysFileTree' file 'failed'
    /* Assume caller knows if arg contains wildcards */
    yes = file \== '.' & file \== '..' & files.0 \= 0
  end
  return yes

/* end IsFile */

/*=== RunExtCmdNoDie(cmd, die) Run external command that sets RC ===*/

RunExtCmdNoDie: procedure expose (Globals) SIGL

  /* Requires GetEnv */

  /* Return error code on error unless overridden */
  parse arg cmd, die

  if cmd = '' then
    call Die 'Required cmd omitted at' SIGL

  if pos('echo off', cmd) = 0 then do
    say
    '@echo on'
  end
  signal off Error
  cmd
  signal on Error
  '@echo off'

  /* if piped cmd of form ( extcmd %+ set E=%_? | rxqueue ) */
  if RC = 0 & (pos('E=%?', cmd) > 0 | pos('E=%_?', cmd) > 0 )then
    RC = GetEnv('E')			/* Retrieve status */

  if RC \= 0 then do
    /* Return error code unless die requested */
    die = die = 1 | die == 'die' & die \== ''
    if die then
      call Die '', cmd 'failed with error' RC
  end

  return RC

  return

/* end RunExtCmdNoDie */

/*=== SyncFileTimes(fileName1, fileName2[, mode]) Sync file timestamps ===*/

/**
 * Sync fileName2 timestamp and archive attribute with fileName1
 * @param fileName1 is base file
 * @param fileName2 is file to have timestamp synchronized
 * @param mode is one or more space separated operating mode modifier keywords
 * @note mode batch suppresses prompting
 * @note mode info indicates fileName1 argument is a SysFileTree TL file info string
 */

SyncFileTimes: procedure expose (Globals)

  parse arg fileName1, fileName2, mode

  batch = pos('batch', mode) > 0	/* Map to true/false */
  info = pos('info', mode) > 0		/* Map to true/false */

  /* Get base info */
  if info then do
    fileList.1 = fileName1
    fileList.0 = 1
  end
  else do
    call SysFileTree fileName1, 'fileList', 'FTL'
    if RESULT \= 0 then
      call Die 'SysFileTree' fileName1 'failed'
    if fileList.0 > 1 then
      call Die fileName1 'must match exactly one file'
    else if fileList.0 = 0 then
      call Die 'Cannot access' fileName1
  end
  parse var fileList.1 fileDate1 fileTime1 fileBytes1 fileAttrib1 fileName1
  fileName1 = strip(fileName1)

  /* Get target file info */
  call SysFileTree fileName2, 'fileList', 'FTL'
  if RESULT \= 0 then
    call Die 'SysFileTree' fileName2 'failed'
  if fileList.0 > 1 then
    call Die fileName2 'must match exactly one file'
  else if fileList.0 = 0 then
    call Die 'Cannot access' fileName2
  parse var fileList.1 fileDate2 fileTime2 fileBytes2 fileAttrib2 fileName2
  fileName2 = strip(fileName2)

  if fileDate1 fileTime1 \== fileDate2 fileTime2 then do
    if batch then
      RESULT = 0
    else do

      /* Try to use relative path to shorten prompts */
      cwd = translate(directory() || '\')
      if pos(cwd, translate(fileName2)) = 1 then do
	shortName = substr(fileName2, length(cwd) + 1)
	if \ IsFile(shortName) then
	  call Die 'Cannot access' fileName2 'by relative name' shortName
	fileName2 = shortName
      end

      call AskYNQ 'Set' fileName2 'timestamp to' fileDate1 fileTime1
    end
    if RESULT >= 2 then exit
    if RESULT == 0 then do
      err = SysSetFileDateTime(fileName2, fileDate1, fileTime1)
      if err \= 0 then do
	if batch then
	  call Die 'SysSetFileDate' fileName2 'failed with error' err
	call AskYNQ 'SysSetFileDate failed with error' err '- continue'
	if RESULT \= 0 then exit
      end
      /* Sync archive attribute too */
      if left(fileAttrib1, 1) \== 'A' then do
	call SysFileTree fileName2, 'fileList', 'O', '+****', '-****'
	if RESULT \= 0 then
	  call Die 'SysFileTree' fileName2 'set attributes failed'
      end
    end
  end

  return

/* end SyncFileTimes */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Die([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting at line' SIGL || '.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end Die */

/*=== Error() Set gErrCondition; report to STDOUT; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  say gErrCondition 'signaled at line' SIGL 'of' cmd || '.'
  if condition('D') \== '' then
    say 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    say 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    say 'RC =' RC || '.'
  say 'Source =' sourceline(SIGL)

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end

  return

/* end Error */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/*=== ScanArgs(cmdLine) Scan command line ===*/

ScanArgs: procedure expose (Globals)

  /* Calls user exits to process arguments and switches */

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  call ScanArgsInit

  /* Ensure optional settings initialized */
  if symbol('SWCTL') \== 'VAR' then
    swCtl = ''				/* Switches that take args, append ? if optional */
  if symbol('KEEPQUOTED') \== 'VAR' then
    keepQuoted = 0			/* Set to 1 to keep arguments quoted */

  /* Scan */
  curArg = ''				/* Current arg string */
  curSwList = ''			/* Current switch list */
  /* curSwArg = '' */			/* Current switch argument, if needed */
  noMoreSw = 0				/* End of switches */

  do while cmdTail \== '' | curArg \== '' | curSwList \== ''

    /* If arg buffer empty, refill */
    if curArg == '' then do
      qChar = left(cmdTail, 1)		/* Remember quote */
      if \ verify(qChar,'''"', 'M') then
	parse var cmdTail curArg cmdTail	/* Not quoted */
      else do
	/* Arg is quoted */
	curArg = ''
	do forever
	  /* Parse dropping quotes */
	  parse var cmdTail (qChar)quotedPart(qChar) cmdTail
	  curArg = curArg || quotedPart
	  /* Check for escaped quote within quoted string (i.e. "" or '') */
	  if left(cmdTail, 1) \== qChar then do
	    cmdTail = strip(cmdTail)	/* Strip leading whitespace */
	    leave			/* Done with this quoted arg */
	  end
	  curArg = curArg || qChar	/* Append quote */
	  if keepQuoted then
	    curArg = curArg || qChar	/* Append escaped quote */
	  parse var cmdTail (qChar) cmdTail	/* Strip quote */
	end /* do forever */
	if keepQuoted then
	  curArg = qChar || curArg || qChar	/* requote */
      end /* if quoted */
    end /* if curArg empty */

    /* If switch buffer empty, refill */
    if curSwList == '' & \ noMoreSw then do
      if left(curArg, 1) == '-' & curArg \== '-' then do
	if curArg == '--' then
	  noMoreSw = 1
	else
	  curSwList = substr(curArg, 2)	/* Remember switch string */
	curArg = ''			/* Mark empty */
	iterate				/* Refill arg buffer */
      end /* if switch */
    end /* if curSwList empty */

    /* If switch in progress */
    if curSwList \== '' then do
      curSw = left(curSwList, 1)	/* Next switch */
      curSwList = substr(curSwList, 2)	/* Drop from pending */
      /* Check switch allows argument, avoid matching ? */
      if pos(curSw, translate(swCtl,,'?')) \= 0 then do
	if curSwList \== '' then do
	  curSwArg = curSwList		/* Use rest of switch string for switch argument */
	  curSwList = ''
	end
	else if curArg \== '' & left(curArg, 1) \== '-' then do
	  curSwArg = curArg		/* Arg string is switch argument */
	  curArg = ''			/* Mark arg string empty */
	end
	else if pos(curSw'?', swCtl) = 0 then
	  call ScanArgsUsage 'Switch "-' || curSw || '" requires an argument'
	else
	  curSwArg = ''			/* Optional arg omitted */
      end

      call ScanArgsSwitch		/* Passing curSw and curSwArg */
      drop curSwArg			/* Must be used by now */
    end /* if switch */

    /* If arg */
    else if curArg \== '' then do
      noMoreSw = 1
      call ScanArgsArg			/* Passing curArg */
      curArg = ''
    end

  end /* while not done */

  call ScanArgsTerm

  return

/* end ScanArgs */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/* eof */
