unit REWads;
{$I-}
interface
uses
  REGraph;

type
  PWadEntry=^TWadEntry;
  TWadEntry=record
    Offset:longint;
    Size:longint;
    Name:array[0..8] of char;
  end;

  PWadDirectory=^TWadDirectory;
  TWadDirectory=record
    NumEntries:longint;
    Entries:array[0..0] of PWadEntry;
  end;

  PWall=^TWall;
  TWall=array[0..4095] of byte; { 64 * 64 }

  PFloor=^TFloor;
  TFloor=record
    Width:integer;
    Height:integer;
    DeltaX:integer;
    DeltaY:integer;
    Data:array[0..16383] of byte; { 128 * 128 }
  end;

  PSky=^TSky;
  TSky=array[0..51199] of byte; { 256 * 200 }

  PPicture=^TPicture;
  TPicture=record
    OrigSize:integer;
    Width:integer;
    Height:integer;
    DeltaX:integer;
    DeltaY:integer;
    Columns:array[0..319] of word;
    Data:array[0..0] of byte;
  end;

  PTransPicture=^TTransPicture;
  TTransPicture=record
    OrigSize:integer;
    Width:integer;
    Height:integer;
    DeltaX:integer;
    DeltaY:integer;
    TransLevel:integer;
    Columns:array[0..319] of word;
    Data:array[0..0] of byte;
  end;

function CreateWadDirectory(NumEntries:longint):PWadDirectory;
procedure DeleteWadDirectory(Dir:PWadDirectory);
function ReadWadDirectory(Filename:PChar):PWadDirectory;
function ReadWadLump(Filename:PChar; Offset,Size:longint):pointer;
function ReadWall(Filename:PChar; Offset,Size:longint):PBitmap;
function ReadFloor(Filename:PChar; Offset,Size:longint):PBitmap;
function ReadSky(Filename:PChar; Offset,Size:longint):PBitmap;
function ReadPicture(Filename:PChar; Offset,Size:longint):PBitmap;
function ReadTransPicture(Filename:PChar; Offset,Size:longint):PBitmap;

implementation

const
  IWADId=$44415749; { "IWAD" }

function CreateWadDirectory(NumEntries:longint):PWadDirectory;
var
  WD:PWadDirectory;
begin
  GetMem(WD,4+NumEntries*4);
  FillChar(WD^,4+NumEntries*4,0);
  WD^.NumEntries:=NumEntries;
  CreateWadDirectory:=WD;
end;

procedure DeleteWadDirectory(Dir:PWadDirectory);
var
  I:longint;
begin
  if Dir=nil then
    Exit;
  for I:=0 to Dir^.NumEntries-1 do
    if Dir^.Entries[I]<>nil then
      Dispose(Dir^.Entries[I]);
  FreeMem(Dir,4+Dir^.NumEntries*4);
end;

function ReadWadDirectory(Filename:PChar):PWadDirectory;
var
  F:file;
  FileId,NumEntries,DirOffset,I:longint;
  WD:PWadDirectory;
begin
  ReadWadDirectory:=nil;
  Assign(F,Filename);
  Reset(F,1);
  if IOResult<>0 then begin
    WriteLn('Error opening file ''',Filename,'''.');
    Exit;
  end;

  BlockRead(F,FileId,4);
  BlockRead(F,NumEntries,4);
  BlockRead(F,DirOffset,4);
  if IOResult<>0 then begin
    WriteLn('Error reading from file ''',Filename,'''.');
    Close(F);
    Exit;
  end;

  if FileId<>IWADId then begin
    WriteLn('Invalid file format.');
    Close(F);
    Exit;
  end;

  Seek(F,DirOffset);
  WD:=CreateWadDirectory(NumEntries);
  for I:=0 to NumEntries-1 do begin
    New(WD^.Entries[I]);
    BlockRead(F,WD^.Entries[I]^,16);
    if IOResult<>0 then begin
      WriteLn('Error reading from file ''',Filename,'''.');
      DeleteWadDirectory(WD);
      Close(F);
      Exit;
    end;
    WD^.Entries[I]^.Name[8]:=#0;
  end;

  Close(F);
  ReadWadDirectory:=WD;
end;

function ReadWadLump(Filename:PChar; Offset,Size:longint):pointer;
var
  F:file;
  Data:pointer;
begin
  ReadWadLump:=nil;
  Assign(F,Filename);
  Reset(F,1);
  if IOResult<>0 then
    Exit;

  Seek(F,Offset);
  GetMem(Data,Size);
  BlockRead(F,Data^,Size);
  if IOResult<>0 then begin
    FreeMem(Data,Size);
    Close(F);
    Exit;
  end;

  Close(F);
  ReadWadLump:=Data;
end;

function ReadWall(Filename:PChar; Offset,Size:longint):PBitmap;
var
  Wall:PWall;
  Bitmap:PBitmap;
  I,J:integer;
begin
  ReadWall:=nil;
  Wall:=ReadWadLump(Filename,Offset,Size);
  if Wall=nil then
    Exit;

  Bitmap:=CreateBitmap(64,64);
  for I:=0 to 63 do
    for J:=0 to 63 do
      Bitmap^.Data[J*64+I]:=Wall^[I*64+J];
  FreeMem(Wall,Size);
  ReadWall:=Bitmap;
end;

function ReadFloor(Filename:PChar; Offset,Size:longint):PBitmap;
var
  Floor:PFloor;
  Bitmap:PBitmap;
  I,J:integer;
begin
  ReadFloor:=nil;
  Floor:=ReadWadLump(Filename,Offset,Size);
  if Floor=nil then
    Exit;

  Bitmap:=CreateBitmap(128,128);
  for I:=0 to 127 do
    for J:=0 to 127 do
      Bitmap^.Data[J*128+I]:=Floor^.Data[I*128+J];
  FreeMem(Floor,Size);
  ReadFloor:=Bitmap;
end;

function ReadSky(Filename:PChar; Offset,Size:longint):PBitmap;
var
  Sky:PSky;
  Bitmap:PBitmap;
  I,J:integer;
begin
  ReadSky:=nil;
  Sky:=ReadWadLump(Filename,Offset,Size);
  if Sky=nil then
    Exit;

  Bitmap:=CreateBitmap(256,200);
  for I:=0 to 255 do
    for J:=0 to 199 do
      Bitmap^.Data[J*256+I]:=Sky^[I*200+J];
  FreeMem(Sky,Size);
  ReadSky:=Bitmap;
end;

function ReadPicture(Filename:PChar; Offset,Size:longint):PBitmap;
var
  Picture:PPicture;
  Bitmap:PBitmap;
  I,J,K:word;
  RowStart,NumPixels:byte;
begin
  ReadPicture:=nil;
  Picture:=ReadWadLump(Filename,Offset,Size);
  if Picture=nil then
    Exit;

  Bitmap:=CreateBitmap(Picture^.Width,Picture^.Height);
  for I:=0 to Bitmap^.Width-1 do begin
    K:=Picture^.Columns[I]-650;
    RowStart:=Picture^.Data[K];
    while RowStart<>255 do begin
      K:=K+1;
      NumPixels:=Picture^.Data[K];
      if RowStart+NumPixels>Bitmap^.Height then begin { voor de zekerheid }
        FreeMem(Picture,Size);
        DeleteBitmap(Bitmap);
        Exit;
      end;
      for J:=0 to NumPixels-1 do begin
        K:=K+1;
        Bitmap^.Data[(J+RowStart)*Bitmap^.Width+I]:=Picture^.Data[K];
      end;
      K:=K+1;
      RowStart:=Picture^.Data[K];
    end;
  end;

  FreeMem(Picture,Size);
  ReadPicture:=Bitmap;
end;

function ReadTransPicture(Filename:PChar; Offset,Size:longint):PBitmap;
var
  TransPicture:PTransPicture;
  Bitmap:PBitmap;
  I,J,K:integer;
  RowStart,NumPixels:byte;
begin
  ReadTransPicture:=nil;
  TransPicture:=ReadWadLump(Filename,Offset,Size);
  if TransPicture=nil then
    Exit;

  Bitmap:=CreateBitmap(TransPicture^.Width,TransPicture^.Height);
  for I:=0 to Bitmap^.Width-1 do begin
    K:=TransPicture^.Columns[I]-652;
    RowStart:=TransPicture^.Data[K];
    while RowStart<>255 do begin
      K:=K+1;
      NumPixels:=TransPicture^.Data[K];
      if RowStart+NumPixels>Bitmap^.Height then begin
        FreeMem(TransPicture,Size);
        DeleteBitmap(Bitmap);
        Exit;
      end;
      if TransPicture^.Data[K+1]=254 then begin
        K:=K+1;
        for J:=0 to NumPixels-1 do
          Bitmap^.Data[(J+RowStart)*Bitmap^.Width+I]:=32;
      end else
        for J:=0 to NumPixels-1 do begin
          K:=K+1;
          Bitmap^.Data[(J+RowStart)*Bitmap^.Width+I]:=TransPicture^.Data[K];
        end;
      K:=K+1;
      RowStart:=TransPicture^.Data[K];
    end;
  end;

  FreeMem(TransPicture,Size);
  ReadTransPicture:=Bitmap;
end;

begin
end.
