unit REGraph;
interface

var
  VideoSegment:word;

const
  mb_Left=$0001;
  mb_Right=$0002;
  mb_Middle=$0004;

  ke_Nothing=0;
  ke_Esc=$001B;
  ke_F1=$3B00;
  ke_F2=$3C00;
  ke_F3=$3D00;
  ke_F4=$3E00;
  ke_F5=$3F00;
  ke_F6=$4000;
  ke_F7=$4100;
  ke_F8=$4200;
  ke_SF8=$5B00;
  ke_CF8=$6500;
  ke_AF8=$6F00;
  ke_F9=$4300;
  ke_F10=$4400;
  ke_Up=$4800;
  ke_PgUp=$4900;
  ke_Left=$4B00;
  ke_Right=$4D00;
  ke_Down=$5000;
  ke_PgDn=$5100;
  ke_Ins=$5200;

  me_Nothing=0;
  me_Move=1;
  me_LeftDown=2;
  me_LeftUp=3;
  me_RightDown=4;
  me_RightUp=5;
  me_MiddleDown=6;
  me_MiddleUp=7;

type
  PRect=^TRect;
  TRect=record
    X:integer;
    Y:integer;
    W:integer;
    H:integer;
  end;

  PPalette=^TPalette;
  TPalette=array[0..767] of byte;

  PFont=^TFont;
  TFont=record
    Width:integer;
    Height:integer;
    Data:array[0..0] of byte;
  end;

  PBitmap=^TBitmap;
  TBitmap=record
    Width:integer;
    Height:integer;
    Data:array[0..0] of byte;
  end;

procedure InitGraphics;
procedure DoneGraphics;
procedure SetRect(var R:TRect; X,Y,W,H:integer);
function LoadPalette(Filename:PChar):PPalette;
procedure DeletePalette(Palette:PPalette);
function SetPalette(Palette:PPalette):PPalette;
function RGB(Red,Green,Blue:byte):byte;
function CreateFont(Width,Height:integer):PFont;
function LoadFont(Filename:PChar):PFont;
procedure DeleteFont(Font:PFont);
function SetFont(Font:PFont):PFont;
function CreateBitmap(Width,Height:integer):PBitmap;
function LoadBitmap(Filename:PChar):PBitmap;
procedure DeleteBitmap(Bitmap:PBitmap);
procedure SetClipRect(X,Y,W,H:integer);
procedure ClearScreen(Color:byte);
procedure Pixel(X,Y:integer; Color:byte);
procedure Rectangle(X,Y,W,H:integer; Color:byte);
procedure WriteChar(X,Y:integer; Ch:char);
procedure WriteStr(X,Y:integer; Str:PChar);
procedure DrawBitmap(X,Y:integer; Bitmap:PBitmap);
procedure ShowMouse;
procedure HideMouse;
procedure GetMouseState(var Buttons,X,Y:word);
function GetKeyEvent:word;
function GetMouseEvent:word;
procedure ClearEvents;

implementation
uses
  Crt,WinDos,Strings;

var
  Regs:TRegisters;
  CurClipRect:TRect;
  CurPalette:PPalette;
  CurFont:PFont;
  CurButtons,CurX,CurY:word;

procedure InitGraphics;
begin
  Regs.AX:=$0013;
  Intr($10,Regs);

  Regs.AX:=0;
  Regs.BX:=0;
  Intr($33,Regs);
  Regs.AX:=13;
  Intr($33,Regs);
end;

procedure DoneGraphics;
begin
  Regs.AX:=$0003;
  Intr($10,Regs);
end;

procedure SetRect(var R:TRect; X,Y,W,H:integer);
begin
  R.X:=X;
  R.Y:=Y;
  R.W:=W;
  R.H:=H;
end;

function LoadPalette(Filename:PChar):PPalette;
var
  F:file;
  Palette:PPalette;
  I:integer;
begin
  LoadPalette:=nil;
  Assign(F,Filename);
  Reset(F,1);
  if IOResult<>0 then
    Exit;

  New(Palette);
  BlockRead(F,Palette^,SizeOf(TPalette));
  if IOResult<>0 then begin
    Dispose(Palette);
    Close(F);
    Exit;
  end;

  for I:=0 to SizeOf(TPalette)-1 do
    Palette^[I]:=Palette^[I] div 4;
  Close(F);
  LoadPalette:=Palette;
end;

procedure DeletePalette(Palette:PPalette);
begin
  if Palette<>nil then
    Dispose(Palette);
end;

function SetPalette(Palette:PPalette):PPalette;
type
  TPointer=record
    Ofs,Seg:word;
  end;
begin
  if Palette<>nil then begin
    Regs.AX:=$1012;
    Regs.BX:=0;
    Regs.CX:=256;
    Regs.ES:=TPointer(Palette).Seg;
    Regs.DX:=TPointer(Palette).Ofs;
    Intr($10,Regs);
  end;

  SetPalette:=CurPalette;
  CurPalette:=Palette;
end;

function RGB(Red,Green,Blue:byte):byte;
var
  D,I:byte;
begin
  if CurPalette=nil then
    Exit;

  for D:=0 to 255 do
    for I:=0 to 255 do
      if (Abs(Red-CurPalette^[I*3])<=D) and
        (Abs(Green-CurPalette^[I*3+1])<=D) and
        (Abs(Blue-CurPalette^[I*3+2])<=D) then begin
          RGB:=I;
          Exit;
        end;
end;

function CreateFont(Width,Height:integer):PFont;
var
  Font:PFont;
begin
  GetMem(Font,4+Width*16*Height*16);
  Font^.Width:=Width;
  Font^.Height:=Height;
  FillChar(Font^.Data,Width*16*Height*16,255);
  CreateFont:=Font;
end;

function LoadFont(Filename:PChar):PFont;
var
  F:file;
  W,H:integer;
  Font:PFont;
begin
  LoadFont:=nil;
  Assign(F,Filename);
  Reset(F,1);
  if IOResult<>0 then
    Exit;

  BlockRead(F,W,2);
  BlockRead(F,H,2);
  if IOResult<>0 then begin
    Close(F);
    Exit;
  end;

  Font:=CreateFont(W,H);
  BlockRead(F,Font^.Data,W*16*H*16);
  if IOResult<>0 then begin
    DeleteFont(Font);
    Close(F);
    Exit;
  end;

  Close(F);
  LoadFont:=Font;
end;

procedure DeleteFont(Font:PFont);
begin
  if Font<>nil then
    FreeMem(Font,4+Font^.Width*16*Font^.Height*16);
end;

function SetFont(Font:PFont):PFont;
begin
  SetFont:=CurFont;
  CurFont:=Font;
end;

function CreateBitmap(Width,Height:integer):PBitmap;
var
  Bitmap:PBitmap;
begin
  GetMem(Bitmap,4+Width*Height);
  Bitmap^.Width:=Width;
  Bitmap^.Height:=Height;
  FillChar(Bitmap^.Data,Width*Height,255);
  CreateBitmap:=Bitmap;
end;

function LoadBitmap(Filename:PChar):PBitmap;
var
  F:file;
  Bitmap:PBitmap;
begin
  LoadBitmap:=nil;
  Assign(F,Filename);
  Reset(F,1);
  if IOResult<>0 then
    Exit;

  GetMem(Bitmap,FileSize(F));
  BlockRead(F,Bitmap^,FileSize(F));
  if IOResult<>0 then begin
    FreeMem(Bitmap,FileSize(F));
    Close(F);
    Exit;
  end;

  Close(F);
  LoadBitmap:=Bitmap;
end;

procedure DeleteBitmap(Bitmap:PBitmap);
begin
  if Bitmap<>nil then
    FreeMem(Bitmap,4+Bitmap^.Width*Bitmap^.Height);
end;

procedure SetClipRect(X,Y,W,H:integer);
begin
  SetRect(CurClipRect,X,Y,W,H);
end;

procedure ClearScreen(Color:byte);
begin
  FillChar(mem[VideoSegment:$0000],64000,Color);
end;

procedure Pixel(X,Y:integer; Color:byte);
begin
  if (X>=CurClipRect.X) and (Y>=CurClipRect.Y) and
    (X<CurClipRect.X+CurClipRect.W) and (Y<CurClipRect.Y+CurClipRect.H) then
      mem[VideoSegment:Y*320+X]:=Color;
end;

procedure Rectangle(X,Y,W,H:integer; Color:byte);
var
  I:integer;
begin
  if (Y>=CurClipRect.Y) and (Y<CurClipRect.Y+CurClipRect.H) then
    for I:=0 to W-1 do
      if (X+I>=CurClipRect.X) and (X+I<CurClipRect.X+CurClipRect.W) then
        mem[VideoSegment:Y*320+X+I]:=Color;
  if (Y+H-1>=CurClipRect.Y) and (Y+H-1<CurClipRect.Y+CurClipRect.H) then
    for I:=0 to W-1 do
      if (X+I>=CurClipRect.X) and (X+I<CurClipRect.X+CurClipRect.W) then
        mem[VideoSegment:(Y+H-1)*320+X+I]:=Color;
  if (X>=CurClipRect.X) and (X<CurClipRect.X+CurClipRect.W) then
    for I:=0 to H-1 do
      if (Y+I>=CurClipRect.Y) and (Y+I<CurClipRect.Y+CurClipRect.H) then
        mem[VideoSegment:(Y+I)*320+X]:=Color;
  if (X+W-1>=CurClipRect.X) and (X+W-1<CurClipRect.X+CurClipRect.W) then
    for I:=0 to H-1 do
      if (Y+I>=CurClipRect.Y) and (Y+I<CurClipRect.Y+CurClipRect.H) then
        mem[VideoSegment:(Y+I)*320+X+W-1]:=Color;
end;

procedure WriteChar(X,Y:integer; Ch:char);
var
  FX,FY:integer;
  I,J:integer;
  C:byte;
begin
  if CurFont=nil then
    Exit;

  FX:=Byte(Ch) mod 16*CurFont^.Width;
  FY:=Byte(Ch) div 16*CurFont^.Height;
  for I:=0 to CurFont^.Width-1 do
    for J:=0 to CurFont^.Height-1 do
      if (X+I>=CurClipRect.X) and (X+I<CurClipRect.X+CurClipRect.W) and
        (Y+J>=CurClipRect.X) and (Y+J<CurClipRect.Y+CurClipRect.H) then begin
          C:=CurFont^.Data[(J+FY)*CurFont^.Width*16+I+FX];
          if C<>255 then
            mem[VideoSegment:(J+Y)*320+I+X]:=C;
        end;
end;

procedure WriteStr(X,Y:integer; Str:PChar);
var
  I:integer;
begin
  if (CurFont=nil) or (Str=nil) or (StrLen(Str)=0) then
    Exit;

  for I:=0 to StrLen(Str)-1 do
    WriteChar(X+I*CurFont^.Width,Y,Str[I]);
end;

procedure DrawBitmap(X,Y:integer; Bitmap:PBitmap);
var
  I,J:integer;
  C:byte;
begin
  if Bitmap=nil then
    Exit;

  for I:=0 to Bitmap^.Width-1 do
    for J:=0 to Bitmap^.Height-1 do begin
      C:=Bitmap^.Data[J*Bitmap^.Width+I];
      if (C<>255) and (X+I>=CurClipRect.X) and (Y+J>=CurClipRect.Y) and
        (X+I<CurClipRect.X+CurClipRect.W) and (Y+J<CurClipRect.Y+CurClipRect.H) then
          mem[VideoSegment:(Y+J)*320+X+I]:=C;
    end;
end;

procedure ShowMouse;
begin
  Regs.AX:=1;
  Intr($33,Regs);
end;

procedure HideMouse;
begin
  Regs.AX:=2;
  Intr($33,Regs);
end;

procedure GetMouseState(var Buttons,X,Y:word);
begin
  Regs.AX:=3;
  Intr($33,Regs);
  Buttons:=Regs.BX;
  X:=Regs.CX div 2;
  Y:=Regs.DX;
end;

function GetKeyEvent:word;
var
  Key:word;
begin
  GetKeyEvent:=ke_Nothing;
  if not KeyPressed then
    Exit;

  Key:=Word(ReadKey);
  if Key=0 then
    GetKeyEvent:=Word(ReadKey) shl 8
  else
    GetKeyEvent:=Key;
end;

function GetMouseEvent:word;
var
  B,X,Y:word;
begin
  GetMouseEvent:=me_Nothing;

  GetMouseState(B,X,Y);
  if (X<>CurX) or (Y<>CurY) then begin
    CurX:=X;
    CurY:=Y;
    GetMouseEvent:=me_Move;
    Exit;
  end;

  if B<>CurButtons then begin
    if (B and mb_Left)<>(CurButtons and mb_Left) then begin
      if B and mb_Left=mb_Left then
        GetMouseEvent:=me_LeftDown
      else
        GetMouseEvent:=me_LeftUp;
    end else if (B and mb_Right)<>(CurButtons and mb_Right) then begin
      if B and mb_Right=mb_Right then
        GetMouseEvent:=me_RightDown
      else
        GetMouseEvent:=me_RightUp;
    end else
      if B and mb_Middle=mb_Middle then
        GetMouseEvent:=me_MiddleDown
      else
        GetMouseEvent:=me_MiddleUp;
    CurButtons:=B;
    Exit;
  end;
end;

procedure ClearEvents;
begin
  repeat
  until GetMouseEvent=me_Nothing;
  repeat
  until GetKeyEvent=ke_Nothing;
end;

begin
  VideoSegment:=$A000;
  SetRect(CurClipRect,0,0,320,200);
  CurPalette:=nil;
  CurFont:=nil;
  CurButtons:=0;
  CurX:=0;
  CurY:=0;
end.